#!/usr/bin/env python3
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable-msg=protected-access

from ArnetModel import (
   IpGenericAddress,
)
from CliModel import (
   Bool,
   Dict,
   Enum,
   Float,
   Int,
   List,
   Model,
   Str,
   Submodel,
)
from FlowTrackerCliUtil import (
   ftrTypeHardware,
   ftrTypeShowStr,
   addressStr,
   protocolStr,
   timeStr,
)
from IntfModels import Interface
from TypeFuture import TacLazyType
import Tac
import TacSigint
import TableOutput

IpProtoType = TacLazyType( 'Arnet::IpProtocolNumber' )
FtConsts = TacLazyType( 'FlowTracking::Constants' )

def pindent( indent, *args ):
   print( ' ' * ( 2 * indent - 1 ), *args )

def templateIdNum( staticTemplateIdEnum ):
   return FtConsts.reservedTemplateId( staticTemplateIdEnum )

class FlowKeyModel( Model ):
   vrfName = Str( help="VRF name" )
   srcAddr = IpGenericAddress( help="Source IP address" )
   dstAddr = IpGenericAddress( help="Destination IP address" )
   ipProtocol = Enum( help="IP protocol", values=IpProtoType.attributes,
                      optional=True )
   ipProtocolNumber = Int( help="IP protocol number" )
   srcPort = Int( help="Source port" )
   dstPort = Int( help="Destination port" )
   _combinedKey = Int( help="Combined flow key" )
   _direction = Enum( values=( 'fcRight', 'fcLeft' ),
                     help="Combined flow key direction" )

class FlowDetailModel( Model ):
   tcpFlags = Str( help="Flow TCP flags" )
   lastPktTime = Float( help="Last packet received time" )
   ingressIntf = Interface( optional=True, help="Flow ingress interface" )
   # egressIntf may not be an IntfId, i.e. multicast/discard/unknown
   egressIntf = Str( help="Flow egress interface" )
   appCategory = Str( optional=True,
                      help="Name of category of the application" )
   dpsPathName = Str( optional=True,
                      help="Name of a DPS path" )
   avtName = Str( optional=True,
                  help="Name of AVT profile" )
   dpsPathLatency = Float( optional=True, help="Latency of DPS path in msecs" )
   dpsPathLossRate = Float( optional=True, help="Loss in DPS path in percentage" )
   dpsSourceDevice = Bool( optional=True,
                           help="Device is the source device selecting "
                                 "the DPS path to destination" )

class FlowModel( Model ):
   key = Submodel( valueType=FlowKeyModel, help="Flow key" )
   # bytes/pkts for forward dir. kept as is without prefix to be backward compatible
   bytesReceived = Int( help="Number of bytes received in forward direction" )
   pktsReceived = Int( help="Number of packets received in forward direction" )
   revBytesReceived = Int( optional=True, help="Number of bytes received " +
                          "in reverse direction" )
   revPktsReceived = Int( optional=True, help="Number of packets received " +
                         "in reverse direction" )
   startTime = Float( help="Flow start time" )
   application = Str( optional=True, help="Name of application" )
   appService = Str( optional=True,
                     help="Name of service of the application" )
   complete = Int( optional=True, help="The number of fully established flows. " +
                  "Meaning handshake complete for TCP, or bidirectional flow seen " +
                  "for other protocols" )
   halfOpen = Int( optional=True, help="The number of half-open flows. " +
                  "Meaning handshake incomplete for TCP, or only one direction " +
                  "of flow seen for other protocols" )
   flowDetail = Submodel( valueType=FlowDetailModel, optional=True,
                          help="Flow detailed information" )

   _exportedTime = Float( help="Time last exported to collector" )
   _exportedBytes = Int( help="Number of bytes exported to collector" )
   _exportedPkts = Int( help="Number of packet exported to collector" )

class GroupModel( Model ):
   flows = List( valueType=FlowModel, help="List of flows" )

class TrackerModel( Model ):
   groups = Dict( keyType=str, valueType=GroupModel,
                  help="A mapping bewteen group name and group" )
   numFlows = Int( help="Total number of flows" )

class TrackingModel( Model ):
   _detail = Bool( 'Print info in details' )
   trackers = Dict( keyType=str, valueType=TrackerModel,
                    help="A mapping between tracker name and tracker" )
   running = Bool( help="Flow tracking is active" )
   softwareFlowTable = Bool( help="Software flow table is available" )

   def renderDetail( self ):
      for trackerName, tracker in sorted( self.trackers.items() ):
         print( f"Tracker: {trackerName}, Flows: {tracker.numFlows}" )
         for groupName, group in sorted( tracker.groups.items() ):
            if not group.flows:
               continue
            print( f"  Group: {groupName}, Flows: {len( group.flows )}" )
            for flow in group.flows:
               key = flow.key
               detail = flow.flowDetail
               tab = " " * 4
               print( f"{tab}Flow: "
                      f"{protocolStr( key.ipProtocol, key.ipProtocolNumber )} "
                      f"{addressStr( key.srcAddr, key.srcPort )} - "
                      f"{addressStr( key.dstAddr, key.dstPort )}, "
                      f"VRF: {key.vrfName}" )
               tab = " " * 6
               print( f"{tab}Combined Key: {str( key._combinedKey )} "
                      f"Direction: {str( key._direction )}" )
               print( f"{tab}Start time: {timeStr( flow.startTime )}, "
                      f"Last packet time: {timeStr( detail.lastPktTime )}" )
               print( f"{tab}Packets: {flow.pktsReceived}, Bytes: "
                      f"{flow.bytesReceived}, TCP Flags: {detail.tcpFlags}" )
               ingressIntf = "unknown" if detail.ingressIntf is None\
                     else detail.ingressIntf.stringValue
               print( f"{tab}Ingress Interface: {ingressIntf}, "
                      f"Egress Interface: {detail.egressIntf}" )
               print( f"{tab}Application: {flow.application}, "
                      f"Application service: {flow.appService}, "
                      f"Application category: {detail.appCategory}" )
               print( f"{tab}AVT: {detail.avtName}, "
                      f"DPS path: {detail.dpsPathName}" )
               print( f"{tab}DPS path latency: {detail.dpsPathLatency:.3f}ms, "
                      f"DPS path loss rate: {detail.dpsPathLossRate:.4f}%" )
               print( f"{tab}DPS source device: {detail.dpsSourceDevice}" )
               print( f"{tab}Last exported time: {timeStr( flow._exportedTime )}, "
                      f"Packets: {flow._exportedPkts}, "
                      f"Bytes: {flow._exportedBytes}" )
               TacSigint.check()

   def renderSummary( self ):
      headings = None
      headings = ( "VRF", "Source", "Destination", "Protocol", "Application",
                   "Application Service", "Start Time", "Pkts", "Bytes" )
      formatLeft = TableOutput.Format( justify="left" )
      formatLeft.noPadLeftIs( True )
      formatCenter = TableOutput.Format( justify="center" )
      formatRight = TableOutput.Format( justify="right" )
      for trackerName, tracker in sorted( self.trackers.items() ):
         print( f"Tracker: {trackerName}, Flows: {tracker.numFlows}" )
         for groupName, group in sorted( tracker.groups.items() ):
            if not group.flows:
               continue
            table = TableOutput.createTable( headings )
            table.formatColumns( formatLeft, formatLeft, formatLeft,
                                 formatCenter, formatLeft, formatLeft,
                                 formatRight, formatRight, formatRight )
            print( f"Group: {groupName}, Flows: {len( group.flows )}" )
            for flow in group.flows:
               key = flow.key
               table.newRow( key.vrfName,
                             addressStr( key.srcAddr, key.srcPort ),
                             addressStr( key.dstAddr, key.dstPort ),
                             protocolStr( key.ipProtocol, key.ipProtocolNumber ),
                             flow.application,
                             flow.appService,
                             timeStr( flow.startTime ),
                             flow.pktsReceived,
                             flow.bytesReceived )
               TacSigint.check()
            print( table.output() )

   def render( self ):
      ftrStr = ftrTypeShowStr[ ftrTypeHardware ]
      if not self.running:
         print( ftrStr, 'flow tracking is not active' )
         return

      if self._detail:
         self.renderDetail()
      else:
         self.renderSummary()

class PlatformCountersModel( Model ):
   """ Represents sfeFtShowCounters """
   grpcErr = Int( help="Number of errors in GRPC processing",
                  default=0 )
   flowAdd = Int( help="Number flow added", default=0 )
   flowDel = Int( help="Number flow deleted", default=0 )

   exportInfoReq = Int( help="Number of export GRPC "
                        "request message sent", default=0 )
   exportInfoRsp = Int( help="Number of export GRPC response "
                        "message received", default=0 )
   exportInfoInRspAdd = Int( help="Number of add export message received",
                             default=0 )
   exportInfoInRspRemove = Int( help="Number of remove export message received",
                                default=0 )
   exportInfoInRspRemoveNoFlow = Int( help="Number of remove export message "
                                      "received for invalid flow", default=0 )
   exportInfoRspInvalidFt = Int( help="Number of export received "
                                 "for invalid flow traker", default=0 )
   exportInfoRspInvalidIntf = Int( help="Number of export received "
                                   "for invalid flow tracker interface", default=0 )
   exportInfoRspNoModule = Int( help="Number of export received "
                                "for invalid flow tracker module", default=0 )
   exportInfoRspInvalidFlow = Int( help="Number of export received "
                                   "for stale flows", default=0 )

   flowUdReq = Int( help="Number of user data GRPC request sent", default=0 )
   flowUdRsp = Int( help="Number of user data GRPC response received",
                    default=0 )
   flowKeyUdInReq = Int( help="Number of user data requested", default=0 )
   flowKeyUdInRsp = Int( help="Number of user data response received",
                         default=0 )
   flowKeyUdInRspNotFound = Int( help="Number of user data response not found",
                                 default=0 )
   flowKeyUdInRspNoFlow = Int( help="Number of user data response for "
                               "invalid flow", default=0 )
   flowKeyUdInRspNoPktSeen = Int( help="Number of user data response "
                                  "with no update", default=0 )
   flowKeyUdInRspInvalidFt = Int( help="Number of user data response "
                                 "for invalid flow tracker", default=0 )
   flowKeyUdInRspInvalidIntf = Int( help="Number of user data response "
                                    "for invalid flow tracker interface", default=0 )
   flowKeyUdInRspNoModule = Int( help="Number of user data response "
                                "for invalid flow tracker module", default=0 )
   flowKeyUdInRspInvalidFlow = Int( help="Number of user data response "
                                         "for stale flows", default=0 )

   exportQueueWriteUnavailable = Int(
      help="Number of times export queue write unavailable",
      default=0 )
   exportQueuePushAttempt = Int( help="Number of export record enqueue attempted",
                                 default=0 )
   exportQueuePushFail = Int( help="Number of export record enqueue failed",
                              default=0 )

   getflowUd = Int( help="Number of BESS get user data requests", default=0 )
   getExportInfo = Int( help="Number of BESS get export info requests", default=0 )

   fcudPoolSize = Int( help="Size of BESS flow-cache user data pool", default=0 )
   fcudPoolGet = Int( help="Number of BESS flow-cache user data pool allocation",
                      default=0 )
   fcudPoolPut = Int( help="Number of BESS flow-cache user data pool "
                      "deallocation", default=0 )
   fcudPoolFull = Int( help="Number of BESS flow-cache user data pool allocation "
                       "failure", default=0 )
   fcudPoolAvailable = Int( help="Number of BESS flow-cache user data available",
                            default=0 )
   fcudPoolInuse = Int( help="Number of BESS flow-cache user data in use",
                        default=0 )

   exportRingSize = Int( help="Size of BESS export ring", default=0 )
   exportRingInuse = Int( help="Number of export ring data in use", default=0 )
   exportRingAddEnqFail = Int( help="Number of BESS export ring add enqueue "
                               "failure", default=0 )
   exportRingDelEnqFail = Int( help="Number of BESS export ring delete enqueue "
                               "failure", default=0 )

   exportfd = Int( help="Export ring BESS file descriptor", default=0 )
   exportfdReq = Int( help="Number of export ring read indication", default=0 )
   exportfdReqFail = Int( help="Number of export ring read indication failure",
                          default=0 )

   def render( self ):
      print( f"Number of flows added: {self.flowAdd}" )
      print( f"Number of flows deleted: {self.flowDel}" )

      print( f"Number of errors in GRPC processing: {self.grpcErr}" )
      print( f"Number of export GRPC request message sent: {self.exportInfoReq}" )
      print( "Number of export GRPC response message received: "
              f"{self.exportInfoRsp}" )
      print( f"Number of add export message received: {self.exportInfoInRspAdd}" )
      print( "Number of remove export message received: "
             f"{self.exportInfoInRspRemove}" )
      print( "Number of remove export message received "
             f"for non existing flow: {self.exportInfoInRspRemoveNoFlow}" )
      print( "Number of export received for invalid flow tracker: "
             f"{self.exportInfoRspInvalidFt}" )
      print( "Number of export received for "
             f"invalid flow tracker interface : {self.exportInfoRspInvalidIntf}" )
      print( "Number of export received for invalid flow tracker "
             f"module: {self.exportInfoRspNoModule}" )
      print( "Number of export received for stale flows: "
             f"{self.exportInfoRspInvalidFlow}" )

      print( f"Number of user data GRPC request sent: {self.flowUdReq}" )
      print( f"Number of user data GRPC response received: {self.flowUdRsp}" )
      print( f"Number of user data requested: {self.flowKeyUdInReq}" )
      print( f"Number of user data response received: {self.flowKeyUdInRsp}" )
      print( "Number of user data response not found: "
             f"{self.flowKeyUdInRspNotFound}" )
      print( "Number of user data response for invalid flow: "
             f"{self.flowKeyUdInRspNoFlow}" )
      print( "Number of user data response with no update: "
             f"{self.flowKeyUdInRspNoPktSeen}" )
      print( "Number of user data response for invalid flow tracker: "
             f"{self.flowKeyUdInRspInvalidFt}" )
      print( "Number of user data response for invalid "
             f"flow tracker interface: {self.flowKeyUdInRspInvalidIntf}" )
      print( "Number of user data response for invalid "
             f"flow tracker module: {self.flowKeyUdInRspNoModule}" )
      print( "Number of user data response for stale flows: "
             f"{self.flowKeyUdInRspInvalidFlow}" )

      print( "Number of times export queue write unavailable: "
             f"{self.exportQueueWriteUnavailable}" )
      print( "Number of export record enqueue attempts: "
             f"{self.exportQueuePushAttempt}" )
      print( "Number of export record enqueue failed: "
             f"{self.exportQueuePushFail}" )

      print( f"Number of BESS get user data requests: {self.getflowUd}" )
      print( f"Number of BESS get export info requests: {self.getExportInfo}" )

      print( f"Size of BESS flow-cache user data pool: {self.fcudPoolSize}" )
      print( "Number of BESS flow-cache user data pool allocation: "
             f"{self.fcudPoolGet}" )
      print( "Number of BESS flow-cache user data pool deallocation: "
             f"{self.fcudPoolPut}" )
      print( "Number of BESS flow-cache user data pool allocation failure: "
             f"{self.fcudPoolFull}" )
      print( "Number of BESS flow-cache user data available: "
             f"{self.fcudPoolAvailable}" )
      print( "Number of BESS flow-cache user data in use: "
             f"{self.fcudPoolInuse}" )

      print( f"Size of BESS export ring: {self.exportRingSize}" )
      print( f"Number of BESS export ring data in use: {self.exportRingInuse}" )
      print( "Number of BESS export ring add enqueue failure: "
             f"{self.exportRingAddEnqFail}" )
      print( "Number of BESS export ring delete enqueue failure: "
             f"{self.exportRingDelEnqFail}" )

      print( f"Export ring BESS file descriptor: {self.exportfd}" )
      print( "Number of BESS export ring read indication: "
             f"{self.exportfdReq}" )
      print( "Number of BESS export ring read indication failure: "
             f"{self.exportfdReqFail}" )
