# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import BasicCliModes
import CliCommand
import CliMatcher
import CliParser
import CliGlobal
import ConfigMount
from TypeFuture import TacLazyType
from CliToken.Router import routerMatcherForConfig
from CliPlugin.SfeInternetExitCliLib import (
        IeExitGroupConfigContext,
        IePolicyConfigContext,
)
from CliPlugin.SfeServiceInsertionCli import (
        connectionConfigNameMatcher,
        serviceGroupConfigNameMatcher,
)
from CliMode.SfeInternetExit import (
        RouterInternetExitConfigMode,
        RouterIeExitGroupConfigMode,
        RouterIePolicyConfigMode,
)
from Toggles.WanTECommonToggleLib import (
      toggleAvtRemoteInternetExitEnabled,
)

ExitType = TacLazyType( 'SfeInternetExit::ExitType' )
ExitKey = TacLazyType( 'SfeInternetExit::ExitKey' )
PolicyConfig = TacLazyType( 'SfeInternetExit::PolicyConfig' )
ExitGroupConfig = TacLazyType( 'SfeInternetExit::ExitGroupConfig' )

gv = CliGlobal.CliGlobal( dict( ieCliConfig=None ) )

def getPolicyNames( mode ):
   return [ k for k in gv.ieCliConfig.policyConfig
                    if k != PolicyConfig.defaultName ]

def getExitGroupNames( mode ):
   return list( gv.ieCliConfig.exitGroupConfig )

def getExitGroupKwMatcher( helpdesc='Configure Internet exit group' ):
   return CliMatcher.KeywordMatcher( 'exit-group',
             helpdesc=helpdesc, autoCompleteMinChars=5 )

# ===================================================================================
# (config)# router internet-exit
# ===================================================================================

ieMatcher = CliMatcher.KeywordMatcher( 'internet-exit',
        helpdesc='Configure Internet exit' )

class RouterInternetExitConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''router internet-exit'''
   noOrDefaultSyntax = syntax
   data = {
           'router' : routerMatcherForConfig,
           'internet-exit' : ieMatcher,
          }

   @staticmethod
   def _createIeDefaults( mode ):
      # creating default internet exit policy and its exit group
      policyContext = IePolicyConfigContext( gv.ieCliConfig,
                         mode, PolicyConfig.defaultName )
      policyContext.addPolicyConfig( PolicyConfig.defaultName )
      policyContext.addPolicyExitGroup( ExitGroupConfig.defaultName, '' )
      exitGroupContext = IeExitGroupConfigContext( gv.ieCliConfig,
                            mode, ExitGroupConfig.defaultName )
      exitGroupContext.addExitGroupConfig( ExitGroupConfig.defaultName )
      exitGroupContext.addExitOption( ExitKey( ExitType.fibDefault, '' ) )

   @staticmethod
   def handler( mode, args ):
      gv.ieCliConfig.enabled = True
      childMode = mode.childMode( RouterInternetExitConfigMode )
      mode.session_.gotoChildMode( childMode )
      RouterInternetExitConfigCmd._createIeDefaults( mode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      BasicCliModes.removeCommentWithKey( 'internet-exit' )
      for grpName in gv.ieCliConfig.exitGroupConfig:
         BasicCliModes.removeCommentWithKey( f'exit-group-{grpName}' )
      for polName in gv.ieCliConfig.policyConfig:
         BasicCliModes.removeCommentWithKey( f'policy-{polName}' )
      gv.ieCliConfig.exitGroupConfig.clear()
      gv.ieCliConfig.policyConfig.clear()
      gv.ieCliConfig.enabled = False

# ===================================================================================
# (config-internet-exit)# exit-group <group-name>
# (config-exit-group-name)# local connection <name> | remote service group <name> |
#                           fib-default
# ===================================================================================

exitGroupConfigNameMatcher = CliMatcher.DynamicNameMatcher(
        getExitGroupNames,
        helpdesc='Name of the exit group',
        helpname='WORD',
        priority=CliParser.PRIO_LOW )
keywordHelpDescMapping = {
    'fib-default' : 'Fib default exit for default routes',
}
exitOptionWithoutNameMatcher = CliMatcher.EnumMatcher( keywordHelpDescMapping )

class IeExitGroupConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''exit-group EXIT_GROUP_NAME'''
   noOrDefaultSyntax = syntax
   data = {
           'exit-group' : getExitGroupKwMatcher(),
           'EXIT_GROUP_NAME' : exitGroupConfigNameMatcher,
          }

   @staticmethod
   def handler( mode, args ):
      exitGroupName = args.get( 'EXIT_GROUP_NAME' )
      context = IeExitGroupConfigContext( gv.ieCliConfig, mode, exitGroupName )
      context.addExitGroupConfig( exitGroupName )
      childMode = mode.childMode( RouterIeExitGroupConfigMode, context=context )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      exitGroupName = args.get( 'EXIT_GROUP_NAME' )
      # Not allowing to delete default exit-group
      if exitGroupName == ExitGroupConfig.defaultName:
         if args.get( '__no__' ):
            mode.addError( f'Cannot delete or modify {exitGroupName}' )
         return
      BasicCliModes.removeCommentWithKey( f'exit-group-{exitGroupName}' )
      context = IeExitGroupConfigContext( gv.ieCliConfig, mode, exitGroupName )
      context.delExitGroupConfig( exitGroupName )

class IeExitOptionBase:
   @staticmethod
   def _exitKeyGenerator( args ):
      # generate an exitKey based on exitType and exitName
      exitTypeMapper = {
               'local connection' : ExitType.localExit,
               'remote service group' : ExitType.remoteExit,
               'fib-default' : ExitType.fibDefault,
              }
      return ExitKey( exitTypeMapper.get( args.get( 'EXIT_TYPE' ) ),
                      args.get( 'EXIT_NAME', '' ) )

   @staticmethod
   def handler( mode, args ):
      # Throw error if user setting other exit-options to default exitgroup
      if mode.context.exitGroupName() == ExitGroupConfig.defaultName:
         if args.get( 'EXIT_TYPE' ) != 'fib-default':
            mode.addError( f'Cannot delete or modify {ExitGroupConfig.defaultName}' )
         return
      mode.context.addExitOption( IeExitOptionBase._exitKeyGenerator( args ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      # Not allowing to delete default exit-group options
      if mode.context.exitGroupName() == ExitGroupConfig.defaultName:
         mode.addError( f'Cannot delete or modify {ExitGroupConfig.defaultName}' )
         return
      mode.context.delExitOption( IeExitOptionBase._exitKeyGenerator( args ) )

   @staticmethod
   def modifyHandler( exitTypeStr, noOrDefault=False ):
      def innerHandler( mode, args ):
         args[ 'EXIT_TYPE' ] = exitTypeStr
         handler = IeExitOptionBase.noOrDefaultHandler if noOrDefault else \
                      IeExitOptionBase.handler
         return handler( mode, args )
      return innerHandler

class IeExitOptionLocalExitConfigCli( CliCommand.CliCommandClass,
                                      IeExitOptionBase ):
   syntax = '''local connection EXIT_NAME'''
   noOrDefaultSyntax = syntax
   data = {
           'local' : 'Local exit',
           'connection' : 'Connection of the local exit',
           'EXIT_NAME' : connectionConfigNameMatcher,
          }
   handler = IeExitOptionBase.modifyHandler( 'local connection' )
   noOrDefaultHandler = IeExitOptionBase.modifyHandler( 'local connection', True )

class IeExitOptionRemoteExitConfigCli( CliCommand.CliCommandClass,
                                       IeExitOptionBase ):
   syntax = '''remote service group EXIT_NAME'''
   noOrDefaultSyntax = syntax
   data = {
           'remote' : 'Remote exit',
           'service' : 'Network service',
           'group' : 'Network service group',
           'EXIT_NAME' : serviceGroupConfigNameMatcher,
          }
   handler = IeExitOptionBase.modifyHandler( 'remote service group' )
   noOrDefaultHandler = IeExitOptionBase.modifyHandler(
                           'remote service group', True )

class IeExitOptionsWithoutNameConfigCli( CliCommand.CliCommandClass,
                                         IeExitOptionBase ):
   syntax = '''EXIT_TYPE'''
   noOrDefaultSyntax = syntax
   data = {
           'EXIT_TYPE' : exitOptionWithoutNameMatcher,
          }

# ===================================================================================
# (config-internet-exit)# policy <policy-name>
# (config-policy-name)# exit-group <group-name> [<before | after> exit-group
#                                                                      <group-name>]
# ===================================================================================

policyConfigNameMatcher = CliMatcher.DynamicNameMatcher(
        getPolicyNames,
        helpdesc='Name of the policy',
        helpname='WORD',
        priority=CliParser.PRIO_LOW )
policyExitGroupNameMatcher = CliMatcher.DynamicNameMatcher(
        getExitGroupNames,
        helpdesc='Name of the exit group',
        helpname='WORD',
        priority=CliParser.PRIO_LOW )
policyExitGroupOrderMatcher = CliMatcher.EnumMatcher( {
    'after' : 'Insert the previous exit group '
             'AFTER the following exit group',
    'before' : 'Insert the previous exit group '
              'BEFORE the following exit group',
    } )

class IePolicyConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''policy POLICY_NAME'''
   noOrDefaultSyntax = syntax
   data = {
           'policy' : 'Configure Internet exit policy',
           'POLICY_NAME' : policyConfigNameMatcher,
          }

   @staticmethod
   def handler( mode, args ):
      policyName = args.get( 'POLICY_NAME' )
      context = IePolicyConfigContext( gv.ieCliConfig, mode, policyName )
      context.addPolicyConfig( policyName )
      childMode = mode.childMode( RouterIePolicyConfigMode, context=context )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      policyName = args.get( 'POLICY_NAME' )
      # Not allowing user to delete default policy
      if policyName == PolicyConfig.defaultName:
         if args.get( '__no__' ):
            mode.addError( f'Cannot delete or modify {policyName}' )
         return
      BasicCliModes.removeCommentWithKey( f'policy-{policyName}' )
      context = IePolicyConfigContext( gv.ieCliConfig, mode, policyName )
      context.delPolicyConfig( policyName )

class IePolicyExitGroupCli( CliCommand.CliCommandClass ):
   syntax = '''exit-group GROUP [ORDER GROUP2]'''
   noOrDefaultSyntax = '''exit-group GROUP'''
   data = {
           'exit-group' : getExitGroupKwMatcher( helpdesc='Select the exit-group' ),
           'GROUP' : policyExitGroupNameMatcher,
           'ORDER' : policyExitGroupOrderMatcher,
           'GROUP2' : policyExitGroupNameMatcher,
          }

   @staticmethod
   def handler( mode, args ):
      name = args.get( 'GROUP' )
      order = args.get( 'ORDER' )
      name2 = args.get( 'GROUP2' )
      # Not allowing to change default policy exit-groups
      if mode.context.policyName() == PolicyConfig.defaultName:
         if name != ExitGroupConfig.defaultName:
            mode.addError( f'Cannot delete or modify {PolicyConfig.defaultName}' )
            return
      mode.context.addPolicyExitGroup( name, order, orderName=name2 )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      name = args.get( 'GROUP' )
      # Not allowing to delete default exit-group
      if ( mode.context.policyName() == PolicyConfig.defaultName and
          name == ExitGroupConfig.defaultName ):
         mode.addError( f'Cannot delete or modify {PolicyConfig.defaultName}' )
         return
      mode.context.delPolicyExitGroup( name )

# Commands that define router internet-exit general attributes (if any)
BasicCli.GlobalConfigMode.addCommandClass( RouterInternetExitConfigCmd )

# Commands that define router internet-exit exit-group mode clis
RouterInternetExitConfigMode.addCommandClass( IeExitGroupConfigCmd )
RouterIeExitGroupConfigMode.addCommandClass( IeExitOptionLocalExitConfigCli )
RouterIeExitGroupConfigMode.addCommandClass( IeExitOptionsWithoutNameConfigCli )
if toggleAvtRemoteInternetExitEnabled():
   RouterIeExitGroupConfigMode.addCommandClass( IeExitOptionRemoteExitConfigCli )

# Commands that define router internet-exit policy mode clis
RouterInternetExitConfigMode.addCommandClass( IePolicyConfigCmd )
RouterIePolicyConfigMode.addCommandClass( IePolicyExitGroupCli )

# ===================================================================================
# Register the plugin
# ===================================================================================
def Plugin( entityManager ):
   gv.ieCliConfig = ConfigMount.mount( entityManager,
                                 'ie/cli/config',
                                 'SfeInternetExit::InternetExitCliConfig', 'w' )
