# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from TypeFuture import TacLazyType

ExitGroup = TacLazyType( 'SfeInternetExit::ExitGroupConfig' )

class IeExitGroupConfigContext:
   def __init__( self, config, mode, name ):
      self.config = config
      self.mode = mode
      self._exitGroupName = name
      self._exitGroup = None

   def exitGroupName( self ):
      return self._exitGroupName

   def exitGroupConfig( self ):
      return self._exitGroup

   def addExitGroupConfig( self, exitGroupName ):
      exitGroup = self.config.newExitGroupConfig( exitGroupName )
      self._exitGroup = exitGroup

   def delExitGroupConfig( self, exitGroupName ):
      del self.config.exitGroupConfig[ exitGroupName ]
      self._exitGroup = None

   def addExitOption( self, exitKey ):
      exitOption = self._exitGroup.exitOption.get( exitKey )
      if not exitOption:
         if len( self._exitGroup.exitOption ) >= ExitGroup.maxExitOption:
            self.mode.addError(
               f"Can't configure more than {ExitGroup.maxExitOption} exit-options "
               "in a group" )
            return
         self._exitGroup.exitOption.add( exitKey )

   def delExitOption( self, exitKey ):
      del self._exitGroup.exitOption[ exitKey ]

class IePolicyConfigContext:
   def __init__( self, config, mode, name ):
      self.config = config
      self.mode = mode
      self._policyName = name
      self._policy = None

   def policyName( self ):
      return self._policyName

   def policyConfig( self ):
      return self._policy

   def addPolicyConfig( self, policyName ):
      policy = self.config.newPolicyConfig( policyName )
      self._policy = policy

   def delPolicyConfig( self, policyName ):
      del self.config.policyConfig[ policyName ]
      self._policy = None

   def findPolicyExitGroup( self, exitGroupName ):
      ( before, current, after ) = ( 0.0, None, None )
      matched = False
      for ( seqNo, exitGroup ) in self._policy.exitGroup.items():
         if matched:
            after = seqNo
            break
         if exitGroupName == exitGroup.exitGroupName:
            current = seqNo
            matched = True
         else:
            before = seqNo
      return ( before, current, after )

   def advanceSeqNo( self ):
      SEQNO_STEP = 10.0
      seqNo = self._policy.lastSeqNo + SEQNO_STEP
      self._policy.lastSeqNo = seqNo
      return seqNo

   def addPolicyExitGroup( self, exitGroupName, order, orderName=None ):
      if orderName is not None:
         ( before, orderSeqNo, after ) = self.findPolicyExitGroup( orderName )
         if orderSeqNo is None:
            self.mode.addError( f'{orderName} does not exist.' )
            return
         # no need to do anything if both the exit-group names are the same
         if orderName == exitGroupName:
            return
         # if the exit-group already exists, delete it.
         self.delPolicyExitGroup( exitGroupName )
         if order == 'before':
            seqNo = ( before + orderSeqNo ) / 2
         else:
            seqNo = self.advanceSeqNo() if after is None else \
                                    ( ( orderSeqNo + after ) / 2 )
      else:
         seqNo = self.findPolicyExitGroup( exitGroupName )[ 1 ]
         if seqNo is not None:
            return
         seqNo = self.advanceSeqNo()
      # We found the seqNo, now insert.
      eg = self._policy.newExitGroup( seqNo )
      if eg is not None:
         eg.exitGroupName = exitGroupName

   def delPolicyExitGroup( self, exitGroupName ):
      ( before, seqNo, _ ) = self.findPolicyExitGroup( exitGroupName )
      if seqNo is not None:
         if seqNo == self._policy.lastSeqNo:
            self._policy.lastSeqNo = before
         del self._policy.exitGroup[ seqNo ]
