#!/usr/bin/env python3
# Copyright (c) 2017-2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
from CliMatcher import KeywordMatcher
from CliMatcher import IntegerMatcher
import AgentCommandRequest
from io import StringIO
import ast
import SfeAgent
from CliPlugin import VrfCli
from CliPlugin.IpAddrMatcher import IpAddrMatcher, IpPrefixMatcher
from CliPlugin.IpAddrMatcher import ipPrefixExpr, PREFIX_OVERLAP_AUTOZERO
from CliPlugin.Ip6AddrMatcher import ip6PrefixExpr, Ip6AddrMatcher, Ip6PrefixMatcher
from CliPlugin.SfeCliLib import nodeSfe, notHwPlatform, notRubyDfw
from CliPlugin.SfeL3UnicastShowCliModel import L3AgentAsyncIpRoutes
from CliPlugin.SfeL3UnicastShowCliModel import L3AgentAsyncNextHops
from CliPlugin.SfeL3UnicastShowCliModel import L3AgentIpRoutes
from CliPlugin.SfeL3UnicastShowCliModel import L3AgentNextHops
from CliPlugin.SfeL3UnicastShowCliModel import L3AgentSummarySoftware
from CliPlugin.SfeL3UnicastShowCliModel import L3AgentSummaryUnicast
from CliPlugin.SfeL3UnicastShowCliModel import PeerGroupIdModel
from CliPlugin.SfeL3UnicastShowCliModel import IpRouteMcastModel
from Toggles.EosInitToggleLib import toggleSfeVrfScaleEnabled
from Toggles.EosInitToggleLib import toggleSfeVrfScalev2Enabled

import ShowCommand
import CliCommand
import CliModel
from CliToken.Platform import platformMatcherForShow

matcherDiff = CliCommand.guardedKeyword( 'diff', helpdesc='Display cModule diff',
                                         guard=notHwPlatform )
matcherL3 = CliCommand.guardedKeyword( 'l3',
                                       helpdesc='Display routing information',
                                       guard=notRubyDfw )
matcherIP = CliCommand.guardedKeyword( 'ip',
                                       helpdesc='IPV4 related commands',
                                       guard=notRubyDfw )
matcherIPv6 = CliCommand.guardedKeyword( 'ipv6',
                                         helpdesc='IPV6 related commands',
                                         guard=notRubyDfw )
matcherSummary = KeywordMatcher( 'summary', 'Display L3 software summary' )
matcherFecId = IntegerMatcher( 1, 0xFFFFFFFFFFFFFFFF, helpdesc='FEC identifier' )
matcherFib = KeywordMatcher( 'fib', 'FIB related commands' )

sfeIpPrefixExpr = ipPrefixExpr(
      'Destination prefix', 'Destination prefix mask',
      'Destination address with prefix',
      partial=True, overlap=PREFIX_OVERLAP_AUTOZERO )

sfeIp6PrefixExpr = ip6PrefixExpr(
      'Destination prefix', 'Destination prefix mask',
      'Destination address with prefix',
      overlap=PREFIX_OVERLAP_AUTOZERO )

#---------------------------------------------------------------------------------
# show platform sfe ip route [ vrf <vrfName> [ <prefix> ] ] [ diff ]
#---------------------------------------------------------------------------------
def doShowIpRouteVrf( mode, args ):
   buff = StringIO()
   cmd = "SIPRD"
   AgentCommandRequest.runSocketCommand( mode.entityManager, SfeAgent.name(),
                                         "sfe", cmd, stringBuff=buff, timeout=50 )
   try:
      siprDict = ast.literal_eval( buff.getvalue() )
   except SyntaxError:
      mode.addError( buff.getvalue() )
      return None

   sipr = L3AgentIpRoutes()
   #
   # Need these to prevent CliModel warning
   #
   sipr.initTables()
   sipr.setAttrsFromDict( siprDict )
   sipr.diffTables()

   return sipr

class PlatformSfeIpRouteDiffCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform sfe ip route diff'
   data = {
      'platform' : platformMatcherForShow,
      'sfe' : nodeSfe,
      'ip' : matcherIP,
      'route' : 'Show hardware routing table',
      'diff' : matcherDiff,
   }

   handler = doShowIpRouteVrf
   cliModel = L3AgentIpRoutes
   privileged = True

BasicCli.addShowCommandClass( PlatformSfeIpRouteDiffCmd )

class PlatformSfeIpRouteCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show platform sfe ip route [ VRF [ PREFIX ] ] '
              ' [ ROUTE | ( prefix PREFIXONLY ) ]' )
   data = {
      'platform' : platformMatcherForShow,
      'sfe' : nodeSfe,
      'ip' : matcherIP,
      'route' : 'Show hardware routing table',
      'VRF' : VrfCli.VrfExprFactory( helpdesc='Show VRF routing table' ),
      'PREFIX' : sfeIpPrefixExpr,
      'ROUTE' : IpAddrMatcher( helpdesc='Show route using IPV4 address' ),
      'prefix' : 'Show route using IPV4 specific prefix',
      'PREFIXONLY' : IpPrefixMatcher(
         helpdesc='Match this prefix',
         overlap=PREFIX_OVERLAP_AUTOZERO ),
   }

   cliModel = L3AgentAsyncIpRoutes
   privileged = True

   @staticmethod
   def handler( mode, args ):
      outputFormat = mode.session_.outputFormat_
      cmd = "SIPRN+"
      errors = [ 'l3AgentSm is not initialized' ]

      if 'VRF' in args:
         cmd += "VRF+" + args[ 'VRF' ] + "+"

      if 'PREFIX' in args:
         cmd += "PREFIX+" + args[ 'PREFIX' ] + "+"

      if 'ROUTE' in args:
         cmd += "ROUTE+" + args[ 'ROUTE' ] + "+"
      elif 'PREFIXONLY' in args:
         cmd += "PREFIXONLY+" + args[ 'PREFIXONLY' ] + "+"

      try:
         AgentCommandRequest.runSocketCommand( mode.entityManager, SfeAgent.name(),
                                               "SfeCliWithFormat", cmd,
                                                outputFormat=outputFormat,
                                                throwException=True,
                                                errors=errors,
                                                asyncCommand=True )
         return CliModel.cliPrinted( L3AgentAsyncIpRoutes )
      except AgentCommandRequest.RunSocketCommandException as e:
         mode.addError( str( e ) )
         return None

BasicCli.addShowCommandClass( PlatformSfeIpRouteCmd )

#---------------------------------------------------------------------------------
# show platform sfe ipv6 route [ vrf <vrfName> [ <prefix> ] ]
#---------------------------------------------------------------------------------
class PlatformSfeIpv6RouteCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show platform sfe ipv6 route [ VRF [ PREFIX ] ] '
              '[ ROUTE | ( prefix PREFIXONLY ) ]' )
   data = {
      'platform' : platformMatcherForShow,
      'sfe' : nodeSfe,
      'ipv6' : matcherIPv6,
      'route' : 'Show hardware routing table',
      'VRF' : VrfCli.VrfExprFactory( helpdesc='Show VRF routing table' ),
      'PREFIX' : sfeIp6PrefixExpr,
      'ROUTE' : Ip6AddrMatcher( helpdesc='Show route using IPV6 address' ),
      'prefix' : 'Show route using IPV6 specific prefix',
      'PREFIXONLY' : Ip6PrefixMatcher(
         helpdesc='Match this prefix',
         overlap=PREFIX_OVERLAP_AUTOZERO ),
   }
   cliModel = L3AgentAsyncIpRoutes
   privileged = True

   @staticmethod
   def handler( mode, args ):
      outputFormat = mode.session_.outputFormat_
      cmd = "SIP6RN+"
      errors = [ 'l3AgentSm is not initialized' ]

      if 'VRF' in args:
         cmd += "VRF+" + args[ 'VRF' ] + "+"

      if 'PREFIX' in args:
         cmd += "PREFIX+" + str( args[ 'PREFIX' ] ) + "+"

      if 'ROUTE' in args:
         cmd += "ROUTE+" + str( args[ 'ROUTE' ] ) + "+"
      elif 'PREFIXONLY' in args:
         cmd += "PREFIXONLY+" + str( args[ 'PREFIXONLY' ] ) + "+"

      try:
         AgentCommandRequest.runSocketCommand( mode.entityManager, SfeAgent.name(),
                                               "SfeCliWithFormat", cmd,
                                                outputFormat=outputFormat,
                                                throwException=True,
                                                errors=errors,
                                                asyncCommand=True )
         return CliModel.cliPrinted( L3AgentAsyncIpRoutes )
      except AgentCommandRequest.RunSocketCommandException as e:
         mode.addError( str( e ) )
         return None

BasicCli.addShowCommandClass( PlatformSfeIpv6RouteCmd )

# display fibIpv4 route for multicast traffic
class PlatformSfeIpRouteMcastCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show platform sfe ip route multicast' )
   data = {
      'platform' : platformMatcherForShow,
      'sfe' : nodeSfe,
      'ip' : matcherIP,
      'route' : 'Show hardware routing table',
      'multicast' : 'Show routes used for Multicast'
   }
   cliModel = IpRouteMcastModel
   privileged = True

   @staticmethod
   def handler( mode, args ):
      outputFormat = mode.session_.outputFormat_
      cmd = "MCASTIPRN"
      errors = [ 'l3AgentSm is not initialized' ]
      try:
         AgentCommandRequest.runSocketCommand( mode.entityManager, SfeAgent.name(),
                                               "SfeCliWithFormat", cmd,
                                               outputFormat=outputFormat,
                                               throwException=True,
                                               errors=errors,
                                               asyncCommand=True )
         return CliModel.cliPrinted( IpRouteMcastModel )
      except AgentCommandRequest.RunSocketCommandException as e:
         mode.addError( str( e ) )
         return None

BasicCli.addShowCommandClass( PlatformSfeIpRouteMcastCmd )

#---------------------------------------------------------------------------------
# show platform sfe l3 unicast summary
#---------------------------------------------------------------------------------
class PlatformSfeL3UnicastSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform sfe l3 unicast summary'
   data = {
      'platform' : platformMatcherForShow,
      'sfe' : nodeSfe,
      'l3' : matcherL3,
      'unicast' : 'Display unicast routing state',
      'summary' : matcherSummary,
   }

   cliModel = L3AgentSummaryUnicast
   privileged = True

   @staticmethod
   def handler( mode, args ):
      outputFormat = mode.session_.outputFormat_
      errors = [ 'SfeAgent is not initialized' ]
      try:
         AgentCommandRequest.runSocketCommand( mode.entityManager,
                                               SfeAgent.name(),
                                               "SfeCliWithFormat",
                                               "SUMM",
                                               stringBuff=None,
                                               forceBuffering=True,
                                               outputFormat=outputFormat,
                                               throwException=True,
                                               errors=errors )
         return CliModel.cliPrinted( L3AgentSummaryUnicast )
      except AgentCommandRequest.RunSocketCommandException as e:
         mode.addError( str( e ) )
         return None

BasicCli.addShowCommandClass( PlatformSfeL3UnicastSummaryCmd )

#---------------------------------------------------------------------------------
# show platform sfe l3 software summary
#---------------------------------------------------------------------------------
def doShowL3SummarySoftware( mode, args ):
   buff = StringIO()
   AgentCommandRequest.runSocketCommand( mode.entityManager, SfeAgent.name(),
                                         "sfe", "SSus", stringBuff=buff )
   try:
      summDict = ast.literal_eval( buff.getvalue() )
   except SyntaxError:
      mode.addError( buff.getvalue() )
      return None
   summ = L3AgentSummarySoftware()
   summ.setAttrsFromDict( summDict )
   return summ

class PlatformSfeL3SoftwareSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform sfe l3 software summary'
   data = {
      'platform' : platformMatcherForShow,
      'sfe' : nodeSfe,
      'l3' : matcherL3,
      'software' : 'Display L3 software state',
      'summary' : matcherSummary,
   }

   handler = doShowL3SummarySoftware
   cliModel = L3AgentSummarySoftware
   privileged = True

BasicCli.addShowCommandClass( PlatformSfeL3SoftwareSummaryCmd )

#---------------------------------------------------------------------------------
# show platform sfe l3 next-hops [ diff ]
#---------------------------------------------------------------------------------
class PlatformSfeL3NextHopsDiffCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform sfe l3 next-hops diff'
   data = {
      'platform' : platformMatcherForShow,
      'sfe' : nodeSfe,
      'l3' : matcherL3,
      'next-hops' : 'Show next-hop entries',
      'diff' : matcherDiff,
   }

   cliModel = L3AgentNextHops
   privileged = True

   @staticmethod
   def handler( mode, args ):
      cmd = "L3NHD"
      keep = False
      buff = StringIO()
      AgentCommandRequest.runSocketCommand( mode.entityManager, SfeAgent.name(),
                                         "sfe", cmd, stringBuff=buff, timeout=50,
                                         keepalive=keep )
      try:
         nhDict = ast.literal_eval( buff.getvalue() )
      except SyntaxError:
         mode.addError( buff.getvalue() )
         return None

      l3Nh = L3AgentNextHops()
      #
      # Need these to prevent CliModel warning
      #
      l3Nh.initTables()

      l3Nh.setAttrsFromDict( nhDict )
      l3Nh.diffTables()

      return l3Nh

BasicCli.addShowCommandClass( PlatformSfeL3NextHopsDiffCmd )

#---------------------------------------------------------------------------------
# show platform sfe l3 next-hops [ FECID ]
#---------------------------------------------------------------------------------

class PlatformSfeL3NextHopsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform sfe l3 next-hops [ FECID ]'
   data = {
      'platform' : platformMatcherForShow,
      'sfe' : nodeSfe,
      'l3' : matcherL3,
      'next-hops' : 'Show next-hop entries',
      'FECID' : matcherFecId,
   }

   cliModel = L3AgentAsyncNextHops
   privileged = True

   @staticmethod
   def handler( mode, args ):
      cmd = "L3NH"
      fecId = args.get( 'FECID' )
      errors = [ 'l3AgentSm is not initialized' ]
      if fecId:
         cmd += '#F' + str( fecId )
      outputFormat = mode.session_.outputFormat_
      try:
         AgentCommandRequest.runSocketCommand( mode.entityManager, SfeAgent.name(),
                                               "SfeCliWithFormat", cmd,
                                                outputFormat=outputFormat,
                                                throwException=True, errors=errors,
                                                asyncCommand=True )
         return CliModel.cliPrinted( L3AgentAsyncNextHops )
      except AgentCommandRequest.RunSocketCommandException as e:
         mode.addError( str( e ) )
         return None

BasicCli.addShowCommandClass( PlatformSfeL3NextHopsCmd )

# ---------------------------------------------------------------------------------
# show platform sfe l3 peer-group
# ---------------------------------------------------------------------------------

class PlatformSfePeerGroupIdCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform sfe l3 peer-group'
   data = {
      'platform' : platformMatcherForShow,
      'sfe' : nodeSfe,
      'l3' : matcherL3,
      'peer-group' : 'Display peer group to peer IP mapping',
   }

   cliModel = PeerGroupIdModel
   privileged = True

   @staticmethod
   def handler( mode, args ):
      errors = [ 'SfeAgent is not initialized' ]
      try:
         buff = StringIO()
         AgentCommandRequest.runSocketCommand( mode.entityManager,
                                               SfeAgent.name(),
                                               "sfe", "peergroup", stringBuff=buff,
                                               throwException=True,
                                               errors=errors )
         peerGroupIdDict = ast.literal_eval( buff.getvalue() )
      except AgentCommandRequest.RunSocketCommandException as e:
         mode.addError( str( e ) )
         return None
      model = PeerGroupIdModel()
      model.setAttrsFromDict( peerGroupIdDict )
      return model

BasicCli.addShowCommandClass( PlatformSfePeerGroupIdCmd )

# ---------------------------------------------------------------------------------
# show platform sfe ip fib state
# ---------------------------------------------------------------------------------

class PlatformSfeIpFibStateCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform sfe ip fib state'
   data = {
      'platform' : platformMatcherForShow,
      'sfe' : nodeSfe,
      'ip' : matcherIP,
      'fib' : matcherFib,
      'state' : 'Display FIB state',
   }

   # No cliModel for this one
   privileged = True

   @staticmethod
   def handler( mode, args ):
      AgentCommandRequest.runSocketCommand( mode.entityManager,
                                            SfeAgent.name(),
                                            'sfe',
                                            'fibstate' )

if toggleSfeVrfScaleEnabled() or toggleSfeVrfScalev2Enabled():
   BasicCli.addShowCommandClass( PlatformSfeIpFibStateCmd )
