#!/usr/bin/env python3
# Copyright (c) 2016-2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
from CliModel import Bool, Float, Int, Model, Enum, Dict, List, Submodel, Str
from IntfModels import Interface
from ArnetModel import MacAddress
from socket import inet_aton
import Arnet
import ast

import TableOutput
import Tac
# pylint: disable-msg=superfluous-parens
# pylint: disable-msg=consider-using-f-string

# NOTE: This list currently duplicates the values in the SfeL3Unicast::NextHopPktCmd
# enum. Loading the enum type was causing too much slowdown in loading this plugin.
# BUG395932 tracks deduplicating this information in a way that does not cause as
# much slowdown in the CLI.
NextHopPktCmdTypes = [
      'pktCmdNone',
      'pktCmdForward',
      'pktCmdArpTrap',
      'pktCmdReceive',
      'pktCmdSlowReceive',
      'pktCmdTrapVxlanULFlood',
      'pktCmdDrop',
      'pktCmdForwardAndTrap' ]

bessdTxModesToDisplayStr = {
   'txModeDirect' : 'Direct',
   'txModeAdaptive' : 'Adaptive',
   'txModeEnqueue' : 'Enqueue',
   'txModeInvalid' : 'Invalid'
}
rateUnitsToDisplayStr = {
   'rateUnitPps' : 'Pps',
   'rateUnitbps' : 'Bps',
   'rateUnitKbps' : 'Kbps',
   'rateUnitMbps' : 'Mbps',
   'rateUnitInvalid' : 'Invalid'
}

u64MaxSpaces = 22
macAddressMaxSpaces = 18
prefixMaxSpaces = 20
packetCmdMaxSpaces = 22

counterUnit = Tac.Type( "Sfe::CounterUnit" )

def getCounterTypes():
   counterUnitAttributes = []
   for attr in counterUnit.__dict__.keys():
      if not attr.startswith( "__" ):
         counterUnitAttributes.append( attr )
   return counterUnitAttributes

def createColumnFormat( minLen, maxLen ):
   fmt = TableOutput.Format( justify="left", minWidth=minLen,
                             maxWidth=maxLen )
   fmt.noPadLeftIs( True )
   fmt.padLimitIs( True )
   return fmt

class L3AgentSummaryUnicast( Model ):

   # Routing state
   ucastEnabledV4 = Bool( help="Status of IPV4 unicast routing" )
   ucastEnabledV6 = Bool( help="Status of IPV6 unicast routing" )

   mcastEnabledV4 = Bool( help="Status of IPV4 multicast routing" )
   mplsEnabled = Bool( help="Status of MPLS routing" )

   # Forwarding state
   forwardingMode = Enum( values=[ "defaultMode", "cutThrough",
                          "storeAndForward" ], help="Forwarding mode" )
   platformAdjSharing = Bool( help="Status of Platform Adjacency sharing" )
   routeDeletionDelay = Float( help="Route deletion delay" )

   # Unicast routes
   lpmRoutesV4 = Int( help="Number of IPV4 LPM routes" )
   lpmRoutesV6 = Int( help="Number of IPV6 LPM routes" )
   lpmRoutes = Int( help="Number of IPV4 and IPV6 LPM routes" )

   lpmCreateV4 = Int( help="Number of IPV4 LPM creates" )
   lpmDeleteV4 = Int( help="Number of IPV4 LPM deletes" )
   lpmUpdateV4 = Int( help="Number of IPV4 LPM updates" )

   lpmCreateV6 = Int( help="Number of IPV6 LPM creates" )
   lpmDeleteV6 = Int( help="Number of IPV6 LPM deletes" )
   lpmUpdateV6 = Int( help="Number of IPV6 LPM updates" )

   def render( self ):
      # Routing state
      print( "IPV4 unicast routing: %s" %
         ( "Enabled" if self.ucastEnabledV4 else "Disabled" ) )
      print( "IPV6 unicast routing: %s" %
         ( "Enabled" if self.ucastEnabledV6 else "Disabled" ) )

      print( "IPV4 multicast routing: %s\n" %
         ( "Enabled" if self.mcastEnabledV4 else "Disabled" ) )
      print( "MPLS routing: %s" %
            ( "Enabled" if self.mplsEnabled else "Disabled" ) )

      # Forwarding State
      print( "Forwarding mode: %s\n" % self.forwardingMode )
      print( "FIB ADJ sharing: enabled" )
      print( "Platform ADJ sharing: %s" %
            ( "enabled" if self.platformAdjSharing else "disabled" ) )
      print( "Route deletion delay: %0.0f seconds\n" %
            ( self.routeDeletionDelay ) )

      # LPM table
      print( "LPM IPV4 routes: %u" % ( self.lpmRoutesV4 ) )
      print( "LPM IPV4 creates: %u" % ( self.lpmCreateV4 ) )
      print( "LPM IPV4 updates: %u" % ( self.lpmUpdateV4 ) )
      print( "LPM IPV4 deletes: %u" % ( self.lpmDeleteV4 ) )

      # Unicast routes
      print( "LPM routes (IPV4 + IPV6): %u" % ( self.lpmRoutes ) )

      print()

class Licensing( Model ):
   tcLicenseCreated = Bool( help="License TC has been created" )
   throttledInterfaces = Dict( keyType=Interface, valueType=int,
         help="A mapping of interface to its throttled speed in Mbps" )

   def render( self ):
      print( "Licensing Information" )
      print( "---------------------" )
      print( " License TC created: %s" %
            ( "yes" if self.tcLicenseCreated else "no" ) )
      print( " Number of throttled interfaces: %u" %
            ( len( self.throttledInterfaces ) ) )
      if self.throttledInterfaces:
         print( " Interfaces throttled:" )
         for intf in Arnet.sortIntf( self.throttledInterfaces ):
            speed = self.throttledInterfaces[ intf ]
            print( "  %s: %u Mbps" % ( intf, speed ) )
      print( "" )

class Counter( Model ):
   name = Str( help="Name of the counter", optional=True )
   ownerId = Int( help="ID used to look up the module owner of the counter "
                  "in the owners dictionary", optional=True )
   counterType = Enum( values=( "gatehook", "module" ), help="Counter type" )
   unit = Enum( values=getCounterTypes(), help="Basic unit of the counter" )
   count = Int( help="Number of occurrences" )

class CounterModel( Model ):
   counters = Dict( keyType=int, valueType=Counter,
                    help="A list of counters used by Bessd keyed by the counterId" )
   owners = Dict( keyType=int, valueType=str,
                  help="A list of counter owners keyed by the ownerId" )

   def getCounterName( self, counterId, counter ):
      # FIX for symptom of BUG745268
      if counter.name:
         return counter.name.encode( 'utf-8', 'replace' ).decode()
      name = "ID:%d" % counterId
      return name

   def render( self ):
      header = ( "Name", "Owner", "Counter Type", "Unit", "Count" )

      numOfColumns = 5
      # Name
      nameMinLen = len( header[ 0 ] )
      nameMaxLen = nameMinLen
      # Owner
      ownerMinLen = len( header[ 1 ] )
      ownerMaxLen = ownerMinLen
      # Counter Type
      cTypeMinLen = len( header[ 2 ] )
      cTypeMaxLen = max( cTypeMinLen, 10 )
      # Unit
      unitMinLen = len( header[ 3 ] )
      unitMaxLen = max( unitMinLen, 10 )
      # Count
      countMinLen = len( header[ 4 ] )
      countMaxLen = u64MaxSpaces

      # Calculate dynamic column's widths first
      for ( counterId, entry ) in self.counters.items():
         name = self.getCounterName( counterId, entry )
         owner = self.owners.get( entry.ownerId, "-" )
         nameMaxLen = max( len( name ), nameMaxLen )
         ownerMaxLen = max( len( owner ), ownerMaxLen )

      tableMaxWidth = ( nameMaxLen + ownerMaxLen + cTypeMaxLen +
                        unitMaxLen + countMaxLen +
                        # space between columns
                        numOfColumns - 1 )

      # tableWidth is set to enforce a "terminal wrap" instead of
      # a "table wrap"
      table = TableOutput.createTable( header, tableWidth=tableMaxWidth )

      # Name
      f1 = createColumnFormat( nameMinLen, nameMaxLen )

      # Owner
      f2 = createColumnFormat( ownerMinLen, ownerMaxLen )

      # Counter Type
      f3 = createColumnFormat( cTypeMinLen, cTypeMaxLen )

      # Unit
      f4 = createColumnFormat( unitMinLen, unitMaxLen )

      # Count
      f5 = createColumnFormat( countMinLen, countMaxLen )

      table.formatColumns( f1, f2, f3, f4, f5 )

      for ( counterId, entry ) in self.counters.items():
         name = self.getCounterName( counterId, entry )
         owner = self.owners.get( entry.ownerId, "-" )
         table.newRow( name, owner, entry.counterType, entry.unit, entry.count )

      print( table.output() )

class Qos( Model ):
   txModeInterface = Enum( values=Tac.Type( 'Sfe::BessdTxMode' ).attributes,
         help="Tx Mode Interface" )
   txModeCpu = Enum( values=Tac.Type( 'Sfe::BessdTxMode' ).attributes,
         help="Tx Mode CPU" )
   cpuReceiveLimit = Int( "Control plane receive limit" )
   rateUnit = Enum( values=Tac.Type( 'Qos::RateUnit' ).attributes,
         help="Rate unit of Control plane receive limit" )

   def setAttrsFromDict( self, data ):
      self.txModeInterface = Tac.enumName( 'Sfe::BessdTxMode',
            data[ 'txModeInterface' ] )
      self.txModeCpu = Tac.enumName( 'Sfe::BessdTxMode',
            data[ 'txModeCpu' ] )
      self.cpuReceiveLimit = data[ 'cpuReceiveLimit' ]
      self.rateUnit = Tac.enumName( 'Qos::RateUnit', data[ 'rateUnit' ] )

   def render( self ):
      print( "Qos Information" )
      print( "-----------------" )
      print( " Tx Mode Interface: %s"
            % ( bessdTxModesToDisplayStr[ self.txModeInterface ] ) )
      print( " Tx Mode Cpu: %s" %
            ( bessdTxModesToDisplayStr[ self.txModeCpu ] ) )
      print( " Control plane receive limit: %s %s"
            % ( str( self.cpuReceiveLimit ),
               rateUnitsToDisplayStr[ self.rateUnit ] ) )

class NicInfo( Model ):
   pciAddr = Str( help="PCI address" )
   macAddr = MacAddress( help="MAC address" )
   driver = Str( help="Driver" )
   version = Str( help="Version" )
   busInfo = Str( help="PCI bus information" )
   nicType = Enum( values=Tac.Type( 'Sfe::NicType' ).attributes, help="NIC Type" )
   l3SubIntfCapable = Bool( help="Layer-3 sub-interface capable" )

   def render( self ):

      def nicTypeName( nicType ):
         nicTypeNameDict = {
               'nicUnknown' : 'Unknown',
               'vf' : 'VF',
               'passthrough' : 'Passthrough',
               'physical' : 'Physical',
               'acceleratedNetworking' : 'Accelerated Networking',
               'vSwitch' : 'vSwitch',
               'nac' : 'NAC',
               'xl710' : 'XL710' }
         return nicTypeNameDict.get( nicType, nicType )

      print( 'PCI address: %s' % self.pciAddr )
      print( 'MAC address: %s' % self.macAddr.displayString )
      print( 'Driver: %s' % self.driver )
      print( 'NIC type: %s' % nicTypeName( self.nicType ) )
      print( 'Layer-3 sub-interface can be enabled.'
            if self.l3SubIntfCapable else
            'Layer-3 sub-interface cannot be enabled.' )

class PlatformModel( Model ):
   platform = Str( help="Cloud platform" )
   nCpCores = Int( help="Number of control plane cores" )
   nDpCores = Int( help="Number of datapath cores" )
   dpMem = Int( help="Memory for datapath in MB" )
   cloudInstanceType = Str( help="Cloud instance type" )
   cpuGhz = Float( help="Clock speed of CPU in Ghz" )
   cpuModel = Str( help="Processor model name" )
   vrfCount = Int( help="Number of VRFs in use" )
   maxVrfs = Int( help="Maximum number of VRFs" )
   num2MbHugepages = Int( help="Number of 2MB hugepages allocated" )
   num1GbHugepages = Int( help="Number of 1GB hugepages allocated" )
   bessBuffers = Int( help="Number of bess buffers" )
   threadsPerCore = Int( help="Number of threads per core" )
   isolCpuLow = Int( help="Lower limit of the range of isolated CPUs" )
   isolCpuHigh = Int( help="Upper limit of the range of isolated CPUs" )
   lastReloadTime = Str( help="Last Sfe agent reload time" )
   dpdkVersion = Str( help="DPDK version" )
   nicInfo = Dict( keyType=Interface, valueType=NicInfo,
                   help="Dictionary of NIC information keyed by interface name" )

   def render( self ):
      print( 'Platform: %s' % self.platform )
      print( 'Number of control plane cores: %s' % self.nCpCores )
      print( 'Number of datapath cores: %s' % self.nDpCores )
      print( 'Memory for datapath: %sMB' % self.dpMem )
      print( 'Cloud instance type: %s' % (
         self.cloudInstanceType if self.cloudInstanceType else 'N/A' ) )
      print( 'Processor model name: %s' % self.cpuModel )
      print( 'Clock speed of CPU: %s GHz' % self.cpuGhz )
      print( 'Thread(s) per core: %s' % self.threadsPerCore )
      if self.isolCpuLow != self.isolCpuHigh:
         print( f'Isolated CPUs: {self.isolCpuLow}-{self.isolCpuHigh}' )
      else:
         print( 'Isolated CPUs: %s' % self.isolCpuLow )
      print( 'Number of VRFs in use: %s' % self.vrfCount )
      print( 'Maximum number of VRFs: %s' % self.maxVrfs )
      print( 'Number of 2MB hugepages allocated: %s' % self.num2MbHugepages )
      print( 'Number of 1GB hugepages allocated: %s' % self.num1GbHugepages )
      print( 'Number of packet buffers: %s' % self.bessBuffers )
      print( 'Last Sfe agent reload time: %s' % (
         self.lastReloadTime if self.lastReloadTime else 'N/A' ) )
      print( 'DPDK version: %s' % self.dpdkVersion )
      print ( 'NIC information:' )

      for nic in Arnet.sortIntf( self.nicInfo ):
         print( Tac.newInstance( 'Arnet::IntfId', nic ).shortName.lower() )
         self.nicInfo[ nic ].render()
         print ( '' )

class PeerGroupIdModel( Model ):

   # peerGroupId to peer Ip
   peerGroupIdToVtepIpMap = Dict( keyType=int, valueType=str,
                                  help='Map for peer group id to vtepIp' )

   def setAttrsFromDict( self, data ):
      peergroupidDict = data[ "peerGroupIds" ]
      for key in peergroupidDict:
         # remove last two char=", "
         self.peerGroupIdToVtepIpMap[ key ] = peergroupidDict[ key ][ : -2 ]

   def render( self ):
      header = ( "Peer Group ID", "Peer IP" )
      numOfColumns = 2
      peerGroupIdMinLen = len( header[ 0 ] )
      # max value for peerGroupId = 65535
      peerGroupIdMaxLen = max( peerGroupIdMinLen, 5 )
      peerIpMinLen = len( header[ 1 ] )
      # max two vtepIps as HA peer
      # AvtFecFlat test has 4 peers, so adjust width accordingly.
      peerIpMaxLen = max( peerIpMinLen, 70 )
      tableMaxWidth = ( peerGroupIdMaxLen + peerIpMaxLen +
                        numOfColumns - 1 )

      table = TableOutput.createTable( header, tableWidth=tableMaxWidth )
      # PeerGroupId Right align
      f1 = TableOutput.Format( justify="right", minWidth=peerGroupIdMinLen,
                                maxWidth=peerGroupIdMaxLen )
      # vtepIp left align
      f2 = createColumnFormat( peerIpMinLen, peerIpMaxLen )

      table.formatColumns( f1, f2 )
      for key in sorted( self.peerGroupIdToVtepIpMap ):
         table.newRow( str( key ),
                       str( self.peerGroupIdToVtepIpMap[ key ] ) )
      output = table.output()
      print( output )

#
# Start of Next Hop code
#

class NextHop( Model ):

   pktCmd = Enum( optional=True, help="Next Hop Type",
                  values=NextHopPktCmdTypes )
   intf = Interface( help="Egress interface" )
   ethAddr = MacAddress( help="Destination MAC address" )
   nhLocalIndex = Int( help="Next Hop Local Index" )

class FecInfo( Model ):

   numberOfUniqueNextHops = Int( help="Number of Unique Next Hops" )

   fecInfos = List( valueType=NextHop,
                    help='Lists of NextHop' )

   def setAttrsFromDict( self, data ):
      self.numberOfUniqueNextHops = data[ 0 ] # 1st element in tuple
      # fecInfo is local dict, self.fecInfos is CliModel List obj.
      fecInfo = data[ 1 ]
      for value in fecInfo.values():
         nextHop = NextHop()
         self.fecInfos.append( nextHop )
         nextHop.setAttrsFromDict( value )

class Gates( Model ):

   outputGate = Int( help="Output gate" )

   def setOutputGate( self, data ):
      self.outputGate = data

class pModNextHop( Model ):

   gate = Int( help="pModule Gate" )
   nhLocalIndex = Int( help="Next Hop Local Index" )

class FibFecInfo( Model ):

   numberOfUniqueNextHops = Int( help="Number of Unique Next Hops" )

   fecInfos = List( valueType=pModNextHop,
                    help='Lists of pModNextHop' )

   def setAttrsFromDict( self, data ):
      self.numberOfUniqueNextHops = data[ 0 ] # 1st element in tuple
      # fecInfo is local dict, self.fecInfos is CliModel List obj.
      fecInfo = data[ 1 ]
      for value in fecInfo.values():
         nextHop = pModNextHop()
         self.fecInfos.append( nextHop )
         nextHop.setAttrsFromDict( value )

def keyIpPrefixAndLen( a ):
   """Ip addresses in this form 1.2.3.4/32
      Returns key based on Ip address and prefix length """
   slashPos = a.prefix.find( '/' )
   addr = a.prefix[ 0 : slashPos ]
   num = inet_aton( addr )
   prefixLen = int( a.prefix[ slashPos + 1 : ] )
   return ( num, prefixLen )

def keyIpAndLen( a ):
   """Ip addresses in this form vrf/1.2.3.4/32
      Returns key based on Ip address and prefix length """
   slashPos = a.find( '/' )
   slashPosEnd = a.find( '/', slashPos + 1 )
   addr = a[ slashPos + 1 : slashPosEnd ]
   num = inet_aton( addr )
   prefixLen = int( a[ slashPosEnd + 1 : ] )
   return ( num, prefixLen )

def keyVrf( a ):
   slashPos = a.find( '/' )
   vrf = a[ 0 : slashPos ]
   return vrf

class Fec( Model ):

   fecKey = Int( help="FecKey associated with L2Adj" )

   def setAttr( self, data ):
      self.fecKey = data

class Fib( Model ):

   vrfName = Str( help="VrfName" )
   prefix = Str( help="Prefix" )
   fecKey = Int( help="FecKey associated with L2Adj" )
   nextHops = Str( help="Comma separated list of Next Hop(s)" )

class ModuleTable( Model ):

   nhInfos = List( valueType=NextHop,
                   help='Lists of NextHop' )

   fibs = List( valueType=Fib,
                help='Lists of Fib' )

   fecIndexes = Dict( keyType=int, valueType=FibFecInfo,
                      help='Maps a FecKey index to FibFecInfo' )

   def setNHAttrsFromTuple( self, data ):
      """ Receive a tuple and convert to dict. """

      # v[0] = intfId.id(), v[1] = nhId, v[2] = pktCmd v[3] = macaddr
      # Sample data :('Ethernet1', '2', 0, '2e:10:ef:f6:f8:17')

      nextHop = NextHop()
      valueDict = { 'intf' : data[ 0 ], 'nhLocalIndex' : int( data[ 1 ] ),
                    'pktCmd' : data[ 2 ], 'ethAddr' : data[ 3 ] }
      nextHop.setAttrsFromDict( valueDict )
      self.nhInfos.append( nextHop )

   def setFibAttrsFromTuple( self, data ):
      """ Receive a tuple and convert to dict. """

      # v[0] = prefix v[1] = fecKey, v[2] = vrfName, v[3] = Next Hop
      # Sample data :('5.6.7.8', 1, 'default', '10,11')

      fib = Fib()
      valueDict = { 'prefix' : data[ 0 ], 'fecKey' : data[ 1 ],
                    'vrfName' : data[ 2 ], 'nextHops' : data[ 3 ] }
      fib.setAttrsFromDict( valueDict )
      self.fibs.append( fib )

   def setFecAttrsFromTuple( self, data ):
      """ Receive a tuple and convert to dict. """

      # v[0] = fec, v[1] = tuple of (nh + gates) / could be ECMP

      self.fecIndexes[ int( data[ 0 ] ) ] = FibFecInfo()
      self.fecIndexes[ int( data[ 0 ] ) ].setAttrsFromDict(
            next( iter( data[ 1 ].values() ) ) )

   def setFecAttrsFromDiff( self, data ):
      """ Receive a tuple and convert to dict. """

      self.fecIndexes[ int( data[ 0 ] ) ] = FibFecInfo()
      shimTuple = ( 1, { '0' : { 'nhLocalIndex' : int( data[ 1 ] ), 'gate' : 0 } } )
      self.fecIndexes[ int( data[ 0 ] ) ].setAttrsFromDict( shimTuple )

class PModStatusTable( Model ):

   activeStatus = Bool( help='pModule status' )

   def setAttrsFromDict( self, data ):
      setattr( self, "activeStatus", data )

class AsyncNextHop( Model ):
   l2Adj = Int( help="L2Adj from fecToL2Adjs" )
   nhFecIndex = Int( help="Index of this nh in the NextHopKey hashmap" )
   nhLocalIndex = Int( help="Next Hop Local Index" )
   packetCmd = Enum( optional=True, help="Next Hop Type",
                  values=NextHopPktCmdTypes )
   nhIntf = Interface( help="Egress interface" )
   nhMac = MacAddress( help="Destination MAC address" )
   nhType = Enum( optional=True, help="Next Hop Type",
                  values=( 'single', 'ecmp' ) )
   nhIndex = Int( optional=True, help="Next Hop Index" )
   nhVlanId = Int( optional=True, help="Next Hop VLAN ID" )
   nhGate = Int( optional=True, help="Next Hop Gate" )

class AsyncFecInfo( Model ):
   numberOfUniqueNextHops = Int( help="Number of Unique Next Hops" )
   nhInfos = List( valueType=AsyncNextHop,
                    help='Lists of NextHops in this fec' )

class L3AgentAsyncNextHops( Model ):
   """Replacement class for async NextHops, needed for testing"""
   fecs = Dict( keyType=int, valueType=AsyncFecInfo,
         help="Maps a FecKey index to FecInfo" )

class L3AgentNextHops( Model ):
   """Major class for NextHops."""

   # key defaults to a string (l2Adj index)
   l2Adjs = Dict( valueType=Fec,
                  help='Maps a L2Adj index to FecKey' )

   # map fec to l2Adj string
   fecToL2Adjs = Dict( keyType=int, valueType=str,
                       help='Maps a FecKey to L2Adj' )

   fecs = Dict( keyType=int, valueType=FecInfo,
                help='Maps a FecKey index to FecInfo' )

   diffNH = Bool( help='Diff Next Hop packet Module with control Module' )

   pModIntfs = Dict( keyType=Interface, valueType=PModStatusTable,
                     help='pModule interfaces' )

   pModule = Submodel( valueType=ModuleTable,
                       help='packet Module Next Hop Table' )

   cModule = Submodel( valueType=ModuleTable,
                       help='control Module Next Hop Table' )

   diffModule = Submodel( valueType=ModuleTable,
                          help='Next Hop Table Diff' )

   def convertFromPktCmd( self, pktCmd ):
      return {
         'pktCmdArpTrap' : 0,
         'pktCmdForward' : 1,
         'pktCmdDrop' : 2,
         }[ pktCmd ]

   def convertToPktCmd( self, pktCmd ):
      return {
         0 : 'pktCmdArpTrap',
         1 : 'pktCmdForward',
         2 : 'pktCmdDrop',
         # 3 : 'pktCmdFwdMirror',
         }[ pktCmd ]

   def render( self ):

      if not self.diffNH:
         header = ( "FEC", "L2ADJ", "Next Hop 1 Of N", "Next Hop",
                    "Packet CMD", "Interface", "Ethernet MAC Address" )

         numOfColumns = 7
         # FEC
         fecMinLen = len( header[ 0 ] )
         fecMaxLen = max( fecMinLen, u64MaxSpaces )
         # L2ADJ
         adjMinLen = len( header[ 1 ] )
         adjMaxLen = max( adjMinLen, u64MaxSpaces )
         # Next Hop 1 Of N
         nHop1NMinLen = len( header[ 2 ] )
         nHop1NMaxLen = max( nHop1NMinLen, u64MaxSpaces )
         # Next Hop
         nextHopMinLen = len( header[ 3 ] )
         nextHopMaxLen = max( nextHopMinLen, u64MaxSpaces )
         # Packet CMD
         cmdMinLen = len( header[ 4 ] )
         cmdMaxLen = max( cmdMinLen, packetCmdMaxSpaces )
         # Interface
         ifMinLen = len( header[ 5 ] )
         ifMaxLen = ifMinLen
         # Ethernet MAC Address
         macMinLen = len( header[ 6 ] )
         macMaxLen = max( macMinLen, macAddressMaxSpaces )

         for key in sorted( self.fecs ):
            for index, nh in enumerate( sorted( self.fecs[ key ].fecInfos ), 1 ):
               ifMaxLen = max( len( str( nh.intf ).strip( '\'' )
                               if nh.intf else '-' ), ifMaxLen )

         tableMaxWidth = ( fecMaxLen + adjMaxLen + nHop1NMaxLen, cmdMaxLen,
                           ifMaxLen + macMaxLen +
                           numOfColumns - 1 )

         table = TableOutput.createTable( header, tableWidth=tableMaxWidth )

         # FEC
         f1 = createColumnFormat( fecMinLen, fecMaxLen )

         # L2ADJ
         f2 = createColumnFormat( adjMinLen, adjMaxLen )

         # Next Hop 1 of N
         f3 = createColumnFormat( nHop1NMinLen, nHop1NMaxLen )

         # Next Hop
         f4 = createColumnFormat( nextHopMinLen, nextHopMaxLen )

         # Packet CMD
         f5 = createColumnFormat( cmdMinLen, cmdMaxLen )

         # Interfacei
         f6 = createColumnFormat( ifMinLen, ifMaxLen )

         # Ethernet Mac Address
         f7 = createColumnFormat( macMinLen, macMaxLen )

         table.formatColumns( f1, f2, f3, f4, f5, f6, f7 )

         for key in sorted( self.fecs ):
            for index, nh in enumerate( sorted( self.fecs[ key ].fecInfos ), 1 ):
               table.newRow( str( key ),
                             self.fecToL2Adjs[ key ],
                             str( index ),
                             str( nh.nhLocalIndex ),
                             nh.pktCmd,
                             str( nh.intf ).strip( '\'' ) if nh.intf else '-',
                             nh.ethAddr )
         output = table.output()
         print( "                                             SFE Control Module" )
         print( "                                             ==================" )
         print( output )
      else:
         #
         # Three tables
         # cModule, pModule and diffModule

         # cModule
         if self.cModule:
            header = ( "Next Hop", "Packet CMD", "Interface",
                       "Ethernet MAC Address" )

            numOfColumns = 4
            # Next Hop
            nextHopMinLen = len( header[ 0 ] )
            nextHopMaxLen = max( nextHopMinLen, u64MaxSpaces )
            # Packet CMD
            cmdMinLen = len( header[ 1 ] )
            cmdMaxLen = max( cmdMinLen, packetCmdMaxSpaces )
            # Interface
            ifMinLen = len( header[ 2 ] )
            ifMaxLen = ifMinLen
            # Ethernet MAC Address
            macMinLen = len( header[ 3 ] )
            macMaxLen = max( macMinLen, macAddressMaxSpaces )

            for nh in sorted( self.cModule.nhInfos ):
               ifMaxLen = max( len( str( nh.intf ).strip( '\'' ) ), ifMaxLen )

            tableMaxWidth = ( nextHopMaxLen + cmdMaxLen +
                              ifMaxLen + macMaxLen +
                              numOfColumns - 1 )

            cModTable = TableOutput.createTable( header,
                                                 tableWidth=tableMaxWidth )
            # Next Hop
            f1 = createColumnFormat( nextHopMinLen, nextHopMaxLen )

            # Packet CMD
            f2 = createColumnFormat( cmdMinLen, cmdMaxLen )

            # Interface
            f3 = createColumnFormat( ifMinLen, ifMaxLen )

            # Ethernet Mac Address
            f4 = createColumnFormat( macMinLen, macMaxLen )

            cModTable.formatColumns( f1, f2, f3, f4 )

            for nh in sorted( self.cModule.nhInfos ):
               cModTable.newRow( str( nh.nhLocalIndex ),
                                 nh.pktCmd,
                                 str( nh.intf ).strip( '\'' ),
                                 nh.ethAddr )
            output = cModTable.output()
            print( "                            SFE Control Module" )
            print( "                            ==================" )
            print( output )
            print()

         # pModule
         if self.pModule:
            header = ( "Next Hop", "Packet CMD", "Interface",
                       "Ethernet MAC Address" )

            numOfColumns = 4
            # Next Hop
            nextHopMinLen = len( header[ 0 ] )
            nextHopMaxLen = max( nextHopMinLen, u64MaxSpaces )
            # Packet CMD
            cmdMinLen = len( header[ 1 ] )
            cmdMaxLen = max( cmdMinLen, packetCmdMaxSpaces )
            # Interface
            ifMinLen = len( header[ 2 ] )
            ifMaxLen = ifMinLen
            # Ethernet MAC Address
            macMinLen = len( header[ 3 ] )
            macMaxLen = max( macMinLen, macAddressMaxSpaces )

            for nh in sorted( self.pModule.nhInfos ):
               ifMaxLen = max( len( str( nh.intf ).strip( '\'' ) ), ifMaxLen )

            tableMaxWidth = ( nextHopMaxLen + cmdMaxLen +
                              ifMaxLen + macMaxLen +

                              numOfColumns - 1 )

            pModTable = TableOutput.createTable( header,
                                                 tableWidth=tableMaxWidth )

            # Next Hop
            f1 = createColumnFormat( nextHopMinLen, nextHopMaxLen )

            # Packet CMD
            f2 = createColumnFormat( cmdMinLen, cmdMaxLen )

            # Interface
            f3 = createColumnFormat( ifMinLen, ifMaxLen )

            # Ethernet Mac Address
            f4 = createColumnFormat( macMinLen, macMaxLen )

            pModTable.formatColumns( f1, f2, f3, f4 )

            for nh in sorted( self.pModule.nhInfos ):
               pModTable.newRow( str( nh.nhLocalIndex ),
                                 nh.pktCmd,
                                 str( nh.intf ).strip( '\'' ),
                                 nh.ethAddr )
            output = pModTable.output()
            print( "                            SFE Packet Module" )
            print( "                            ==================" )
            print( output )
            print()

         # diffModule
         if self.diffModule:
            header = ( "Next Hop", "Packet CMD", "Interface",
                       "Ethernet MAC Address" )

            numOfColumns = 4
            # Next Hop
            nextHopMinLen = len( header[ 0 ] )
            nextHopMaxLen = max( nextHopMinLen, u64MaxSpaces )
            # Packet CMD
            cmdMinLen = len( header[ 1 ] )
            cmdMaxLen = max( cmdMinLen, packetCmdMaxSpaces )
            # Interface
            ifMinLen = len( header[ 2 ] )
            ifMaxLen = ifMinLen
            # Ethernet MAC Address
            macMinLen = len( header[ 3 ] )
            macMaxLen = max( macMinLen, macAddressMaxSpaces )

            for nh in self.diffModule.nhInfos:
               ifMaxLen = max( len( str( nh.intf ).strip( '\'' ) ), ifMaxLen )

            tableMaxWidth = ( nextHopMaxLen + cmdMaxLen +
                              ifMaxLen + macMaxLen +
                              numOfColumns - 1 )

            diffModTable = TableOutput.createTable( header,
                                                    tableWidth=tableMaxWidth )

            # Next Hop
            f1 = createColumnFormat( nextHopMinLen, nextHopMaxLen )

            # Packet CMD
            f2 = createColumnFormat( cmdMinLen, cmdMaxLen )

            # Interface
            f3 = createColumnFormat( ifMinLen, ifMaxLen )

            # Ethernet Mac Address
            f4 = createColumnFormat( macMinLen, macMaxLen )

            diffModTable.formatColumns( f1, f2, f3, f4 )

            for nh in self.diffModule.nhInfos:
               diffModTable.newRow( str( nh.nhLocalIndex ),
                                    nh.pktCmd,
                                    str( nh.intf ).strip( '\'' ),
                                    nh.ethAddr )
            output = diffModTable.output()
            print( "                   Difference between pModule and cModule" )
            print( "                   ======================================" )
            print( output )
            print()

         #
         # Dump the pModule Interface list
         #
         if self.pModIntfs:
            header = ( "Interface", "Status" )

            numOfColumns = 2
            # Interface
            ifMinLen = len( header[ 0 ] )
            ifMaxLen = ifMinLen
            # Status
            statusMinLen = len( header[ 1 ] )
            statusMaxLen = statusMinLen

            for k, v in sorted( self.pModIntfs.items() ):
               ifMaxLen = max( len( str( nh.intf ).strip( '\'' ) ), ifMaxLen )
               statusMaxLen = max( len( "Active" if v.activeStatus else "Missing" ),
                                   statusMaxLen )
            tableMaxWidth = ( ifMaxLen + statusMaxLen +
                              numOfColumns - 1 )

            pModIntfTable = TableOutput.createTable( header,
                                                     tableWidth=tableMaxWidth )

            # Interface
            f1 = createColumnFormat( ifMinLen, ifMaxLen )

            # Status
            f2 = createColumnFormat( statusMinLen, statusMaxLen )

            pModIntfTable.formatColumns( f1, f2 )

            for k, v in sorted( self.pModIntfs.items() ):
               statusString = "Active" if v.activeStatus else "Missing"
               pModIntfTable.newRow( k, statusString )

            output = pModIntfTable.output()
            print( "  pModule Interface(s)" )
            print( "  ====================" )
            print( output )
            print()

   def initTables( self ):
      self.cModule = ModuleTable()
      self.pModule = ModuleTable()
      self.diffModule = ModuleTable()

   def diffTables( self ):
      tableOneSet = set() # cModule
      for key in sorted( self.fecs ):
         for nh in self.fecs[ key ].fecInfos:
            if not nh.intf.stringValue:
               continue
            # Filter out Management ports
            if "Management" in nh.intf.stringValue:
               continue
            tupleEntry = ( nh.intf.stringValue,
                           str( nh.nhLocalIndex ),
                           nh.pktCmd,
                           nh.ethAddr.stringValue )
            tableOneSet.add( tupleEntry )

      # Populate cModule with all data
      if tableOneSet:
         self.cModule = ModuleTable()
         for value in tableOneSet:
            self.cModule.setNHAttrsFromTuple( value )

      # We must do a symmetric diff on the two tables

      if self.pModule:
         tableTwoSet = set()
         for nh in self.pModule.nhInfos:
            tupleEntry = ( nh.intf.stringValue,
                           str( nh.nhLocalIndex ),
                           nh.pktCmd,
                           nh.ethAddr.stringValue )
            tableTwoSet.add( tupleEntry ) # Add tuple to set
         symmetric_diff = tableOneSet ^ tableTwoSet

         # Convert back into the model List
         if symmetric_diff:
            self.diffModule = ModuleTable()
            for value in symmetric_diff:
               self.diffModule.setNHAttrsFromTuple( value ) # Add to diff table

   def setAttrsFromDict( self, data ):
      tableTwoSet = set() # pModule
      for key in data:
         if key == 'l2Adj':
            l2AdjInfo = data[ key ]
            for l2AdjIndex, value in l2AdjInfo.items():
               self.l2Adjs[ l2AdjIndex ] = Fec()
               self.l2Adjs[ l2AdjIndex ].setAttr( value )
               self.fecToL2Adjs[ value ] = l2AdjIndex
         elif key == 'fec':
            fecInfo = data[ key ]
            for fecKey, value in fecInfo.items():
               self.fecs[ int( fecKey ) ] = FecInfo()
               self.fecs[ int( fecKey ) ].setAttrsFromDict( value )
         elif key == 'pModIntf':
            pModIntf = data[ key ]
            for intf, value in pModIntf.items():
               # Filter out Management ports
               if "Management" in intf:
                  continue
               self.pModIntfs[ intf ] = PModStatusTable()
               self.pModIntfs[ intf ].setAttrsFromDict( value )
         elif key.startswith( 'pModule' ):
            pModDict = data[ key ]
            for v in pModDict.values():
               IntfId = Tac.Type( 'Arnet::IntfId' )
               nhIntfId = IntfId()
               nhIntfId.intfId = v[ 0 ]
               nhId = str( v[ 1 ] )
               nhPktCmd = self.convertToPktCmd( v[ 2 ] )
               updatedVal = ( nhIntfId.stringValue, nhId, nhPktCmd, v[ 3 ] )
               tableTwoSet.add( updatedVal )
         else:
            setattr( self, key, data[ key ] )

      # Populate pModule with all data
      if tableTwoSet:
         self.pModule = ModuleTable()
         for value in tableTwoSet:
            self.pModule.setNHAttrsFromTuple( value ) # send a tuple

class L3AgentSummarySoftware( Model ):

   # Route Helper
   ipVersion = Enum( values=( "IPV4", "IPV6" ), help="Version of IP" )

   # Route stats
   routeInserts = Int( help="Number of routes inserted" )
   routeDeletes = Int( help="Number of routes deleted" )
   routeUpdates = Int( help="Number of routes updated" )
   routeUpdatesIgnored = Int( help="Number of route updates ignored" )
   routeUpdatesFecChanged = Int( help="Number of route updates where Fec Changed" )
   routeUpdatesFecUpdated = Int( help="Number of route updates where Fec Updated" )

   # LPM stats
   lpmCreates = Int( help="Number of LPM creates" )
   lpmDeletes = Int( help="Number of LPM deletes" )
   lpmUpdates = Int( help="Number of LPM updates" )
   lpmUpdatesFecChanged = Int( help="Number of LPM updates where Fec Changed" )
   lpmUpdatesFecUpdated = Int( help="Number of LPM updates where Fec Updated" )

   # Adjacencies
   adjCreates = Int( help="Number of adjacency creates" )
   adjUpdates = Int( help="Number of adjacency updates" )
   adjDeletes = Int( help="Number of adjacency deletes" )

   def render( self ):
      print()
      print( f"{self.ipVersion} Summary" )
      print( f" Route inserts: {self.routeInserts}, deletes: {self.routeDeletes}, "
             f"updates: {self.routeUpdates}, ignores: {self.routeUpdatesIgnored}" )

      print( f" Route updates fec changed: {self.routeUpdatesFecChanged}, "
             f"updated: {self.routeUpdatesFecUpdated}" )

      print( f" LPM creates: {self.lpmCreates}, deletes: {self.lpmDeletes}, "
             f"updates: {self.lpmUpdates}" )
      print( f" LPM updates fec changed: {self.lpmUpdatesFecChanged}, "
             f"updated: {self.lpmUpdatesFecUpdated}" )

      print()
      print( "Adjacency Summary" )
      print( f" ADJ creates: {self.adjCreates}, updates: {self.adjUpdates}, "
             f"deletes: {self.adjDeletes}" )

# FibV4
class FibNextHop( Model ):

   # key defaults to a string (next hop index)
   nextHops = List( valueType=int,
                    help="nexthop local index" )

   def setAttrsFromDictForNextHop( self, data ):
      self.nextHops.append( data )

class AsyncNhLocalIndex( Model ):
   """Used only for testing"""
   nhLocalIndex = Int( help="Next Hop Local Index" )

class AsyncGate( Model ):
   vrfName = Str( help="VRF associated with this interface" )
   outputGate = Int( help="Output gate" )

class AsyncFec( Model ):
   """Used only for testing"""
   fecKey = Int( help="FecKey associated with L2Adj" )
   # List cannot have valueType=int, due to a limitation in the cpp renderer
   nhLocalIndexes = List( valueType=AsyncNhLocalIndex, help="nexthop local indexes" )

class AsyncPrefix( Model ):
   routes = Dict( keyType=str, valueType=AsyncFec, help="Maps prefix to FEC info" )

class McastGates( Model ):
   dataGate = Int( help="ogate used for multicast data packet" )
   cpuGate = Int( help="ogate used for multicast control packet" )

class IpRouteMcastModel( Model ):
   routes = Dict( keyType=str, valueType=McastGates,
                  help="Maps vrf/prefix to its ogate" )
   def render( self ):
      sortedList = sorted( self.routes, key=keyVrf )

      header = ( "VRF", "prefix", "Output Gate" )
      # VRF
      vrfMinLen = len( header[ 0 ] )
      vrfMaxLen = vrfMinLen

      pMinLen = len( header[ 1 ] )
      pMaxLen = pMinLen
      # Output Gate
      oGateMinLen = len( header[ 2 ] )
      oGateMaxLen = max( oGateMinLen, u64MaxSpaces )

      tableMaxWidth = ( vrfMaxLen + pMaxLen + oGateMaxLen )
      table = TableOutput.createTable( header,
                                       tableWidth=tableMaxWidth )
      # VRF
      f1 = createColumnFormat( vrfMinLen, vrfMaxLen )
      # prefix
      f2 = createColumnFormat( pMinLen, pMaxLen )
      # Output Gate
      f3 = createColumnFormat( oGateMinLen, oGateMaxLen )
      table.formatColumns( f1, f2, f3 )

      for key in sortedList:
         mcastGates = self.routes[ key ]
         table.newRow( key, "224.0.0.0/4", mcastGates.dataGate )
         table.newRow( key, "224.0.0.0/24", mcastGates.cpuGate )
      output = table.output()
      print( output )
      print()


class L3AgentAsyncIpRoutes( Model ):
   """Replacement class for async IpRoutes, needed for testing"""
   cmdErr = Str( optional=True, help="Error Message" )
   vrfs = Dict( keyType=str, valueType=AsyncPrefix, help="Maps vrf to prefix" )
   # Key is vrfInterface string
   gates = Dict( keyType=str, valueType=AsyncGate,
                 help='Output Gates keyed by interface string' )

class L3AgentIpRoutes( Model ):
   """Major class for Ipv4 Routes."""

   routes = Dict( keyType=str, valueType=Fec,
                  help="Maps vrf/prefix to its FecKey" )

   fecs = Dict( keyType=int, valueType=FibFecInfo,
                help='Maps a FecKey index to FecInfo' )

   # Key is vrfInterface string
   gates = Dict( keyType=str, valueType=Gates,
                 help='Output Gates keyed by interface string' )

   diffFIB = Bool( help='Diff FIBV4 packet Module with control Module' )

   pModVrfs = Dict( keyType=str, valueType=PModStatusTable,
                    help='pModule VRFS' )

   pModuleFib = Submodel( valueType=ModuleTable,
                          help='packet Module FIBV4 Table Dump' )

   pModuleFec = Submodel( valueType=ModuleTable,
                          help='packet Module FEC Table Diff' )

   cModule = Submodel( valueType=ModuleTable,
                       help='control Module FIBV4 Table' )

   diffModuleFec = Submodel( valueType=ModuleTable,
                             help='FIBV4 (FEC) Table Diff' )

   def render( self ):

      if not self.diffFIB:
         header = ( "VRF", "Prefix", "FEC", "Next Hops" )

         numOfColumns = 4
         # VRF
         vrfMinLen = len( header[ 0 ] )
         vrfMaxLen = vrfMinLen
         # Prefix
         prefixMinLen = len( header[ 1 ] )
         prefixMaxLen = max( prefixMinLen, prefixMaxSpaces )
         # FEC
         fecMinLen = len( header[ 2 ] )
         fecMaxLen = max( fecMinLen, u64MaxSpaces )
         # Next Hops
         nextHopsMinLen = len( header[ 3 ] )
         nextHopsMaxLen = u64MaxSpaces

         sortedList = sorted( self.routes, key=keyIpAndLen )
         sortedList = sorted( sortedList, key=keyVrf )
         for key in sortedList:
            slashPos = key.find( '/' )
            vrfName = key[ 0 : slashPos ]

            vrfMaxLen = max( len( vrfName ), vrfMaxLen )

         tableMaxWidth = ( vrfMaxLen + prefixMaxLen + fecMaxLen +
                           nextHopsMaxLen +
                           numOfColumns - 1 )

         table = TableOutput.createTable( header,
                                          tableWidth=tableMaxWidth )

         # VRF
         f1 = createColumnFormat( vrfMinLen, vrfMaxLen )

         # Prefix
         f2 = createColumnFormat( prefixMinLen, prefixMaxLen )

         # FEC
         f3 = createColumnFormat( fecMinLen, fecMaxLen )

         # Next Hops
         f4 = createColumnFormat( nextHopsMinLen, nextHopsMaxLen )

         table.formatColumns( f1, f2, f3, f4 )

         for key in sortedList:
            slashPos = key.find( '/' )
            vrfName = key[ 0 : slashPos ]
            prefix = key[ slashPos + 1 : ]
            fecKey = self.routes[ key ].fecKey

            nhs = []
            # Walk fecInfo
            for nh in sorted( self.fecs[ fecKey ].fecInfos ):
               nhs.append( str( nh.nhLocalIndex ) )
            nhsStr = ",".join( sorted( nhs ) )
            table.newRow( vrfName,
                          prefix,
                          fecKey,
                          nhsStr )
         output = table.output()
         print( "                  SFE Control Module" )
         print( "                  ==================" )
         print( output )
         print()
         print()

         header = ( "Interface", "VRF", "Output Gate" )

         numOfColumns = 3
         # Interface
         ifMinLen = len( header[ 0 ] )
         ifMaxLen = ifMinLen
         # VRF
         vrfMinLen = len( header[ 1 ] )
         vrfMaxLen = vrfMinLen
         # Output Gate
         oGateMinLen = len( header[ 2 ] )
         oGateMaxLen = max( oGateMinLen, u64MaxSpaces )

         for key in sorted( self.gates ):
            slashPos = key.find( '/' )
            ifMaxLen = max( len( key[ slashPos + 1 : ] ), ifMaxLen )
            vrfMaxLen = max( len( key[ 0 : slashPos ] ), vrfMaxLen )

         tableMaxWidth = ( ifMaxLen + vrfMaxLen + oGateMaxLen +
                           numOfColumns - 1 )

         table = TableOutput.createTable( header,
                                          tableWidth=tableMaxWidth )

         # Interface
         f1 = createColumnFormat( ifMinLen, ifMaxLen )

         # VRF
         f2 = createColumnFormat( vrfMinLen, vrfMaxLen )

         # Output Gate
         f3 = createColumnFormat( oGateMinLen, oGateMaxLen )

         table.formatColumns( f1, f2, f3 )

         for key in sorted( self.gates ):
            slashPos = key.find( '/' )
            vrfName = key[ 0 : slashPos ]
            intf = key[ slashPos + 1 : ]
            gate = self.gates[ key ].outputGate
            table.newRow( intf,
                          vrfName,
                          gate )
         output = table.output()
         print( "                  Gate Info" )
         print( "                  =========" )
         print( output )
         print()
         print()
      else:
         #
         # Four tables
         # cModule, pModuleFib, pModuleFec and diffModuleFec

         # cModule
         if self.cModule:
            header = ( "VRF", "Prefix", "FEC", "Next Hops" )

            numOfColumns = 4
            # VRF
            vrfMinLen = len( header[ 0 ] )
            vrfMaxLen = vrfMinLen
            # Prefix
            prefixMinLen = len( header[ 1 ] )
            prefixMaxLen = max( prefixMinLen, prefixMaxSpaces )
            # FEC
            fecMinLen = len( header[ 2 ] )
            fecMaxLen = max( fecMinLen, u64MaxSpaces )
            # Next Hops
            nextHopsMinLen = len( header[ 3 ] )
            nextHopsMaxLen = u64MaxSpaces

            sortedList = sorted( self.cModule.fibs, key=keyIpPrefixAndLen )
            sortedList = sorted( sortedList, key=lambda fib: fib.vrfName )
            for fib in sortedList:
               vrfMaxLen = max( len( fib.vrfName ), vrfMaxLen )

            tableMaxWidth = ( vrfMaxLen + prefixMaxLen + fecMaxLen +
                              nextHopsMaxLen +
                              numOfColumns - 1 )

            cModTable = TableOutput.createTable( header,
                                                 tableWidth=tableMaxWidth )

            # VRF
            f1 = createColumnFormat( vrfMinLen, vrfMaxLen )

            # Prefix
            f2 = createColumnFormat( prefixMinLen, prefixMaxLen )

            # FEC
            f3 = createColumnFormat( fecMinLen, fecMaxLen )

            # Next Hops
            f4 = createColumnFormat( nextHopsMinLen, nextHopsMaxLen )

            cModTable.formatColumns( f1, f2, f3, f4 )
            for fib in sortedList:
               cModTable.newRow( fib.vrfName,
                                 fib.prefix,
                                 fib.fecKey,
                                 fib.nextHops )

            output = cModTable.output()
            print( "                  SFE Control Module" )
            print( "                  ==================" )
            print( output )
            print()

         # pModuleFib
         if self.pModuleFib:
            header = ( "VRF", "Prefix/Depth", "FEC" )

            numOfColumns = 3
            # VRF
            vrfMinLen = len( header[ 0 ] )
            vrfMaxLen = vrfMinLen
            # Prefix
            prefixMinLen = len( header[ 1 ] )
            prefixMaxLen = max( prefixMinLen, prefixMaxSpaces )
            # FEC
            fecMinLen = len( header[ 2 ] )
            fecMaxLen = max( fecMinLen, u64MaxSpaces )

            sortedList = sorted( self.pModuleFib.fibs, key=keyIpPrefixAndLen )
            sortedList = sorted( sortedList, key=lambda fib: fib.vrfName )
            for fib in sortedList:
               vrfMaxLen = max( len( fib.vrfName ), vrfMaxLen )

            tableMaxWidth = ( vrfMaxLen + prefixMaxLen + fecMaxLen +
                              numOfColumns - 1 )

            pModFibTable = TableOutput.createTable( header,
                                                    tableWidth=tableMaxWidth )

            # VRF
            f1 = createColumnFormat( vrfMinLen, vrfMaxLen )

            # Prefix
            f2 = createColumnFormat( prefixMinLen, prefixMaxLen )

            # FEC
            f3 = createColumnFormat( fecMinLen, fecMaxLen )

            pModFibTable.formatColumns( f1, f2, f3 )

            for fib in sortedList:
               pModFibTable.newRow( fib.vrfName,
                                    fib.prefix,
                                    fib.fecKey )
            output = pModFibTable.output()
            print( "      SFE Packet Module (FIB)" )
            print( "      =======================" )
            print( output )
            print()

         # pModuleFec
         if self.pModuleFec:
            header = ( "FEC", "Next Hop", "Gate" )

            numOfColumns = 3
            # FEC
            fecMinLen = len( header[ 0 ] )
            fecMaxLen = max( fecMinLen, u64MaxSpaces )
            # Next Hop
            nextHopMinLen = len( header[ 1 ] )
            nextHopMaxLen = max( nextHopMinLen, u64MaxSpaces )
            # Gate
            gateMinLen = len( header[ 2 ] )
            gateMaxLen = max( gateMinLen, u64MaxSpaces )

            tableMaxWidth = ( fecMaxLen + nextHopMaxLen + gateMaxLen +
                              numOfColumns - 1 )

            pModFecTable = TableOutput.createTable( header,
                                                    tableWidth=tableMaxWidth )

            # FEC
            f1 = createColumnFormat( fecMinLen, fecMaxLen )

            # Next Hop
            f2 = createColumnFormat( nextHopMinLen, nextHopMaxLen )

            # Gate
            f3 = createColumnFormat( gateMinLen, gateMaxLen )

            pModFecTable.formatColumns( f1, f2, f3 )

            for key in sorted( self.pModuleFec.fecIndexes ):
               for nh in sorted( self.pModuleFec.fecIndexes[ key ].fecInfos ):
                  pModFecTable.newRow( str( key ),
                                       str( nh.nhLocalIndex ),
                                       str( nh.gate ) )
            output = pModFecTable.output()
            print( "    SFE Packet Module (FEC)" )
            print( "    =======================" )
            print( output )
            print()

         # diffModuleFec
         if self.diffModuleFec:
            header = ( "FEC", "Next Hop" )

            numOfColumns = 2
            # FEC
            fecMinLen = len( header[ 0 ] )
            fecMaxLen = max( fecMinLen, u64MaxSpaces )
            # Next Hop
            nextHopMinLen = len( header[ 1 ] )
            nextHopMaxLen = max( nextHopMinLen, u64MaxSpaces )

            tableMaxWidth = ( fecMaxLen + nextHopMaxLen +
                              numOfColumns - 1 )

            diffModFecTable = TableOutput.createTable( header,
                                                       tableWidth=tableMaxWidth )

            # FEC
            f1 = createColumnFormat( fecMinLen, fecMaxLen )

            # Next Hop
            f2 = createColumnFormat( nextHopMinLen, nextHopMaxLen )

            diffModFecTable.formatColumns( f1, f2 )

            for key in sorted( self.diffModuleFec.fecIndexes ):
               for nh in sorted( self.diffModuleFec.fecIndexes[ key ].fecInfos ):
                  diffModFecTable.newRow( str( key ),
                                          str( nh.nhLocalIndex ) )
            output = diffModFecTable.output()

            print( "Difference between pModule and cModule (FEC)" )
            print( "============================================" )
            print( output )
            print()

         #
         # Dump the pModule Vrf list
         #
         if self.pModVrfs:
            header = ( "Vrf", "Status" )

            numOfColumns = 2
            # VRF
            vrfMinLen = len( header[ 0 ] )
            vrfMaxLen = vrfMinLen
            # Status
            statusMinLen = len( header[ 1 ] )
            statusMaxLen = statusMinLen

            for k, v in sorted( self.pModVrfs.items() ):
               vrfMaxLen = max( len( k ), vrfMaxLen )
               statusString = "Active" if v.activeStatus else "Missing"
               statusMaxLen = max( len( statusString ), statusMaxLen )

            tableMaxWidth = ( vrfMaxLen + statusMaxLen +
                              numOfColumns - 1 )

            pModVrfTable = TableOutput.createTable( header,
                                                    tableWidth=tableMaxWidth )

            # VRF
            f1 = createColumnFormat( vrfMinLen, vrfMaxLen )

            # Status
            f2 = createColumnFormat( statusMinLen, statusMaxLen )

            pModVrfTable.formatColumns( f1, f2 )

            for k, v in sorted( self.pModVrfs.items() ):
               statusString = "Active" if v.activeStatus else "Missing"
               pModVrfTable.newRow( k, statusString )

            output = pModVrfTable.output()
            print( "  pModule VRF(s)" )
            print( "  ==============" )
            print( output )
            print()

   def initTables( self ):
      self.cModule = ModuleTable()
      self.pModuleFib = ModuleTable()
      self.pModuleFec = ModuleTable()
      self.diffModuleFec = ModuleTable()

   def diffTables( self ):

      tableOneSet = set() # cModule
      tableOneTrimSet = set() # cModule

      sortedList = sorted( self.routes, key=keyIpAndLen )
      sortedList = sorted( sortedList, key=keyVrf )
      for key in sortedList:

         slashPos = key.find( '/' )
         vrfName = key[ 0 : slashPos ]
         prefix = key[ slashPos + 1 : ]
         fecKey = self.routes[ key ].fecKey

         nhs = []
         # Walk fecInfo
         for nh in sorted( self.fecs[ fecKey ].fecInfos ):
            nhs.append( str( nh.nhLocalIndex ) )
         nhsStr = ",".join( sorted( nhs ) )
         tupleEntry = ( prefix, fecKey, vrfName, nhsStr )
         tableOneSet.add( tupleEntry )

         # TRIM
         for nh in sorted( self.fecs[ fecKey ].fecInfos ):
            trimTupleEntry = ( str( fecKey ),
                               str( nh.nhLocalIndex ) )
            tableOneTrimSet.add( trimTupleEntry )

      # Populate cModule with all data
      if tableOneSet:
         self.cModule = ModuleTable()
         for value in tableOneSet:
            self.cModule.setFibAttrsFromTuple( value )

      # We must do a symmetric diff on the two tables

      if self.pModuleFec:
         tableTwoFecSet = set()

         for key in sorted( self.pModuleFec.fecIndexes ):
            for nh in sorted( self.pModuleFec.fecIndexes[ key ].fecInfos ):
               tupleEntry = ( str( key ),
                              str( nh.nhLocalIndex ) )
               tableTwoFecSet.add( tupleEntry ) # Add tuple to set

         symmetric_diff = tableOneTrimSet ^ tableTwoFecSet

         # Convert back into the model List
         if symmetric_diff:
            self.diffModuleFec = ModuleTable()

            for value in symmetric_diff:
               self.diffModuleFec.setFecAttrsFromDiff( value ) # Add to diff table

   def setAttrsFromDict( self, data ):
      tableTwoFec = [] # pModule
      tableTwoFib = set() # pModule
      for key in data:
         if key == 'route':
            routeInfo = data[ key ]
            for aleVrfKey, value in routeInfo.items():
               self.routes[ aleVrfKey ] = Fec()
               self.routes[ aleVrfKey ].setAttr( value )
         elif key == 'fec':
            fecInfo = data[ key ]
            for fecKey, value in fecInfo.items():
               self.fecs[ int( fecKey ) ] = FibFecInfo()
               self.fecs[ int( fecKey ) ].setAttrsFromDict( value )
         elif key == 'gate':
            gateInfo = data[ key ]
            for vrfIntfKey, value in gateInfo.items():
               self.gates[ vrfIntfKey ] = Gates()
               self.gates[ vrfIntfKey ].setOutputGate( value )
         elif key == 'pModVrf':
            pModVrf = data[ key ]
            for vrf, value in pModVrf.items():
               self.pModVrfs[ vrf ] = PModStatusTable()
               self.pModVrfs[ vrf ].setAttrsFromDict( value )
         elif key.startswith( 'pModuleFIB' ):
            pModTupl = data[ key ]
            prefixes = []
            vrfs = []
            fecs = []

            for prefix in pModTupl[ : : 2 ]:
               prefixes.append( prefix )
            for value in pModTupl[ 1 : : 2 ]:
               vrfs.append( value[ 0 ] )
               fecs.append( value[ 1 ] )

            #
            # TableTwoFib:
            # Will be a set of FIB info used for a dump
            # of pModule FIB state and not a diff.
            # Diff is hard to do because the FIB
            # info cannot be rebuilt because of prefix len is
            # not easily reconstructed.
            #
            for index, prefix in enumerate( prefixes ):
               tableTwoFib.add( ( prefix, fecs[ index ], vrfs[ index ],
                                     "" ) )

         elif key.startswith( 'pModuleFEC' ):
            pModDict = data[ key ]

            # Test data, standard and ecmp
            # {'11': ( 10, 0 ), '1': ( 0, 1 ), '13': ( 11, 0 )}
            # ECMP
            # {'12': ( 1, 2 ), '14': ( 12, 0, 13, 1, 14, 2, 15, 3 ) }
            #pModDict = {'12': ( 1, 2 ), '14': ( 12, 0, 13, 1, 14, 2, 15, 3 ) }

            iterCount = 0
            for fec, v in pModDict.items():
               iterCount = iterCount + 1
               nhCount = len( v ) // 2
               updatedVal = "(" + str( fec ) + \
                            ", {'" + str( fec ) + "':( " + str( nhCount ) + ", "
               count = len( v )
               for i in range( 0, count, 2 ):
                  updatedVal += "{ '" + str( iterCount ) + \
                                "': { 'nhLocalIndex':" + \
                                str( v[ i ] ) + \
                                ", 'gate':" + str( v[ i + 1 ] ) + "} }, "
               updatedVal += "),})"
               flyTupl = ast.literal_eval( updatedVal )
               tableTwoFec.append( flyTupl )
         else:
            setattr( self, key, data[ key ] )

      # Populate pModule with all data
      if tableTwoFib:
         self.pModuleFib = ModuleTable()
         for value in tableTwoFib:
            self.pModuleFib.setFibAttrsFromTuple( value ) # send a tuple

      if tableTwoFec:
         self.pModuleFec = ModuleTable()
         for value in tableTwoFec:
            self.pModuleFec.setFecAttrsFromTuple( value ) # send a tuple
