#!/usr/bin/env python3
# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import Model, Int, Dict, Bool, Enum, List, Str
from ArnetModel import Ip4Address, MacAddress
from IntfModels import Interface
import TableOutput

fl = TableOutput.Format( justify='left' )
fl.padLimitIs( True )
fr = TableOutput.Format( justify='right' )
fr.padLimitIs( True )

class SfeL3UnicastTunnelIvifEntry( Model ):
   hwIndex = Int( help="Tunnel Ivif table index" )
   tunnelInfoIndex = Int( help="Tunnel-info table hardware index" )
   setIngressVif = Bool( help="set ingress vif in the token" )
   macSaMissCmd =  Enum(
      values=( "forward", "drop", "trap", "forwardAndMirror" ),
      help="Action taken on MAC SA miss" )

   def myRender( self, table, index ):
      table.newRow( index, self.hwIndex, self.tunnelInfoIndex,
                    self.setIngressVif, self.macSaMissCmd )

class SfeL3UnicastTunnelIvifModel( Model ):
   remoteVteps = Dict( keyType=Ip4Address, valueType=SfeL3UnicastTunnelIvifEntry,
                       help="Tunnel Ivif table entries" )

   def setAttrsFromDict( self, data ):
      for key, value in data[ 'tunnelIvifEntries' ].items():
         model = SfeL3UnicastTunnelIvifEntry()
         model.setAttrsFromDict( value )
         self.remoteVteps[ key ] = model

   def render( self ):
      header = [ "VtepIp", "HwIndex", "TunnelInfoIndex",
                 "SetIngressVif", "MacSaMissCmd"  ]
      table = TableOutput.createTable( header )
      table.formatColumns( fl, fr, fr, fl, fl )
      for key, entry in self.remoteVteps.items():
         entry.myRender( table, key )

      print( table.output() )

class SfeL3UnicastLocalVtepEntry( Model ):
   isVarpVtepAddr = Bool( help="VARP VTEP IP Address" )
   hwIndex = Int( help="Tunnel Info table index" )
   localVtepIpOut = \
         Ip4Address( help="Outer Source IP Address in vxlan encapsulated packets" )

   def myRender( self, table, index ):
      table.newRow( index, self.isVarpVtepAddr, self.hwIndex,
                    self.localVtepIpOut )

class SfeL3UnicastLocalVtepModel( Model ):
   localVteps = Dict( keyType=Ip4Address, valueType=SfeL3UnicastLocalVtepEntry,
                      help="Tunnel Local Vtep table entries" )

   def setAttrsFromDict( self, data ):
      for key, value in data[ 'localVtepEntries' ].items():
         model = SfeL3UnicastLocalVtepEntry()
         model.setAttrsFromDict( value )
         self.localVteps[ key ] = model

   def render( self ):
      header = [ "VtepIp", "IsVarpVtepAddr", "HwIndex", "EncapSourceVtepIp" ]
      table = TableOutput.createTable( header )
      table.formatColumns( fl, fr, fr )
      for key, entry in self.localVteps.items():
         entry.myRender( table, key )

      print( table.output() )

class SfeL3UnicastVxlanUdpPortModel( Model ):
   vxlanUdpPortIn = Int( help="Ingress Vxlan UDP Port" )
   vxlanUdpPortOut = Int( help="Egress Vxlan UDP Port" )

   def render( self ):
      print( f" Vxlan UDP Port for Decapsulation: {self.vxlanUdpPortIn} " )
      print( f" Vxlan UDP Port for Encapsulation: {self.vxlanUdpPortOut} " )

class SfeL3UnicastRemoteVtepEvif( Model ):
   interfaces = List( valueType=Interface, help="Interfaces in Egress Port-set" )
   macAddress = MacAddress( help="MAC address" )
   vlanId = Int( help="VLAN ID" )

   def myRender( self, index ):
      return index, self.macAddress, self.vlanId, ', '.join( self.interfaces )

class SfeL3UnicastTunnelInfo( Model ):
   hwIndex = Int( help="Tunnel Info table index" )
   vtepIp = Ip4Address( help="Vtep IP address" )
   ecmpSize = Int( help="ECMP size" )
   pktCmd = Enum(
      values=( "forward", "drop", "trap", "forwardAndMirror" ),
      help="Action taken on tunnel info table hit" )
   egressVifs = Dict( keyType=int, valueType=SfeL3UnicastRemoteVtepEvif,
                      help="Egress VIF entries" )

   def setAttrsFromDict( self, data ):
      for key in data:
         if key == 'egressVifs':
            for vif, value in data[ 'egressVifs' ].items():
               model = SfeL3UnicastRemoteVtepEvif()
               model.setAttrsFromDict( value )
               self.egressVifs[ vif ] = model
         else:
            setattr( self, key, data[key] )

   def myRender( self, table=None ): # pylint: disable=inconsistent-return-statements
      if table:
         firstRow = True
         for vif, value in self.egressVifs.items():
            if firstRow:
               table.newRow( self.vtepIp, self.hwIndex, self.ecmpSize, self.pktCmd,
                             *value.myRender( vif ) )
               firstRow = False
            else:
               table.newRow( '', '', '', '', *value.myRender( vif ) )
      else:
         interfaces = []
         for vif, value in self.egressVifs.items():
            interfaces += value[ 'interfaces' ]
         return 'remoteVtep', self.vtepIp, self.hwIndex, interfaces

class SfeL3UnicastRemoteVtepModel( Model ):
   __revision__ = 2
   remoteVteps = Dict( keyType=Ip4Address, valueType=SfeL3UnicastTunnelInfo,
                       help="Remote vtep entries" )

   def setAttrsFromDict( self, data ):
      for key, value in data[ 'vteps' ].items():
         model = SfeL3UnicastTunnelInfo()
         model.setAttrsFromDict( value )
         self.remoteVteps[ key ] = model

   def render( self ):
      header = [ "VtepIp", "HwIndex", "EcmpSize", "PktCmd", "EgressVif",
                 "MacAddress", "VlanId", "Interface" ]
      table = TableOutput.createTable( header )
      fIntf = TableOutput.Format( justify='left', maxWidth=35, wrap=True )
      fIntf.padLimitIs( True )
      table.formatColumns( fl, fr, fr, fl, fr, fl, fr, fIntf )
      for entry in self.remoteVteps.values():
         entry.myRender( table )

      print( table.output() )

   def degrade( self, dictRepr, revision ):
      if revision == 1:
         for value in dictRepr[ 'remoteVteps' ].values():
            del value[ 'vtepIp' ]
      return dictRepr

class SfeVxlanSummaryModel( Model ):
   numRemoteMacLearntEvent = Int( help="Remote MAC learned event count" )
   numRemoteMacMoveEvent = Int( help="Remote MAC move event count" )
   numRemoteMacAgeEvent = Int( help="Remote MAC age event count" )
   numRemoteVtep = Int( help="Remote VTEPs count" )
   numLocalVtep = Int( help="Local VTEPs count" )
   numRemoteVtepAddEvent = Int( help="Count discovered VTEPs" )
   numRemoteVtepDeleteEvent = Int( help="Count deleted VTEPs" )
   numVxlanVlan = Int( help="VXLAN VLAN count" )
   numRemoteMac = Int( help="Remote MACs count" )

   def render( self ):
      formatString = " %-45s%-10d"
      print( formatString % ( 'Number of Remote MAC Learnt Events',
                             self.numRemoteMacLearntEvent ) )
      print( formatString % ( 'Number of Remote MAC Move Events',
                             self.numRemoteMacMoveEvent ) )
      print( formatString % ( 'Number of Remote MAC Age Events',
                             self.numRemoteMacAgeEvent ) )
      print( formatString % ( 'Number of Remote VTEPs', self.numRemoteVtep ) )
      print( formatString % ( 'Number of Local VTEPs ', self.numLocalVtep ) )
      print( formatString % ( 'Number of Remote VTEP Add Events',
                             self.numRemoteVtepAddEvent ) )
      print( formatString % ( 'Number of Remote VTEP Delete Events',
                             self.numRemoteVtepDeleteEvent ) )
      print( formatString % ( 'Number of VXLAN VLANs', self.numVxlanVlan ) )
      print( formatString % ( 'Number of Remote MACs', self.numRemoteMac ) )
      print()

class TunnelMdtModel( Model ):
   tunnelInfoDetails = Dict( keyType=int, valueType=SfeL3UnicastTunnelInfo,
                             help="Remote vtep entries" )

   def setAttrsFromDict( self, data ):
      for key, entry in data[ 'tunnelInfo' ].items():
         if key != 'mdtType':
            model = SfeL3UnicastTunnelInfo()
            model.setAttrsFromDict( entry )
            self.tunnelInfoDetails[ int( key ) ] = model

   def myRender( self ):
      for entry in self.tunnelInfoDetails.values():
         return  entry.myRender( )

class BridgeMdtModel( Model ):
   mdtType = Enum( values=( "localUnTagged", "localTagged" ),
                   help="Bridge MDT entry type" )
   egressVif = Int( help="Egress vif associated with MDT" )
   interfaces = List( valueType=Interface,
                      help="Ports associated with eVifs of MDT" )

   def myRender( self ):
      return self.mdtType, '-', self.egressVif, self.interfaces

class SfeMcastMdtCollModel( Model ):
   bridgeMdts = Dict( keyType=int, valueType=BridgeMdtModel,
                      help="Mapping between Bridge MDT index and MDT details",
                      optional=True )
   tunnelMdts = Dict( keyType=int, valueType=TunnelMdtModel,
                      help="Mapping between tunnel MDT index and MDT details",
                      optional=True )

   def setAttrsFromDict( self, data ):
      for key, value in data.items():
         if data[key]['mdtType'] == 'remoteVtep':
            self.tunnelMdts[ int( key ) ] = TunnelMdtModel()
            self.tunnelMdts[ int( key ) ].setAttrsFromDict( value )
         else:
            self.bridgeMdts[ int( key ) ] = BridgeMdtModel()
            self.bridgeMdts[ int( key ) ].setAttrsFromDict( value )

   def myRender( self, table, vlan ):
      bridgeMdtList = list( self.bridgeMdts.items() )
      tunnelMdtList = list( self.tunnelMdts.items() )
      firstEntry = []
      mdtList = []
      if self.bridgeMdts:
         firstEntry = bridgeMdtList[0]
         mdtList = bridgeMdtList[1:] + tunnelMdtList[0:]
      else:
         firstEntry = self.tunnelMdtList[0]
         mdtList = self.tunnelMdtList[1:]
      mdtIndex, mdtEntry = firstEntry
      mdtInfo = mdtEntry.myRender()
      table.newRow( vlan, len( self.bridgeMdts ) + len ( self.tunnelMdts ) ,
                    mdtInfo[0], mdtInfo[1], mdtIndex, mdtInfo[2],
                    ', '.join( mdtInfo[3] ) )
      for mdtIndex, mdtEntry in mdtList:
         mdtInfo = mdtEntry.myRender()
         table.newRow( '', '', mdtInfo[0], mdtInfo[1], mdtIndex, mdtInfo[2],
                       ', '.join( mdtInfo[3] ) )

class SfeMcastVlanFloodVifInfoModel( Model ):
   cmdErr = Str( optional=True, help="Error flag" )
   vlans = Dict( keyType=int, valueType=SfeMcastMdtCollModel, optional=True,
                         help="Mapping between vlan and MDTs" )

   def setAttrsFromDict( self, data ):
      for key, value in data[ 'vlanfloodVifInfo' ].items():
         if  key == 'cmdErr':
            setattr( self, key, value )
         else:
            self.vlans[ int( key ) ] = SfeMcastMdtCollModel()
            self.vlans[ int( key ) ].setAttrsFromDict( value )

   def render( self ):
      if self.cmdErr:
         print( f'{self.cmdErr}' )
      else:
         header = [ "Vlan", "NumOfMdt", "MdtTypes", "TunnelIp",
                    "MdtIndex", "TunInfoId/Evif", "Ports" ]
         table = TableOutput.createTable( header )
         fPortList = TableOutput.Format( justify='left', maxWidth=15, wrap=True )
         fPortList.noPadLeftIs( True )
         table.formatColumns( fr, fr, fl, fl, fr, fr,
                              fPortList )
         for key, entry in self.vlans.items():
            entry.myRender( table, key )
         print( table.output() )
