#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import Tracing
from CliPlugin import SfeMcastShowCliModel
from CliPlugin.SfeCliLib import nodeSfeEos
from CliPlugin.VrfCli import DEFAULT_VRF, VrfExprFactory, nsFromVrf
import BasicCli
import ShowCommand
import CliMatcher
import CliToken.Platform
import SmashLazyMount

traceHandle = Tracing.Handle( "SfeMcastShowCli" )
t0 = traceHandle.trace0

class ShowMroute( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform sfe mroute AF [ VRF ]'

   data = {
      'platform' : CliToken.Platform.platformMatcherForShow,
      'sfe' : nodeSfeEos,
      'mroute' : 'multicast route information',
      'AF' : CliMatcher.EnumMatcher( { "ipv4" : "IPV4",
                                       "ipv6" : "IPV6" } ),
      'VRF' : VrfExprFactory( helpdesc='Use a specific VRF' ),
   }
   cliModel = SfeMcastShowCliModel.MfibMrouteGroups

   @staticmethod
   def handler( mode, args ):
      vrfName = args.get( 'VRF', DEFAULT_VRF )
      af = args.get( 'addressFamily', 'ipv4' )
      model = SfeMcastShowCliModel.MfibMrouteGroups()

      ns = nsFromVrf( vrfName )
      # Namespace required for BessProc file name
      if not ns:
         mode.addWarning( f"VRF name {vrfName} is not defined." )
         return model

      try:
         model.fromBessProc( af, ns )
      except ( OSError, Tac.SystemCommandError ):
         mode.addWarning( f"BESS mroute data for VRF {vrfName} "
                           "in IPv{af[ -1 ]} is not available." )
         return SfeMcastShowCliModel.MfibMrouteGroups()

      return model

BasicCli.addShowCommandClass( ShowMroute )

def showSfeMcastIntfCounter( mode, args ):
   vrfName = args.get( 'VRF', DEFAULT_VRF )
   af = args.get( 'AF', 'ipv4' )
   path = Tac.Type( 'Sfe::Multicast::IntfCounterTable' ).mountPath( af, vrfName )
   intfCounterTable = SmashLazyMount.mount(
      mode.entityManager, path, 'Sfe::Multicast::IntfCounterTable',
      SmashLazyMount.mountInfo( 'reader' ) )

   intfCounterModel = SfeMcastShowCliModel.McastIntfCounters()
   intfCounterModel.fromTacc( vrfName, intfCounterTable )
   return intfCounterModel

class ShowMcastIntfCounterCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform sfe multicast interface counter AF [ VRF ]'

   data = {
      'platform' : CliToken.Platform.platformMatcherForShow,
      'sfe' : nodeSfeEos,
      'multicast' : 'multicast information',
      'interface' : 'multicast interface',
      'counter' : 'interface counter',
      'AF' : CliMatcher.EnumMatcher( { "ipv4" : "IPV4",
                                       "ipv6" : "IPV6" } ),
      'VRF' : VrfExprFactory( helpdesc='Use a specific VRF' ),
   }
   handler = showSfeMcastIntfCounter
   cliModel = SfeMcastShowCliModel.McastIntfCounters

BasicCli.addShowCommandClass( ShowMcastIntfCounterCmd )

