#!/usr/bin/env python3
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import AgentCommandRequest
import CliPlugin.SfeMirroringShowCliModel as MirroringModel
from CliPlugin import SfeCliLib
from io import StringIO
import SfeAgent
from CliToken.Platform import platformMatcherForShow
import ShowCommand
from ast import literal_eval

#---------------------------------------------------------------------------------
# show platform sfe mirroring
#---------------------------------------------------------------------------------
def doShowSfeMirroring( mode, args ):
   buff = StringIO()
   AgentCommandRequest.runSocketCommand( mode.entityManager, SfeAgent.name(),
                                         "sfe", "MIRROR", stringBuff=buff )
   model = MirroringModel.SfeMirrorSessions()
   output = buff.getvalue()
   if not output:
      return model

   try:
      mirrorDict = literal_eval( output )
   except SyntaxError:
      mode.addError( output )
      return model

   for data in mirrorDict:
      sessionName = data[ 'Session' ]
      sessionModel = model.MirrorSession()
      sessionModel.mirrorGroup = data[ 'mirrorGroup' ]
      sessionModel.globalId = data[ 'globalId' ]

      # populate source interface and direction
      for srcIntfAndDir in data[ 'srcIntf' ]:
         srcIntfModel = sessionModel.SourceIntf()
         srcIntfModel.direction = srcIntfAndDir[ 1 ]
         sessionModel.srcIntfs[ srcIntfAndDir[ 0 ] ] = srcIntfModel

      # populate target interface and encapType
      for tgtIntfAndEncap in data[ 'tgtIntf' ]:
         dstIntfModel = sessionModel.TargetIntf()
         dstIntfModel.encapType = tgtIntfAndEncap[ 1 ]
         sessionModel.tgtIntfs[ tgtIntfAndEncap[ 0 ] ] = dstIntfModel

      # populate Sfe module
      for sfeModuleInfo in data[ 'Sfe module' ]:
         moduleName = sfeModuleInfo[ 0 ]
         sfeModuleModel = sessionModel.SfeModule()

         srcIntfInfo = sfeModuleInfo[ 1 ][ 0 ] # only 1 source
         sfeSourceIntfModel = sfeModuleModel.SfeSourceIntf()
         sfeSourceIntfModel.direction = srcIntfInfo[ 1 ]
         sfeSourceIntfModel.gate = srcIntfInfo[ 2 ]
         sfeModuleModel.sfeSourceIntfs[ srcIntfInfo[ 0 ] ] = sfeSourceIntfModel

         for tgtIntfInfo in sfeModuleInfo[ 2 ]:
            sfeTargetIntfModel = sfeModuleModel.SfeTargetIntf()
            sfeTargetIntfModel.gate = tgtIntfInfo[ 1 ]
            sfeModuleModel.sfeTargetIntfs[ tgtIntfInfo[ 0 ] ] = sfeTargetIntfModel

         for i in range( 3, 5 ):
            cntr = sfeModuleInfo[ i ]
            sfeCounterModel = sfeModuleModel.SfeSuccessFailCounter()
            sfeCounterModel.success = cntr[ 1 ]
            sfeCounterModel.fail = cntr[ 2 ]
            sfeModuleModel.sfeCounters[ cntr[ 0 ] ] = sfeCounterModel

         sessionModel.sfeModules[ moduleName ] = sfeModuleModel

      model.sessions[ sessionName ] = sessionModel
   return model

class PlatformSfeMirroringCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform sfe mirroring'
   data = {
      'platform' : platformMatcherForShow,
      'sfe' : SfeCliLib.nodeSfe,
      'mirroring' : 'Show mirroring table information',
   }

   handler = doShowSfeMirroring
   cliModel = MirroringModel.SfeMirrorSessions
   privileged = True

BasicCli.addShowCommandClass( PlatformSfeMirroringCmd )
