# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
from CliToken.Platform import platformMatcherForShow
from CliPlugin import VrfCli
from CliPlugin import RouteCacheCliLib
from CliPlugin import SfeCliLib
from CliPlugin import SfeRouteCacheCliModels
import ShowCommand
from collections import defaultdict
import AgentCommandRequest
import CliModel
import SfeAgent

# show platform sfe route-cache ip hitbit summary
class ShowSfeRouteCacheHitbitSummary(
      ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform sfe route-cache ip hitbit summary'
   data = {
      'platform' : platformMatcherForShow,
      'sfe' : SfeCliLib.nodeSfe,
      'route-cache' : RouteCacheCliLib.routeCacheKw,
      'ip' : 'IPv4 information',
      'hitbit' : RouteCacheCliLib.hitbitKw,
      'summary' : 'Hitbit summary',
   }
   cliModel = SfeRouteCacheCliModels.SfeRouteCacheHitbitSummary

   @staticmethod
   def handler( mode, args ):
      model = SfeRouteCacheCliModels.SfeRouteCacheHitbitSummary()

      if SfeCliLib.v4RouteCacheHitbitStatus is None:
         return model

      # Iterate over the hitbit status and consolidate per VRF
      hitbitStatus = SfeCliLib.v4RouteCacheHitbitStatus
      vrfHitbitSummary = defaultdict( int )
      for entry in hitbitStatus.routeHitBit.values():
         vrfHitbitSummary[ entry.vrfId ] += 1

      # Fill the model
      for vrf in RouteCacheCliLib.routingHwRouteStatus.vrfStatus.values():
         model.vrfs[ vrf.vrfName ] = vrfHitbitSummary.get( vrf.vrfId, 0 )

      return model

BasicCli.addShowCommandClass( ShowSfeRouteCacheHitbitSummary )

# show platform sfe route-cache ip hitbit VRF
class ShowSfeRouteCacheHitbitVrf(
      ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform sfe route-cache ip hitbit VRF'
   data = {
      'platform' : platformMatcherForShow,
      'sfe' : SfeCliLib.nodeSfe,
      'route-cache' : RouteCacheCliLib.routeCacheKw,
      'ip' : 'IPv4 information',
      'hitbit' : RouteCacheCliLib.hitbitKw,
      'VRF' : VrfCli.VrfExprFactory( helpdesc='VRF specific information' ),
   }
   cliModel = SfeRouteCacheCliModels.SfeRouteCacheHitbitVrf

   @staticmethod
   def handler( mode, args ):
      model = SfeRouteCacheCliModels.SfeRouteCacheHitbitVrf()

      if SfeCliLib.v4RouteCacheHitbitStatus is None:
         return model

      vrfName = args.get( 'VRF' )
      vrfStatus = RouteCacheCliLib.routingHwRouteStatus.vrfStatus.get( vrfName )

      if vrfStatus is None:
         mode.addError( "Invalid VRF name " )
         return None

      # Iterate over the hitbit status and consolidate per VRF
      hitbitStatus = SfeCliLib.v4RouteCacheHitbitStatus
      for entry in hitbitStatus.routeHitBit.values():
         if entry.vrfId != vrfStatus.vrfId:
            continue
         hitbitEntry = SfeRouteCacheCliModels.HitbitEntry()
         hitbitEntry.prefix = entry.key.prefix.stringValue
         hitbitEntry.hitbit = entry.hitbit
         model.hitbitEntries.append( hitbitEntry )

      return model

BasicCli.addShowCommandClass( ShowSfeRouteCacheHitbitVrf )

# show platform sfe route-cache summary
class ShowSfeRouteCacheSummary(
      ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform sfe route-cache summary'
   data = {
      'platform' : platformMatcherForShow,
      'sfe' : SfeCliLib.nodeSfe,
      'route-cache' : RouteCacheCliLib.routeCacheKw,
      'summary' : 'Route cache summary',
   }
   cliModel = SfeRouteCacheCliModels.SfeRouteCacheSummary

   @staticmethod
   def handler( mode, args ):
      outputFormat = mode.session_.outputFormat_
      try:
         AgentCommandRequest.runSocketCommand( mode.entityManager, SfeAgent.name(),
                                            "SfeCliWithFormat",
                                            "showRouteCacheSummary",
                                            stringBuff=None,
                                            forceBuffering=True,
                                            outputFormat=outputFormat,
                                            throwException=True,
                                            keepalive=True,
                                            errors=[] )
         return CliModel.cliPrinted( SfeRouteCacheCliModels.SfeRouteCacheSummary )
      except AgentCommandRequest.RunSocketCommandException as e:
         mode.addError( str( e ) )
         return None

BasicCli.addShowCommandClass( ShowSfeRouteCacheSummary )
