# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
from TypeFuture import TacLazyType

IntfId = TacLazyType( 'Arnet::IntfId' )
IpGenAddr = TacLazyType( 'Arnet::IpGenAddr' )
ConnectionInterfaceKey = TacLazyType( 'SfeServiceInsertion::ConnectionInterfaceKey' )

class SiConnectionConfigContext:
   def __init__( self, config, mode, connName ):
      self.config = config
      self.mode = mode
      self._connName = connName
      self._connection = None

   def isNextHopValid( self, nextHop ):
      return not nextHop.isAddrZero

   def connectionName( self ):
      return self._connName

   def connectionConfig( self ):
      return self._connection

   def addConnectionConfig( self, connName ):
      connection = self.config.newConnectionConfig( connName )
      self._connection = connection

   def delConnectionConfig( self, connName ):
      del self.config.connectionConfig[ connName ]
      self._connection = None

   def setConnectivityMonitor( self, hostName ):
      self._connection.monConnectivityHostName = hostName

   def addConnectionEthInterface( self, intfId, nextHop ):
      if not self.isNextHopValid( nextHop ):
         self.mode.addError( f'{nextHop.v4Addr} is an invalid nexthop address' )
         return
      self._connection.primaryIntf = ConnectionInterfaceKey( intfId, nextHop )
      self._connection.secondaryIntf = ConnectionInterfaceKey(
                                            IntfId( '' ), IpGenAddr() )

   def addConnectionTunnelInterface( self, intfId, isPrimary ):
      intfKey = ConnectionInterfaceKey( intfId, IpGenAddr() )
      if isPrimary:
         self._connection.primaryIntf = intfKey
      else:
         # reset any valid nat configuration if present before setting the secondary
         if self.isNextHopValid( self._connection.primaryIntf.nextHop ):
            self._connection.primaryIntf = ConnectionInterfaceKey(
                                                IntfId( '' ), IpGenAddr() )
         self._connection.secondaryIntf = intfKey

   def delConnectionInterface( self, intfId ):
      defIntfKey = ConnectionInterfaceKey( IntfId( '' ), IpGenAddr() )
      if self._connection.primaryIntf.intfId == intfId.stringValue:
         self._connection.primaryIntf = defIntfKey
      if self._connection.secondaryIntf.intfId == intfId.stringValue:
         self._connection.secondaryIntf = defIntfKey

class SiServiceGroupConfigContext:
   def __init__( self, config, mode, groupName ):
      self.config = config
      self.mode = mode
      self._groupName = groupName
      if groupName in self.config.serviceGroupConfig:
         self._group = self.config.serviceGroupConfig[ groupName ]
      else:
         self._group = None

   def getGroupConfig( self ):
      return self._group

   def groupName( self ):
      return self._groupName

   def groupId( self ):
      return self._group.groupId

   def addGroupConfig( self, groupName, groupId ):
      group = self.config.newServiceGroupConfig( groupName, groupId )
      assert group
      self._group = group

   def delGroupConfig( self, groupName ):
      del self.config.serviceGroupConfig[ groupName ]
      self._group = None

   def setServiceGroupType( self, groupType ):
      self._group.type = groupType

class SiServiceGroupInstanceConfigContext:
   def __init__( self, groupConfig, mode, instanceName ):
      self.groupConfig = groupConfig
      self.mode = mode
      self._instanceName = instanceName
      if instanceName in self.groupConfig.instanceConfig:
         self._instance = self.groupConfig.instanceConfig[ instanceName ]
      else:
         self._instance = None

   def instanceName( self ):
      return self._instanceName

   def instanceId( self ):
      return self._instance.instanceId

   def groupName( self ):
      return self.groupConfig.name

   def addInstanceConfig( self, instanceName, instanceId ):
      instance = self.groupConfig.newInstanceConfig( instanceName, instanceId )
      assert instance
      self._instance = instance

   def delInstanceConfig( self, instanceName ):
      del self.groupConfig.instanceConfig[ instanceName ]
      self._instance = None

   def setConnectionName( self, connectionName ):
      self._instance.connectionName = connectionName
