#!/usr/bin/env python3
# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import TableOutput
from CliModel import Model, Submodel, Int, Str, Bool, Enum, Dict, List
import Tac
import string
from ArnetModel import IpGenericAddress
from Arnet import IpGenAddr
from functools import reduce # pylint: disable=redefined-builtin
#-------------------------------------------------------------------------------
# Utils for creating, formatting, and displaying tables in the Cli
#-------------------------------------------------------------------------------

AccelId = Tac.Type( 'Hardware::SflowAccel::AccelId' )
MessageHelper = Tac.Type( 'Hardware::SflowAccel::MessageHelper' )
CollectorStateReason = Tac.Type('Hardware::SflowAccel::CollecterStateReason')

def _columFormat():
   fmt = TableOutput.Format( justify='left' )
   fmt.noPadLeftIs( True )
   return fmt

def _numColumns( headings ):
   return sum( len( heading[ 1 ] ) if isinstance( heading, tuple ) else 1
               for heading in headings )

def _createTable( *headings ):
   table = TableOutput.createTable( headings )
   colFormats = [ _columFormat() ] * _numColumns( headings )
   table.formatColumns( *colFormats )
   return table

#-------------------------------------------------------------------------------
# Misc utilities
#-------------------------------------------------------------------------------

def _yesNoString( boolVal ):
   return 'Yes' if boolVal else 'No'

def _sortedModuleNames( moduleNames ):
   # Sort by slot number after removing the 'Linecard' prefix
   return sorted( moduleNames, key=lambda name: int( name[ 8: ] ) )

def _sortKeyForFapName( fapName ):
   '''For an input FAP name like 'Jericho1/4', return a list like [ 1, 4 ].
   '''
   # Remove the prefix (Arad, Jericho, etc)
   suffix = fapName.lstrip( string.ascii_letters )
   # Split on '/' in case of modular (eg. 'Jericho4/1')
   suffixSplit = suffix.split( '/' )
   # Transform each string to an int
   return list( map( int, suffixSplit ) )

def accelNameToId( accelName ):
   '''Return the accel Id (as an int) corresponding to the given accel name.
   '''
   expectedNamePrefix = 'SflowAccelFpga'
   prefixLen = len( expectedNamePrefix )
   suffix = accelName[ prefixLen : ]
   if not accelName.startswith( expectedNamePrefix ):
      return AccelId.null
   splitSuffix = suffix.split( ':' )
   if len( splitSuffix ) == 1:
      sliceId = AccelId.nullSlice
   elif len( splitSuffix ) == 2:
      sliceId = int( splitSuffix[ 0 ] )
   else:
      return AccelId.null
   localAccelId = int( splitSuffix[ -1 ] )
   return AccelId.makeAccelId( sliceId, localAccelId )

def sortedItems( collection, sortFunction ):
   return sorted( collection.items(),
                  key=lambda item: sortFunction( item[ 0 ] ) )

def _enabledDisabledString( active ):
   return "enabled" if active else "disabled"

def _ipAndPortToString( ip, port ):
   ipGenAddr = IpGenAddr( ip )
   if ipGenAddr.af == 'ipv4':
      return f'{ipGenAddr}:{port}'
   else:
      return f'[{ipGenAddr}]:{port}'

class ModuleSflowMode( Model ):
   active = Enum( values=[ 'disabled', 'hardware-accelerated', 'software' ],
                  help="Current sFlow status of this module" )
   configured = Enum( values=[ 'hardware-accelerated', 'software' ],
                      help="Whether this module is configured to use software or "
                           "hardware-accelerated sFlow" )

class Collector( Model ):
   port = Int( help="Port of sFlow collector" )
   active = Bool( help="Whether this collector is active" )
   reason = Enum( values=CollectorStateReason.attributes,
                  help="Reason why collector is disabled "                          
                       "it's an empty string if collector is enabled" )

class Vrf( Model ):
   collectorIps = Dict( keyType=IpGenericAddress, valueType=Collector, 
                        help="A mapping of collector IP address to its status" )

class SflowAccelModuleStatus( Model ):
   sflowMode = Submodel( valueType=ModuleSflowMode,
                         help="Type of sFlow running for this module" )
   hasAccelerators = Bool( help="Whether this module has sFlow accelerators" )

class SflowAccelStatus( Model ):
   running = Bool( help="Whether sFlow hardware acceleration is running" )
   reasonsNotRunning = List( valueType=str, optional=True,
                             help="List of reasons why sFlow hardware acceleration "
                                  "is not running" )
   sampleRate = Int( help='Sample rate, 1 sample per N packets' )
   accelPPSupported = Bool( help="Whether sFlow hardware acceleration in "
                                 "pipeline mode is available on the system")
   inactiveFeatures = List( valueType=str, optional=True,
                            help="List of sFlow features that are configured but "
                                 "not supported with hardware-accelerated sFlow, "
                                 "and will only take effect with software sFlow" )
   modules = Dict( keyType=str, valueType=SflowAccelModuleStatus, optional=True,
                   help="Per-module status of hardware-accelerated sFlow" )
   vrfs = Dict( keyType=str, valueType=Vrf,
                help="A mapping from VRF's name to its collectors" )

   def render( self ):
      print( 'Status' )
      print( '------' )
      print( 'Hardware acceleration on:', _yesNoString( self.running ) )

      for reason in self.reasonsNotRunning:
         print( '  -', reason )

      print( 'Sample rate:', self.sampleRate if self.running else 'None' )

      if self.inactiveFeatures:
         print() # newline
         print( 'Inactive features:' )

         for feature in self.inactiveFeatures:
            print( '  -', feature )

      # Print per-module status
      if self.modules:
         table = _createTable( 'Module',
                               ( 'sFlow Mode', ( 'Active', 'Configured' ) ),
                               'Has sFlow accelerators' )

         for moduleName in _sortedModuleNames( list( self.modules ) ):
            moduleStatus = self.modules[ moduleName ]

            table.newRow( moduleName,
                          moduleStatus.sflowMode.active.capitalize(),
                          moduleStatus.sflowMode.configured.capitalize(),
                          _yesNoString( moduleStatus.hasAccelerators ) )

         print() # newline
         print( table.output() )

      # Print configured collectors status
      if self.accelPPSupported:
         table = _createTable( 'VRF', 'Collector', 'Status', 'Reason' ) 
         for vrfName in sorted( self.vrfs ):
            vrf = self.vrfs[ vrfName ]
            for ip in sorted( vrf.collectorIps, key=IpGenAddr ):  
               collector = vrf.collectorIps[ ip ]
               reason = MessageHelper.collecterStateReasonDescription( 
                                                               collector.reason )
               table.newRow( vrfName, 
                             _ipAndPortToString( ip, collector.port ),
                             _enabledDisabledString( collector.active ),
                             reason )
      
         print() # newline
         print( table.output() )

class SflowAcceleratorInfo( Model ):
   sliceName = Str( help="Name of the slice on which " \
                         "this sFlow accelerator is located" )
   sflowAccelFpgaType = Str( help="Type of sFlow accelerator" )
   chips = List( valueType=str,
                 help="List of chips directly connected to this accelerator" )
   pciAddress = Str( help="PCI address of this accelerator" )
   accelRevision = Int( help="Accelerator revision" )

class SflowAccelerators( Model ):
   accelerators = Dict( keyType=str, valueType=SflowAcceleratorInfo,
                        help="Mapping of accelerator name to accelerator info" )

   def render( self ):
      lineFmt = "{:15}{:22}{:6}{:18}{:17}{}"

      headerFields = [ "Slice", "sFlow Accelerator", "Type", "PCI Address",
                       "Accel Revision", "Direct Connections" ]
      print( lineFmt.format( *headerFields ) )

      if not self.accelerators:
         return

      lines = []

      for accelName, accelInfo in sortedItems( self.accelerators, accelNameToId ):
         accelRevisionHex = '-' if accelInfo.accelRevision == 0 else \
                            hex( accelInfo.accelRevision )
         lines.append( lineFmt.format( accelInfo.sliceName,
                                       accelName,
                                       accelInfo.sflowAccelFpgaType,
                                       accelInfo.pciAddress,
                                       accelRevisionHex,
                                       ", ".join( accelInfo.chips ) ) )


      # Print a separator line as wide as the longest rendered line
      maxLineLength = max( len( line ) for line in lines )
      print( "-" * max( maxLineLength, 96 ) )

      for line in lines:
         print( line )


class SflowAcceleratorMappingInfo( Model ):
   sflowAccelerator = Str( help="sFlow accelerator name" )
   directConnection = Bool( help="Direct connection between the chip "
                              "and the accelerator" )

class HardwareSflowMapping( Model ):
   chips = Dict( keyType=str, valueType=SflowAcceleratorMappingInfo,
                 help="Mapping of chip to sFlow accelerator" )

   def render( self ):
      header = "%-15s%-25s%-25s" \
            % ( "Chip", "sFlow Accelerator", "Direct Connection" )

      print( header )
      print( "-" * 60 )

      for chipName, chipToAccelMapping in sortedItems( self.chips,
                                                       _sortKeyForFapName ):
         print( "%-15s%-25s%-25s"
               % ( chipName, chipToAccelMapping.sflowAccelerator,
                   chipToAccelMapping.directConnection ) )

class SflowAcceleratorMappingLagInfo( Model ):
   sflowAccelerator = Str( help="sFlow accelerator name" )

class HardwareSflowMappingLag( Model ):
   lags = Dict( keyType=str, valueType=SflowAcceleratorMappingLagInfo,
                help="Mapping of LAG to sFlow accelerator" )

   def render( self ):
      formatStr = "{:25}{:35}"
      print( formatStr.format( "LAG", "sFlow Accelerator" ) )
      print( "-" * 60 )

      for lagName, lagToAccelMapping in sorted( self.lags.items() ):
         print( formatStr.format( lagName, lagToAccelMapping.sflowAccelerator ) )

class HardwareSflowCountersInfo( Model ):
   inPktCnt = Int( help="Incoming Packet Count", default=0 )
   outSflowDgramCnt = Int( help="Outgoing Sflow Datagram Count", default=0 )
   outFlowSampleCnt = Int( help="Outgoing Flow Sample Count", default=0 )
   inProcPktCnt = Int( help="Incoming Processed Packet Count", default=0 )
   rcvPktDropCnt = Int( help="Receive Packet Drop Count", default=0 )
   rcvPktTrunCnt = Int( help="Packet Truncated Count", default=0 )
   inPktErrCnt = Int( help="Incoming Packet Error Count", default=0 )
   outProcSflowDgramCnt = Int( help="Outgoing Processed Datagram Count", default=0 )
   samplePool = Int( help="Sample Pool", default=0 )

   # __add__ is not allowed for a Model, so defining it as a non-special function
   # that can manually be called.
   def add( self, other ):
      result = HardwareSflowCountersInfo()
      result.inPktCnt = self.inPktCnt + other.inPktCnt
      result.outSflowDgramCnt = self.outSflowDgramCnt + other.outSflowDgramCnt
      result.outFlowSampleCnt = self.outFlowSampleCnt + other.outFlowSampleCnt
      result.inProcPktCnt = self.inProcPktCnt + other.inProcPktCnt
      result.rcvPktDropCnt = self.rcvPktDropCnt + other.rcvPktDropCnt
      result.rcvPktTrunCnt = self.rcvPktTrunCnt + other.rcvPktTrunCnt
      result.inPktErrCnt = self.inPktErrCnt + other.inPktErrCnt
      result.outProcSflowDgramCnt = self.outProcSflowDgramCnt + \
                                    other.outProcSflowDgramCnt
      result.samplePool = self.samplePool + other.samplePool
      return result

   def render( self ):
      outputList = (
         ( "Incoming Packet Count", self.inPktCnt ),
         ( "Outgoing Sflow Datagram Count", self.outSflowDgramCnt ),
         ( "Outgoing Flow Sample Count", self.outFlowSampleCnt ),
         ( "Incoming Processed Packet Count", self.inProcPktCnt ),
         ( "Receive Packet Drop Count", self.rcvPktDropCnt ),
         ( "Packet Truncated Count", self.rcvPktTrunCnt ),
         ( "Incoming Packet Error Count", self.inPktErrCnt ),
         ( "Outgoing Processed Datagram Count", self.outProcSflowDgramCnt ),
         ( "Sample Pool", self.samplePool ),
      )
      for msg, counter in outputList:
         print( f"{msg:35}: {counter:15}" )

class HardwareSflowCounters( Model ):
   counters = Dict( keyType=str, valueType=HardwareSflowCountersInfo,
                    help="sFlow accelerator counters" )

   def renderFpga( self, accelName, counterInfo, printEmptyLine=True ):
      print( "------------------" )
      print( accelName )
      print( "------------------" )
      counterInfo.render()
      if printEmptyLine:
         print()

   def render( self ):
      for accelName, counterInfo in sortedItems( self.counters, accelNameToId ):
         self.renderFpga( accelName, counterInfo )
      if len( self.counters ):
         counterInfoSum = reduce( HardwareSflowCountersInfo.add,
                                  self.counters.values(),
                                  HardwareSflowCountersInfo() )
         self.renderFpga( "Total", counterInfoSum, printEmptyLine=False )
