#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCliModes
import CliCommand
import CliMatcher
import CliParser
import CliPlugin.VrfCli as VrfCli # pylint: disable=consider-using-from-import
import CliPlugin.SflowCli as SflowCli # pylint: disable=consider-using-from-import
from CliPlugin.IntfCli import Intf
from CliPlugin.IpGenAddrMatcher import IpGenAddrMatcher
from CliPlugin.EthIntfCli import EthIntfModelet
from CliPlugin.SwitchIntfCli import SwitchIntfModelet
from CliPlugin.LagIntfCli import LagIntfConfigModelet
from CliPlugin.SubIntfCli import SubIntfModelet
import DscpCliLib
import SflowConst
import Tac
import HostnameCli
import Toggles.SflowLibToggleLib as ToggleSflow

SampleTruncateSizeType = Tac.Type( "Sflow::SampleTruncateSizeType" )
SampleInterfaceIdType = Tac.Type( "Sflow::SampleInterfaceIdType" )

matcherExtension = CliMatcher.KeywordMatcher( 'extension',
      helpdesc='Configure sFlow extension settings' )
matcherSample = CliMatcher.KeywordMatcher( 'sample',
      helpdesc='Set sample characteristics for sFlow' )
matcherSourceInterface = CliMatcher.KeywordMatcher( 'source-interface',
      helpdesc='Configure the source interface for sFlow datagrams' )
sflowNode = CliCommand.guardedKeyword( 'sflow',
      helpdesc='sFlow configuration',
      guard=SflowCli.sflowSupportedGuard )
evpnMatcher = CliMatcher.KeywordMatcher( 'evpn',
      helpdesc='Configure sFlow extension settings for EVPN' )
vrfExprFactory = VrfCli.VrfExprFactory(
      helpdesc='Configure VRFs',
      inclDefaultVrf=True )
matcherInterface = CliMatcher.KeywordMatcher( 'interface',
      helpdesc='Include the output interface in reported packet samples' )

#--------------------------------------------------------------------------------
# [ no | default ] sflow [ vrf VRF ] destination DESTINATION [ PORT ]
#--------------------------------------------------------------------------------
class SflowDestinationCmd( CliCommand.CliCommandClass ):
   syntax = 'sflow [ VRF ] destination DESTINATION [ PORT ]'
   noOrDefaultSyntax = 'sflow [ VRF ] destination DESTINATION [ PORT ] ...'
   data = {
      'sflow' : sflowNode,
      'VRF' : vrfExprFactory,
      'destination' : 'Set the sFlow collector destination',
      'DESTINATION' : HostnameCli.IpAddrOrHostnameMatcher(
         ipv6=True,
         helpdesc='IPv4 address or IPv6 address or fully qualified '
         'domain name' ),
      'PORT' : CliMatcher.IntegerMatcher( SflowConst.minPort,
         SflowConst.maxPort, helpdesc='Port' ),
   }
   handler = SflowCli.setDestination
   noOrDefaultHandler = SflowCli.noDestination

BasicCliModes.GlobalConfigMode.addCommandClass( SflowDestinationCmd )

def addExtensionCmd( cmdSyntax, cmdData, attrName ):
   cmdSyntax = 'sflow extension ' + cmdSyntax
   cmdData[ 'sflow' ] = sflowNode
   cmdData[ 'extension' ] = 'Configure sFlow extension settings'

   class ExtensionCmd( CliCommand.CliCommandClass ):
      syntax = cmdSyntax
      noOrDefaultSyntax = cmdSyntax
      data = cmdData

      @staticmethod
      def handler( mode, args ):
         SflowCli.setExtension( attrName, True )

      @staticmethod
      def noHandler( mode, args ):
         SflowCli.setExtension( attrName, False )

      @staticmethod
      def defaultHandler( mode, args ):
         SflowCli.setExtension( attrName, None )
   BasicCliModes.GlobalConfigMode.addCommandClass( ExtensionCmd )

addExtensionCmd(
      'bgp',
      { 'bgp' : 'Enable extended gateway information for BGP' },
      'bgpExtension' )
# Disable MPLS extension for now, it will be enabled when we have support for
# independently enable/disable this feature from the FPGA (BUG357099). Currently
# loading the SflowAccel2 FPGA image is hooked up to
# `sflow hardware acceleration profile sflowaccel-v2`.
addExtensionCmd(
      'mpls',
      { 'mpls' : CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'mpls',
            helpdesc='Enable MPLS extension' ),
         guard=lambda mode, token : CliParser.guardNotThisPlatform ) },
      'mplsExtension' )
addExtensionCmd(
      'router',
      { 'router' : 'Include router extension in reported packet samples' },
      'routerExtension' )
addExtensionCmd(
      'switch',
      { 'switch' : 'Include switch extension in reported packet samples' },
      'switchExtension' )
addExtensionCmd(
      'tunnel ipv4 egress',
      {
         'tunnel' : CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'tunnel',
               helpdesc='Configure tunnel extension settings' ),
            guard=SflowCli.sflowTunnelExtensionGuard ),
         'ipv4' : 'Configure IPv4 tunnel extension settings',
         'egress' : 'Configure egress tunnel extension in reported packet samples',
      },
      'tunnelIpv4EgrExtension' )

if ToggleSflow.toggleSflowVplsExtensionEnabled():
   addExtensionCmd(
         'vpls',
         { 'vpls' : 'Include MPLS VC extension for VPLS packet samples' },
         'vplsExtension' )

if ToggleSflow.toggleSflowEvpnMplsEnabled():
   addExtensionCmd(
         'evpn mpls',
         {
            'evpn' : CliCommand.Node( matcher=evpnMatcher ),
            'mpls' : 'Include MPLS VC extension for EVPN packet samples',
         },
         'evpnMplsExtension' )

if ToggleSflow.toggleSflowVxlanEnabled():
   addExtensionCmd(
         'vxlan',
         { 'vxlan' : 'Include VXLAN extension in reported packet samples' },
         'vxlanExtension' )

#--------------------------------------------------------------------------------
# [ no | default ] sflow interface disable default
#--------------------------------------------------------------------------------
class SflowInterfaceIngressDisableDefaultCmd( CliCommand.CliCommandClass ):
   syntax = 'sflow interface disable default'
   noOrDefaultSyntax = syntax
   data = {
      'sflow' : sflowNode,
      'interface' : 'Global sFlow configuration for interfaces',
      'disable' : 'Set default for ingress sFlow to Disabled on interfaces',
      'default' : 'Set default ingress sFlow configuration for interfaces',
   }
   handler = SflowCli.disableIngressInterfacesByDefault
   noOrDefaultHandler = SflowCli.enableIngressInterfacesByDefault

BasicCliModes.GlobalConfigMode.addCommandClass(
                                    SflowInterfaceIngressDisableDefaultCmd )

#--------------------------------------------------------------------------------
# [ no | default ] sflow interface egress enable default
#--------------------------------------------------------------------------------
class SflowInterfaceEgressEnableDefaultCmd( CliCommand.CliCommandClass ):
   syntax = 'sflow interface egress enable default'
   noOrDefaultSyntax = syntax
   data = {
      'sflow' : sflowNode,
      'interface' : 'Global sFlow configuration for interfaces',
      'egress' : CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'egress',
                   helpdesc='Global egress sFlow configuration for interfaces' ),
                   guard=SflowCli.sflowEgressSflowIntfGuard ),
      'enable' : CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'enable',
             helpdesc='Set default for egress sFlow to Enabled on interfaces' ),
             guard=SflowCli.sflowEgressModifiedSflowIntfGuard ),
      'default' : 'Set default egress sFlow configuration for interfaces',
   }
   handler = SflowCli.enableEgressInterfacesByDefault
   noOrDefaultHandler = SflowCli.disableEgressInterfacesByDefault

BasicCliModes.GlobalConfigMode.addCommandClass(
                                    SflowInterfaceEgressEnableDefaultCmd )

#--------------------------------------------------------------------------------
# [ no | default ] sflow interface egress unmodified enable default
#--------------------------------------------------------------------------------
class SflowInterfaceEgressUnmodifiedEnableDefaultCmd( CliCommand.CliCommandClass ):
   syntax = 'sflow interface egress unmodified enable default'
   noOrDefaultSyntax = syntax
   data = {
      'sflow' : sflowNode,
      'interface' : 'Global sFlow configuration for interfaces',
      'egress' : CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'egress',
                   helpdesc='Global egress sFlow configuration for interfaces' ),
                   guard=SflowCli.sflowEgressSflowIntfGuard ),
      'unmodified' : CliCommand.Node(
                           matcher=CliMatcher.KeywordMatcher( 'unmodified',
                           helpdesc='Unmodified egress sFlow' ),
                           guard=SflowCli.sflowEgressUnmodifiedSflowIntfGuard ),
      'enable' : 'Set default for egress sFlow to Enabled on interfaces',
      'default' : 'Set default egress sFlow configuration for interfaces',
   }
   handler = SflowCli.enableEgressInterfacesByDefault
   noOrDefaultHandler = SflowCli.disableEgressInterfacesByDefault

BasicCliModes.GlobalConfigMode.addCommandClass(
                                 SflowInterfaceEgressUnmodifiedEnableDefaultCmd )

#--------------------------------------------------------------------------------
# [ no | default ] sflow polling-interval POLLING_INTERVAL
#--------------------------------------------------------------------------------
class SflowPollingIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'sflow polling-interval POLLING_INTERVAL'
   noOrDefaultSyntax = 'sflow polling-interval ...'
   data = {
      'sflow' : sflowNode,
      'polling-interval' : 'Set polling interval (secs) for sFlow',
      'POLLING_INTERVAL' : CliMatcher.IntegerMatcher( SflowConst.minPollingInterval,
         SflowConst.maxPollingInterval, helpdesc='Seconds' ),
   }
   handler = SflowCli.setPollingInterval
   noHandler = SflowCli.noPollingInterval
   defaultHandler = SflowCli.defaultPollingInterval

BasicCliModes.GlobalConfigMode.addCommandClass( SflowPollingIntervalCmd )

#--------------------------------------------------------------------------------
# [ no | default ] sflow datagram size maximum MAX_DATAGRAM_SIZE
#--------------------------------------------------------------------------------
class SflowMaxDatagramSizeCmd( CliCommand.CliCommandClass ):
   syntax = 'sflow datagram size maximum MAX_DATAGRAM_SIZE'
   noOrDefaultSyntax = 'sflow datagram size maximum ...'
   data = {
      'sflow' : sflowNode,
      'datagram' : CliCommand.guardedKeyword( 'datagram',
         helpdesc='Configure datagram attributes',
         guard=SflowCli.sflowMaxDatagramSizeGuard ),
      'size' : 'Configure datagram payload size',
      'maximum' : 'Configure datagram maximum payload size',
      'MAX_DATAGRAM_SIZE' : CliMatcher.IntegerMatcher( SflowConst.minMaxDatagramSize,
         SflowConst.maxMaxDatagramSize, helpdesc='Bytes' )
   }
   handler = SflowCli.setMaxDatagramSize
   noOrDefaultHandler = SflowCli.defaultMaxDatagramSize

BasicCliModes.GlobalConfigMode.addCommandClass( SflowMaxDatagramSizeCmd )

#--------------------------------------------------------------------------------
# [ no | default ] sflow qos dscp DSCP
#--------------------------------------------------------------------------------
DscpCliLib.addQosDscpCommandClass( BasicCliModes.GlobalConfigMode,
                                   SflowCli.setDscp, SflowCli.noDscp,
                                   tokenProto=sflowNode )

#--------------------------------------------------------------------------------
# [ no | default ] sflow run
#--------------------------------------------------------------------------------
class SflowRunCmd( CliCommand.CliCommandClass ):
   syntax = 'sflow run'
   noOrDefaultSyntax = 'sflow run ...'
   data = {
      'sflow' : sflowNode,
      'run' : 'Run sFlow globally'
   }
   handler = SflowCli.runSflow
   noOrDefaultHandler = SflowCli.noRunSflow

BasicCliModes.GlobalConfigMode.addCommandClass( SflowRunCmd )

#--------------------------------------------------------------------------------
# [ no | default ] sflow sample ( RATE | ( dangerous DANGEROUS_RATE ) )
#--------------------------------------------------------------------------------
class SflowSampleCmd( CliCommand.CliCommandClass ):
   syntax = 'sflow sample ( RATE | ( dangerous DANGEROUS_RATE ) )'
   noOrDefaultSyntax = 'sflow sample [ ( RATE | ( dangerous ... ) ) ]'
   data = {
      'sflow' : sflowNode,
      'sample' : matcherSample,
      'RATE' : CliMatcher.DynamicIntegerMatcher( SflowCli.rateRangeFn,
         helpdesc='Rate' ),
      'dangerous' : CliCommand.Node(
                     matcher=CliMatcher.KeywordMatcher( 'dangerous',
                        helpdesc='DANGEROUS: Set any sample rate for sFlow' ),
                     hidden=True ),
      'DANGEROUS_RATE' : CliMatcher.DynamicIntegerMatcher( SflowCli.dangRateRangeFn,
         helpdesc='Rate' ),
   }
   handler = SflowCli.setSampleRate
   noOrDefaultHandler = SflowCli.defaultSampleRate

BasicCliModes.GlobalConfigMode.addCommandClass( SflowSampleCmd )

#--------------------------------------------------------------------------------
# [ no | default ] sflow sample include drop reason acl
#--------------------------------------------------------------------------------
class SflowSampleIncludeDropReasonAclCmd( CliCommand.CliCommandClass ):
   syntax = 'sflow sample include drop reason acl'
   noOrDefaultSyntax = 'sflow sample include drop reason acl ...'
   data = {
      'sflow' : sflowNode,
      'sample' : matcherSample,
      'include' : 'Set configuration for packets included in sampling',
      'drop' : 'Include dropped packets',
      'reason' : 'Include dropped packets',
      'acl' : CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'acl',
            helpdesc='Include packets dropped by ACL' ),
         guard=SflowCli.sflowSampleIncludeDropReasonAclSupportedGuard ),
   }
   handler = SflowCli.includeDropReasonAcl
   noOrDefaultHandler = SflowCli.noIncludeDropReasonAcl

BasicCliModes.GlobalConfigMode.addCommandClass( SflowSampleIncludeDropReasonAclCmd )

#--------------------------------------------------------------------------------
# [ no | default ] sflow sample output portchannel ifindex IFINDEX_MODE
#--------------------------------------------------------------------------------
class SflowSampleOutputPortchannelCmd( CliCommand.CliCommandClass ):
   syntax = 'sflow sample output portchannel ifindex IFINDEX_MODE'
   noOrDefaultSyntax = 'sflow sample output portchannel ifindex ...'
   data = {
      'sflow' : sflowNode,
      'sample' : matcherSample,
      'output' : 'Configure sFlow output settings',
      'portchannel' : 'Configure sFlow Port Channel settings',
      'ifindex' : CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'ifindex',
            helpdesc='Port Channel\'s output ifindex mode' ),
         guard=SflowCli.sflowPortChannelOutputIfindexConfigGuard ),
      'IFINDEX_MODE' : CliMatcher.EnumMatcher( {
         'portchannel' : 'Ifindex of the portchannel',
         'member' : 'Ifindex of the physical member of the portchannel',
      } )
   }
   handler = SflowCli.setPortChannelIfindexMode
   noOrDefaultHandler = SflowCli.defaultPortChannelIfindex

BasicCliModes.GlobalConfigMode.addCommandClass( SflowSampleOutputPortchannelCmd )

#--------------------------------------------------------------------------------
# [ no | default ] sflow sample output multicast interface
# --------------------------------------------------------------------------------
class SflowSampleOutputMulticastInterfaceCmd( CliCommand.CliCommandClass ):
   syntax = 'sflow sample output multicast interface'
   noOrDefaultSyntax = 'sflow sample output multicast interface'
   data = {
      'sflow' : sflowNode,
      'sample' : matcherSample,
      'output' : 'Configure sFlow output settings',
      'multicast' : CliCommand.guardedKeyword( 'multicast',
         helpdesc='Include the output interface for egress multicast packet samples',
         guard=SflowCli.sflowSampleOutputMulticastInterfaceConfigGuard ),
      'interface' : matcherInterface
   }
   handler = SflowCli.outputMuticastInterface
   noOrDefaultHandler = SflowCli.noOutputMuticastInterface

BasicCliModes.GlobalConfigMode.addCommandClass(
                              SflowSampleOutputMulticastInterfaceCmd )

#--------------------------------------------------------------------------------
# [ no | default ] sflow sample input svi ifindex ( svi | member )
#--------------------------------------------------------------------------------
class SflowSampleInputSviIfindexCmd( CliCommand.CliCommandClass ):
   syntax = 'sflow sample input svi ifindex INDEX'
   noOrDefaultSyntax = syntax
   data = {
      'sflow' : sflowNode,
      'sample' : matcherSample,
      'input' : 'Configure sFlow input settings',
      'svi' : CliCommand.guardedKeyword( 'svi',
         helpdesc='Configure sFlow SVI settings',
         guard=SflowCli.sflowSampleInputSviIfindexGuard ),
      'ifindex' : 'Configure sFlow SVI ifindex sampling',
      'INDEX' : CliMatcher.EnumMatcher( {
         'svi' : 'Report ifindex of the SVI in flow sample',
         'member' : 'Report ifindex of the physical member of the SVI in the '
         'flow sample',
         } )
      }

   handler = SflowCli.sflowSampleInputSviHandler
   noOrDefaultHandler = SflowCli.noSflowSampleInputSviHandler

BasicCliModes.GlobalConfigMode.addCommandClass( SflowSampleInputSviIfindexCmd )

#--------------------------------------------------------------------------------
# [ no | default ] sflow sample output svi ifindex ( svi | member )
#--------------------------------------------------------------------------------
class SflowSampleOutputSviIfindexCmd( CliCommand.CliCommandClass ):
   syntax = 'sflow sample output svi ifindex INDEX'
   noOrDefaultSyntax = syntax
   data = {
      'sflow': sflowNode,
      'sample' : matcherSample,
      'output' : 'Configure sFlow output settings',
      'svi' : CliCommand.guardedKeyword( 'svi',
         helpdesc='Configure sFlow SVI settings',
         guard=SflowCli.sflowSampleOutputSviIfindexGuard ),
      'ifindex' : 'Configure sFlow SVI ifindex sampling',
      'INDEX' : CliMatcher.EnumMatcher( {
         'svi' : 'Report ifindex of the SVI in flow sample',
         'member' : 'Report ifindex of the physical member of the SVI in the '
         'flow sample',
         } )
      }

   handler = SflowCli.sflowSampleOutputSviHandler
   noOrDefaultHandler = SflowCli.noSflowSampleOutputSviHandler

BasicCliModes.GlobalConfigMode.addCommandClass( SflowSampleOutputSviIfindexCmd )

#--------------------------------------------------------------------------------
# [ no | default ] sflow sample rewrite dscp
#--------------------------------------------------------------------------------
class SflowSampleRewriteDscpCmd( CliCommand.CliCommandClass ):
   syntax = 'sflow sample rewrite dscp'
   noOrDefaultSyntax = 'sflow sample rewrite dscp ...'
   data = {
      'sflow' : sflowNode,
      'sample' : matcherSample,
      'rewrite' : 'Sample rewrite settings',
      'dscp' : 'Rewrite DSCP value in sFlow samples'
   }
   handler = SflowCli.rewriteDscp
   noOrDefaultHandler = SflowCli.noRewriteDscp

BasicCliModes.GlobalConfigMode.addCommandClass( SflowSampleRewriteDscpCmd )

#--------------------------------------------------------------------------------
# [ no | default ] sflow [ vrf VRF ] source SOURCE
#--------------------------------------------------------------------------------
class SflowSourceCmd( CliCommand.CliCommandClass ):
   syntax = 'sflow [ VRF ] source SOURCE'
   noOrDefaultSyntax = 'sflow [ VRF ] source [ SOURCE ]'
   data = {
      'sflow' : sflowNode,
      'VRF' : vrfExprFactory,
      'source' : 'Set the source IP address',
      'SOURCE' : IpGenAddrMatcher( 'IPv4 or IPv6 Address',
                                   helpdesc4='IPv4 address',
                                   helpdesc6='IPv6 address' ),
   }
   handler = SflowCli.setSource
   noOrDefaultHandler = SflowCli.noSource

BasicCliModes.GlobalConfigMode.addCommandClass( SflowSourceCmd )

#--------------------------------------------------------------------------------
# [ no | default ] sflow [ vrf VRF ] source-interface IPINTF
#--------------------------------------------------------------------------------
class SflowSourceInterfaceIpintfCmd( CliCommand.CliCommandClass ):
   syntax = 'sflow [ VRF ] source-interface IPINTF'
   noOrDefaultSyntax = 'sflow [ VRF ] source-interface ...'
   data = {
      'sflow' : sflowNode,
      'VRF' : vrfExprFactory,
      'source-interface' : matcherSourceInterface,
      'IPINTF' : Intf.matcherWithIpSupport,
   }
   handler = SflowCli.setSourceIntf
   noOrDefaultHandler = SflowCli.noSourceIntf

BasicCliModes.GlobalConfigMode.addCommandClass( SflowSourceInterfaceIpintfCmd )

#--------------------------------------------------------------------------------
# [ no | default ] sflow [ vrf VRF ] source address SOURCE
# --------------------------------------------------------------------------------
class SflowSourceAddressCmd ( CliCommand.CliCommandClass ):
   syntax = 'sflow [ VRF ] source address SOURCE'
   noOrDefaultSyntax = 'sflow [ VRF ] source address [ SOURCE ]'
   data = {
      'sflow' : sflowNode,
      'VRF' : vrfExprFactory,
      'source' : 'Set the source IP address',
      'address' : CliCommand.guardedKeyword( 'address',
                  helpdesc='Set the source IP address independently' +
                           ' of agent IP address',
                  guard=SflowCli.sflowIndependentAddressConfigGuard ),
      'SOURCE' : IpGenAddrMatcher( 'IPv4 or IPv6 Address',
                                   helpdesc4='IPv4 Address',
                                   helpdesc6='IPv6 Address' ),
   }
   handler = SflowCli.setSourceAddress
   noOrDefaultHandler = SflowCli.noSourceAddress

BasicCliModes.GlobalConfigMode.addCommandClass( SflowSourceAddressCmd )

# -------------------------------------------------------------------------------
# [no | default] sflow agent address ADDR
# -------------------------------------------------------------------------------
class SflowAgentAddressCmd( CliCommand.CliCommandClass ):
   syntax = 'sflow [VRF] agent address ADDR'
   noOrDefaultSyntax = 'sflow [ VRF ] agent address [ ADDR ]'
   data = {
      'sflow' : sflowNode,
      'VRF' : vrfExprFactory,
      'agent' : CliCommand.guardedKeyword( 'agent',
                     helpdesc='Agent field in the sFlow header',
                     guard=SflowCli.sflowIndependentAddressConfigGuard ),
      'address' : CliCommand.guardedKeyword( 'address',
                     helpdesc='Set agent address in the sFlow header',
                     guard=SflowCli.sflowIndependentAddressConfigGuard ),
      'ADDR' : IpGenAddrMatcher( 'IPv4 or IPv6 Address',
                                 helpdesc4='IPv4 Address',
                                 helpdesc6='IPv6 Address' ),
   }
   handler = SflowCli.setAgentAddress
   noOrDefaultHandler = SflowCli.noAgentAddress

BasicCliModes.GlobalConfigMode.addCommandClass( SflowAgentAddressCmd )

# --------------------------------------------------------------------------------
# [ no | default ] sflow sample output ( interface | subinterface )
#--------------------------------------------------------------------------------
class SflowSampleOutputCmd( CliCommand.CliCommandClass ):
   syntax = 'sflow sample output ( interface | subinterface )'
   noOrDefaultSyntax = syntax
   data = {
      'sflow' : sflowNode,
      'sample' : matcherSample,
      'output' : 'Configure sFlow output settings',
      'interface' : matcherInterface,
      'subinterface' : CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'subinterface',
            helpdesc=( 'Include the output subinterface in samples using expanded '
                       'sample format' ) ),
         guard=SflowCli.sflowEgressSubintfGuard ),
   }
   handler = SflowCli.sflowSampleOutputHandler
   noHandler = SflowCli.sflowSampleOutputNoHandler
   defaultHandler = SflowCli.sflowSampleOutputDefaultHandler

BasicCliModes.GlobalConfigMode.addCommandClass( SflowSampleOutputCmd )

#--------------------------------------------------------------------------------
# [ no | default ] sflow sample input subinterface
#--------------------------------------------------------------------------------
class SflowSampleInputCmd( CliCommand.CliCommandClass ):
   syntax = 'sflow sample input subinterface'
   noOrDefaultSyntax = syntax
   data = {
      'sflow' : sflowNode,
      'sample' : matcherSample,
      'input' : 'Configure sFlow input settings',
      'subinterface' : ( 'Include the input subinterface in samples '
                         'using expanded sample format' ),
   }
   handler = SflowCli.sflowSampleInputHandler
   noHandler = SflowCli.sflowSampleInputNoHandler
   defaultHandler = SflowCli.sflowSampleInputDefaultHandler

BasicCliModes.GlobalConfigMode.addCommandClass( SflowSampleInputCmd )

# --------------------------------------------------------------------------------
# [ no | default ] sflow sample output vlan
# --------------------------------------------------------------------------------
if ToggleSflow.toggleSflowL2SubIntfVlanEnabled():

   class SflowSampleOutputVlanCmd( CliCommand.CliCommandClass ):
      syntax = 'sflow sample output vlan'
      noOrDefaultSyntax = syntax
      data = {
         'sflow' : sflowNode,
         'sample' : matcherSample,
         'output' : 'Configure sFlow output settings',
         'vlan' : 'Report VLAN ID changes on L2 subinterfaces',
      }
      handler = SflowCli.sflowSampleOutputVlanHandler
      noHandler = SflowCli.sflowSampleOutputVlanNoHandler
      defaultHandler = SflowCli.sflowSampleOutputVlanDefaultHandler

   BasicCliModes.GlobalConfigMode.addCommandClass( SflowSampleOutputVlanCmd )

#--------------------------------------------------------------------------------
# [ no | default ] sflow sample truncate size SIZE
#--------------------------------------------------------------------------------
class SflowSampleTruncateSizeCmd( CliCommand.CliCommandClass ):
   syntax = 'sflow sample truncate size SIZE'
   noOrDefaultSyntax = 'sflow sample truncate size ...'
   data = {
      'sflow' : sflowNode,
      'sample' : matcherSample,
      'truncate' : CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'truncate',
            helpdesc=( 'Limit the number of packet bytes included in '
                       'sFlow samples' ) ) ),
      'size' : 'Maximum sample size limit',
      'SIZE' : CliMatcher.IntegerMatcher( SampleTruncateSizeType.min,
                                          SampleTruncateSizeType.max,
                                          helpdesc='Sample size in bytes' ),
   }
   handler = SflowCli.sflowSampleTruncateSizeHandler
   noOrDefaultHandler = handler

BasicCliModes.GlobalConfigMode.addCommandClass( SflowSampleTruncateSizeCmd )

#--------------------------------------------------------------------------------
# [ no | default ] sflow enable
#--------------------------------------------------------------------------------
class SflowIngressEnableCmd( CliCommand.CliCommandClass ):
   syntax = 'sflow enable'
   noOrDefaultSyntax = syntax
   data = {
      'sflow' : sflowNode,
      'enable' : 'Enable ingress sFlow on an interface'
   }
   handler = SflowCli.intfIngressSflowEnableHandler
   noHandler = SflowCli.intfIngressSflowEnableNoHandler
   defaultHandler = SflowCli.intfIngressSflowEnableDefaultHandler

SwitchIntfModelet.addCommandClass( SflowIngressEnableCmd )
LagIntfConfigModelet.addCommandClass( SflowIngressEnableCmd )
EthIntfModelet.addCommandClass( SflowIngressEnableCmd )

# --------------------------------------------------------------------------------
# [ no | default ] sflow enable
# --------------------------------------------------------------------------------
class SflowIngressSubintfEnableCmd( CliCommand.CliCommandClass ):
   syntax = 'sflow enable'
   noOrDefaultSyntax = syntax
   data = {
      'sflow'  : sflowNode,
      'enable' : CliCommand.guardedKeyword( 'enable',
                                  helpdesc='Enable ingress sFlow on subinterface',
                                  guard=SflowCli.sflowIngressSubIntfGuard )

   }
   handler = SflowCli.intfIngressSflowEnableHandler
   noHandler = SflowCli.intfIngressSflowEnableNoHandler
   defaultHandler = SflowCli.intfIngressSflowEnableDefaultHandler

SubIntfModelet.addCommandClass( SflowIngressSubintfEnableCmd )

#--------------------------------------------------------------------------------
# [ no | default ] sflow egress enable
#--------------------------------------------------------------------------------
class SflowEgressEnableCmd( CliCommand.CliCommandClass ):
   syntax = 'sflow egress enable'
   noOrDefaultSyntax = syntax
   data = {
      'sflow' : sflowNode,
      'egress' : CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'egress',
                                  helpdesc='Egress sFlow' ),
                                  guard=SflowCli.sflowEgressSflowIntfGuard ),
      'enable' : CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'enable',
                                  helpdesc='Enable egress sFlow on an interface' ),
                                  guard=SflowCli.sflowEgressModifiedSflowIntfGuard ),
   }
   handler = SflowCli.intfEgressSflowEnableHandler
   noHandler = SflowCli.intfEgressSflowEnableNoHandler
   defaultHandler = SflowCli.intfEgressSflowEnableDefaultHandler

SwitchIntfModelet.addCommandClass( SflowEgressEnableCmd )
LagIntfConfigModelet.addCommandClass( SflowEgressEnableCmd )
EthIntfModelet.addCommandClass( SflowEgressEnableCmd )

#--------------------------------------------------------------------------------
# [ no | default ] sflow egress enable
#--------------------------------------------------------------------------------
class SflowEgressSubIntfEnableCmd( CliCommand.CliCommandClass ):
   syntax = 'sflow egress enable'
   noOrDefaultSyntax = syntax
   data = {
      'sflow' : sflowNode,
      'egress' : CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'egress',
                                  helpdesc='Egress sFlow' ),
                                  guard=SflowCli.sflowEgressSubIntfGuard ),
      'enable' : 'Enable egress sFlow on the subinterface'
   }
   handler = SflowCli.intfEgressSflowEnableHandler
   noHandler = SflowCli.intfEgressSflowEnableNoHandler
   defaultHandler = SflowCli.intfEgressSflowEnableDefaultHandler

SubIntfModelet.addCommandClass( SflowEgressSubIntfEnableCmd )

#--------------------------------------------------------------------------------
# [ no | default ] sflow egress unmodified enable
#--------------------------------------------------------------------------------
class SflowEgressUnmodifiedEnableCmd( CliCommand.CliCommandClass ):
   syntax = 'sflow egress unmodified enable'
   noOrDefaultSyntax = syntax
   data = {
      'sflow' : sflowNode,
      'egress' : CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'egress',
                                  helpdesc='Egress sFlow' ),
                                  guard=SflowCli.sflowEgressSflowIntfGuard ),
      'unmodified' : CliCommand.Node(
                           matcher=CliMatcher.KeywordMatcher( 'unmodified',
                           helpdesc='Unmodified egress sFlow' ),
                           guard=SflowCli.sflowEgressUnmodifiedSflowIntfGuard ),
      'enable' : 'Enable egress sFlow on an interface'
   }
   handler = SflowCli.intfEgressSflowEnableHandler
   noHandler = SflowCli.intfEgressSflowEnableNoHandler
   defaultHandler = SflowCli.intfEgressSflowEnableDefaultHandler

SwitchIntfModelet.addCommandClass( SflowEgressUnmodifiedEnableCmd )
LagIntfConfigModelet.addCommandClass( SflowEgressUnmodifiedEnableCmd )
EthIntfModelet.addCommandClass( SflowEgressUnmodifiedEnableCmd )

#--------------------------------------------------------------------------------
# [ no | default ] sflow id ID
#--------------------------------------------------------------------------------
class SflowConfigInterfaceCmd( CliCommand.CliCommandClass ):
   syntax = 'sflow id ID'
   noOrDefaultSyntax = 'sflow id ...'
   data = {
      'sflow' : sflowNode,
      'id' : CliCommand.Node(
          matcher=CliMatcher.KeywordMatcher( 'id',
              helpdesc='ID for interface used in sFlow packets' ),
          hidden=True ),
      'ID' : CliMatcher.IntegerMatcher( SampleInterfaceIdType.min,
                                        SampleInterfaceIdType.max,
                                        helpdesc='24 bit unsigned integer' ),
   }
   handler = SflowCli.intfIdHandler
   noOrDefaultHandler = SflowCli.intfIdNoDefaultHandler

SwitchIntfModelet.addCommandClass( SflowConfigInterfaceCmd )
LagIntfConfigModelet.addCommandClass( SflowConfigInterfaceCmd )
EthIntfModelet.addCommandClass( SflowConfigInterfaceCmd )

#--------------------------------------------------------------------------------
# [ no | default ] sflow sample vxlan header strip
#--------------------------------------------------------------------------------
class SflowSampleVxlanHeaderStripCmd( CliCommand.CliCommandClass ):
   syntax = 'sflow sample vxlan header strip'
   noOrDefaultSyntax = syntax
   data = {
      'sflow' : sflowNode,
      'sample' : matcherSample,
      'vxlan' : 'Set sample characteristics for VXLAN packets',
      'header' : 'Set sample characteristics for VXLAN packets',
      'strip' : CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'strip',
            helpdesc='Strip VXLAN tunnel headers of VXLAN encapsulated '
                     'sample packets' ),
         guard=SflowCli.sflowSampleVxlanHeaderStripSupportedGuard ),
   }
   handler = SflowCli.vxlanHeaderStrip
   noOrDefaultHandler = SflowCli.noVxlanHeaderStrip

BasicCliModes.GlobalConfigMode.addCommandClass( SflowSampleVxlanHeaderStripCmd )
