# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliCommand
from CliMatcher import (
   DynamicNameMatcher,
   EnumMatcher,
   IntegerMatcher,
   KeywordMatcher,
)
from CliPlugin.FlowTrackingCliLib import guardSampledHwOffload
from CliPlugin.Ip6AddrMatcher import Ip6AddrMatcher
from CliPlugin.IpAddrMatcher import IpAddrMatcher
from CliPlugin.IpGenAddrMatcher import IpGenAddrMatcher
from CliPlugin.VrfCli import getAllPlusReservedVrfNames
from FlowTrackerCliUtil import ftrTypeSampled
from Intf.IntfRange import intfRangeMatcher
import Smash
from SftCliUtil import HoFlowStateEnum
from TypeFuture import TacLazyType

IpProtoType = TacLazyType( 'Arnet::IpProtocolNumber' )
SubIntfId = TacLazyType( 'Arnet::SubIntfId' )
AllowedPacketDirection = TacLazyType( 'Bridging::AllowedTrafficDirection' )

tokenSrcIp = KeywordMatcher( 'src-ip', "Flow source IP address" )

tokenDstIp = KeywordMatcher( 'dst-ip', "Flow destination IP address" )

tokenIPv4 = CliCommand.Node( IpAddrMatcher( helpdesc='IPv4 address' ), maxMatches=1 )

tokenIPv6 = CliCommand.Node( Ip6AddrMatcher( helpdesc='IPv6 address' ),
                             maxMatches=1 )

tokenIPGen = CliCommand.Node( IpGenAddrMatcher( helpdesc='IP address (v4 or v6)' ),
                              maxMatches=1 )

tokenSrcPort = KeywordMatcher( 'src-port', "Flow source port" )

tokenDstPort = KeywordMatcher( 'dst-port', "Flow destination port" )

tokenPort = CliCommand.Node( IntegerMatcher( 0, 65535, helpdesc='IP port' ),
                             maxMatches=1 )

tokenVrf = KeywordMatcher( 'vrf', "Flow VRF" )

tokenVrfValue = CliCommand.Node( DynamicNameMatcher( getAllPlusReservedVrfNames,
                                                     "VRF" ),
                                 maxMatches=1 )

tokenVlan = KeywordMatcher( 'vlan', "Flow VLAN ID" )

_vlanMatcher = IntegerMatcher( 0, 4094, helpdesc='Identifier for a Virtual LAN' )
tokenVlanValue = CliCommand.Node( _vlanMatcher, maxMatches=1 )

protocols = [ ( p[ len( 'ipProto' ) : ] if p.startswith( 'ipProto' ) else p )
              for p in IpProtoType.attributes ]

tokenProtocol = KeywordMatcher( 'protocol', "Flow IP protocol" )

tokenProtocolValue = CliCommand.Node( DynamicNameMatcher( lambda mode: protocols,
                                                          "IP protocol" ),
                                      maxMatches=1 )

tokenInterface = KeywordMatcher( 'interface', "ingress/egress interface" )

tokenState = KeywordMatcher( 'state', "Hardware offload flow state" )

tokenDirections = {
   'ingress' : 'ingress',
   'egress' : 'egress'
}

tokenDirectionValue = CliCommand.Node( EnumMatcher( tokenDirections ),
                                       maxMatches=1 )

hoFlowStateShowStr = {
   HoFlowStateEnum.hoFlowStateUnknown : 'unknown',
   HoFlowStateEnum.inHardware : 'active',
   HoFlowStateEnum.inProgress : 'in-progress',
   HoFlowStateEnum.outOfResources : 'hw-limit',
}

def _hoFlowStateValues( mode ):
   return list( hoFlowStateShowStr.values() )

_hoFlowStateMatcher = DynamicNameMatcher( _hoFlowStateValues, "Flow state" )

tokenStateValue = CliCommand.Node( _hoFlowStateMatcher, maxMatches=1 )

intfRangeRule = CliCommand.Node( intfRangeMatcher, maxMatches=1 )

sampledHardwareKw = CliCommand.guardedKeyword( 'hardware', helpdesc='Hardware',
                                            guard=guardSampledHwOffload )

def getFlowTable( shmemEm, trackerName ):
   mountPath = f'flowtracking/{ftrTypeSampled}/flowTable/{trackerName}'
   entityType = 'FlowTracking::FlowTable'
   smashFlowTable = shmemEm.doMount( mountPath, entityType,
                                     Smash.mountInfo( 'reader' ) )
   return smashFlowTable

def getTrackerName( ftConfig, intfId, direction ):
   tracker = 'Unknown'
   # For tracker configured at subinterface, subIntf can be in hwIntfFtConfig
   if not direction:
      direction = 'ingress'
   intfFtConfig = ftConfig.hwIntfFtConfig.get( intfId ) if direction == 'ingress' \
      else ftConfig.hwEgressIntfFtConfig.get( intfId )
   if not intfFtConfig:
      if SubIntfId.isSubIntfId( intfId ):
         intfId = SubIntfId.parentIntfId( intfId )
         intfFtConfig = ftConfig.hwIntfFtConfig.get( intfId ) if direction == \
            'ingress' else ftConfig.hwEgressIntfFtConfig.get( intfId )
   if intfFtConfig:
      tracker = intfFtConfig.hwFtConfig.name
   return tracker

# May be further impacted by
# https://tools.ietf.org/html/draft-tgraf-ipfix-mpls-sr-label-type-05#section-3
#
# some discussion on ipfix-dev@:
# "IPFIX export of MPLS related flows - LFIB source and Tunnel Type mapping"
# https://groups.google.com/a/arista.com/g/ipfix-dev/c/tEwtfJa9SD8/m/pFctxC_vBQAJ
topLabelTypeMapping = {
   'teMidpt' : {
      'lfibSourceGribi',
      'lfibSourceRsvp',
      'lfibSourceSrTePolicy',
   },
   'vpn' : {
      'lfibSourceBgpL2Evpn',
      'lfibSourceBgpL3Vpn',
   },
   'pseudowire' : {
      'lfibSourcePseudowire',
   },
   'ldp' : {
      'lfibSourceLdp',
      'lfibSourceMldp',
      'lfibSourceIsisSrToLdp',
      'lfibSourceOspfSrToLdp',
   },
   'bgp' : {
      'lfibSourceBgpLs',
      'lfibSourceBgpLu',
   },
}

lfibSourceToTopLabelType = {}
for _value, _sources in topLabelTypeMapping.items():
   for _source in _sources:
      lfibSourceToTopLabelType[ _source ] = _value
