# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from Ark import timestampToStr
from ArnetModel import (
   IpGenericAddrAndPort,
   IpGenericAddress,
   IpGenericPrefix,
   MacAddress,
)
from CliModel import (
   Bool,
   Dict,
   Enum,
   Float,
   Int,
   List,
   Model,
   Str,
   Submodel,
)
from Ethernet import convertMacAddrToDisplay
from FlowTrackerCliUtil import(
   FtrTypeEnum,
   addressStr,
   ftrTypeShowStr,
   protocolStr,
   renderCounter,
   timeStr,
)
from IntfModels import Interface
from CliPlugin.SftCliLib import hoFlowStateShowStr, AllowedPacketDirection
from SftCliUtil import HoFlowStateEnum
from SftLib import (
   labelStackString,
   valueOrUnknown,
)
import TableOutput
import Tac
import TacSigint
from TypeFuture import TacLazyType

FtConsts = TacLazyType( 'FlowTracking::Constants' )
IpProtoType = TacLazyType( 'Arnet::IpProtocolNumber' )

def pindent( indent, *args ):
   print( " " * ( 2 * indent ) + " ".join( map( str, args ) ) )

def macAddrStrOrNone( macAddr ):
   if macAddr is None:
      return ''
   else:
      return convertMacAddrToDisplay( macAddr.stringValue )

def timeStrOrNone( timestamp ):
   if timestamp is None:
      return ''
   else:
      return timeStr( timestamp )

def utcTimestampToStr( timestamp ):
   return timestampToStr( timestamp, now=Tac.utcNow() )

def lastSentStr( timestamp ):
   if timestamp is None:
      return ''
   else:
      return ', last sent ' + utcTimestampToStr( timestamp )

def getLastClearedStr( timestamp ):
   if timestamp is None:
      return ''
   else:
      return f' (Last cleared {utcTimestampToStr( timestamp )})'

def collectorAddr( addrAndPort ):
   if addrAndPort.ip.af == 'ipv6':
      fmtStr = '[{}]:{}'
   else:
      fmtStr = '{}:{}'
   return fmtStr.format( addrAndPort.ip, addrAndPort.port )

def templateIdNum( staticTemplateIdEnum ):
   return FtConsts.reservedTemplateId( staticTemplateIdEnum )

def getIntfNameForFlowKey( flow, key ):
   if key.direction == AllowedPacketDirection.egress:
      return flow.egressIntf.stringValue
   else:
      return flow.ingressIntf.stringValue

# We will move to Stash when it is ready. For now, redender python
# Json model as a temporary solution

class FlowKeyModel( Model ):
   vrfName = Str( help="VRF name" )
   vlanId = Int( help="Flow VLAN ID" )
   srcAddr = IpGenericAddress( help="Source IP address" )
   dstAddr = IpGenericAddress( help="Destination IP address" )
   ipProtocol = Enum( help="IP protocol", values=IpProtoType.attributes,
                      optional=True )
   ipProtocolNumber = Int( help="IP protocol number" )
   direction = Enum( help="Packet Direction",
                     values=AllowedPacketDirection.attributes )

   srcPort = Int( help="Source port" )
   dstPort = Int( help="Destination port" )

topLabelTypeValues = {
   'teMidpt' : 'TE',
   'pseudowire' : 'pseudowire',
   'vpn' : 'VPN',
   'bgp' : 'BGP',
   'ldp' : 'LDP',
}

class MplsFlowDetail( Model ):
   topLabel = Int( help="Ingress MPLS top label", optional=True )
   labelStack = List( help="Egress MPLS label stack", valueType=int )
   labelStackTruncated = Bool( help="The actual flow had labels beyond "
                               "'labelStack'" )
   fec = IpGenericPrefix( help="Forwarding equivalence class assignment",
                          optional=True )
   topLabelType = Enum( help="Control protocol that allocated 'topLabel'",
                        values=topLabelTypeValues,
                        optional=True )

class FlowDetailModel( Model ):
   srcEthAddr = MacAddress( help="Flow source MAC address" )
   dstEthAddr = MacAddress( help="Flow destination MAC address" )
   sampledBytesReceived = Int( help="Number of sampled bytes received",
                               optional=True )
   sampledPktsReceived = Int( help="Number of sampled packets received",
                              optional=True )
   hwBytesReceived = Int( help="Number of bytes received in hardware",
                          optional=True )
   hwPktsReceived = Int( help="Number of packets received in hardware",
                         optional=True )
   flowLabel = Int( help="IPv6 flow label", optional=True )
   tos = Int( help="TOS in IP header" )
   tcpFlags = Str( help="Flow TCP flags" )
   lastPktTime = Float( help="Last packet received time" )
   ingressVlanId = Int( help="Flow ingress VLAN ID" )
   ingressIntf = Interface( help="Flow ingress interface" )
   egressVlanId = Int( help="Flow egress VLAN ID" )
   # egressIntf may not be an IntfId, i.e. multicast/discard
   egressIntf = Str( help="Flow egress interface" )
   srcAs = Int( help="BGP source AS" )
   dstAs = Int( help="BGP destination AS" )
   dot1qVlanId = Int( help="IEEE 802.1Q tag VLAN ID", optional=True )
   dot1qCustomerVlanId = Int( help="IEEE 802.1Q inner tag VLAN ID",
                              optional=True )
   nextHopIp = IpGenericAddress( help="Next hop IP address" )
   bgpNextHopIp = IpGenericAddress( help="BGP next hop IP address" )
   srcPrefixLen = Int( help="Source prefix length" )
   dstPrefixLen = Int( help="Destination prefix length" )
   vni = Str( help="VXLAN Network Identifier" )
   mpls = Submodel( valueType=MplsFlowDetail,
                    help="MPLS flow details",
                    optional=True )

class FlowModel( Model ):
   key = Submodel( valueType=FlowKeyModel, help="Flow key" )
   bytesReceived = Int( help="Number of bytes received" )
   pktsReceived = Int( help="Number of packets received" )
   startTime = Float( help="Flow start time" )
   flowDetail = Submodel( valueType=FlowDetailModel, optional=True,
                          help="Flow detailed information" )

class GroupModel( Model ):
   flows = List( valueType=FlowModel, help="List of flows" )

class TrackerModel( Model ):
   groups = Dict( keyType=str, valueType=GroupModel,
                  help="A mapping bewteen group name and group" )
   numFlows = Int( help="Total number of flows" )

# Sample Tracking model Json output
# {
#     "running": true,
#     "trackers": {
#         "ftTest1": {
#             "groups": {
#                 "IPv4": {
#                     "flows": [
#                         {
#                             "bytesReceived": 0,
#                             "key": {
#                                 "vrfName": "default",
#                                 "vlanId": 0,
#                                 "srcAddr": "1.1.1.1",
#                                 "dstPort": 1,
#                                 "ipProtocol": "ipProtoTcp",
#                                 "dstAddr": "2.2.2.1",
#                                 "srcPort": 1
#                             },
#                             "startTime": 1543015649.679666,
#                             "pktsReceived": 0,
#                             "flowDetail": {
#                                 "lastPktTime": 0.0,
#                                 "bgpNextHopIp": "0.0.0.0",
#                                 "srcAs": 0,
#                                 "srcEthAddr": "00:00:00:00:00:00",
#                                 "dstPrefixLen": 0,
#                                 "tos": 0,
#                                 "ingressIntf": "",
#                                 "tcpFlags": "none",
#                                 "dstAs": 0,
#                                 "egressVlanId": 0,
#                                 "egressIntf": "",
#                                 "dstEthAddr": "00:00:00:00:00:00",
#                                 "nextHopIp": "0.0.0.0",
#                                 "vni": "unknown",
#                                 "dot1qVlanId": "10",
#                                 "dot1qCustomerVlanId": "100",
#                                 "sampledBytesReceived": 0,
#                                 "sampledPktsReceived": 0,
#                                 "hwBytesReceived": 0,
#                                 "hwPktsReceived": "0,
#                             }
#                         }
#                     ]
#                 },
#                 "IPv6": {
#                     "flows": [
#                         {
#                             "bytesReceived": 0,
#                             "key": {
#                                 "vrfName": "default",
#                                 "vlanId": 0,
#                                 "srcAddr": "2001:db8:1:1::1",
#                                 "dstPort": 1,
#                                 "ipProtocol": "ipProtoTcp",
#                                 "dstAddr": "2001:db8:2:2::1",
#                                 "srcPort": 1
#                             },
#                             "startTime": 1543015649.681361,
#                             "pktsReceived": 0,
#                             "flowDetail": {
#                                 "lastPktTime": 0.0,
#                                 "bgpNextHopIp": "::",
#                                 "srcAs": 0,
#                                 "srcEthAddr": "00:00:00:00:00:00",
#                                 "dstPrefixLen": 0,
#                                 "tos": 0,
#                                 "ingressIntf": "",
#                                 "tcpFlags": "none",
#                                 "srcPrefixLen": 0,
#                                 "dstAs": 0,
#                                 "egressVlanId": 0,
#                                 "egressIntf": "",
#                                 "dstEthAddr": "00:00:00:00:00:00",
#                                 "nextHopIp": "::",
#                                 "vni": "unknown",
#                                 "dot1qVlanId": "10",
#                                 "dot1qCustomerVlanId": "100",
#                                 "flowLabel": 0,
#                                 "hwBytesReceived": 0,
#                                 "hwPktsReceived": "0,
#                             }
#                         }
#                     ]
#                 }
#             },
#             "numFlows": 2
#         }
#     }
# }
class TrackingModel( Model ):
   trackers = Dict( keyType=str, valueType=TrackerModel,
                    help="A mapping between tracker name and tracker" )
   running = Bool( help="Sampled flow tracking is active" )
   ftrType = Enum( help="Flow tracker type", values=FtrTypeEnum.attributes )

   def vlanIdOrRouted( self, vlanId, intf=None ):
      """
      Render the VLAN ID numerically or 'routed' or 'unknown' depending on if the
      sampled packet was bridged within a VLAN, routed, or if this information could
      not be determine (e.g. the ingress interface is unknown for egress samples for
      certain platforms).

      vlanId  : VLAN ID to render.
      intf    : Interface name in the VLAN; may be a string or an
                IntfModels.Interface. When displaying a flow key, it may be entirely
                unspecified. This is signaled using None.
      """
      if intf in ( '', 'unknown', Tac.Value( 'Arnet::IntfId', '' ) ):
         return 'unknown'
      return vlanId or 'routed'

   def renderFlowDetail( self, flow, key ):
      detail = flow.flowDetail
      tab = " " * 4
      print( "%sFlow: %s %s - %s, VRF: %s, VLAN: %s" %
             ( tab, protocolStr( key.ipProtocol, key.ipProtocolNumber ),
               addressStr( key.srcAddr,
                           key.srcPort ),
               addressStr( key.dstAddr, key.dstPort ),
               key.vrfName,
               self.vlanIdOrRouted( detail.ingressVlanId, detail.ingressIntf ) ) )
      tab = " " * 6
      print( "%sStart time: %s, Last packet time: %s" %
             ( tab, timeStr( flow.startTime ), timeStr( detail.lastPktTime ) ) )
      flowLabel = ", Flow label: " + ( str( detail.flowLabel )
                                       if detail.flowLabel is not None else '' )
      print( "%sPackets: %d, Bytes: %d, TOS: %d, TCP flags: %s%s" %
             ( tab, flow.pktsReceived, flow.bytesReceived, detail.tos,
               detail.tcpFlags, flowLabel ) )
      if detail.sampledPktsReceived is not None:
         print( "%sSampled packets: %d, Sampled bytes: %d, Hardware packets: %d,"
                " Hardware bytes: %d" %
                ( tab, detail.sampledPktsReceived, detail.sampledBytesReceived,
                  detail.hwPktsReceived, detail.hwBytesReceived ) )
      print( "%sSource MAC: %s, Destination MAC: %s" %
             ( tab, convertMacAddrToDisplay( detail.srcEthAddr.stringValue ),
               convertMacAddrToDisplay( detail.dstEthAddr.stringValue ) ) )
      if detail.dot1qCustomerVlanId is not None:
         print( "%s802.1Q VLAN ID: %s, 802.1Q customer VLAN ID: %s" %
             ( tab, valueOrUnknown( detail.dot1qVlanId ), valueOrUnknown(
                  detail.dot1qCustomerVlanId ) ) )
      elif detail.dot1qVlanId is not None:
         print( "%s802.1Q VLAN ID: %s" %
             ( tab, valueOrUnknown( detail.dot1qVlanId ) ) )
      print( "%sIngress interface: %s, Egress VLAN: %s, Egress interface: %s" %
             ( tab,
               valueOrUnknown( detail.ingressIntf ),
               self.vlanIdOrRouted( detail.egressVlanId, detail.egressIntf ),
               detail.egressIntf ) )
      print( "%sNext hop: %s, BGP next hop: %s (AS %s), Source AS: %s" %
             ( tab, valueOrUnknown( detail.nextHopIp ),
               valueOrUnknown( detail.bgpNextHopIp ), valueOrUnknown(
                  detail.dstAs ), valueOrUnknown( detail.srcAs ) ) )
      print( "%sSource prefix length: %s, Destination prefix length: %s" %
             ( tab, valueOrUnknown(
                detail.srcPrefixLen ), valueOrUnknown( detail.dstPrefixLen ) ) )
      if detail.mpls:
         print( "%sIngress MPLS label: %s, Egress MPLS label stack: %s" %
                ( tab, detail.mpls.topLabel,
                  labelStackString( detail.mpls.labelStack,
                                    detail.mpls.labelStackTruncated ) ) )
         fecSrc = "%s" % valueOrUnknown( detail.mpls.fec )
         if detail.mpls.fec:
            fecSrc += " (%s)" % valueOrUnknown(
               topLabelTypeValues.get( detail.mpls.topLabelType ) )
         print( f"{tab}MPLS FEC: {fecSrc}" )

   def render( self ):
      if not self.running:
         print( "%s flow tracking is not active" % ftrTypeShowStr[ self.ftrType ] )
         return

      headings = ( "VRF", "VLAN", "Source", "Destination", "Protocol",
                   "Direction", "Start Time", "Pkts", "Bytes" )
      formatLeft = TableOutput.Format( justify="left" )
      formatLeft.noPadLeftIs( True )
      formatCenter = TableOutput.Format( justify="center" )
      formatRight = TableOutput.Format( justify="right" )

      if self.trackers:
         print()
      for trackerName, tracker in sorted( self.trackers.items() ):
         print( f"Tracker: {trackerName}, Flows: {tracker.numFlows}" )
         for groupName, group in sorted( tracker.groups.items() ):
            if not group.flows:
               continue

            # check the first flow to see if we need to display detailed format
            if group.flows[ 0 ].flowDetail:
               print( f"  Group: {groupName}, Flows: {len( group.flows )}" )
            else:
               table = TableOutput.createTable( headings )
               table.formatColumns( formatLeft, formatLeft, formatLeft,
                                    formatLeft, formatCenter, formatLeft,
                                    formatRight, formatRight, formatRight )
               print( f"Group: {groupName}, Flows: {len( group.flows )}\n" )

            for flow in group.flows:
               key = flow.key
               if flow.flowDetail:
                  self.renderFlowDetail( flow, key )
               else:
                  table.newRow( key.vrfName, self.vlanIdOrRouted( key.vlanId ),
                                addressStr( key.srcAddr, key.srcPort ),
                                addressStr( key.dstAddr, key.dstPort ),
                                protocolStr( key.ipProtocol,
                                             key.ipProtocolNumber ),
                                key.direction,
                                timeStr( flow.startTime ),
                                flow.pktsReceived,
                                flow.bytesReceived )
               TacSigint.check()

            if not group.flows[ 0 ].flowDetail:
               print( table.output() )

# hardware offload flow-table counters
class HoFlowCounterEntryModel( Model ):
   key = Submodel( valueType=FlowKeyModel,
                   help="Hardware offload flow counter key" )
   ingressIntf = Interface( help="Flow ingress interface", optional=True )
   egressIntf = Interface( help="Flow egress interface", optional=True )
   pktsReceived = Int( help="Number of packets received" )
   bytesReceived = Int( help="Number of bytes received" )
   createTime = Float( help="Hardware offload flow counter entry create time" )
   updateTime = Float( help="Hardware offload flow counter entry update time" )

class HoGroupCounterModel( Model ):
   flows = List( valueType=HoFlowCounterEntryModel, help="List of flows" )

class HoTrackerCounterModel( Model ):
   groups = Dict( keyType=str, valueType=HoGroupCounterModel,
                  help="A mapping bewteen group name and group" )
   numFlows = Int( help="Total number of flows" )

# Sample Hardware Offload FlowTable Counters model Json output
# {
#     "running": true,
#     "ftrType": sampled,
#     "trackers": {
#         "ftr1": {
#             "groups": {
#                 "IPv4": {
#                     "flows": [
#                         {
#                             "key": {
#                                 "srcAddr": "1.1.1.1",
#                                 "ipProtocolNumber": 6,
#                                 "dstAddr": "2.2.2.1",
#                                 "vrfName": "default",
#                                 "ipProtocol": "ipProtoTcp",
#                                 "dstPort": 1,
#                                 "srcPort": 1
#                                 "vlanId": 0,
#                             },
#                             "ingressIntf": "Ethernet1",
#                             "pktsReceived": 10,
#                             "bytesReceived": 1280,
#                             "createTime": 1587357689.51372,
#                             "updateTime": 1587357689.51372,
#                         }
#                     ]
#                 },
#             },
#             "numFlows": 1
#         }
#     }
# }

class HoFlowCountersModel( Model ):
   trackers = Dict( keyType=str, valueType=HoTrackerCounterModel,
                    help="A mapping between tracker name and tracker" )
   running = Bool( help="Sampled flow tracking is active" )
   ftrType = Enum( help="Flow tracker type", values=FtrTypeEnum.attributes )

   def vlanIdOrRouted( self, vlanId ):
      return vlanId or 'routed'

   def render( self ):
      if not self.running:
         print( "%s flow tracking is not active" % ftrTypeShowStr[ self.ftrType ] )
         return

      headings = ( "VRF", "VLAN", "Source", "Destination", "Protocol",
                   "Interface", "Direction", "Packets", "Bytes", "Create Time",
                   "Update Time" )
      formatLeft = TableOutput.Format( justify="left" )
      formatLeft.noPadLeftIs( True )
      formatLeft.padLimitIs( True )
      formatCenter = TableOutput.Format( justify="center" )
      formatCenter.padLimitIs( True )
      formatRight = TableOutput.Format( justify="right" )
      formatRight.padLimitIs( True )

      for trackerName in sorted( self.trackers, key=str.lower ):
         tracker = self.trackers[ trackerName ]
         print( f"Tracker: {trackerName}, Flows: {tracker.numFlows}\n" )
         for groupName, group in sorted( tracker.groups.items() ):
            if not group.flows:
               continue

            table = TableOutput.createTable( headings )

            table.formatColumns( formatLeft, formatLeft, formatLeft, formatLeft,
                                 formatCenter, formatRight, formatLeft,
                                 formatRight, formatRight, formatRight,
                                 formatRight )
            print( f"Group: {groupName}, Flows: {len( group.flows )}\n" )

            for flow in group.flows:
               key = flow.key

               table.newRow( key.vrfName, self.vlanIdOrRouted( key.vlanId ),
                             addressStr( key.srcAddr, key.srcPort ),
                             addressStr( key.dstAddr, key.dstPort ),
                             protocolStr( key.ipProtocol, key.ipProtocolNumber ),
                             getIntfNameForFlowKey( flow, key ),
                             key.direction,
                             flow.pktsReceived, flow.bytesReceived,
                             timeStr( flow.createTime ),
                             timeStr( flow.updateTime ) )

               TacSigint.check()

            print( table.output() )

# hardware offload flow-table
class HoFlowDetailModel( Model ):
   srcEthAddr = MacAddress( help="Flow source MAC address", optional=True )
   dstEthAddr = MacAddress( help="Flow destination MAC address", optional=True )
   swCreateTime = Float( help="Hardware offload flow config entry create time",
                         optional=True )
   hwUpdateTime = Float( help="Hardware offload flow status entry update time" )

class HoFlowEntryModel( Model ):
   key = Submodel( valueType=FlowKeyModel,
                   help="Hardware offload flow Status key" )
   ingressIntf = Interface( help="Flow ingress interface", optional=True )
   egressIntf = Interface( help="Flow egress interface", optional=True )
   state = Enum( values=HoFlowStateEnum.attributes,
                 help="Hardware offload flow state" )
   hwCreateTime = Float( help="Hardware offload flow status entry create time" )
   flowDetail = Submodel( valueType=HoFlowDetailModel, optional=True,
                          help="Hardware offload flow detailed information" )

class HoGroupModel( Model ):
   flows = List( valueType=HoFlowEntryModel, help="List of flows" )

class HoTrackerModel( Model ):
   groups = Dict( keyType=str, valueType=HoGroupModel,
                  help="A mapping bewteen group name and group" )
   numFlows = Int( help="Total number of flows" )

# Sample Hardware Offload FlowTable model Json output
# {
#     "running": true,
#     "ftrType": sampled,
#     "trackers": {
#         "ftr1": {
#             "groups": {
#                 "IPv4": {
#                     "flows": [
#                         {
#                             "hwCreateTime": 1587357689.51372,
#                             "state": "inHardware",
#                             "key": {
#                                 "srcAddr": "1.1.1.1",
#                                 "ipProtocolNumber": 6,
#                                 "dstAddr": "2.2.2.1",
#                                 "vrfName": "default",
#                                 "ipProtocol": "ipProtoTcp",
#                                 "dstPort": 1,
#                                 "srcPort": 1
#                                 "vlanId": 0,
#                             },
#                             "flowDetail": {
#                                 "swCreateTime": 1587357689.51372,
#                                 "hwUpdateTime": 1587357689.51372,
#                                 "srcEthAddr": "00:01:02:03:04:05",
#                                 "dstEthAddr": "00:00:01:01:01:01",
#                             }
#                             "ingressIntf": "Ethernet1",
#                         }
#                     ]
#                 },
#             },
#             "numFlows": 1
#         }
#     }
# }

class HoFlowTableModel( Model ):
   trackers = Dict( keyType=str, valueType=HoTrackerModel,
                    help="A mapping between tracker name and tracker" )
   running = Bool( help="Sampled flow tracking is active" )
   ftrType = Enum( help="Flow tracker type", values=FtrTypeEnum.attributes )

   def vlanIdOrRouted( self, vlanId ):
      return vlanId or 'routed'

   def renderHoFlowDetail( self, flow, key ):
      detail = flow.flowDetail
      tab = " " * 4
      print(
         "%sFlow: %s %s - %s, VRF: %s, VLAN: %s, HW state: %s" %
         ( tab, protocolStr( key.ipProtocol, key.ipProtocolNumber ),
           addressStr( key.srcAddr, key.srcPort ),
           addressStr( key.dstAddr, key.dstPort ), key.vrfName,
           self.vlanIdOrRouted( key.vlanId ), hoFlowStateShowStr[ flow.state ] ) )
      tab = " " * 6
      direction = key.direction
      print( "%sInterface: %s, SW create time: %s, Direction: %s" %
             ( tab, getIntfNameForFlowKey( flow, key ), timeStrOrNone(
                detail.swCreateTime ), direction ) )
      print( "%sHW create time: %s, HW last update time: %s" %
             ( tab, timeStr( flow.hwCreateTime ), timeStr( detail.hwUpdateTime ) ) )
      print( "%sSource MAC: %s, Destination MAC: %s" %
             ( tab, macAddrStrOrNone(
                detail.srcEthAddr ), macAddrStrOrNone( detail.dstEthAddr ) ) )

   def render( self ):
      '''
      Sample detail output:

      Tracker: ftr1, Flows: 1
        Group: IPv4, Flows: 1
          Flow: UDP 1.1.1.1:0 - 1.2.1.2:0, VRF: red, VLAN: 2, HW state: active
               Ingress interface: Ethernet1, SW create time: 2020-04-19 21:41:29.0
               HW create time: 2020-04-19 21:41:29.513720, HW last update time:
2020-04-19 21:41:29.513720
               Source MAC: 0001.0203.0405, Destination MAC: 0000.0101.0101

      '''

      if not self.running:
         print( "%s flow tracking is not active" % ftrTypeShowStr[ self.ftrType ] )
         return

      headings = ( "VRF", "VLAN", "Source", "Destination", "Protocol",
                   "Interface", "Direction", "State", "Create Time" )
      formatLeft = TableOutput.Format( justify="left" )
      formatLeft.noPadLeftIs( True )
      formatCenter = TableOutput.Format( justify="center" )
      formatRight = TableOutput.Format( justify="right" )

      if self.trackers:
         print()
      for trackerName in sorted( self.trackers, key=str.lower ):
         tracker = self.trackers[ trackerName ]
         print( f"Tracker: {trackerName}, Flows: {tracker.numFlows}" )
         for groupName, group in sorted( tracker.groups.items() ):
            if not group.flows:
               continue

            # check the first flow to see if we need to display detailed format
            if group.flows[ 0 ].flowDetail:
               print( f"  Group: {groupName}, Flows: {len( group.flows )}" )
            else:
               table = TableOutput.createTable( headings )

               table.formatColumns( formatLeft, formatLeft, formatLeft,
                                    formatLeft, formatCenter, formatRight,
                                    formatLeft, formatRight, formatRight )
               print( f"Group: {groupName}, Flows: {len( group.flows )}\n" )

            for flow in group.flows:
               key = flow.key
               if flow.flowDetail:
                  self.renderHoFlowDetail( flow, key )
               else:
                  table.newRow( key.vrfName, self.vlanIdOrRouted( key.vlanId ),
                                addressStr( key.srcAddr, key.srcPort ),
                                addressStr( key.dstAddr, key.dstPort ),
                                protocolStr( key.ipProtocol,
                                             key.ipProtocolNumber ),
                                getIntfNameForFlowKey( flow, key ),
                                key.direction,
                                hoFlowStateShowStr[ flow.state ],
                                timeStr( flow.hwCreateTime ) )
               TacSigint.check()

            if not group.flows[ 0 ].flowDetail:
               print( table.output() )

# flow tracking counters
class CollectorSetCounters( Model ):
   flowRecords = Int(
      help="Number of flow records sent to the collector" )

class CollectorTemplateCounters( Model ):
   templateType = Enum(
      help="IPFIX template type",
      values=( "template", "optionsTemplate", ) )
   templates = Int(
      help="Number of templates sent to the collector" )

class CollectorTimestamp( Model ):
   message = Float(
      help="UTC time of last exported message",
      optional=True )
   template = Float(
      help="UTC time of last exported template",
      optional=True )
   dataRecord = Float(
      help="UTC time of last exported data record",
      optional=True )
   optionsData = Float(
      help="UTC time of last exported options data record",
      optional=True )

class CollectorCounters( Model ):
   addrAndPort = Submodel(
      help="Collector IP address and port number",
      valueType=IpGenericAddrAndPort )
   exportedMessageTotalCount = Int(
      help="The total number of messages sent to the collector" )
   exportedFlowRecordTotalCount = Int(
      help="The total number of flow records sent to the collector" )
   exportedOctetTotalCount = Int(
      help="The total number of bytes sent to the collector" )
   lastUpdates = Submodel(
      help="UTC time of last events",
      valueType=CollectorTimestamp )
   sets = Dict(
      help="A mapping of IPFIX template ID (set ID) to per-set-id counters",
      keyType=int,
      valueType=CollectorSetCounters )
   templates = List(
      help="A list of per-template counters",
      valueType=CollectorTemplateCounters )

class ExporterCounters( Model ):
   exporterType = Enum(
      help="Exporter type",
      values=( "ipfix", ) )
   clearTime = Float(
      help="UTC time when counters were last cleared",
      optional=True )
   collectors = List(
      help="A list of per-collector counters",
      valueType=CollectorCounters )

def exporterTypeStr( exporterType ):
   mapping = {
      'ipfix' : 'IPFIX',
   }
   return mapping[ exporterType ]

class HardwareOffloadCounters( Model ):
   flowsActive = Int( help="Number of active flows in hardware", optional=True )
   flowsCreated = Int( help="Number of flows created in hardware", optional=True )
   flowsDeleted = Int( help="Number of flows deleted from hardware", optional=True )
   flowsPending = Int(
         help="Number of flows pending to be created in hardware", optional=True )
   hoLimitIpv4 = Int( help="Maximum number of IPv4 flows to try to offload",
                      optional=True )
   hoLimitIpv6 = Int( help="Maximum number of IPv6 flows to try to offload",
                      optional=True )
   pktsReceived = Int(
         help="Number of packets received for hardware flows", optional=True )
   pktsDiscarded = Int(
         help="Number of packets discarded in CPU for hardware flows",
         optional=True )
   pktsHardwareMiss = Int(
         help="Number of packets that didn't hit hardware flows", optional=True )
   pktsHardwareFailed = Int(
         help="Number of packets that failed to create hardware flow",
         optional=True )
   clearTime = Float(
         help="UTC time when counters were last cleared", optional=True )

class FlowGroupCounters( Model ):
   activeFlows = Int(
      help="Number of active flows" )
   expiredFlows = Int(
      help="Cumulative expired flow count" )
   flows = Int(
      help="Cumulative flow count" )
   packets = Int(
      help="Cumulative count of sampled packets" )
   hardwareOffload = Submodel(
      valueType=HardwareOffloadCounters,
      help="Hardware flow group counters",
      optional=True )

class TrackerCounters( Model ):
   activeFlows = Int(
      help="Number of active flows" )
   expiredFlows = Int(
      help="Cumulative expired flow count" )
   flows = Int(
      help="Cumulative flow count" )
   packets = Int(
      help="Cumulative count of sampled packets" )
   clearTime = Float(
      help="UTC time when counters were last cleared",
      optional=True )
   flowGroups = Dict(
      help="A mapping of flow group name to counters",
      keyType=str,
      valueType=FlowGroupCounters )
   exporters = Dict(
      help="A mapping of exporter name to counters",
      valueType=ExporterCounters )

class SampleCounters( Model ):
   received = Int( help="Number of samples received" )
   discarded = Int( help="Number of samples discared", optional=True )
   poolSize = Int( help="Sample pool size", optional=True )
   hardwareReceived = Int( help="Number of hardware samples", optional=True )

class FtrCounters( Model ):
   running = Bool( help="Flow tracking agent is running" )
   trackers = Dict(
      help="A mapping of tracker name to counters",
      keyType=str,
      valueType=TrackerCounters )
   flows = Int( help="Cumulative flow count", optional=True )
   activeFlows = Int( help="Cumulative active flow count", optional=True )
   expiredFlows = Int( help="Cumulative expired flow count", optional=True )
   packets = Int( help="Cumulative count of sampled packets", optional=True )
   clearTime = Float(
         help="UTC time when total counters were last cleared",
         optional=True )
   ftrType = Enum( help="Flow tracker type", values=FtrTypeEnum.attributes )
   sampleCounters = Submodel(
      valueType=SampleCounters,
      help="Sample counters",
      optional=True )
   hardwareOffload = Submodel(
      valueType=HardwareOffloadCounters,
      help="Hardware flow group counters",
      optional=True )

   def renderFlowGroup( self, indent, groupInfo ):
      pindent( indent, '{} flows, {} RX packets'.format(
         renderCounter( groupInfo.activeFlows ),
         renderCounter( groupInfo.packets ) ) )
      self.renderHardwareOffload( indent, groupInfo.hardwareOffload )

   def renderCollector( self, indent, collector ):
      pindent( indent, '{} messages{}'.format( renderCounter(
         collector.exportedMessageTotalCount ),
         lastSentStr( collector.lastUpdates.message ) ) )
      flowRecords = 0
      optionsDataRecords = 0
      for setId, setCounter in collector.sets.items():
         if setId in range( templateIdNum( FtConsts.dataTemplateMin ),
                            templateIdNum( FtConsts.dataTemplateMax ) + 1 ):
            flowRecords += setCounter.flowRecords
         else:
            optionsDataRecords += setCounter.flowRecords
      pindent( indent, '{} flow records{}'.format( renderCounter( flowRecords ),
         lastSentStr( collector.lastUpdates.dataRecord ) ) )
      pindent( indent, '{} options data records{}'.format(
         renderCounter( optionsDataRecords ),
         lastSentStr( collector.lastUpdates.optionsData ) ) )
      templates = sum( x.templates for x in collector.templates )
      pindent( indent, '{} templates{}'.format( renderCounter( templates ),
         lastSentStr( collector.lastUpdates.template ) ) )

   def renderExporter( self, indent, expInfo ):
      for collector in expInfo.collectors:
         pindent( indent, 'Collector:', collector.addrAndPort.ip, "port",
                  collector.addrAndPort.port )
         self.renderCollector( indent + 1, collector )

   def renderTrackerDetails( self, indent, tracker ):
      lastCleared = getLastClearedStr( tracker.clearTime )
      pindent( indent, '{} flows, {} RX packets{}'.format(
         renderCounter( tracker.activeFlows ), renderCounter( tracker.packets ),
         lastCleared ) )
      pindent( indent,
               'Flows created: {}, expired: {}'.format(
                  renderCounter( tracker.flows ),
                  renderCounter( tracker.expiredFlows ) ) )
      for fgName in sorted( tracker.flowGroups ):
         groupInfo = tracker.flowGroups[ fgName ]
         pindent( indent, 'Group:', fgName )
         self.renderFlowGroup( indent + 1, groupInfo )

      for expName in sorted( tracker.exporters ):
         expInfo = tracker.exporters[ expName ]
         expType = exporterTypeStr( expInfo.exporterType )
         lastCleared = getLastClearedStr( expInfo.clearTime )
         pindent( indent, 'Exporter:', expName, '({}){}'.format( expType,
                                                                 lastCleared ) )
         self.renderExporter( indent + 1, expInfo )

   def renderTrackers( self, indent ):
      for trName in sorted( self.trackers ):
         tracker = self.trackers[ trName ]
         pindent( indent, 'Tracker:', trName )
         self.renderTrackerDetails( indent + 1, tracker )

   def renderSamples( self, indent, counter ):
      if counter:
         pindent( indent, 'Samples:' )
         line = f'{renderCounter( counter.received )} received'
         if counter.discarded is not None:
            line += f', {renderCounter( counter.discarded )} discards'
         if counter.poolSize is not None:
            line += f', {renderCounter( counter.poolSize )} sample pool'
         pindent( indent + 1, line )
         line = ''
         if counter.hardwareReceived is not None:
            line += '{} hardware samples'.format(
                                    renderCounter( counter.hardwareReceived ) )
            line += ', {} hardware-software difference'.format(
                  renderCounter(
                     abs( counter.received - counter.hardwareReceived ) ) )
         pindent( indent + 1, line )

   def renderHardwareOffload( self, indent, counter ):
      if counter:
         pindent( indent, 'Hardware offload:' )
         flowsStr = 'Flows: '
         if counter.flowsActive is not None:
            flowsStr += '{} active, '.format(
                  renderCounter( counter.flowsActive ) )
         if counter.flowsPending is not None:
            flowsStr += '{} pending, '.format(
                  renderCounter( counter.flowsPending ) )
         flowsStr += '{} created, {} deleted'.format(
                  renderCounter( counter.flowsCreated ),
                  renderCounter( counter.flowsDeleted ) )
         pindent( indent + 1, flowsStr )
         pindent( indent + 1,
               'Packets: {} received, {} discards, {} offload miss, '
               '{} offload failed'.format(
                  renderCounter( counter.pktsReceived ),
                  renderCounter( counter.pktsDiscarded ),
                  renderCounter( counter.pktsHardwareMiss ),
                  renderCounter( counter.pktsHardwareFailed ) ) )
         if counter.hoLimitIpv4 is not None:
            pindent( indent + 1,
                     'IPv4 flow table size: {}'.format(
                        counter.hoLimitIpv4 ) )
         if counter.hoLimitIpv6 is not None:
            pindent( indent + 1,
                     'IPv6 flow table size: {}'.format(
                        counter.hoLimitIpv6 ) )


   def render( self ):
      '''
 Render FtrCounters and all submodels.

 Sample output:
 Total active flows: 1.05K (1050), RX packets: 3.139T (3139123456789)
  (Last cleared 4 days, 19:55:12 ago)
 Total flows created: 4K (4000), expired: 3.05K (3050)
 Samples:
   2.851M (2851100) received, 1K (1000) discards, 1.427B (1426550000) sample pool
   2.852M (2852232) hardware samples, 1.132K (1132) hardware-software difference
 Hardware offload:
   Flows: 9K (9000) active, 1K (1000) pending, 10K (10000) created, 1K (1000) deleted
   Packets: 9K (9000) received, 1K (1000) discards, 0 offload miss, 0 offload failed
 Tracker: ftr1
   9K (9000) flows, 1T (1000000000000) RX packets (Last cleared 4 days, 19:55:12 ago)
   Flows created: 959.47K (959470), expired: 950.47K (950470)
   Group: IPv4
     2K (2000) flows, 5.04K (5040) RX packets
     Hardware offload:
       Flows: 9K (9000) created, 8K (8000) deleted
       Packets: 999 received, 0 discards, 1K (1000) offload miss, 0 offload failed
   Group: IPv6
     300 flows, 2Q (2000000000000000) RX packets
   Exporter: exp1 (IPFIX) (Last cleared 3 days, 18:55:12 ago)
     Collector: 10.0.1.6 port 2055
       1.5K (1500) messages, last sent 0:03:30 ago
       2K (2000) flow records, last sent 0:03:30 ago
       1K (1000) options data records, last sent 0:02:30 ago
       1K (1000) templates, last sent 0:05:00 ago
 Tracker: ftr2
   550 flows, 5.09K (5090) RX packets (Last cleared 3 days, 17:55:12 ago)
   Flows created: 2K (2000), expired: 1.55K (1550)
   Group: IPv4
     220 flows, 5.04K (5040) RX packets
     Hardware offload:
       Flows: 230 created, 40 deleted
       Packets: 499 received, 0 discards, 15 offload miss, 4K (4000) offload failed
   Group: IPv6
     330 flows, 2B (2000000000) RX packets
   Exporter: exp1 (IPFIX) (Last cleared 2 days, 16:55:12 ago)
     Collector: 10.0.5.6 port 2055
       1.5K (1500) messages, last sent 0:03:30 ago
       2M (2000000) flow records, last sent 0:03:30 ago
       100 options data records, last sent 0:02:30 ago
       400 templates, last sent 0:05:00 ago
     Collector: b::6 port 1345
       1.5K (1500) messages, last sent 0:03:30 ago
       2 flow records, last sent 0:03:30 ago
       100 options data records, last sent 0:02:30 ago
       5 templates, last sent 0:05:00 ago
     Collector: b::6 port 5454
       1.5K (1500) messages, last sent 0:05:00 ago
       0 flow records
       0 options data records
       30 templates, last sent 0:05:00 ago
      '''
      if not self.running:
         print( "%s flow tracking is not active" % ftrTypeShowStr[ self.ftrType ] )
         return

      indent = 0
      lastCleared = getLastClearedStr( self.clearTime )
      pindent( indent,
            'Total active flows: {}, RX packets: {}{}'.format(
               renderCounter( self.activeFlows ), renderCounter( self.packets ),
            lastCleared ) )
      pindent( indent, 'Total flows created: {}, expired: {}'.format(
         renderCounter( self.flows ), renderCounter( self.expiredFlows ) ) )
      self.renderSamples( indent, self.sampleCounters )
      self.renderHardwareOffload( indent, self.hardwareOffload )
      self.renderTrackers( indent )

class PacketErrorCounter( Model ):
   flowTableLimit = Int(
      help="Number of packets dropped due to flow table limit" )
   ftInputQueueFull = Int(
      help="Number of packets dropped due to flow table manager input queue full" )
   fttSimReqQueueFull = Int(
      help="Number of packets dropped due to simulation request queue full" )
   fttSimRespQueueFull = Int(
      help="Number of packets dropped due to simulation response queue full" )
   routeInfoQueueFull = Int(
      help="Number of times route information queue is full" )

class FtrCountersDebug( Model ):
   __revision__ = 2
   ftrType = Enum( help="Flow tracker type", values=FtrTypeEnum.attributes )
   running = Bool( help="Flow tracking agent is running" )
   # Present if Flow tracking is running
   packetErrorCounters = Submodel(
      valueType=PacketErrorCounter,
      help="Packet error counters debug",
      optional=True )

   def degrade( self, dictRepr, revision ):
      def degradePacketErrorCounter():
         # Older rev showCountersDebug wasn't allocating packetErrorCounters
         packetErrorCounters = {}
         packetErrorCounters[ 'flowTableLimit' ] = 0
         packetErrorCounters[ 'ftInputQueueFull' ] = 0
         packetErrorCounters[ 'fttSimReqQueueFull' ] = 0
         packetErrorCounters[ 'fttSimRespQueueFull' ] = 0
         packetErrorCounters[ 'routeInfoQueueFull' ] = 0
         return packetErrorCounters

      if revision == 1 and not 'packetErrorCounters' in dictRepr:
         dictRepr[ 'packetErrorCounters' ] = degradePacketErrorCounter()
      return dictRepr

   def render( self ):
      '''
      Flow table limit: 0
      Flow table input queue full: 0
      Simulation request queue full: 0
      Simulation response queue full: 0
      Routing information queue full: 0
      '''
      if not self.running:
         print( "%s flow tracking is not active" % ftrTypeShowStr[ self.ftrType ] )
         return

      indent = 0
      pindent( indent, 'Flow table limit: {}'.format(
         self.packetErrorCounters.flowTableLimit ) )
      pindent( indent, 'Flow table input queue full: {}'.format(
         self.packetErrorCounters.ftInputQueueFull ) )
      pindent( indent, 'Simulation request queue full: {}'.format(
         self.packetErrorCounters.fttSimReqQueueFull ) )
      pindent( indent, 'Simulation response queue full: {}'.format(
         self.packetErrorCounters.fttSimRespQueueFull ) )
      pindent( indent, 'Routing information queue full: {}'.format(
         self.packetErrorCounters.routeInfoQueueFull ) )

class PerInterfaceFlowErrCounters( Model ):
   parseFail = Int( help="Packets that could not be parsed" )
   unknownProtocol = Int( help="Packets with unsuppored protocol type" )
   other = Int( help="Packets dropped for other reasons" )

   def renderCounters( self, table, intfName ):
      table.newRow( intfName, self.parseFail, self.unknownProtocol, self.other )

class InterfaceCountersFlowErr( Model ):
   # Sample JSON output
   # {
   #     "running": true,
   #     "ftrType": sampled,
   #     "ingressCounters": {
   #                 "Ethernet1":{
   #                     "parseFail": 0,
   #                     "unknownProtocol": 0,
   #                     "other": 0
   #                 },
   #                 "Ethernet2":
   #                 {
   #                     "parseFail": 0,
   #                     "unknownProtocol": 0,
   #                     "other": 0
   #                 },
   #     },
   #     "egressCounters": {
   #                 "Ethernet1":{
   #                     "parseFail": 0,
   #                     "unknownProtocol": 0,
   #                     "other": 0
   #                 },
   #     },
   # }

   running = Bool( help="Sampled flow tracking is active" )
   ftrType = Enum( help="Flow tracker type", values=FtrTypeEnum.attributes )
   ingressCounters = Dict( keyType=Interface,
                           valueType=PerInterfaceFlowErrCounters,
                           help="Ingress counters per interface",
                           optional=True )
   egressCounters = Dict( keyType=Interface,
                          valueType=PerInterfaceFlowErrCounters,
                          help="Egress counters per interface",
                          optional=True )

   def render( self ):
      if not self.running:
         print( "%s flow tracking is not active" % ftrTypeShowStr[ self.ftrType ] )
         return
      formatLeft = TableOutput.Format( justify="left" )
      formatLeft.noPadLeftIs( True )
      headings = ( "Interface",
                   "Parsing Failures",
                   "Untracked Protocols",
                   "Other Errors" )
      table = TableOutput.createTable( headings )
      table.formatColumns( formatLeft, formatLeft, formatLeft, formatLeft )

      for ( direction, ctr ) in [ ( "ingress", self.ingressCounters ),
                                  ( "egress", self.egressCounters ) ]:
         if ctr:
            print( "Direction: %s" % direction )
            for idx, counter in ctr.items():
               TacSigint.check()
               counter.renderCounters( table, idx )
            print( table.output() )
