#!/usr/bin/env python3
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import Int, List, Model, Str, Dict
from SharedSecretProfileLib import LifetimeHelper

class SecretModel( Model ):
   secretID = Int( help='The id of the secret when it can be '
                   'represented as an integer [deprecated]' )
   id = Str( help='The id of the secret' )
   secret = Str( help='The encrypted secret used for authentication' )
   rxLifetimeStart = Int( help='The time (UTC) when the secret is valid for '
                          'receiving data' )
   rxLifetimeEnd = Int( help='The time (UTC) when the secret is no longer valid for '
                        'receiving data' )
   txLifetimeStart = Int( help='The time (UTC) when the secret is valid for '
                          'transmitting data' )
   txLifetimeEnd = Int( help='The time (UTC) when the secret is no longer valid for '
                        'transmitting data' )
   timezone = Str( help='The timezone of the secret lifetime', optional=True )

   def render( self ):
      print( "   ID %s" % self.id )
      print( "      Secret: %s" % self.secret )
      rxStr = LifetimeHelper.displayDuration( self.rxLifetimeStart,
                                              self.rxLifetimeEnd, self.timezone )
      print( "      Receive lifetime: %s" % rxStr )
      txStr = LifetimeHelper.displayDuration( self.txLifetimeStart,
                                              self.txLifetimeEnd, self.timezone )
      print( "      Transmit lifetime: %s" % txStr )

class SharedSecretProfile( Model ):
   profileName = Str( help='The name associated with the shared secret profile',
                      optional=True )
   currentRxId = Int( help='The id of the current secret being used for '
                           'reception if it can be represented as an integer '
                           '[deprecated]', optional=True )
   currentTxId = Int( help='The id of the current secret being used for '
                           'transmition if it can be represented as an integer '
                           '[deprecated]', optional=True )
   rxId = Str( help='The id of the current secret being used for '
               'reception', optional=True )
   txId = Str( help='The id of the current secret being used for '
               'transmition', optional=True )
   currentRxExpiration = Int( help='The date and time when the current receive '
                              'secret expires', optional=True )
   currentTxExpiration = Int( help='The date and time when the current transmit '
                              'secret expires', optional=True )
   rxRotation = List( help='The order in which receive secrets will be used',
                      valueType=str )
   txRotation = List( help='The order in which transmit secrets will be used',
                      valueType=str )
   secrets = List( help='The secrets currently configured in the profile',
                   valueType=SecretModel )

   def render( self ):
      if self.profileName is None:
         return

      print( 'Profile: %s\n' % self.profileName )
      if self.rxId is not None:
         print( 'Current receive secret: ID: %s, Expires: %s' % (
           self.rxId,
           LifetimeHelper.displayExpiration( self.currentRxExpiration ) ) )
      else:
         print( 'No current receive secret' )
      if self.txId is not None:
         print( 'Current transmit secret: ID: %s, Expires: %s\n' % (
            self.txId,
            LifetimeHelper.displayExpiration( self.currentTxExpiration ) ) )
      else:
         print( 'No current transmit secret\n' )

      print( 'Receive secret rotation order: %s' % (
         ", ".join( self.rxRotation ) ) )
      print( 'Transmit secret rotation order: %s\n' % (
         ", ".join( self.txRotation ) ) )

      print( 'Secrets:' )
      for secret in self.secrets:
         secret.render()

class SharedSecretProfiles( Model ):
   profiles = Dict( valueType=SharedSecretProfile,
                    help="Shared secret profiles indexed by name" )

   def render( self ):
      for profile in self.profiles.values():
         profile.render()
         print( '\n' )
