# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import sys

import AgentDirectory
import ShowAllocsHelper
from ShowAllocsHelper import (
   ShowAllocsHelperLegacy,
   ShowAllocsHelperTracktor,
   TracktorSnapshot
)
from Toggles.ArkCliToggleLib import toggleTracktorCliEnabled
import BasicCli
import Cell
import CliCommand
import CliMatcher
import CliPlugin.AgentCli as AgentCli # pylint: disable=consider-using-from-import
# pylint: disable-next=consider-using-from-import
import CliPlugin.AgentLibShowCommands as AgentLibShowCommands
from CliPlugin.ShowAllocsModel import ShowAllocsDiffModel, ShowAllocsModel
import CliToken.Agent
import LazyMount
import ShowCommand
import Tac

showAllocsHelper = None
showAllocsClearHelper = None

class ShowAllocsClearHelper:
   def __init__( self, sysname, agentConfigDir ):
      self.sysname = sysname
      self.agentConfigDir = agentConfigDir

   def clearHighWatermark( self, agentName=None ):
      if agentName is None:
         for agent in self.agentConfigDir.values():
            agent.allocTrackResetHighWatermarkRequest = Tac.now()
      else:
         self.agentConfigDir[ agentName ].allocTrackResetHighWatermarkRequest\
            = Tac.now()

def showImpl( mode, args, agent, snapshot, filterTacc ):
   typesFromLastRun = showAllocsHelper.typesFromLastRun( mode )
   renderOptions = {}
   renderOptions[ '_maxNameLen' ] = 100
   renderOptions[ '_tableWidth' ] = 200
   renderOptions[ '_limit' ] = args.get( 'LIMIT', 0 )
   sort = args.get( 'SORT' )

   if sort:
      if sort == 'high-watermark':
         renderOptions[ '_sortOrder' ] = 'highWatermarkMemory'
      elif sort == 'total-allocations':
         renderOptions[ '_sortOrder' ] = 'totalAllocations'
      else:
         renderOptions[ '_sortOrder' ] = 'currentMemory'

   filterTransient = args.get( 'FILTER' )
   model, allTypes = ShowAllocsHelper.showAllocsSnapshotToModel(
      snapshot,
      typeFilter=args.get( 'REGEX' ),
      filterTacc=filterTacc, filterTransient=filterTransient, **renderOptions )
   typesFromLastRun[ agent ] = allTypes

   return model

def showDiffImpl( mode, args, agent, snapshot, filterTacc ):
   typesFromLastRun = showAllocsHelper.typesFromLastRun( mode )
   renderOptions = {}
   renderOptions[ '_maxNameLen' ] = 100
   renderOptions[ '_tableWidth' ] = 200
   renderOptions[ '_limit' ] = args.get( 'LIMIT', 0 )
   noDeltaBase = False
   delta = args.get( 'DELTA' )
   sortOrder = 'currentMemory' if delta == 'current' else 'transientMemory'
   renderOptions[ '_sortOrder' ] = sortOrder

   baseTypes = typesFromLastRun.get( agent )
   if baseTypes is None:
      noDeltaBase = True

   # use alias for function to avoid complications with 85 char limit per line rule
   snapshotToTypes = ShowAllocsHelper.showAllocsSnapshotToTypes
   allCurrentTypes = snapshotToTypes( snapshot, typeFilter=args.get( 'REGEX' ),
                                      filterTacc=filterTacc )
   typesFromLastRun[ agent ] = allCurrentTypes
   if noDeltaBase:
      print( 'No base allocation data to diff against.' )
      print( 'Setting base for next command invocation.' )
      return ShowAllocsDiffModel( **renderOptions )

   return ShowAllocsHelper.showAllocsDiffModel( baseTypes, allCurrentTypes,
                                                    **renderOptions )

def getAgents( includeBuggyAgents=False ):
   # this logic to create the agent name comes from Agent.py
   def sysdbAgentName( agentNameAndSubsystemId ):
      ( agentName, subsystemId ) = agentNameAndSubsystemId
      if subsystemId is None or subsystemId == "":
         return agentName
      return agentName + "-" + subsystemId
   agents = set( map( sysdbAgentName,
                      AgentDirectory.agents( showAllocsHelper.sysname ) ) )
   if includeBuggyAgents:
      return agents
   else:
      # Skip all Smbus agents and Stp/StpTxRx, we hit a bug in
      # EntityManager for these agents triggered by having a non
      # default listenSocketName. See BUG353782
      return [ agent for agent in agents
               if not agent.startswith( 'Smbus' ) and agent != 'Stp'
               and agent != 'StpTxRx' ]

def clearAgentHighWatermarkHandler( args ):
   agentArg = args.get( 'AGENT', '*' )
   if agentArg != '*':
      showAllocsClearHelper.clearHighWatermark( agentArg )
   else:
      showAllocsClearHelper.clearHighWatermark()

def showAgentMemoryAllocationsHandler( mode, args, diff=False ):
   filterTacc = 'all' not in args
   agentArg = args.get( 'AGENT', '*' )
   snapshot = None
   if agentArg != '*':
      allAgents = AgentCli.allAgentNames( mode )
      if agentArg not in allAgents:
         mode.addErrorAndStop( f'Invalid agent name {agentArg}' )
      sysname = mode.entityManager.sysname()
      if not AgentDirectory.agentIsRunning( sysname, agentArg ):
         mode.addWarning( f'{agentArg} agent is not running' )
         return ShowAllocsModel()
      snapshot = showAllocsHelper.getSnapshot( agentArg, mode=mode )
      if snapshot is None:
         return ShowAllocsModel()
   else:
      sumSnapshot = TracktorSnapshot() if toggleTracktorCliEnabled() else\
         Tac.newInstance( 'Tac::AllocTrackSnapshotEnt', 0, Tac.now() )
      allAgents = getAgents()
      snapshots, err = showAllocsHelper.getSnapshots( agents=allAgents,
                                                          blocking=False )
      for snapshot in snapshots.values():
         sumSnapshot.add( snapshot )
      for _err in err.values():
         mode.addError( _err )
      snapshot = sumSnapshot

   if not diff:
      return showImpl( mode, args, agentArg, snapshot, filterTacc )
   else:
      return showDiffImpl( mode, args, agentArg, snapshot, filterTacc )

#--------------------------------------------------------------------------------
# show agent [ AGENT ] memory allocations [ summary ] [ ( sort SORT )
#                                   [ limit LIMIT ] ] [ filter REGEX ] [ all ]
#                                   [ FILTER ]
#--------------------------------------------------------------------------------
class AgentMemoryAllocationsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show agent [ AGENT ] memory allocations [ summary ] '
              '[ ( sort SORT ) [ limit LIMIT ] ] '
              '[ filter REGEX ] [ all ] [ FILTER ]' )
   data = {
      'agent': CliToken.Agent.agentKwForShow,
      'AGENT': AgentCli.agentNameNewMatcher,
      'memory': AgentLibShowCommands.matcherMemory,
      'allocations': AgentLibShowCommands.matcherAllocations,
      'summary': 'Show sum of memory allocations across all running agents',
      'sort': 'Type information sort order',
      'SORT': CliMatcher.EnumMatcher( {
         'high-watermark': 'Sort types by memory use high watermark',
         'current': 'Sort types by current memory use',
         'total-allocations': 'Sort types by total allocations',
      } ),
      'limit': 'Number of types to display',
      'LIMIT': CliMatcher.IntegerMatcher( 1, sys.maxsize, helpdesc='Type limit' ),
      'filter': 'Display only type names matching this regular expression',
      'REGEX': CliMatcher.PatternMatcher( '[^|].*',
                                          helpdesc='Type name regular expression',
                                          helpname='REGEX' ),
      'all': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'all',
                                            helpdesc='Disable default filtering' ),
         hidden=True ),
      'FILTER': CliMatcher.EnumMatcher( {
         'transients': 'Show only transient types',
         'non-transients': 'Show only non-transient types',
      } ),
   }

   @staticmethod
   def handler( mode, args ):
      return showAgentMemoryAllocationsHandler( mode, args )

   cliModel = ShowAllocsModel

BasicCli.addShowCommandClass( AgentMemoryAllocationsCmd )

#--------------------------------------------------------------------------------
# show agent [ AGENT ] memory allocations [ summary ] ( delta DELTA )
#                                   [ limit LIMIT ] [ filter REGEX ] [ all ]
#--------------------------------------------------------------------------------
class AgentMemoryAllocationsDiffCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show agent [ AGENT ] memory allocations [ summary ] '
              '( delta DELTA ) [ limit LIMIT ] '
              '[ filter REGEX ] [ all ]' )
   data = {
      'agent': CliToken.Agent.agentKwForShow,
      'AGENT': AgentCli.agentNameNewMatcher,
      'memory': AgentLibShowCommands.matcherMemory,
      'allocations': AgentLibShowCommands.matcherAllocations,
      'summary': 'Show sum of memory allocations across all running agents',
      'delta': 'Allocation changes over a period of time',
      'DELTA': CliMatcher.EnumMatcher( {
         'current': 'Display delta in current memory use',
         'transient': 'Display transient allocations',
      } ),
      'limit': 'Number of types to display',
      'LIMIT': CliMatcher.IntegerMatcher( 1, sys.maxsize, helpdesc='Type limit' ),
      'filter': 'Display only type names matching this regular expression',
      'REGEX': CliMatcher.PatternMatcher( '[^|].*',
                                          helpdesc='Type name regular expression',
                                          helpname='REGEX' ),
      'all': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'all',
                                            helpdesc='Disable default filtering' ),
         hidden=True )
   }

   @staticmethod
   def handler( mode, args ):
      return showAgentMemoryAllocationsHandler( mode, args, diff=True )

   cliModel = ShowAllocsDiffModel

BasicCli.addShowCommandClass( AgentMemoryAllocationsDiffCmd )

class AgentMemoryAllocationsResetHighWatermark( CliCommand.CliCommandClass ):
   syntax = 'clear agent [ AGENT ] memory allocations SORT'

   data = {
      'clear': CliToken.Clear.clearKwNode,
      'agent': CliToken.Agent.agentKwForClear,
      'AGENT': AgentCli.agentNameNewMatcher,
      'memory': AgentLibShowCommands.matcherMemory,
      'allocations': AgentLibShowCommands.matcherAllocations,
      'SORT': CliMatcher.EnumMatcher( {
         'high-watermark': 'Clear high-watermark measurement',
      } ),
   }

   @staticmethod
   def handler( mode, args ):
      clearAgentHighWatermarkHandler( args )

BasicCli.EnableMode.addCommandClass( AgentMemoryAllocationsResetHighWatermark )

def Plugin( entityManager ):
   global showAllocsHelper
   global showAllocsClearHelper
   agentConfigDir = LazyMount.mount( entityManager,
                                     Cell.path( 'agent/config' ),
                                     'Tac::Dir', 'wi' )
   agentStatusDir = LazyMount.mount( entityManager,
                                     Cell.path( 'agent/status' ),
                                     'Tac::Dir', 'ri' )

   HelperType = ShowAllocsHelperTracktor if toggleTracktorCliEnabled()\
      else ShowAllocsHelperLegacy

   showAllocsHelper = HelperType(
      entityManager.sysname(), agentConfigDir, agentStatusDir )
   showAllocsClearHelper = ShowAllocsClearHelper( entityManager.sysname(),
                                                  agentConfigDir )
