# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os
import re
import threading

import Ark
import BasicCli
import CliCommand
import CliMatcher
import CliModel
import CliToken.Cli
import ShowCommand
import Tac
import Tracing

# pkgdeps: rpmwith %{_bindir}/oomadj

__defaultTraceHandle__ = Tracing.Handle( 'ShowCliCommands' )
t3 = Tracing.trace3

ruleKeyCompiledRe = re.compile( '(\\W|no|default)*' )

class Clis( CliModel.Model ):
   clis = CliModel.List( valueType=str, help='CLI commands in the current mode' )

   def render( self ):
      for cli in self.clis:
         print( cli )

class AllClis( CliModel.Model ):
   allClis = CliModel.Dict(
               keyType=str, valueType=Clis,
               help='CLI commands in all modes, keyed by mode name' )

   def render( self ):
      for mode, clis in sorted( self.allClis.items() ):
         for cli in clis.clis:
            print( f'{mode}: {cli}' )

verbatimOrTruncate = object()

class ShowCliCommandsOptionsExpr( CliCommand.CliExpression ):
   '''Expression is: options in any order,
   but 'verbatim' and 'truncate' are mutually exclusive.
   '''
   expression = '{ hidden | functions | ( truncate LENGTH ) | verbatim }'
   data = {
         'verbatim': CliCommand.singleKeyword( 'verbatim',
                                                'Show syntaxes as-is',
                                                sharedMatchObj=verbatimOrTruncate ),
         'hidden': CliCommand.singleKeyword( 'hidden',
                                              'Show hidden commands as well',
                                              hidden=True ),
         'functions': CliCommand.singleKeyword( 'functions',
                                                 'Show callback functions as well',
                                                 hidden=True ),
         'truncate': CliCommand.singleKeyword( 'truncate',
                                                'Truncate commands to N characters '
                                                '(200 by default, 0 means no limit)',
                                                sharedMatchObj=verbatimOrTruncate ),
         'LENGTH': CliMatcher.IntegerMatcher( 0, 2**32 - 1,
                                               helpdesc='Command length limit' ),
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      args.pop( 'truncate', None )
      options = {
            'verbatim': bool( args.pop( 'verbatim', False ) ),
            'hidden': bool( args.pop( 'hidden', False ) ),
            'functions': bool( args.pop( 'functions', False ) ),
            'truncate': args.pop( 'LENGTH', [ 200 ] )[ 0 ],
      }
      args[ 'OPTIONS' ] = options

#-----------------------------------------------------------------------------------
# show cli commands [ OPTIONS ]
#-----------------------------------------------------------------------------------
def ruleKey( s ):
   # start with the first word character after any combination of no|default
   m = ruleKeyCompiledRe.match( s )
   return s[ m.end() : ]

def getModeSyntaxes( mode, args ):
   clis = list( mode.getSyntaxes( **args[ 'OPTIONS' ] ) )
   for modeletClass in getattr( mode, 'modeletClasses', [] ):
      clis.extend( modeletClass.getSyntaxes( **args[ 'OPTIONS' ] ) )
   t3( 'found', len( clis ), 'rules for', mode )
   return Clis( clis=sorted( clis, key=ruleKey ) )

# This will start a new full Cli process, which is slow and uses quite some memory,
# so ensure we don't run 2 of them at the same time. The info displayed is totally
# static so people can just store the result if they complain it is slow...
lock = threading.Lock()

@Ark.synchronized( lock )
def delegateToTranscientCli( mode, args, model, warning ):
   cmd = "oomadj --oom_score_adj 1000 -- rCli -Ap15 --load-dynamic-plugins -c".split(
                                                                                " " )
   cliCmd = [ "show cli commands" ]
   if args.get( "all-modes" ):
      cliCmd.append( 'all-modes' )
   if args[ 'OPTIONS' ].get( 'hidden' ):
      cliCmd.append( 'hidden' )
   if args[ 'OPTIONS' ].get( 'functions' ):
      cliCmd.append( 'functions' )
   if args[ 'OPTIONS' ].get( 'verbatim' ):
      cliCmd.append( 'verbatim' )
   else:
      truncate = args[ 'OPTIONS' ].get( 'truncate' )
      if truncate:
         cliCmd.append( 'truncate' )
         cliCmd.append( str( truncate ) )
   if mode.session_.outputFormat_ == "json":
      cliCmd.append( '|' )
      cliCmd.append( 'json' )
   try:
      mode.addWarningInteractOnly( "using the 'all-modes' option will take a while" )
      cmd.append( " ".join( cliCmd ) )
      out = Tac.run( cmd, stdout=Tac.CAPTURE )
      print( out )
      return CliModel.cliPrinted( model )
   except Tac.SystemCommandError:
      pass # try it inline, maybe not enough mem for an extra Cli
   mode.addWarning( warning )
   return None

class ShowCliCommands( ShowCommand.ShowCliCommandClass ):
   syntax = 'show cli commands [ OPTIONS ]'
   data = {
         'cli': CliToken.Cli.cliForShowMatcher,
         'commands': 'Show CLI commands',
         'OPTIONS': ShowCliCommandsOptionsExpr,
   }
   cliModel = Clis
   privileged = True

   @staticmethod
   def handler( mode, args ):
      """Show rules from the current mode."""
      # 'function' option requires delazification: delegate to a transcient cli so
      # that memory is not wasted for ever.
      if args[ 'OPTIONS' ].get( 'functions' ):
         if not os.environ.get( "LOAD_DYNAMIC_PLUGIN" ): # first time here
            # exec a new Cli that will be unlazy
            warning = "Some of the functions may have unknown line numbers"
            ret = delegateToTranscientCli( mode, args, Clis, warning )
            if ret:
               return ret
            # some error: try it the classic way
      lastMode = mode.session_.modeOfLastPrompt() or mode
      return getModeSyntaxes( lastMode, args )

BasicCli.addShowCommandClass( ShowCliCommands )

#-----------------------------------------------------------------------------------
# show cli commands all-modes [ OPTIONS ]
#-----------------------------------------------------------------------------------
class ShowCliCommandsAllModes( ShowCommand.ShowCliCommandClass ):
   syntax = 'show cli commands all-modes [ OPTIONS ]'
   data = {
         'cli': CliToken.Cli.cliForShowMatcher,
         'commands': 'Show CLI commands',
         'all-modes': 'Show commands from all modes (not recommended on low '
                      'memory systems',
         'OPTIONS': ShowCliCommandsOptionsExpr,
   }
   cliModel = AllClis
   privileged = True

   @staticmethod
   def handler( mode, args ):
      """Show rules from all modes in the system."""
      # For the 'all-modes' option, because of lazification (most command not there)
      # and to prevent delazyfication of all modes, we spawn a tmp standalone Cli.
      # Bad for short term memory, good for long term.
      # This option should not be used on low memory systems.
      if not os.environ.get( "LOAD_DYNAMIC_PLUGIN" ): # first time here
         # exec a new Cli to handle it unlazified.
         warning = "Will only show modes that exist in the current config"
         ret = delegateToTranscientCli( mode, args, AllClis, warning )
         if ret:
            return ret
         # some error: try it the classic way

      # Normal case / second pass (standalone Cli)
      submodes = [ BasicCli.UnprivMode, BasicCli.EnableMode,
            BasicCli.GlobalConfigMode ]
      submodes.extend( BasicCli.ConfigModeBase.__subclasses__() )
      visited = set()
      model = AllClis()
      while submodes:
         submode = submodes.pop( 0 )
         if submode in visited:
            continue

         visited.add( submode )
         model.allClis[ submode.__name__ ] = getModeSyntaxes( submode, args )
         submodes = submode.__subclasses__() + submodes
      return model

BasicCli.addShowCommandClass( ShowCliCommandsAllModes )
