#!/usr/bin/env python3
# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import BasicCli
import CliCommand
import ShowCommand
from Toggles import ConfigConsistencyToggleLib

from CliPlugin.ShowConfigConsistencyCliModels import (
    AclTypeModel,
    CommListTypeModel,
    ConfigConsistencyModel,
    FeatureModel,
    PfxListTypeModel,
    UndefinedReferenceCheckerModel,
)
from ConfigConsistencyChecker import UndefinedReferenceChecker

featuresData = {
   "access-list": "Filter by access lists",
   "prefix-list": "Filter by prefix lists",
   "community-list": "Filter by community lists",
}

if ConfigConsistencyToggleLib.toggleConfigConsistencyRouteMapCmdEnabled():
   featuresData[ "route-map" ] = "Filter by route maps"

allFeatures = []
accessListFeatures = [ "IP access list", "IPv6 access list", "MAC access list" ]
allFeatures.extend( accessListFeatures )
prefixListFeatures = [ "IP prefix list", "IPv6 prefix list" ]
allFeatures.extend( prefixListFeatures )
commListFeatures = [ "IP community list" ]
allFeatures.extend( commListFeatures )
routeMapFeatures = [ "Route map" ]
if ConfigConsistencyToggleLib.toggleConfigConsistencyRouteMapCmdEnabled():
   allFeatures.extend( routeMapFeatures )

def showConfigConsistency( mode, args ):
   model = ConfigConsistencyModel()

   if args.get( "policy" ) is not None:
      model.undefinedReferences = UndefinedReferenceCheckerModel()
      features = []
      featureFilterTypes = args.get( "FEATURES" )
      if featureFilterTypes is not None:
         for featureFilterType in featureFilterTypes:
            if featureFilterType == 'access-list':
               features.extend( accessListFeatures )
            if featureFilterType == 'prefix-list':
               features.extend( prefixListFeatures )
            if featureFilterType == 'community-list':
               features.extend( commListFeatures )
            if featureFilterType == 'route-map':
               features.extend( routeMapFeatures )
      else:
         features = allFeatures
      undefinedReferences = UndefinedReferenceChecker.check( features )
      for feature in undefinedReferences:
         featureModel = FeatureModel()
         featureModel.details = []
         # undefined reference names are printed in alphabetical order
         for undefinedReference in sorted( undefinedReferences[ feature ] ):
            featureModel.details.append( undefinedReference + " is undefined" )
         if featureModel.details:
            if feature in accessListFeatures:
               if model.undefinedReferences.acls is None:
                  model.undefinedReferences.acls = AclTypeModel()
            elif feature in prefixListFeatures:
               if model.undefinedReferences.pfxLists is None:
                  model.undefinedReferences.pfxLists = PfxListTypeModel()
            elif feature in commListFeatures:
               if model.undefinedReferences.commLists is None:
                  model.undefinedReferences.commLists = CommListTypeModel()
            # Route map feature model set below

            if feature == "IP access list":
               model.undefinedReferences.acls.ip = featureModel
            elif feature == "IPv6 access list":
               model.undefinedReferences.acls.ipv6 = featureModel
            elif feature == "MAC access list":
               model.undefinedReferences.acls.mac = featureModel
            elif feature == "IP prefix list":
               model.undefinedReferences.pfxLists.ip = featureModel
            elif feature == "IPv6 prefix list":
               model.undefinedReferences.pfxLists.ipv6 = featureModel
            elif feature == "IP community list":
               model.undefinedReferences.commLists.ip = featureModel
            elif feature == "Route map":
               model.undefinedReferences.routeMaps = featureModel

   return model

class ConfigConsistencyCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "show configuration consistency ( policy [ FEATURES ] )"
   data = {
         "configuration": "Show configuration information",
         "consistency": "Verify configuration consistency",
         "policy": "Check for undefined references to policy type configs",
         "FEATURES": CliCommand.SetEnumMatcher( featuresData )
   }

   handler = showConfigConsistency
   cliModel = ConfigConsistencyModel

BasicCli.addShowCommandClass( ConfigConsistencyCmd )
