# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
import BasicCli
import CliMatcher
import CliToken.Hardware
import CliToken.Resource
import CliGlobal
import CliExtensions
import LazyMount
import ShowCommand
import Tac
import TableOutput
import Tracing
from CliPlugin import ResourceMgrCliLib
from CliModel import Model, Dict, List, Str, Int, Enum
from TypeFuture import TacLazyType

__defaultTraceHandle__ = Tracing.Handle( "AsicHwTableCli" )
t4 = Tracing.trace4

IndexTableType = TacLazyType( 'Asic::IndexTable' )

globalVar = CliGlobal.CliGlobal(
      allConfigPathDir=None,
      )

tableHandlerHook = CliExtensions.CliHook()

"""The HardwareMappingTable module handles getting/mounting tables and printing
the output to the CLI"""

class MemoryData( Model ):
   """CliModel for memory entry"""
   memName = Str( help="Name of the memory/register" )
   width = Int( help="Width of the memory/register in bytes", optional=True )
   minIndex = Int( help="Mininum entry index of the depth of the memory/register",
                   optional=True )
   maxIndex = Int( help="Maximum entry index of the depth of the memory/register",
                   optional=True )

class TableGroup( Model ):
   """CliModel for memory table entry"""
   physicalTable = List( valueType=MemoryData,
                         help="Memories/Registers within the table" )
   depth = Int( help="Depth of the table", optional=True )

class Table( Model ):
   """CliModel for hardware table"""
   groupFormation = List( valueType=TableGroup, help="Table groups" )
   uniformity = Enum( values=( "uniform", "divergent" ),
                      help="Value of instance programming" )
   instances = List( valueType=int, help="Instance ids", optional=True )

def getHwTable( mode=None ):
   """Cli matcher function for available table options"""
   setOfTables = set()
   hwTableMountInfos = ResourceMgrCliLib.collectMountInfos(
         globalVar.allConfigPathDir )
   for table in hwTableMountInfos:
      setOfTables.add( table.split( '::' )[ -1 ] )
      setOfTables.add( table )
   setOfTables.add( '*' )

   return { table: "Show table info for " + table
            for table in setOfTables }

def printTableRegular( hwGroups ):
   """Printing the table without extra information"""
   groupHeader = [ "Group" ]
   hwGroup = hwGroups[ 0 ]
   headers = []
   # get memories for headers
   for memIndex in range( len( hwGroup.physicalTable ) ):
      if ( hwGroup.physicalTable[ 0 ] and
           hwGroup.physicalTable[ 0 ].memName[ -1 ] == 'm'
      ):
         tableType = 'Mem'
      else:
         tableType = 'Reg'
      headers += [ f"{ tableType }{ memIndex }" ]
   # create table with the the headers
   table = TableOutput.createTable( groupHeader + headers )
   leftJustify = TableOutput.Format( justify='left' )
   leftJustify.padLimitIs( True )
   rightJustify = TableOutput.Format( justify='right' )
   rightJustify.padLimitIs( True )
   fmts = [ rightJustify ] + [ leftJustify for _ in headers ]
   table.formatColumns( *fmts )
   # for each of the groups create a row and append the memories
   for groupIndex, hwGroup in enumerate( hwGroups ):
      groupRow = list( map( lambda x: x.memName, hwGroup.physicalTable ) )
      newRow = [ groupIndex ] + groupRow
      table.newRow( *newRow )
   print( table.output() )


def printTableVerbose( hwGroups ):
   """Printing the table with extra table information"""
   # create column headers
   if ( hwGroups[ 0 ].physicalTable[ 0 ] and
        hwGroups[ 0 ].physicalTable[ 0 ].memName[ -1 ] == 'm'
   ):
      tableType = 'Memories'
   else:
      tableType = 'Registers'
   headers = [ 'Group', tableType, 'Width (Bytes)', 'MinIndex', 'MaxIndex' ]
   # create the table with the headers
   table = TableOutput.createTable( headers )
   leftJustify = TableOutput.Format( justify='left' )
   leftJustify.padLimitIs( True )
   rightJustify = TableOutput.Format( justify='right' )
   rightJustify.padLimitIs( True )
   fmts = [ rightJustify, leftJustify, rightJustify, rightJustify, rightJustify ]
   table.formatColumns( *fmts )
   # for each of the groups create a row and append the memories
   for groupIndex, hwGroup in enumerate( hwGroups ):
      for memory in hwGroup.physicalTable:
         table.newRow( *[
            groupIndex,
            memory.memName,
            f'{memory.width}',
            f'{memory.minIndex}',
            f'{memory.maxIndex}',
            ]
         )
   print( table.output() )

class HardwareMapping( Model ):
   """CliModel to render the table information"""
   tables = Dict( valueType=Table,
         help="Hardware mapping tables, keyed by their name" )
   _outputFormat = Str( help="Output format" )

   def render( self ):
      for tableName, hwTable in self.tables.items():
         uniformity = hwTable.uniformity
         print( 'Table: ' + tableName )
         print( 'Uniformity: ' + uniformity )
         if uniformity == 'divergent' and \
               self._outputFormat == 'verbose':
            print( f'Instances: {len( hwTable.instances )}' )
         # No format passed, print regular
         if not self._outputFormat:
            if hwTable.groupFormation:
               printTableRegular( hwTable.groupFormation )
         # Verbose format passed, print verbose
         if self._outputFormat == "verbose":
            if hwTable.groupFormation:
               printTableVerbose( hwTable.groupFormation )


def doShowHwMap( mode, args ):
   """Implementation of getting mounting information and handlers
   for displaying mappings"""
   hwTbl = args.get( "TABLE" )
   outputFormat = args.get( "FORMAT", "" )
   tableDict = {}
   storedTables = {}
   hwTableMountInfos = ResourceMgrCliLib.collectMountInfos(
         globalVar.allConfigPathDir )
   for table in hwTableMountInfos:
      t4( "figure out the configPath table" )
      if hwTbl in table or hwTbl == '*':
         tableType = Tac.Type( table ).tacType
         storedTables[ table ] = tableType
         # map every table to their model
   for tableHook in tableHandlerHook.extensions():
      tableDict = tableHook( storedTables, tableDict, outputFormat )
   if tableDict is None:
      return HardwareMapping( tables={}, _outputFormat=outputFormat )
   else:
      return HardwareMapping( tables=tableDict, _outputFormat=outputFormat )

nodeTable = CliMatcher.DynamicKeywordMatcher( getHwTable )

# -----------------------------------------------------------------------------------
# The "show hardware resource mapping table TABLE [ FORMAT ]" command
# -----------------------------------------------------------------------------------

class ShowHwMappingCmd( ShowCommand.ShowCliCommandClass ):
   """The command class to handle syntax and data for the cli"""
   syntax = '''show hardware resource mapping table
               TABLE [ FORMAT ] '''
   data = {
         'hardware': CliToken.Hardware.hardwareForShowMatcher,
         'resource': CliToken.Resource.resourceMatcherForShow,
         'mapping': 'Display mapping between logical and physical tables',
         'table': 'Specify the logical table',
         'TABLE': nodeTable,
         'FORMAT': CliMatcher.EnumMatcher( { 'verbose': 'Raw dump of table data' } ),
   }
   handler = doShowHwMap
   cliModel = HardwareMapping

BasicCli.addShowCommandClass( ShowHwMappingCmd )

def Plugin( entityManager ):
   """Plugin to get picked up by the CLI"""
   globalVar.allConfigPathDir = LazyMount.mount(
         entityManager, 'hardware/resource', 'Tac::Dir', 'ri' )
