# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import (
   Dict,
   Int,
   Model,
   Str,
)
from TableOutput import (
   Format,
   createTable,
)

# fields from C struct mallinfo; see mallinfo(3)
class Mallinfo( Model ):
   status = Str( help="Heap summary query status (anything other than 'success' "
                      "indicates an error)" )
   arena = Int( help="Non-mmapped space allocated (bytes)", optional=True )
   ordblks = Int( help="Number of free chunks", optional=True )
   smblks = Int( help="Number of free fastbin blocks", optional=True )
   hblks = Int( help="Number of mmapped regions", optional=True )
   hblkhd = Int( help="Space allocated in mmapped regions (bytes)", optional=True )
   usmblks = Int( help="Maximum total allocated space (bytes)", optional=True )
   fsmblks = Int( help="Space in freed fastbin blocks (bytes)", optional=True )
   uordblks = Int( help="Total allocated space (bytes)", optional=True )
   fordblks = Int( help="Total free space (bytes)", optional=True )
   keepcost = Int( help="Top-most, releasable space (bytes)", optional=True )

class HeapSummary( Model ):
   agents = Dict( keyType=str, valueType=Mallinfo, help="Heap summary by agent" )

   def render( self ):
      if not self.agents:
         return
      table = createTable( ( "Agent", "Total kB", "Used kB", "Free kB",
                             "Keep kB", "# Mmap/kB", "# Free/Fast/kB",  ) )
      formatLeft = Format( justify="left" )
      formatLeft.noPadLeftIs( True )
      formatLeft.padLimitIs( True )
      formatRight = Format( justify="right" )
      formatRight.noPadLeftIs( True )
      formatRight.padLimitIs( True )
      table.formatColumns( formatLeft, *[ formatRight ] * 6 )
      for agentName, mallInfo in sorted( self.agents.items(),
                                         key=lambda item: item[ 1 ].arena,
                                         reverse=True ):
         table.newRow( agentName, mallInfo.arena // 1024, mallInfo.uordblks // 1024,
                       mallInfo.fordblks // 1024, mallInfo.keepcost // 1024,
                       # pylint: disable-next=consider-using-f-string
                       "%d / %d" % ( mallInfo.hblks, mallInfo.hblkhd // 1024 ),
                       # pylint: disable-next=consider-using-f-string
                       "%d / %d / %d" % ( mallInfo.ordblks, mallInfo.smblks,
                                          mallInfo.fsmblks // 1024 ),
                        )
      print( table.output() )
