# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
import Tac
from CliPlugin import ShowHardwareMappingTable, ResourceMgrCliLib

def doShowLogicalFapTables( storedTables, tableDict, outputFormat ):
   """The LogicalFapHandler takes in a dictionary of stored tables
   creates the CliModel for tables that have accessInfo and returns a dictionary
   of CliModeled tables for display"""
   for table in storedTables:
      # get accessInfo to use for building cliModel
      if table:
         Tac.Type( table )
         accessInfo = ""
         try:
            tableDesc = ResourceMgrCliLib.ResourceStub.tableDesc[ table ]
            # needed populate the accessInfo
            _ = tableDesc.accessInfo
            accInfo = tableDesc.accInfo
            # get current mapping from current uft mode
            accessInfo = accInfo[ tableDesc.currMode() ]
         except AttributeError:
            print( 'no accessInfo attribute for table desc' )
      # check to see if already in dict
      if table not in tableDict and accessInfo:
         tableDict[ table ] = ShowHardwareMappingTable.Table()
         # cliModel table
         updateTable = tableDict[ table ]
         # update table information
         updateTable.uniformity = accessInfo.instanceProgramming
         if outputFormat == 'verbose' and updateTable.uniformity == 'divergent':
            for instanceId in accessInfo.instance:
               updateTable.instances.append( instanceId )
         else:
            updateTable.instances = None
         # get accessInfo groups and create group/memory in cliModel
         groupList = accessInfo.group
         memDepthCounter = {}
         for groupNum in groupList:
            updateTable.groupFormation.append(
                  ShowHardwareMappingTable.TableGroup() )
            updateGroup = updateTable.groupFormation[ groupNum ]
            if outputFormat == 'verbose':
               updateGroup.depth = groupList[ groupNum ].depth
            for memId, memInfoList in groupList[ groupNum ].ptInfoList.items():
               memInfo = memInfoList.primaryPtInfo
               if memId not in memDepthCounter:
                  memDepthCounter[ memId ] = ( 0, memInfo.depth - 1 )
               else:
                  memDepthCounter[ memId ] = (
                     memDepthCounter[ memId ][ 1 ] + 1,
                     memDepthCounter[ memId ][ 1 ] + memInfo.depth )
               updateMemory = ShowHardwareMappingTable.MemoryData()
               updateMemory.memName = memInfo.name
               if outputFormat == 'verbose':
                  updateMemory.width = memInfo.width
                  updateMemory.minIndex = memDepthCounter[ memId ][ 0 ]
                  updateMemory.maxIndex = memDepthCounter[ memId ][ 1 ]
               updateGroup.physicalTable.append( updateMemory )
      if not accessInfo:
         print( f"Table: {table}" )
         print( 'Currently no support for older table models' )
   return tableDict

ShowHardwareMappingTable.tableHandlerHook.addExtension( doShowLogicalFapTables )
