# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import ArPyUtils
import BasicCli
import CliMatcher
import LazyMount
import ShowCommand
import Tac
from CliModel import (
   Bool,
   Dict,
   Enum,
   Float,
   Int,
   Model,
)
from CliPlugin import TechSupportCli
from TableOutput import createTable, Format

hwLedConfig = None
tokenLedHelpDesc = 'Show LED Information'

class LedSetting( Model ):
   red = Bool( help="Red LED is on" )
   green = Bool( help="Green LED is on" )
   blue = Bool( help="Blue LED is on" )
   yellow = Bool( help="Yellow LED is on" )
   status = Enum( values=[ "off", "good", "warning", "bad", "inactive",
                           "unknown", "beacon" ],
                  help="LED status" )
   period = Float( help="Flash period (seconds)" )
   maxBright = Int( help="Maximum brightness (%)" )
   minBright = Int( help="Minimum brightness (%)" )

class LedSettings( Model ):
   systemLeds = Dict( keyType=str, valueType=LedSetting,
                      help="A mapping of system LED to its settings" )

   def render( self ):
      # Table Formatting
      columns = ( 'Name', 'Red', 'Green', 'Blue', 'Yellow', 'Status',
                  ( 'Flash', ( 'Rate', ) ),
                  ( 'Brightness',  ( 'Max', 'Min' ) ) )
      table = createTable( columns )
      fl = Format( justify="left" )
      table.formatColumns( *( fl for _ in columns ) )
      boolMapper = [ "off", "on" ]
      for led in ArPyUtils.naturalsorted( self.systemLeds ):
         table.newRow( led,
                       boolMapper[ self.systemLeds[ led ].red ],
                       boolMapper[ self.systemLeds[ led ].green ],
                       boolMapper[ self.systemLeds[ led ].blue ],
                       boolMapper[ self.systemLeds[ led ].yellow ],
                       self.systemLeds[ led ].status,
                       str( self.systemLeds[ led ].period ),
                       str( self.systemLeds[ led ].maxBright ),
                       str( self.systemLeds[ led ].minBright ) )

      print( table.output() )

# -----------------------------------------------------------------------------
# show led [ name NAME ]
#
# Shows the lightSetting values of leds
# -----------------------------------------------------------------------------
class ShowLedSettings( ShowCommand.ShowCliCommandClass ):
   syntax = 'show led [ name NAME ]'
   data = {
      'led': tokenLedHelpDesc,
      'name': 'Show LED Light Settings by name',
      'NAME': CliMatcher.PatternMatcher( pattern=r'.+',
                                         helpdesc='LED name',
                                         helpname='WORD' ),
   }
   cliModel = LedSettings

   @staticmethod
   def handler( mode, args ):
      result = LedSettings()

      hwLeds = hwLedConfig.leds
      if not hwLeds:
         mode.addError( "No LEDs found" )
         return None

      name = args.get( 'NAME' )
      if name:
         lower = name.lower()
         # Check for led name match
         ledNames = [ led for led in hwLeds if led.lower() == lower ]
         if not ledNames:
            # pylint: disable-next=consider-using-f-string
            mode.addError( "%s LED not found" % name )
            return None
      else:
         ledNames = hwLeds

      for led in ledNames:
         lightSetting = hwLeds[ led ].lightSetting
         ledSetting = LedSetting()
         if lightSetting.red and lightSetting.green:
            ledSetting.yellow = True
            ledSetting.red = ledSetting.green = False
         else:
            ledSetting.red = lightSetting.red
            ledSetting.green = lightSetting.green
            ledSetting.yellow = lightSetting.yellow
         ledSetting.blue = lightSetting.blue
         ledSetting.status = \
            str( lightSetting.plutoStatus ).lower().replace( 'plutostatus', '' )
         ledSetting.period = lightSetting.flashRate
         ledSetting.maxBright = lightSetting.maxBright
         ledSetting.minBright = lightSetting.minBright
         result.systemLeds[ led ] = ledSetting

      return result

BasicCli.addShowCommandClass( ShowLedSettings )

TechSupportCli.registerShowTechSupportCmd(
   '2019-02-26 16:35:00',
   cmds=[ 'show led' ] )

# -----------------------------------------------------------------------------
# Cli Plugin initialization
# -----------------------------------------------------------------------------
def Plugin( entityManager ):
   global hwLedConfig
   hwLedConfig = LazyMount.mount( entityManager, "hardware/led/config",
                                  "Hardware::Led::LedSystemConfigDir", "r" )

