# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import CliModel
import TableOutput
from operator import attrgetter

warningStrSuffix = " (not configured)"
warningEngIdSuffix = " (does not match system)"


class SnmpV3HostParams( CliModel.Model ):
   user = CliModel.Str( help="SNMPv3 user" )
   securityLevel = CliModel.Enum( values=[ "noAuthNoPriv", "authNoPriv",
                                           "authPriv" ],
                                  help="SNMPv3 security level" )

class SnmpV1V2cHostParams( CliModel.Model ):
   communityString = CliModel.Str( help="Community string for SNMPv1/v2c" )

class SnmpRemoteEngineIdPortToEngIdModel( CliModel.Model ):
   ports = CliModel.Dict( keyType=int, valueType=str,
                          help="A mapping of UDP-port to engineID" )

class SnmpRemoteEngineIdHostToPortModel( CliModel.Model ):
   hosts = CliModel.Dict( keyType=str, valueType=SnmpRemoteEngineIdPortToEngIdModel,
                          help="A mapping of hostname to a collection of UDP-ports" )

class SnmpEngineIdModel( CliModel.Model ):
   localEngineId = CliModel.Str( help="Local SNMP EngineID" )
   remoteEngineIds = CliModel.Submodel( valueType=SnmpRemoteEngineIdHostToPortModel,
                                 help='Remote SNMP EngineIDs' )

   def render( self ):
      print( "Local SNMP EngineID: %s" % self.localEngineId )
      if self.remoteEngineIds.hosts:
         print( "Remote EngineID           IP-addr          Port" )
         for host in sorted( self.remoteEngineIds.hosts ):
            submodel = self.remoteEngineIds.hosts[ host ]
            for port in sorted( submodel.ports ):
               engId = submodel.ports[ port ]
               print( "%-025s %-016s %s" % ( engId, host, port ) )

class HostModel( CliModel.Model ):
   hostname = CliModel.Str( help="Host name or IP address" )
   port = CliModel.Int( help="UDP port number" )
   vrf = CliModel.Str( help="Name of the VRF", optional=True )
   notificationType = CliModel.Enum( values=[ "trap", "inform" ],
                                     help="SNMP notification type" )
   refresh = CliModel.Bool( help="Refresh hostname DNS resolution", optional=True )
   protocolVersion = CliModel.Enum( values=[ 'v1', 'v2c', 'v3' ],
                                    help='SNMP version' )
   v3Params = CliModel.Submodel( valueType=SnmpV3HostParams, 
                                 help='SNMPv3 parameters', 
                                 optional=True )
   v1v2cParams = CliModel.Submodel( valueType=SnmpV1V2cHostParams,
                                   help='SNMPv1/v2c parameters',
                                   optional=True )
   
   def render( self ):
      securityLevelStrMap = { 'noAuthNoPriv' : 'noauth', 
                              'authNoPriv' : 'auth',
                              'authPriv' : 'priv' }
      line = "Notification host: %-15s udp-port: %-5s" % \
            ( self.hostname, self.port ) 
      line += " type: %-6s" % ( self.notificationType )
      line += " refresh: %-8s" % self.refresh
      if self.vrf:
         line += " VRF: %s" % ( self.vrf )
      print( line )
      if self.protocolVersion == "v3":
         print( "user: %-28s security model: %s %s\n" % ( self.v3Params.user,
               self.protocolVersion, 
               securityLevelStrMap[ self.v3Params.securityLevel ] ) )
      else:
         print( "user: %-28s security model: %s\n" % (
               self.v1v2cParams.communityString, self.protocolVersion ) )

class ShowSnmpHosts( CliModel.Model ):
   hosts = CliModel.List( help='SNMP notification recipients',
                          valueType=HostModel )
      
   def render( self ):
      for h in self.hosts:
         h.render()

class SnmpChassis( CliModel.Model ):
   chassisId = CliModel.Str( help="Chassis ID" )
   
   def render( self ):
      print( "Chassis: %s" % ( self.chassisId or None ) )

class Contact( CliModel.Model ):
   contact = CliModel.Str( help="SNMP Contact" )

   def render( self ):
      if self.contact != "":
         print( "Contact: %s" % ( self.contact ) )

class SnmpLocation( CliModel.Model ):
   location = CliModel.Str( help="SNMP Location" )
   
   def render( self ):
      if self.location != "":
         print( "Location: %s" % ( self.location ) )

class SnmpInterface( CliModel.Model ):
   sourceInterfaces = CliModel.Dict( valueType=str, help="A mapping between " 
                                     "a VRF and the source interface" )

   def render( self ):
      if self.sourceInterfaces:
         # pylint: disable-msg=E1103
         print( "SNMP source interfaces:" )
         for v in sorted( self.sourceInterfaces ):
            srcIntf = self.sourceInterfaces[ v ]
            print( "   %s in VRF %s" % ( srcIntf, v ) )

class SnmpCounters( CliModel.Model ):
   serviceEnabled = CliModel.Bool( help="status Snmp agent and snmpd")
   inPkts = CliModel.Int( help="SNMP packet input" )
   inVersionErrs = CliModel.Int( help="Bad SNMP version errors" )
   inBadCommunityNames = CliModel.Int( help="Unknown community name" )
   inBadCommunityUses = CliModel.Int( help="Illegal operation for community name" )
   inParseErrs = CliModel.Int( help="Encoding errors" )
   inRequestVars = CliModel.Int( help="Requested variables" )
   inSetVars = CliModel.Int( help="Altered variables" )
   inGetPdus = CliModel.Int( help="Get-request PDUs" )
   inGetNextPdus = CliModel.Int( help="Get-next PDUs" )
   inSetPdus = CliModel.Int( help="Set-request PDUs" )
   outPkts = CliModel.Int( help="SNMP packets output" )
   outTooBigErrs = CliModel.Int( help="Too big errors" )
   outNoSuchNameErrs = CliModel.Int( help="No such name errors" )
   outBadValueErrs = CliModel.Int( help="Bad value errors" )
   outGeneralErrs = CliModel.Int( help="General errors" )
   outGetResponsePdus = CliModel.Int( help="Response PDUs" )
   outTrapPdus = CliModel.Int( help="Trap PDUs" )
   outTrapDrops = CliModel.Int( help="Trap drops" )
   groupLength = CliModel.Int( help="Users in Group" )
   numberOfGroups = CliModel.Int( help="Groups" )
   numberOfViews = CliModel.Int( help="Views" )

   def render( self ):
      if not self.serviceEnabled:
         return
      print( "%d SNMP packets input" % self.inPkts )
      print( "    %d Bad SNMP version errors" % self.inVersionErrs )
      print( "    %d Unknown community name" % self.inBadCommunityNames )
      print( "    %d Illegal operation for community name supplied" % (
         self.inBadCommunityUses ) )
      print( "    %d Encoding errors" % self.inParseErrs )
      print( "    %d Number of requested variables" % self.inRequestVars )
      print( "    %d Number of altered variables" % self.inSetVars )
      print( "    %d Get-request PDUs" % self.inGetPdus )
      print( "    %d Get-next PDUs" % self.inGetNextPdus )
      print( "    %d Set-request PDUs" % self.inSetPdus )
      # we're missing the "Input queue packet drops" stat from the
      # industry-standard output.
      print( "%d SNMP packets output" % self.outPkts )
      print( "    %d Too big errors" % self.outTooBigErrs )
      print( "    %d No such name errors" % self.outNoSuchNameErrs )
      print( "    %d Bad value errors" % self.outBadValueErrs )
      print( "    %d General errors" % self.outGeneralErrs )
      print( "    %d Response PDUs" % self.outGetResponsePdus )
      print( "    %d Trap PDUs" % self.outTrapPdus )
      print( "    %d Trap drops" % self.outTrapDrops )
      print( "Access Control" )
      print( "    %d Users" % self.groupLength )
      print( "    %d Groups" % self.numberOfGroups )
      print( "    %d Views" % self.numberOfViews )

class SnmpLoggingHost( CliModel.Model ):
   port = CliModel.Int( help="Port for SNMP logger" )
   vrf = CliModel.Str( help="VRF, if applicable, for SNMP logger", optional=True )

class SnmpLogging( CliModel.Model ):
   _snmpEnabled = CliModel.Bool( help="If SNMP is enabled", default=False )
   loggingEnabled = CliModel.Bool( help="If logging is enabled", default=False )
   hosts = CliModel.Dict( valueType=SnmpLoggingHost, 
                          help="Mapping between hostname and host information" )

   def render( self ):
      if not self._snmpEnabled:
         return

      print( "SNMP logging: %s" % ( "enabled" if self.loggingEnabled
                                              else "disabled" ) )
      for hostname in sorted( self.hosts ):
         host = self.hosts[ hostname ]
         line = "    Logging to %s.%d" % ( hostname, host.port )
         if host.vrf:
            line += " VRF: %s" % ( host.vrf )
         print( line )

class SnmpVrfs( CliModel.Model ):
   snmpVrfs = CliModel.List( valueType=str, 
                             help="List of VRFs the SNMP agent is enabled on" )

   def render( self ):
      if not self.snmpVrfs:
         print( "SNMP agent disabled" )
      else:
         print( "SNMP agent enabled in VRFs: %s" % ', '.join( self.snmpVrfs ) )

class SnmpMiscConfig( CliModel.Model ):
   transmitMsgMaxSize = CliModel.Int(
      help="Maximum number of bytes in SNMP message (UDP/TCP payload)" )

   def render( self ):
      print( "Transmit message maximum size: %u" % self.transmitMsgMaxSize )

class SnmpV3UserParams( CliModel.Model ):
   engineId = CliModel.Str( help='SNMPv3 engineID' )
   engineIdMatch = CliModel.Bool(
         help="The system engineID matches the user" )
   authType = CliModel.Enum( help="Authentication protocol for the user",
                             values=[ "MD5", "SHA", "SHA-224", "SHA-256",
                                      "SHA-384", "SHA-512" ], optional=True )
   privType = CliModel.Enum( help="Privacy protocol for the user",
                             values=[ "DES", "AES-128",
                                      "AES-192", "AES-256" ], optional=True )

class UserModel( CliModel.Model ):
   groupName = CliModel.Str( help="Group for the SNMP user" )
   groupConfigured = CliModel.Bool( 
         help="Indicates whether SNMP group is configured" )
   v3Params = CliModel.Submodel( valueType=SnmpV3UserParams,
                                 help="SNMPv3 parameters for the user",
                                 optional=True )

   def renderUser( self, version, userName ):
      groupString = self.groupName
      if not self.groupConfigured:
         groupString += warningStrSuffix
      print( "\nUser name      : %s" % userName )
      print( "Security model : %s" % version )
      if version == 'v3':
         privString = 'None' if self.v3Params.privType is None else \
               self.v3Params.privType
         authString = 'None' if self.v3Params.authType is None else \
               self.v3Params.authType
         engIdString = self.v3Params.engineId
         if not self.v3Params.engineIdMatch:
            engIdString += warningEngIdSuffix
         print( "Engine ID      : %s" % engIdString )
         print( "Authentication : %s" % authString )
         print( "Privacy        : %s" % privString )
      print( "Group          : %s" % groupString )

class UserCollectionModel( CliModel.Model ):
   users = CliModel.Dict( keyType=str, valueType=UserModel, help='SNMP users' )

   def renderUserCollection( self, version ):
      for name in sorted( self.users ):
         self.users[ name ].renderUser( version, name )

class ShowSnmpUsers( CliModel.Model ):
   usersByVersion = CliModel.Dict( keyType=str, valueType=UserCollectionModel,
                              help='SNMP users grouped by version v1, v2c or v3' )
   def render( self ):
      for v in sorted( self.usersByVersion ):
         self.usersByVersion[ v ].renderUserCollection( v )

class SnmpBuffer( CliModel.Model ):
   clientRecvBuf = CliModel.Int( help="Client receive buffer size" )
   clientSendBuf = CliModel.Int( help="Client send buffer size" )
   serverRecvBuf = CliModel.Int( help="Server receive buffer size" )
   serverSendBuf = CliModel.Int( help="Server send buffer size" )

   def render( self ):
      if self.clientRecvBuf > 0:
         print( "Client receive buffer size: %s" % ( self.clientRecvBuf ) )
      if self.clientSendBuf > 0:
         print( "Client send buffer size: %s" % ( self.clientSendBuf ) )
      if self.serverRecvBuf > 0:
         print( "Server receive buffer size: %s" % ( self.serverRecvBuf ) )
      if self.serverSendBuf > 0:
         print( "Server send buffer size: %s" % ( self.serverSendBuf ) )

class SnmpCommunityDetailModel( CliModel.Model ):
   access = CliModel.Enum( values=( "readOnly", "readWrite" ),
                                   help="Community access level" )
   ipV4AccessList = CliModel.Str( help="IPv4 access-list for this community" )
   ipV4AccessListStatus = CliModel.Enum( values=( "nonExistent",
                                                  "ignoredNotAStandardAcl" ),
                                         help="IPv4 access list status",
                                         optional=True )
   ipV6AccessList = CliModel.Str( help="IPv6 access-list for this community" )
   ipV6AccessListStatus = CliModel.Enum( values=( "nonExistent",
                                                  "ignoredNotAStandardAcl" ),
                                         help="IPv6 access list status",
                                         optional=True )
   view = CliModel.Str( help="Community MIB view name" )
   viewStatus = CliModel.Enum( values=( 'notConfigured', ),
                               help="Snmp view status",
                               optional=True )
   defaultContext = CliModel.Str( help="Context associated for this community" )

class SnmpCommunityModel( CliModel.Model ):
   communities = CliModel.Dict( keyType=str, valueType=SnmpCommunityDetailModel,
                                help="A mapping of SNMP community to detail"
                                " community information" )

   def render( self ):
      for communityName, community in sorted( self.communities.items() ):
         print( "\nCommunity name: %s" % communityName )
         print( "Community access: %s" % (
            "read-only" if community.access == 'readOnly' else 'read-write' ) )
         if community.view:
            view = "Community view: %s" % community.view
            if community.viewStatus and community.viewStatus == 'notConfigured':
               view += " (non-existent)"
            print( view )

         def _maybePrintAccessListStr( prefix, accessList, status ):
            if accessList:
               accessListStr = prefix + accessList
               if status:
                  if status == "nonExistent":
                     accessListStr += " (non-existent)"
                  elif status == "ignoredNotAStandardAcl":
                     accessListStr += " (ignored, not a standard acl)"
               print( accessListStr )

         _maybePrintAccessListStr( "Access list: ", community.ipV4AccessList,
                                   community.ipV4AccessListStatus )
         _maybePrintAccessListStr( "IPv6 access list: ", community.ipV6AccessList,
                                   community.ipV6AccessListStatus )

         if community.defaultContext:
            print( "Community default context:", community.defaultContext )

class SnmpNotification( CliModel.Model ):
   component = CliModel.Str( help="Component owning notification" )
   name = CliModel.Str( help="Name of notification" )
   enabled = CliModel.Bool(
         help="Notification is enabled" )
   reason = CliModel.Str( help="Reason for notification state" )

class SnmpNotificationModel( CliModel.Model ):
   notifications = CliModel.List( valueType=SnmpNotification,
                                  help="List of notifications that are configured" )

   def render( self ):
      tableHeadings = ( "Type", "Name", "Enabled" )
      table = TableOutput.createTable( tableHeadings, tableWidth=110 )
      f = TableOutput.Format( justify="left" )
      f.padLimitIs( True )
      table.formatColumns( *( [ f ] * len( tableHeadings ) ) )
      for notification in sorted( self.notifications, key=attrgetter( "component",
                                                                      "name" ) ):
         table.newRow( notification.component, notification.name,
                       notification.reason )
      print( table.output() )

class SnmpView( CliModel.Model ):
   name = CliModel.Str( help="SNMP view name" )
   viewType = CliModel.Dict( keyType=str,
                             valueType=bool,
                             help='Mapping from OID subtree to inclusion status' )

class SnmpViewModel( CliModel.Model ):
   views = CliModel.List( help='List of SNMP views',
                          valueType=SnmpView )

   def render( self ):
      for view in sorted( self.views, key=attrgetter( "name" ) ):
         # if a specific view was requested, there will only be
         # one view, if there are no OID-trees configured print not found
         if not view.viewType:
            print( 'SNMP View %s not found' % view.name )
         for viewType in sorted( view.viewType ):
            status = 'included' if view.viewType[ viewType ] else "excluded"
            print( "%s %s - %s" % ( view.name, viewType, status ) )

authLevelStrMap = { 'v1': 'v1', 'v2c': 'v2c', 'v3NoAuth': 'v3 noauth',
                    'v3Auth': 'v3 auth', 'v3Priv': 'v3 priv' }
class SnmpGroupDetailModel( CliModel.Model ):
   secModel = CliModel.Enum( values=authLevelStrMap,
                             help="Security model in effect for this group" )
   readView = CliModel.Str( help="Name of the SNMP read view" )
   readViewConfig = CliModel.Bool( help="True if the read view exists and has been"
                                   " configured",
                                   optional=True )
   writeView = CliModel.Str( help="Name of the SNMP write view" )
   writeViewConfig = CliModel.Bool( help="True if the write view exists and has been"
                                    " configured",
                                    optional=True )
   notifyView = CliModel.Str( help="Name of the SNMP notify view" )
   notifyViewConfig = CliModel.Bool( help="True if the notify view exists and has"
                                     " been configured",
                                     optional=True )

class SnmpGroupVerToDetailModel( CliModel.Model ):
   versions = CliModel.Dict( keyType=str, valueType=SnmpGroupDetailModel,
                           help="A mapping of SNMP version to SNMP group details" )

class SnmpGroupModel( CliModel.Model ):
   groups = CliModel.Dict( keyType=str, valueType=SnmpGroupVerToDetailModel,
                        help="A mapping of SNMP group name to SNMP versions" )

   def render( self ):
      for name, versions in sorted( self.groups.items() ):
         for _, group in sorted( versions.versions.items() ):
            print( "\nGroup name     : %s" % name )
            secModel = authLevelStrMap[ group[ 'secModel' ] ]
            print( "Security model : %s" % secModel )
            groupDict = group.toDict()
            readViewConfig = groupDict.get( 'readViewConfig' )
            writeViewConfig = groupDict.get( 'writeViewConfig' )
            notifyViewConfig = groupDict.get( 'notifyViewConfig' )

            for view, text, cfg, noText in zip(
                  [ group[ 'readView' ], group[ 'writeView' ],
                    group[ 'notifyView' ] ],
                  [ "Read view      :", "Write view     :", "Notify view    :" ],
                  [ readViewConfig, writeViewConfig, notifyViewConfig ],
                  [ "<no read view specified (default: all included)>",
                    "<no write view specified>",
                    "<no notify view specified>" ] ):
               if view:
                  if cfg is False:
                     view += warningStrSuffix
               else:
                  view = noText
               print( "%s %s" % ( text, view ) )

class SnmpTotalModel( CliModel.Model ):
   chassis = CliModel.Submodel( valueType=SnmpChassis,
                                help='Chassis ID' )
   contact = CliModel.Submodel( valueType=Contact,
                                help='SNMP contact' )
   location = CliModel.Submodel( valueType=SnmpLocation,
                                 help='SNMP location' )
   counters = CliModel.Submodel( valueType=SnmpCounters,
                                 help='SNMP counters',
                                 optional=True )
   logging = CliModel.Submodel( valueType=SnmpLogging,
                                help='SNMP logging' )
   vrfs = CliModel.Submodel( valueType=SnmpVrfs,
                             help='SNMP VRFs' )
   srcIntf = CliModel.Submodel( valueType=SnmpInterface,
                                help='SNMP source interfaces' )
   misc = CliModel.Submodel( valueType=SnmpMiscConfig,
                             help='SNMP miscellaneous configuration' )
   enabled = CliModel.Bool( help="Indicates whether SNMP is enabled" )

   def render( self ):
      self.chassis.render()
      self.contact.render()
      self.location.render()
      if self.counters:
         self.counters.render()
      self.logging.render()
      self.vrfs.render()
      self.srcIntf.render()
      self.misc.render()
      if not self.enabled:
         print ( "SNMP agent disabled: Either no communities and no users are"
                 " configured, or no VRFs are configured." )

class SnmpMibTranslateModel( CliModel.Model ):
   numeric = CliModel.Str( help="Numeric form of the OID" )
   symbolic = CliModel.Str( help="Symbolic (textual) form of the OID" )
   _numToSym = CliModel.Bool( help="Private variable used by render to determine"
                              " which value to display" )

   def numToSymIs( self, value ):
      self._numToSym = value

   def render( self ):
      if self._numToSym is True:
         print( self.symbolic )
      elif self._numToSym is False:
         print( self.numeric )

class SnmpOidsModel( CliModel.Model ):
   oids = CliModel.List( help="Registered OIDs",
                         valueType=str )

class SnmpRegisteredOidsModel( CliModel.Model ):
   contexts = CliModel.Dict( valueType=SnmpOidsModel, help="A mapping from "
                             "a context (VRF) to its registered OIDs" )

   def render( self ):
      for context in self.contexts:
         print()
         if context == 'default':
            print( "In default context:" )
         else:
            print( "In context %r:" % context )

         for oid in self.contexts[ context ].oids:
            print( oid )

class SnmpTableIteratorCounter( CliModel.Model ):
   counters = CliModel.Dict( valueType=int,
                                 help="Table iterator counters" )

class SnmpTableIteratorCounterType( CliModel.Model ):
   counterTypes = CliModel.Dict( valueType=SnmpTableIteratorCounter,
                                 help="Table iterator counters by type" )

   def render( self ):
      for counterType, counters in sorted( self.counterTypes.items() ):
         print( "Counter type:", counterType )
         for counterName, value in sorted( counters.counters.items() ):
            print( counterName + ":", value )
         print()

class SnmpTableIteratorCounterModel( CliModel.Model ):
   tables = CliModel.Dict( valueType=SnmpTableIteratorCounterType,
                           help="Counters by table" )

   def render( self ):
      for tableName, table in sorted( self.tables.items() ):
         print( "Table:", tableName )
         table.render()
         print()

class SnmpTableIteratorContextCounterModel( CliModel.Model ):
   contexts = CliModel.Dict( valueType=SnmpTableIteratorCounterModel,
                             help="Counters by context" )

   def render( self ):
      for contextName, context in sorted( self.contexts.items() ):
         print( "Context:", contextName )
         context.render()
