# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import re

import CliMatcher
from CliPlugin.VirtualIntfRule import _VirtualIntfMatcher
from CliPlugin.VirtualIntfRule import VirtualIntfMatcherBase
import Tac

class _SpeedGroupMatcher( _VirtualIntfMatcher ):

   def __init__( self ):
      _VirtualIntfMatcher.__init__( self, "speed-group", SpeedGroupPathMatcher() )

class SpeedGroupMatcher( VirtualIntfMatcherBase ):
   ''' Matches a speed-group name of the format
       speed-group<slotNumber>/<groupNumber> or speed-group<groupNumber>.
       This also matches if only a prefix of speed-group is specified.
       speed-group followed by a whitespace is also allowed.
   ''' 
   def __init__( self, **kwargs ):
      VirtualIntfMatcherBase.__init__( self,
                                       "speed-group",
                                       _SpeedGroupMatcher(),
                                       SpeedGroupPathMatcher(),
                                       **kwargs )

class ModularSystemSpeedGroupMatcher:
   speedGroupPathMatchRe = re.compile( '([0-9]+)/([0-9]+)$' )

   def __init__( self ):
      self.speedGroupNumMatcher_ = CliMatcher.IntegerMatcher(
         1, 100, helpdesc="Group ID number" )

   def findSlot( self, chassis, slotNum ):
      slot = chassis.cardSlot.get( int( slotNum ) )
      if slot and slot.tag == "Linecard":
         return slot
      return None

   def match( self, mode, token, chassis ):
      m = self.speedGroupPathMatchRe.match( token )
      if not m:
         return None

      slotNum = m.group( 1 )
      groupNum = m.group( 2 )

      # Match the slot.
      slot = self.findSlot( chassis, slotNum )
      if not slot:
         return None

      # Match the group.
      if ( self.speedGroupNumMatcher_.match( mode, None, groupNum )
           is CliMatcher.noMatch ):
         return None

      return token

   def completions( self, mode, token, chassis ):
      m = self.speedGroupPathMatchRe.match( token )
      if not m:
         return []

      slotNum = m.group( 1 )
      groupNum = m.group( 2 )

      # Validate slot.
      slot = self.findSlot( chassis, slotNum )
      if not slot:
         return []

      # Return group completion.
      return self.speedGroupNumMatcher_.completions( mode, None, groupNum )

class FixedSystemSpeedGroupMatcher:
   speedGroupPathMatchRe = re.compile( '([0-9]+)$' )

   def __init__( self ):
      self.speedGroupNumMatcher_ = CliMatcher.IntegerMatcher(
         1, 100, helpdesc="Group ID number" )

   def match( self, mode, token ):
      m = self.speedGroupPathMatchRe.match( token )
      if ( ( m is not None ) and
           ( self.speedGroupNumMatcher_.match( mode, None,
             m.group( 1 ) ) is not CliMatcher.noMatch ) ):
         return CliMatcher.MatchResult( token, token )
      else:
         return None

   def completions( self, mode, token ):
      return self.speedGroupNumMatcher_.completions( mode, None, token )

class SpeedGroupPathMatcher:

   def __init__( self ):
      self.noEntityMibRootSlotNumMatcher_ = CliMatcher.IntegerMatcher(
         1, 32, helpdesc="Slot number" )
      self.noModuleSpeedGroupNumMatcher_ = CliMatcher.IntegerMatcher(
         1, 100, helpdesc="Group ID number" )
      self.modularSpeedGroupPathMatchRe_ = re.compile( '([0-9]+)/([0-9]+)$' )
      self.fixedSpeedGroupPathMatchRe_ = re.compile( '([0-9]+)$' )
      self.modularSystemSpeedGroupMatcher_ = ModularSystemSpeedGroupMatcher()
      self.fixedSystemSpeedGroupMatcher_ = FixedSystemSpeedGroupMatcher()

   def entityMibRoot( self, mode ):
      return mode.entityManager.lookup( 'hardware/entmib' ).root

   def match( self, mode, context, token ):
      entityMibRoot = self.entityMibRoot( mode )
      # Handle the case that Fru has not started yet or that
      # the entity mib root has not yet fully been created.
      # this is the best we can do prior to the mib root being created.
      if ( entityMibRoot is None or
           entityMibRoot.initStatus != "ok" ):
         # Attempt to match a fixed system path.
         m = self.fixedSpeedGroupPathMatchRe_.match( token )
         if ( ( m is not None ) and
              ( self.noModuleSpeedGroupNumMatcher_.match( mode, context,
                m.group( 1 ) ) is not CliMatcher.noMatch ) ):
            return CliMatcher.MatchResult( token, token )

         # Attempt to match a modular system path.
         m = self.modularSpeedGroupPathMatchRe_.match( token )
         if ( ( m is not None ) and
              ( self.noEntityMibRootSlotNumMatcher_.match( mode, context,
                m.group( 1 ) ) is not CliMatcher.noMatch ) ):
            if ( self.noModuleSpeedGroupNumMatcher_.match( mode, context,
                   m.group( 2 ) ) is not CliMatcher.noMatch ):
               return CliMatcher.MatchResult( token, token )
         return CliMatcher.noMatch

      # The entity mib root has been created.
      if entityMibRoot.tacType.fullTypeName == "EntityMib::Chassis":
         m = self.modularSystemSpeedGroupMatcher_.match( mode, token, entityMibRoot )
         if m is None:
            return CliMatcher.noMatch
         return CliMatcher.MatchResult( token, token )
      elif entityMibRoot.tacType.fullTypeName == "EntityMib::FixedSystem":
         m = self.fixedSystemSpeedGroupMatcher_.match( mode, token )
         if m is None:
            return CliMatcher.noMatch
         return CliMatcher.MatchResult( token, token )
      else:
         # pylint: disable-next=consider-using-f-string
         raise Tac.InternalException( "Unknown entity mib root class %s"
                                      % entityMibRoot.tacType.fullTypeName )

   def completions( self, mode, context, token ):
      entityMibRoot = self.entityMibRoot( mode )

      if ( not entityMibRoot or
           not entityMibRoot.tacType or
           not entityMibRoot.tacType.fullTypeName ):
         # No completions are returned if the entity mib is not populated
         # with chassis and slot information.
         return []
      elif entityMibRoot.tacType.fullTypeName == "EntityMib::Chassis":
         return self.modularSystemSpeedGroupMatcher_.completions( mode,
                                                                  token,
                                                                  entityMibRoot )
      elif entityMibRoot.tacType.fullTypeName == "EntityMib::FixedSystem":
         return self.fixedSystemSpeedGroupMatcher_.completions( mode, token )
      else:
         # pylint: disable-next=consider-using-f-string
         raise Tac.InternalException( "Unknown entity mib root class %s"
                                      % entityMibRoot.tacType.fullTypeName )
