# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import (
   Bool,
   Dict,
   List,
   Model,
   Str
)

from TableOutput import (
   Format,
   createTable,
)

from CliMode.Intf import IntfMode

from Arnet import sortIntf

# VPLS instances may have bundled VLANs, a "VLAN aware" case.
# In such case, a number of things happen:
#
# When showing VLAN/SHG table, we don't want to list all VLANs
# in a bundle, since 1) there may be many and 2) it would make it look like
# SHGs belong to different VLANs and are, therefore, different groups with
# matching names. Where in fact it's the same group(s) shared among all VLANs
# in the bundle.
#
# At the same time, we don't want to show all groups belonging to different
# bundles as if they belonged to the same one bundle. We want different records in
# JSON model and different lines in the human readable table for different bundles.
#
# Also, we will need to construct a key to get values for group status, and
# that key is constructed differently in bundled case versus non-bundled.
# For the bundled case we need the VPLS instance name and VLAN=0, and for non-bundled
# we need VLAN ID and we don't use the name. This is true for any VLAN that appears
# in a VPLS bundle, even if we're coming from AC angle.
# Therefore we need to carry the "bundled" flag and the VPLS instance name,
# including for VLANs that are used for both VPLS PWEs and ACs.
#
# The model carries VLAN IDs as keys. So we will be using the first VLAN ID
# of a bundle as the key. In a correct config, bundles may not overlap,
# so different bundles will get separate records.
# We must take care of a configuration when there is a bundle 2-7,
# and a VLAN=3 referenced from an AC that is part of the bundle. In this case
# we must detect it and merge the record for VID=3 into VID=2+bundled.
#
# - a VLAN VID=2, not bundled, will be recorded as key=2, vlanBundle=false
# - an AC or PWE with no VLAN assigned will be recorded under key=0
# - a VLAN bundle 4-8 will be recorded as key=4, vlanBundle=true
# - a VLAN VID=5 that is referenced from an AC only (but falls within
#   4-8 VPLS bundle) will be recorded as key=4, vlanBundle=true

# in all Cli commands we accept the first 32 characters of a group name
maxGroupNameSize = 32

class SplitHorizonGoup( Model ):
   """A model of a single Split Horizon group"""
   active = Bool( help="This group is correctly configured and programmed" )
   vplsInstName = Str( help="VPLS instance name", optional=True )
   # for the split horizon show command, we just need the names of the interfaces
   interfaces = List( valueType=str, help="Interface within a split horizon group" )

class VlanWithGroups( Model ):
   """A model of a VLAN that uses split horizon groups"""
   groups = Dict( help='Split horizon groups keyed by group name',
                  keyType=str,
                  valueType=SplitHorizonGoup )
   vlanBundle = Bool( help="This VLAN is part of a VLAN bundle" )

class SplitHorizonGroups( Model ):
   """A model of split horizon group usage by vlans"""
   # key by string so we can key by vid or name later or even mix it
   vlans = Dict( help='VLANs that use split horizon groups, '
                              'keyed by VLAN',
                              keyType=str,
                              valueType=VlanWithGroups )
   def render( self ):
      if not self.vlans:
         return

      table = createTable( ( "VLAN", "Split Horizon Group", "Status", "Members" ) )

      ftVlan = Format( justify="left", minWidth=len( 'VLAN' ), wrap=False )
      ftVlan.noPadLeftIs( True )
      ftVlan.padLimitIs( True )

      ftStatusWidth = max( len( s ) for s in [ 'Status', 'active', 'disabled' ] )
      ftStatus = Format( justify="left",
                         minWidth=ftStatusWidth,
                         wrap=False )
      ftStatus.noPadLeftIs( True )
      ftStatus.padLimitIs( True )

      # we set max width here to force it to wrap where we want
      ftGroup = Format( justify="left",
                        minWidth=len( 'Split Horizon Group' ),
                        maxWidth=( 2 * maxGroupNameSize + len( 'VPLS::' ) ),
                        wrap=True )
      ftGroup.noPadLeftIs( True )
      ftGroup.padLimitIs( True )

      ftMembers = Format( justify="left", minWidth=len( 'Members' ),
                          wrap=True )
      ftMembers.noPadLeftIs( True )
      ftMembers.padLimitIs( True )

      table.formatColumns( ftVlan, ftGroup, ftStatus, ftMembers )

      # a row for every group, so there will be multiple rows for each vlan
      for vid, vlan in sorted( self.vlans.items() ):
         for groupName, group in sorted( vlan.groups.items() ):
            # we leave full interface names in the model, but shorten for humans
            shortIntfNames = [ IntfMode.getShortname( s ) for s in group.interfaces ]
            portString = ', '.join( sortIntf( shortIntfNames ) )

            assert group.active is not None, "group status is not set"
            statusString = 'active' if group.active else 'disabled'

            # for vlan aware VPLS groups we add VPLS and instance name
            # before group name for humans
            if vlan.vlanBundle and group.vplsInstName:
               groupName = f'VPLS:{group.vplsInstName}:{groupName}'

            if vlan.vlanBundle:
               # in json, we return whatever vid we have and a bundled flag,
               # but for humans we show VB.
               assert vid != '0', 'cannot be bundled if there is no VLAN assigned'
               vid = 'VB'
            elif vid == '0':
               # if AC or PWE doesn't have a VLAN assigned, we return 0 in json,
               # but for humans we show empty
               vid = ''
            else:
               # use the assigned vid
               pass

            table.newRow( vid, groupName, statusString, portString )

      print( table.output() )
