# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from operator import attrgetter
import AgentDirectory
import AgentCommandRequest
import BasicCli
import CliCommand
import CliExtensions
import CliMatcher
import CliParser
import ConfigMount
import LazyMount
import ShowCommand
import SharedMem
import Smash
import Tac
from CliPlugin import TeCli
from CliPlugin import TechSupportCli
from CliPlugin import IntfCli
from CliPlugin.IpGenAddrMatcher import IpGenAddrMatcher
from CliPlugin.SrTePolicyLibCli import ( getSegmentListVias,
      cliDisplayOrderLabelList, SrTeSegmentListVia )
from CliPlugin.TechSupportCli import techSupportKwMatcher, extendedKwMatcher
from CliPlugin.BfdGlobalConfigMode import SbfdReflectorDiscExpr, reflectorAdapter
from CliPlugin.SrTePolicyModel import (
   SrTePolicyBase,
   SrTeSegment,
   SrTePolicyBindingSid,
   SrTePolicyOriginator,
   SrTePolicyPathGroupSegmentList,
   SrTePolicyPathGroup,
   SrTePolicyCandidates,
   SrTePolicyCandidatesModel,
   SrTePolicyCandidatesVrfModel,
   SrTePolicyActive,
   SrTePolicyActiveModel,
   SrTePolicyActiveVrfModel,
   SrTePolicyDynBsidLabelRangeModel,
   SrTePolicySegmentList,
   SrTePolicySegmentLists,
   SrTePolicyFecTable,
   SrTePolicyFecInfo,
   SrTePolicyFecVia,
   VrfSrTePolicySegmentLists,
   SrTePolicySegmentListCounters,
   SrTePolicySegmentListsCounters,
   VrfSrTePolicySegmentListsCounters,
   SrTePolicySummaryModel,
   SrTePolicySummaryVrfModel,
   SrTePolicyPathGroupStatisticsEntryModel,
   SrTePolicyPathGroupComputationStatisticsEntryModel,
   SrTePolicyPathGroupEpProvisioningStatisticsModel,
   SrTePolicyStatisticsEntryModel,
   SrTePolicyStatisticsModel,
   SrTePolicyPathGroupStatisticsModel,
   SrTePolicySegmentListStatisticsModel,
)
from CliPlugin.TunnelCli import (
   TunnelTableIdentifier,
   readMountTunnelTable,
)
from CliToken.Bfd import (
   matcherInterval,
   matcherMultiplierVal,
   matcherTxRxIntervalMs,
)
from CliToken.Clear import clearKwNode
from CliToken.SegmentRoutingToken import matcherSegmentRoutingForShow
from Arnet import IpAddress, IpGenAddr, IntfId
from Arnet.MplsLib import ConvertLabelStack
from ArnetLib import bgpFormatAsn
from CliMode.SrTePolicy import ( SrTeModeBase,
                                 SrTePolicyModeBase,
                                 SrTeDynamicPolicyModeBase,
                                 SrTePolicyPathGroupModeBase,
                                 SrTePolicyPathLocalComputationModeBase,
                                 SrTePolicySegmentListModeBase,
                               )
from IpLibConsts import DEFAULT_VRF
import SrTePolicyCommonLib
from SrTePolicyCommonLib import tacColor, tacEnlpEnum, srTePolicyStatusPath
from SrTePolicyLib import (
   policySrcToString,
   tacProtocol,
   tacPreference,
   tacPolicyCostMetric,
   tacWeight,
   tacSegmentListId,
   tacCandidateId,
   tacPolicyId,
   tacPolicyInvalidReason,
   MplsLabel,
   MplsLabelOperation,
   Preference,
   Weight,
)
from SrTePolicy import (
   tacIndex,
   tacPriority,
   tacStaticSegList,
   Index,
   CandidatePathType
)
from TypeFuture import TacLazyType
from Toggles.SrTePolicyToggleLib import (
   toggleSrTePolicyIgpShortcutEnabled,
   toggleOptionalBsidEnabled,
   toggleDynamicallyAssignBsidEnabled,
   toggleInstallUnprotectedSLTilfaBackupExceedsMSDEnabled,
   toggleSrTeTunnelHoldDownTimerEnabled )
from Toggles.SrTePolicyLibToggleLib import (
   toggleOcInvalidSrTeEnabled,
   toggleDynamicSrTeEnabled
)

TunnelType = TacLazyType( "Tunnel::TunnelTable::TunnelType" )
TunnelId = TacLazyType( "Tunnel::TunnelTable::TunnelId" )
DynTunnelIntfId = TacLazyType( "Arnet::DynamicTunnelIntfId" )
FecIdIntfId = TacLazyType( 'Arnet::FecIdIntfId' )
FecId = TacLazyType( 'Smash::Fib::FecId' )
ConfigSessionType = TacLazyType( "Bfd::SessionType" )
BfdOperState = TacLazyType( "Bfd::OperState" )
SegmentListSharing = TacLazyType( "SrTePolicy::SegmentListSharing" )
PolicySbfdConfigType = TacLazyType( "SrTePolicy::Sbfd::PolicySbfdConfig" )
DynamicMplsLabelStack = TacLazyType( 'Arnet::DynamicMplsLabelStack' )
BoundedMplsLabelStack = TacLazyType( 'Arnet::BoundedMplsLabelStack' )
SbfdHoldDownTime = TacLazyType( 'Bfd::HoldDownTime' )

srConfig = None
repSegConfig = None
mplsConfig = None
mplsHwCapability = None
routingHwStatusCommon = None
routingHwStatus = None
policyStatus = None
ecPolicy = None
policyKeyToIdMap = None
allPolicy = None
allSl = None
slStatus = None
slTunnelTable = None
srteForwardingStatus = None
bgpPolicyInput = None
staticPolicyInput = None
policyCounterClear = None
policyCounterTable = None
bfdStatusPeer = None
policyNameToKeyMap = None
lfibStatus = None
tilfaTunnelTable = None
flexAlgoConfig = None
labelManagerStatus = None
dynamicBsidStats = None

discriminatorMSB = 15
candidatePathTypeToComputationTypeDict = {
   CandidatePathType.explicitSegmentLists: 'configured',
   CandidatePathType.segmentRoutingLabelResolution: 'segmentRouting',
   CandidatePathType.segmentRoutingFlexAlgoLabelResolution: 'flexAlgo',
}
candidatePathTypeToPathComputationDict = {
   CandidatePathType.explicitSegmentLists: 'configured',
   CandidatePathType.segmentRoutingLabelResolution: 'isisSr',
   CandidatePathType.segmentRoutingFlexAlgoLabelResolution: 'isisSrFlexAlgo',
}

#--------------------------------------------------------
# Shared configuration tokens
#--------------------------------------------------------
matcherColorValue = CliMatcher.IntegerMatcher( tacColor.min, tacColor.max,
                                               helpdesc='Color value' )
matcherPolicy = CliMatcher.KeywordMatcher( 'policy',
      helpdesc='Show Segment Routing Traffic Engineering policies' )
matcherTrafficEngineering = CliMatcher.KeywordMatcher( 'traffic-engineering',
      helpdesc='Traffic Engineering related information' )
matcherBgp = CliMatcher.KeywordMatcher( 'bgp',
      helpdesc='Show policies learnt via BGP' )
matcherColor = CliMatcher.KeywordMatcher( 'color',
      helpdesc='Color' )
matcherDynamic = CliMatcher.KeywordMatcher( 'dynamic',
      helpdesc='Dynamically learned endpoint' )
matcherEndpoint = CliMatcher.KeywordMatcher( 'endpoint',
      helpdesc='Show policies for a given endpoint' )
matcherId = CliMatcher.KeywordMatcher( 'id',
      helpdesc='Segment list ID' )
matcherSegmentList = CliMatcher.KeywordMatcher( 'segment-list',
      helpdesc='Show Segment Routing Traffic Engineering policy segment lists' )
matcherSource = CliMatcher.KeywordMatcher( 'source',
      helpdesc='Show policies for a given source' )
matcherStatic = CliMatcher.KeywordMatcher( 'static',
      helpdesc='Show policies configured statically' )
#--------------------------------------------------------
# SrTePolicy Agent running ?
#--------------------------------------------------------
def isSrTePolicyAgentRunning( mode ):
   return AgentDirectory.agentIsRunning( mode.entityManager.sysname(), 'SrTePolicy' )

#--------------------------------------------------------
# Segment-routing currently only support mpls dataplane
#--------------------------------------------------------
def mplsSupportedGuard( mode, token ):
   if mplsHwCapability.mplsSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

#-------------------------------------------------------------------------
# Helper Class. This is used as a base class for all dependent classes of
# 'config-te' mode. When the traffic engineering is unconfigured the
# dependents can cleanup their configuration.
#-------------------------------------------------------------------------
class SrTeDependentBase:
   def __init__( self, mode ):
      pass

   def setDefault( self ):
      pass

#----------------------------------------
# Routine to register dependent class
#----------------------------------------
SrTeModeDependents = IntfCli.DependentClassRegistry()

#---------------------------------
# [ no | default ] segment-routing
#---------------------------------
def delSegmentRouting( mode ):
   """Remove all policy configuration on 'no segment-routing'
      or 'no traffic-engineering'
   """
   srConfig.policy.clear()
   if toggleDynamicSrTeEnabled():
      srConfig.dynamicEpPolicy.clear()
   repSegConfig.staticReplicationSegment.clear()
   srConfig.enabled = False

   for cls in SrTeModeDependents:
      cls( mode ).setDefault()

class CfgSegmentRoutingCmd( CliCommand.CliCommandClass ):
   syntax = 'segment-routing'
   noOrDefaultSyntax = syntax
   data = {
      'segment-routing': CliCommand.guardedKeyword( 'segment-routing',
                                                    'Segment Routing configuration',
                                                    guard=mplsSupportedGuard )
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( SrTeMode )
      mode.session_.gotoChildMode( childMode )
      srConfig.enabled = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      delSegmentRouting( mode )

TeCli.RouterGlobalTeMode.addCommandClass( CfgSegmentRoutingCmd )

#---------------------------------------------------------------
# Remove all segment-routing configs when the parent is removed
# i.e., "no router traffic-engineering" in config mode.
#---------------------------------------------------------------
class TeSrSubMode( TeCli.TeDependentBase ):
   def setDefault( self ):
      delSegmentRouting( self.mode )

#-------------------------------------------------------------------------------
# [  no | default ] policy endpoint ( ADDR | dynamic ) color COLOR
#-------------------------------------------------------------------------------
class CfgPolicyCmd( CliCommand.CliCommandClass ):
   syntax = 'policy endpoint ( ADDR | dynamic ) color COLOR' if \
         toggleDynamicSrTeEnabled() else 'policy endpoint ADDR color COLOR'
   noOrDefaultSyntax = syntax
   data = {
         'policy': 'Segment Routing policy configuration',
         'endpoint': 'Endpoint',
         'ADDR': IpGenAddrMatcher( 'IPv4 or IPv6 address' ),
         'dynamic': matcherDynamic,
         'color': matcherColor,
         'COLOR': matcherColorValue,
   }

   @staticmethod
   def handler( mode, args ):
      color = args[ 'COLOR' ]
      policy = None
      if 'dynamic' in args:
         key = SrTePolicyCommonLib.Color( color )
         policy = srConfig.dynamicEpPolicy.get( key )
         if policy is None:
            policy = srConfig.dynamicEpPolicy.newMember( key )
            policy.policyParams = ()
         childMode = mode.childMode( SrTeDynamicPolicyMode, policy=policy,
                                     color=color )
      else:
         endPoint = args[ 'ADDR' ]
         key = SrTePolicyCommonLib.PolicyKey( endPoint, color )
         policy = srConfig.policy.get( key )
         if policy is None:
            policy = srConfig.policy.newMember( key )
            if toggleDynamicSrTeEnabled():
               policy.policyParams = ()
         childMode = mode.childMode( SrTePolicyMode, policy=policy,
                                     endPoint=endPoint, color=color )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      color = args[ 'COLOR' ]
      if 'dynamic' in args:
         key = SrTePolicyCommonLib.Color( color )
         if key in srConfig.dynamicEpPolicy:
            # Remove the policy config.
            del srConfig.dynamicEpPolicy[ key ]
      else:
         endPoint = args[ 'ADDR' ]
         key = SrTePolicyCommonLib.PolicyKey( endPoint, color )
         # Remove the mapping for the policyName as this policy is going
         # away.
         if key in srConfig.policy:
            policyCfg = srConfig.policy[ key ]
            if toggleDynamicSrTeEnabled():
               policyName = policyCfg.policyParams.policyName
            else:
               policyName = policyCfg.policyName
            if policyName in policyNameToKeyMap.policy:
               del policyNameToKeyMap.policy[ policyName ]
            # Remove the policy config.
            del srConfig.policy[ key ]

class SrTeMode( SrTeModeBase, BasicCli.ConfigModeBase ):
   name = "Segment Routing Configuration"

   def __init__( self, parent, session ):
      SrTeModeBase.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

SrTeMode.addCommandClass( CfgPolicyCmd )

class CfgColoredTunnelRibCmd( CliCommand.CliCommandClass ):
   syntax = 'rib system-colored-tunnel-rib'
   noOrDefaultSyntax = syntax
   data = {
      'rib': 'Include policies in a resolution RIB',
      'system-colored-tunnel-rib': 'The system colored tunnel RIB',
   }

   @staticmethod
   def handler( mode, args ):
      srConfig.populateColoredTunnelRib = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      srConfig.populateColoredTunnelRib = False

matcherCostValue = CliMatcher.IntegerMatcher( 0, 255,
      helpdesc='The IGP cost value' )
matcherIgpCost = CliMatcher.KeywordMatcher( 'igp-cost',
      helpdesc='IGP metric and preference values for next'
               'hops resolving via the policy' )
matcherIgpPreference = CliMatcher.KeywordMatcher( 'preference',
      helpdesc='IGP preference for the policy' )
matcherIgpMetric = CliMatcher.KeywordMatcher( 'metric',
      helpdesc='IGP metric for the policy' )
matcherMetricDynamic = CliMatcher.KeywordMatcher( 'dynamic',
      helpdesc='Dynamically derived metric from IGP path' )
matcherMetricAdditive = CliMatcher.KeywordMatcher( '+',
      helpdesc='Value to be added to the dynamically derived metric' )
matcherMetricSubtractive = CliMatcher.KeywordMatcher( '-',
      helpdesc='Value to be subtracted from the dynamically derived metric' )
matcherMetricAddSubValue = CliMatcher.IntegerMatcher( tacPolicyCostMetric.minMetric,
      tacPolicyCostMetric.maxMetric, helpdesc='Value of the route metric' )
matcherMetricUnreach = CliMatcher.KeywordMatcher( 'igp-unreachable',
      helpdesc='Metric to be set when the endpoint is not reachable in IGP domain' )
matcherMetricValue = CliMatcher.IntegerMatcher( tacPolicyCostMetric.minMetric,
                                                tacPolicyCostMetric.maxMetric,
                                                helpdesc='IGP metric value' )

def createPolicyCostMetric( args, costSet, cost, metric,
      costMetricHandle, perPolicyCmd ):
   dynamicMetric = args.get( 'dynamic' )
   metricModifierType \
         = Tac.Type( "SrTePolicy::PolicyCostMetric::MetricModifierType" )
   dynamicMetricModifier = args.get( 'VALUE' )
   dynamicMetricUnreachable = args.get( 'IGPU_VALUE', tacPolicyCostMetric.maxMetric )
   igpPrefCost = Tac.Value( "SrTePolicy::PolicyCostMetric", cost=cost,
         metric=metric, costSet=costSet )
   if dynamicMetric:
      igpPrefCost.dynamicMetric = True
      if dynamicMetricModifier:
         if '+' in args:
            dynamicMetricModifierType = metricModifierType.metricAdditive
         elif '-' in args:
            dynamicMetricModifierType = metricModifierType.metricSubtractive

         igpPrefCost.dynamicMetricModifier = dynamicMetricModifier
         igpPrefCost.dynamicMetricModifierType = dynamicMetricModifierType
      igpPrefCost.dynamicMetricUnreachable = dynamicMetricUnreachable
   if perPolicyCmd:
      costMetricHandle.costMetric = igpPrefCost
   else:
      costMetricHandle.igpPrefCost = igpPrefCost

# ----------------------------------------------------------
# [ no | default ] igp-cost preference COST | metric dynamic
# [(+|-) VALUE] [igp-unreachable UNREACHABLEVALUE]
# ----------------------------------------------------------
class CfgIgpCostPreference( CliCommand.CliCommandClass ):
   syntax = '''igp-cost
         { ( preference COST )
         | ( metric dynamic [(+|-) VALUE]
               [igp-unreachable IGPU_VALUE] ) }'''
   noOrDefaultSyntax = 'igp-cost ...'
   data = {
         'igp-cost': matcherIgpCost,
         'preference': CliCommand.Node( matcher=matcherIgpPreference, maxMatches=1 ),
         'COST': CliCommand.Node( matcherCostValue, maxMatches=1 ),
         'metric': CliCommand.Node( matcher=matcherIgpMetric, maxMatches=1 ),
         'dynamic': CliCommand.Node( matcherMetricDynamic, maxMatches=1 ),
         '+': CliCommand.Node( matcherMetricAdditive, maxMatches=1 ),
         '-': CliCommand.Node( matcherMetricSubtractive, maxMatches=1 ),
         'VALUE': CliCommand.Node( matcherMetricAddSubValue, maxMatches=1 ),
         'igp-unreachable': CliCommand.Node( matcherMetricUnreach, maxMatches=1 ),
         'IGPU_VALUE': CliCommand.Node( matcherMetricValue, maxMatches=1 ),
        }

   @staticmethod
   def handler( mode, args ):
      costSet = 'COST' in args
      cost = args[ 'COST' ] if costSet else srConfig.igpPrefCost.defaultIgpPrefCost

      createPolicyCostMetric( args=args, costSet=costSet, cost=cost,
            metric=srConfig.igpPrefCost.defaultIgpMetric,
            costMetricHandle=srConfig, perPolicyCmd=False )
   noOrDefaultHandler = handler

SrTeMode.addCommandClass( CfgColoredTunnelRibCmd )
SrTeMode.addCommandClass( CfgIgpCostPreference )

class EnlpBase( CliCommand.CliCommandClass ):
   syntax = '''explicit-null ( ipv4 | ipv6 | ( ipv4 ipv6 ) | none )'''
   noOrDefaultSyntax = 'explicit-null ...'

   @staticmethod
   def enlpHandler( mode, args, config ):
      if 'ipv4' in args:
         if 'ipv6' in args:
            config.enlp = tacEnlpEnum.ipv4AndIpv6
         else:
            config.enlp = tacEnlpEnum.ipv4
      elif 'ipv6' in args:
         config.enlp = tacEnlpEnum.ipv6
      elif 'none' in args:
         config.enlp = tacEnlpEnum.none

   data = {
              'explicit-null': 'Explicit null label policy',
              'ipv4': 'Push IPv4 explicit null label for IPv4 packet',
              'ipv6': 'Push IPv6 explicit null label for IPv6 packet',
              'none': 'No explicit null label be pushed'
   }

class CfgEnlpGlobalCmd( EnlpBase ):
   data = EnlpBase.data.copy()

   @staticmethod
   def handler( mode, args ):
      EnlpBase.enlpHandler( mode, args, srConfig )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      srConfig.enlp = tacEnlpEnum.localPolicy

SrTeMode.addCommandClass( CfgEnlpGlobalCmd )

# -------------------------------------------------------------------------------
# [ no | default ] sbfd echo default
# -------------------------------------------------------------------------------
class CfgSBFDEchoDefaultCmd( CliCommand.CliCommandClass ):
   syntax = 'sbfd echo default'
   noOrDefaultSyntax = syntax
   data = {
         'sbfd': 'Seamless BFD',
         'echo': 'Enable SBFD for all SR-Policy based on Interop Echo Mode',
         'default': 'Configure SBFD echo for all policies',
   }

   @staticmethod
   def handler( mode, args ):
      srConfig.globalSBFDEcho = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      srConfig.globalSBFDEcho = False

SrTeMode.addCommandClass( CfgSBFDEchoDefaultCmd )

# -------------------------------------------------------------------------------
# [ no | default ] sbfd hold down TIME seconds
# -------------------------------------------------------------------------------
class CfgSBFDHoldDownCmd( CliCommand.CliCommandClass ):
   syntax = 'sbfd hold down TIME seconds'
   noOrDefaultSyntax = 'sbfd hold down ...'
   data = {
         'sbfd': 'Seamless BFD',
         'hold': 'Configure SBFD SRTE tunnels hold down time',
         'down': 'Configure SBFD SRTE tunnels hold down time',
         'TIME': CliMatcher.IntegerMatcher( SbfdHoldDownTime.min,
                   SbfdHoldDownTime.max,
                   helpdesc='Hold down time in seconds' ),
         'seconds': 'Units in seconds'
   }

   @staticmethod
   def handler( mode, args ):
      seconds = args.get( 'TIME' )
      srConfig.sbfdHoldDownTime = seconds

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      srConfig.sbfdHoldDownTime = SbfdHoldDownTime.invalid

if toggleSrTeTunnelHoldDownTimerEnabled():
   SrTeMode.addCommandClass( CfgSBFDHoldDownCmd )

# -------------------------------------------------------------------------------
# [ no | default ] path selection tie-break discriminator
# -------------------------------------------------------------------------------
class CfgPathSelectionTieBreakDiscriminatorCmd( CliCommand.CliCommandClass ):
   syntax = 'path selection tie-break discriminator'
   noOrDefaultSyntax = 'path ...'
   data = {
      'path': 'Candidate path',
      'selection': 'Selection of candidate paths',
      'tie-break': 'Tie-breaking for candidate selection',
      'discriminator': 'Use discriminator as tie-breaker',
   }

   @staticmethod
   def handler( mode, args ):
      srConfig.useDiscriminatorInTieBreaking = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      srConfig.useDiscriminatorInTieBreaking = False

SrTeMode.addCommandClass( CfgPathSelectionTieBreakDiscriminatorCmd )

class SrTePolicyMode( SrTePolicyModeBase, BasicCli.ConfigModeBase ):
   name = "Segment Routing Policy Configuration"

   def __init__( self, parent, session, policy, endPoint, color ):
      SrTePolicyModeBase.__init__( self, ( endPoint, color ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.policy = policy
      self.endPoint = endPoint
      self.color = color

class SrTeDynamicPolicyMode( SrTeDynamicPolicyModeBase, BasicCli.ConfigModeBase ):
   name = "Segment Routing Dynamic Policy Configuration"

   def __init__( self, parent, session, policy, color ):
      SrTeDynamicPolicyModeBase.__init__( self, ( color ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.policy = policy
      self.color = color
      self.endPoint = None

# -------------------------------------------------------
# [  no | default ] igp shortcut
# -------------------------------------------------------
class CfgIgpShortcutCmd( CliCommand.CliCommandClass ):
   syntax = 'igp shortcut'
   noOrDefaultSyntax = 'igp shortcut'
   data = {
         'igp': 'IGP configuration',
         'shortcut': 'Allow SR-TE tunnel to be used for IGP shortcuts',
   }

   @staticmethod
   def handler( mode, args ):
      if toggleDynamicSrTeEnabled():
         mode.policy.policyParams.configIgpShortcut = True
      else:
         mode.policy.configIgpShortcut = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if toggleDynamicSrTeEnabled():
         mode.policy.policyParams.configIgpShortcut = False
      else:
         mode.policy.configIgpShortcut = False

if toggleSrTePolicyIgpShortcutEnabled():
   SrTePolicyMode.addCommandClass( CfgIgpShortcutCmd )

#-------------------------------------------------------
# [  no | default ] binding-sid SEGMENT_ID
#-------------------------------------------------------
class CfgBindingSidCmd( CliCommand.CliCommandClass ):
   syntax = 'binding-sid SEGMENT_ID'
   noOrDefaultSyntax = 'binding-sid [ SEGMENT_ID ]'
   data = {
         'binding-sid': 'Binding segment identifier',
         'SEGMENT_ID': CliMatcher.IntegerMatcher(
                          MplsLabel.unassignedMin,
                          MplsLabel.max,
                          helpdesc='Value of the MPLS label' ),
   }

   @staticmethod
   def handler( mode, args ):
      bsidVal = args[ 'SEGMENT_ID' ]
      if toggleDynamicSrTeEnabled():
         if mode.policy.policyParams.bindingSid == bsidVal:
            return
      else:
         if mode.policy.bindingSid == bsidVal:
            return
      # Warn the user if the label is not in SRLB range.
      srlbRange = mplsConfig.labelRange[ 'srlb' ]
      if not srlbRange.base <= bsidVal < ( srlbRange.base + srlbRange.size ):
         mode.addWarning( 'Binding SID is out of SRLB range' )
      # Set the Binding SID to the policy.
      if toggleDynamicSrTeEnabled():
         mode.policy.policyParams.bindingSid = MplsLabel( bsidVal )
      else:
         mode.policy.bindingSid = MplsLabel( bsidVal )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      policyParams = mode.policy.policyParams if toggleDynamicSrTeEnabled()\
         else mode.policy
      if policyParams.bindingSid == MplsLabel():
         return
      # Initialize the binding SID in the policy to default.
      if toggleDynamicSrTeEnabled():
         policyParams.bindingSid = MplsLabel()
      else:
         mode.policy.bindingSid = MplsLabel()

SrTePolicyMode.addCommandClass( CfgBindingSidCmd )

policyNamePattern = r'[^\s]*'
policyNameMatcher = CliMatcher.PatternMatcher(
      pattern=policyNamePattern,
      helpdesc='Policy Name',
      helpname='WORD' )

# -------------------------------------------------------
# [  no | default ] binding-sid specified-only disabled
# -------------------------------------------------------
class BindingSidSpecifiedOnly( CliCommand.CliCommandClass ):
   syntax = 'binding-sid specified-only disabled'
   noOrDefaultSyntax = syntax
   data = {
         'binding-sid': 'Binding segment identifier',
         'specified-only': 'Validity of candidate path based on'
         ' validity of its binding SID',
         'disabled': 'Allow policies without binding SID to become active'
   }

   # Default is srConfig.specifiedBsidOnly = True
   @staticmethod
   def handler( mode, args ):
      srConfig.specifiedBsidOnly = False

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      srConfig.specifiedBsidOnly = True

if toggleOptionalBsidEnabled():
   SrTeMode.addCommandClass( BindingSidSpecifiedOnly )

# -------------------------------------------------------
# [  no | default ] binding-sid allocation dynamic
# -------------------------------------------------------
class BindingSidAllocatedDynamically( CliCommand.CliCommandClass ):
   syntax = 'binding-sid allocation dynamic'
   noOrDefaultSyntax = syntax
   data = {
         'binding-sid': 'Binding segment identifier',
         'allocation': 'Binding segment identifier allocation',
         'dynamic': 'Allocate binding segment identifiers dynamically',
   }

   @staticmethod
   def handler( mode, args ):
      srConfig.dynamicallyAllocateBsid = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      srConfig.dynamicallyAllocateBsid = False

if toggleDynamicallyAssignBsidEnabled():
   SrTeMode.addCommandClass( BindingSidAllocatedDynamically )

#------------------------------------------------------
# [ no | default ] name POLICY_NAME
#------------------------------------------------------
class PolicyNameCmd( CliCommand.CliCommandClass ):
   syntax = 'name POLICY_NAME'
   noOrDefaultSyntax = 'name'
   data = {
      'name' : 'Policy Name',
      'POLICY_NAME' : policyNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      policyName = args[ 'POLICY_NAME' ]
      policyKey = mode.policy.key
      policyParams = mode.policy.policyParams if toggleDynamicSrTeEnabled()\
         else mode.policy

      if policyParams.policyName == policyName:
         return

      if policyParams.policyName != '':
         # The policy name is being modified. Remove the mapping for the
         # existing name.
         tacPolicyName = Tac.newInstance( "SrTePolicy::PolicyName",
                                          policyParams.policyName )
         del policyNameToKeyMap.policy[ tacPolicyName ]

      tacPolicyName = Tac.newInstance( "SrTePolicy::PolicyName", policyName )
      if tacPolicyName in policyNameToKeyMap.policy:
         policyKey = policyNameToKeyMap.policy[ tacPolicyName ]
         mode.addError( f"The policy name: {policyName} has already been configured "
                        f"for the policy with endpoint: "
                        f"{policyKey.endpoint.stringValue} and color: "
                        f"{str( policyKey.color )}" )
         return
      policyNameToKeyMap.policy[ tacPolicyName ] = policyKey
      if toggleDynamicSrTeEnabled():
         policyParams.policyName = tacPolicyName
      else:
         mode.policy.policyName = tacPolicyName

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      policyParams = mode.policy.policyParams if toggleDynamicSrTeEnabled() else\
         mode.policy

      if policyParams.policyName == '':
         return
      else:
         tacPolicyName = Tac.newInstance( "SrTePolicy::PolicyName",
                                          policyParams.policyName )
         del policyNameToKeyMap.policy[ tacPolicyName ]
      # There is no policyName configured by default
      if toggleDynamicSrTeEnabled():
         policyParams.policyName = Tac.newInstance( 'SrTePolicy::PolicyName', '' )
      else:
         mode.policy.policyName = Tac.newInstance( 'SrTePolicy::PolicyName', '' )

SrTePolicyMode.addCommandClass( PolicyNameCmd )

#-------------------------------------------------------
# [ no | default ] description <description>
#-------------------------------------------------------

class PolicyDescriptionCmd( CliCommand.CliCommandClass ):
   syntax = 'description DESC'
   noOrDefaultSyntax = 'description ...'
   data = {
      'description' : 'Policy Description',
      'DESC' : CliMatcher.StringMatcher( helpname='LINE',
               helpdesc='Description for the policy' )
   }

   @staticmethod
   def handler( mode, args ):
      if toggleDynamicSrTeEnabled():
         mode.policy.policyParams.policyDescription = args[ 'DESC' ]
      else:
         mode.policy.policyDescription = args[ 'DESC' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if toggleDynamicSrTeEnabled():
         mode.policy.policyParams.policyDescription = ''
      else:
         mode.policy.policyDescription = ''

SrTePolicyMode.addCommandClass( PolicyDescriptionCmd )

def getCandidatePathType( args ):
   if 'local' in args:
      # by default we start with SR label for 'local' computation
      # which can be changed later in the submode commands
      return CandidatePathType.segmentRoutingLabelResolution
   return CandidatePathType.explicitSegmentLists

# -------------------------------------------------------------------
# [ no | default ] path-group preference PREF [ computation local ]
# -------------------------------------------------------------------
class CfgPathGroupCmd( CliCommand.CliCommandClass ):
   syntax = 'path-group preference PREF [ computation local ]' if\
      toggleDynamicSrTeEnabled() else 'path-group preference PREF'
   noOrDefaultSyntax = syntax
   data = {
         'path-group': 'Candidate path group',
         'preference': 'Path group preference',
         'PREF': CliMatcher.IntegerMatcher( tacPreference.min,
                                             tacPreference.max,
                                             helpdesc='Preference value' ),
         'computation': 'Path computation type',
         'local': 'Path computation type local'
   }

   @staticmethod
   def handler( mode, args ):
      pvalue = args[ 'PREF' ]
      pref = Preference( pvalue )
      candidatePath = None
      newComputationType = 'no'
      if toggleDynamicSrTeEnabled():
         policyParams = mode.policy.policyParams
         candidatePath = policyParams.candidatePath.get( pref )
         if candidatePath is None:
            candidatePath = policyParams.candidatePath.newMember( pref )
            candidatePath.candidatePathType = getCandidatePathType(
               args )

         candidateTypeToComputationTypeDict = {
            CandidatePathType.explicitSegmentLists: 'no',
            CandidatePathType.segmentRoutingLabelResolution: 'local',
            CandidatePathType.segmentRoutingFlexAlgoLabelResolution: 'local',
         }

         newComputationType = candidateTypeToComputationTypeDict[
            getCandidatePathType( args ) ]
         oldComputationType = candidateTypeToComputationTypeDict[
            candidatePath.candidatePathType ]

         if newComputationType != oldComputationType:
            mode.addError( "path-group preference is already defined for preference"
                           f" {pref} with {oldComputationType} computation type" )
            return
      else:
         candidatePath = mode.policy.candidatePath.get( pref )
         if candidatePath is None:
            candidatePath = mode.policy.candidatePath.newMember( pref )

      computationTypeToModeDict = {
         'no': SrTePolicyPathGroupMode,
         'local': SrTePolicyPathLocalComputationMode,
      }
      childMode = mode.childMode( computationTypeToModeDict[ newComputationType ],
                                  endPoint=mode.endPoint,
                                  color=mode.color,
                                  preference=pvalue,
                                  candidatePath=candidatePath,
                                )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      pref = Preference( args[ 'PREF' ] )
      if toggleDynamicSrTeEnabled():
         del mode.policy.policyParams.candidatePath[ pref ]
      else:
         del mode.policy.candidatePath[ pref ]

SrTePolicyMode.addCommandClass( CfgPathGroupCmd )

# -------------------------------------------------------------------
# [ no | default ] path-group preference PREF computation local
# -------------------------------------------------------------------
class CfgDynamicPathGroupCmd( CliCommand.CliCommandClass ):
   syntax = 'path-group preference PREF computation local'
   noOrDefaultSyntax = syntax
   data = {
         'path-group': 'Candidate path group',
         'preference': 'Path group preference',
         'PREF': CliMatcher.IntegerMatcher( tacPreference.min,
                                             tacPreference.max,
                                             helpdesc='Preference value' ),
         'computation': 'Path computation type',
         'local': 'Path computation type local'
   }

   @staticmethod
   def handler( mode, args ):
      pvalue = args[ 'PREF' ]
      pref = Preference( pvalue )
      policyParams = mode.policy.policyParams
      candidatePath = policyParams.candidatePath.get( pref )
      if candidatePath is None:
         candidatePath = policyParams.candidatePath.newMember( pref )
         candidatePath.candidatePathType = getCandidatePathType(
            args )

      childMode = mode.childMode( SrTePolicyPathLocalComputationMode,
                                  endPoint=mode.endPoint,
                                  color=mode.color,
                                  preference=pvalue,
                                  candidatePath=candidatePath,
                                )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      pref = Preference( args[ 'PREF' ] )
      del mode.policy.policyParams.candidatePath[ pref ]
 
SrTeDynamicPolicyMode.addCommandClass( CfgDynamicPathGroupCmd )

# ----------------------------------------------------------
# [ no | default ] igp-cost preference COST metric ( METRIC
#  | dynamic [{+|-} VALUE] [igp-unreachable UNREACHABLEVALUE] )
# ----------------------------------------------------------
class CfgIgpCostMetricCmd( CliCommand.CliCommandClass ):
   syntax = '''igp-cost
         { ( preference COST )
         | ( metric METRIC
            | ( dynamic [(+|-) VALUE] [igp-unreachable IGPU_VALUE] ) ) }'''

   noOrDefaultSyntax = 'igp-cost ...'
   data = {
         'igp-cost': matcherIgpCost,
         'preference': CliCommand.Node( matcher=matcherIgpPreference, maxMatches=1 ),
         'COST': CliCommand.Node( matcherCostValue, maxMatches=1 ),
         'metric': CliCommand.Node( matcher=matcherIgpMetric, maxMatches=1 ),
         'METRIC': CliCommand.Node( matcher=matcherMetricValue, maxMatches=1 ),
         'dynamic': CliCommand.Node( matcherMetricDynamic, maxMatches=1 ),
         '+': CliCommand.Node( matcherMetricAdditive, maxMatches=1 ),
         '-': CliCommand.Node( matcherMetricSubtractive, maxMatches=1 ),
         'VALUE': CliCommand.Node( matcherMetricAddSubValue, maxMatches=1 ),
         'igp-unreachable': CliCommand.Node( matcherMetricUnreach, maxMatches=1 ),
         'IGPU_VALUE': CliCommand.Node( matcherMetricValue, maxMatches=1 ),
        }

   @staticmethod
   def handler( mode, args ):
      metric = args.get( 'METRIC', mode.policy.costMetric.defaultIgpMetric )

      costSet = 'COST' in args
      cost = args[ 'COST' ] if costSet else 0

      if toggleDynamicSrTeEnabled():
         createPolicyCostMetric( args=args, costSet=costSet, cost=cost,
               metric=metric, costMetricHandle=mode.policy.policyParams,
               perPolicyCmd=True )
      else:
         createPolicyCostMetric( args=args, costSet=costSet, cost=cost,
               metric=metric, costMetricHandle=mode.policy,
               perPolicyCmd=True )
   noOrDefaultHandler = handler

SrTePolicyMode.addCommandClass( CfgIgpCostMetricCmd )

#-------------------------------------------------------------------
# [no|default] sbfd [ proxy ] remote-discriminator DISC
#                   [ interval <INTERVAL> multiplier <MULTIPLIER> ]
#-------------------------------------------------------------------
def PolicySbfdConfig( disc, interval, multiplier, disIsU32, proxy ):
   return PolicySbfdConfigType( disc, interval, multiplier, disIsU32, proxy )

def sbfdHandler( mode, args ):
   multDef = Tac.Type( 'Bfd::BfdMultiplier' ).defval
   interval = args.get( 'INT', 0 )
   multiplier = args.get( 'MULT', multDef )
   disc, isU32 = args[ 'DISC' ]
   noOrDefault = CliCommand.isNoOrDefaultCmd( args )
   proxy = False if noOrDefault else 'proxy' in args
   if toggleDynamicSrTeEnabled():
      mode.policy.policyParams.sbfdConfig = PolicySbfdConfig( disc, interval,
                                                              multiplier,
                                                              isU32, proxy )
   else:
      mode.policy.sbfdConfig = PolicySbfdConfig( disc, interval, multiplier,
                                                 isU32, proxy )

class CfgSbfdCmd( CliCommand.CliCommandClass ):
   syntax = ( 'sbfd remote-discriminator DISC '
              '[ interval INT multiplier MULT ] [ proxy ]' )
   noOrDefaultSyntax = 'sbfd remote-discriminator ...'
   data = {
         'sbfd': 'Seamless BFD',
         'remote-discriminator': 'The remote discriminator',
         'DISC': SbfdReflectorDiscExpr,
         'interval': matcherInterval,
         'INT': matcherTxRxIntervalMs,
         'multiplier': 'Sets the SBFD multiplier',
         'MULT': matcherMultiplierVal,
         'proxy': 'Proxy',
         }

   adapter = reflectorAdapter

   handler = sbfdHandler
   noOrDefaultHandler = handler

SrTePolicyMode.addCommandClass( CfgSbfdCmd )

class SrTePolicyPathGroupMode( SrTePolicyPathGroupModeBase,
                               BasicCli.ConfigModeBase ):
   name = "Segment Routing Policy Path Group Configuration"

   def __init__( self, parent, session, endPoint, color, preference, candidatePath ):
      SrTePolicyPathGroupModeBase.__init__( self, ( endPoint, color, preference ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.candidatePath = candidatePath

#-------------------------------------------------------------------
# [ no | default ] explicit-null ( ipv4 | ipv6 | ipv4 ipv6 | none )
#-------------------------------------------------------------------
class CfgEnlpCmd( EnlpBase ):
   data = EnlpBase.data.copy()
   @staticmethod
   def handler( mode, args ):
      EnlpBase.enlpHandler( mode, args, mode.candidatePath )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.candidatePath.enlp = tacEnlpEnum.localPolicy

SrTePolicyPathGroupMode.addCommandClass( CfgEnlpCmd )

def maxLabelStackSize():
   """
   From the CLI allow as many labels as Arnet::MplsLabelOperation can hold.
   We don't check platform limits in the CLI because what matters is the resolved
   segment list as that is what gets programmed in hardware. The label stack
   can expand or shrink based on the resolution.
   Arnet::MplsLabelOperation can hold one more than the deepest label stack we can
   impose on any of our platforms as it's possible for the label stack to shrink if
   the top label was popped as result of resolution.
   """
   MplsStackEntryIndex = Tac.Type( 'Arnet::MplsStackEntryIndex' )
   return MplsStackEntryIndex.max + 1

#------------------------------------------------------------------------------------
# [ no | default ] segment-list label-stack STACK [ weight WEIGHT ] [ index INDEX ]
#------------------------------------------------------------------------------------
class CfgSegmentListCmd( CliCommand.CliCommandClass ):
   syntax = '''segment-list label-stack { STACK }
               [ ( weight WEIGHT [ index INDEX   ] )
               | ( index INDEX   [ weight WEIGHT ] ) ]'''
   noOrDefaultSyntax = syntax
   data = {
         'segment-list': 'Segment list for path group',
         'label-stack': 'Segment list as MPLS Label Stack',
         'weight': 'Weight',
         'index': 'Index',
         'WEIGHT': CliMatcher.IntegerMatcher( tacWeight.min, tacWeight.max,
                                              helpdesc='Weight value' ),
         'INDEX': CliMatcher.IntegerMatcher( tacIndex.min + 1, tacIndex.max,
                                             helpdesc='Index value' ),
         'STACK': CliCommand.Node(
                     matcher=CliMatcher.IntegerMatcher( MplsLabel.min,
                                                        MplsLabel.max,
                                                        helpdesc='MPLS label' ),
                     maxMatches=maxLabelStackSize() ),
         }

   @staticmethod
   def _segListStrToLabelStack( segments ):
      newSegList = MplsLabelOperation()
      index = 0
      for index, segment in enumerate( segments ):
         newSegList.labelStackIs( index, MplsLabel( int( segment ) ) )
      newSegList.stackSize = index + 1
      return newSegList

   @staticmethod
   def handler( mode, args ):
      weight = args.get( 'WEIGHT' )
      index = args.get( 'INDEX' )
      segList = CfgSegmentListCmd._segListStrToLabelStack( args[ 'STACK' ] )
      for sl, wIdx in mode.candidatePath.staticSegmentList.items():
         if sl != segList:
            if index == wIdx.index:
               mode.addError( f'Index {index} is already assigned to another '
                              'segment-list' )
               return
         elif wIdx.index and index != wIdx.index:
            mode.addError( f'Index mismatch, expected index {wIdx.index}' )
            return
      slParam = tacStaticSegList( Weight( weight ), Index( index ) )
      if segList in mode.candidatePath.staticSegmentList:
         slParam.sbfdReturnPath = \
               mode.candidatePath.staticSegmentList[ segList ].sbfdReturnPath
      mode.candidatePath.staticSegmentList[ segList ] = slParam
      childMode = mode.childMode( SrTePolicySegmentListMode,
                                  endPoint=mode.endPoint, color=mode.color,
                                  preference=mode.preference, segList=segList,
                                  weight=weight, index=index,
                                  candidatePath=mode.candidatePath )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      segList = CfgSegmentListCmd._segListStrToLabelStack( args[ 'STACK' ] )
      del mode.candidatePath.staticSegmentList[ segList ]

SrTePolicyPathGroupMode.addCommandClass( CfgSegmentListCmd )

class SrTePolicyPathLocalComputationMode( SrTePolicyPathLocalComputationModeBase,
                                     BasicCli.ConfigModeBase ):
   name = "Segment Routing Policy Path Group Computation Configuration"

   def __init__( self, parent, session, color, preference, candidatePath,
                 endPoint=None ):
      SrTePolicyPathLocalComputationModeBase.__init__( self, ( endPoint, color,
                                                               preference ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.candidatePath = candidatePath

# -----------------------------------------------------------------------------------
# [ no | default ] path segment-routing [flex-algo NAME]
# -----------------------------------------------------------------------------------
class CfgCandidatePathComputationCmd( CliCommand.CliCommandClass ):
   syntax = '''path segment-routing [flex-algo NAME]'''
   noOrDefaultSyntax = syntax
   data = {
      'path': 'Path specification',
      'segment-routing': 'Segment routing label of the endpoint',
      'flex-algo': 'Flex algorithm label of the endpoint',
      'NAME': CliMatcher.DynamicNameMatcher( lambda mode: flexAlgoConfig.nameToIdMap.
                                                keys(),
                                             helpdesc='Algorithm name' ),
   }

   @staticmethod
   def handler( mode, args ):
      if 'NAME' in args:
         flexAlgoName = args[ 'NAME' ]
         mode.candidatePath.candidatePathType = CandidatePathType.\
            segmentRoutingFlexAlgoLabelResolution
         mode.candidatePath.flexAlgoName = flexAlgoName
      else:
         mode.candidatePath.candidatePathType = CandidatePathType.\
            segmentRoutingLabelResolution
         mode.candidatePath.flexAlgoName = ''

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.candidatePath.candidatePathType = CandidatePathType.\
         segmentRoutingLabelResolution
      mode.candidatePath.flexAlgoName = ""

SrTePolicyPathLocalComputationMode.addCommandClass( CfgCandidatePathComputationCmd )

class SrTePolicySegmentListMode( SrTePolicySegmentListModeBase,
                                 BasicCli.ConfigModeBase ):
   name = "Segment Routing Policy Path Group Segment List Configuration"

   def __init__( self, parent, session, endPoint, color, preference, segList,
                 weight, index, candidatePath ):
      SrTePolicySegmentListModeBase.__init__( self, ( endPoint, color, preference,
                                                      segList, weight, index ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.candidatePath = candidatePath

#-----------------------------------------------------------------------------------
# [ no | default ] sbfd return-path label-stack STACK
#-----------------------------------------------------------------------------------
class CfgSbfdReturnPathLabelStackCmd( CliCommand.CliCommandClass ):
   syntax = 'sbfd return-path label-stack { STACK }'
   noOrDefaultSyntax = 'sbfd return-path label-stack ...'
   data = {
      'sbfd': 'Seamless BFD',
      'return-path': 'return path',
      'label-stack': 'MPLS Label Stack',
      'STACK': CliCommand.Node(
         matcher=CliMatcher.IntegerMatcher( MplsLabel.min,
                                            MplsLabel.max,
                                            helpdesc='MPLS label' ),
         maxMatches=maxLabelStackSize() ),
   }

   @staticmethod
   def _labelStackStrToLabelStack( labels ):
      newLabelStack = MplsLabelOperation()
      index = 0
      for index, label in enumerate( labels ):
         newLabelStack.labelStackIs( index, MplsLabel( int( label ) ) )
      newLabelStack.stackSize = index + 1
      return newLabelStack

   @staticmethod
   def handler( mode, args ):
      sbfdReturnPath = CfgSbfdReturnPathLabelStackCmd._labelStackStrToLabelStack(
         args[ 'STACK' ] )
      slParam = tacStaticSegList( Weight( mode.weight ), Index( mode.index ) )
      slParam.sbfdReturnPath = sbfdReturnPath
      mode.candidatePath.staticSegmentList[ mode.segmentList ] = slParam

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      slParam = tacStaticSegList( Weight( mode.weight ), Index( mode.index ) )
      mode.candidatePath.staticSegmentList[ mode.segmentList ] = slParam

SrTePolicySegmentListMode.addCommandClass( CfgSbfdReturnPathLabelStackCmd )

def isSlValidAndGetTunEntry( slId, sl ):
   '''
   Return a tuple ( True, tunnelEntry ) if the @sl with id @slId is valid; otherwise
   return ( False, None )
   '''
   tunnelEntry = None
   slResolved = None
   valid = False

   if sl is None or sl.size == 0:
      return ( valid, tunnelEntry )
   else:
      # The segment list is automatically invalid if it has no segments
      slResolved = slStatus.status.get( slId )

   if slResolved is not None and slResolved.resolveError == 'success':
      # Be permissive of the tunnel entry not being present yet (race condition).
      # We'll treat the SL as invalid if we can't find the tunnel yet.
      tunnelId = tacSegmentListId( slId ).getTunnelId()
      tunnelEntry = slTunnelTable.entry.get( tunnelId, None )

   valid = tunnelEntry is not None
   return ( valid, tunnelEntry )

srTeTunnelCountersHook = CliExtensions.CliHook()

def getCountersForTunnelEntry( tunnelEntry ):
   '''
   The function returns three parameters: packet, byte counters, and counterInUse.
   The "counterInUse" parameter indicates that a counter is being used for the given
   tunnel entry.
   Note: srTeTunnelCountersHook return None in case counter feature is not enabled.
   '''
   result = ( None, None, None )
   # Try to use CLI hook if available
   for hook in srTeTunnelCountersHook.extensions():
      result = hook( tunnelEntry.tunnelId )
      break

   return result

def getCountersForPolicy( policyKey ):
   '''Given a policy key, return packet & byte counters if available'''
   result = ( None, None )
   policyCounter = policyCounterTable.policyCounter.get( policyKey )
   policySnapshot = policyCounterTable.policySnapshot.get( policyKey )
   if policyCounter and policySnapshot:
      pkts = policyCounter.pkts - policySnapshot.pkts
      octets = policyCounter.octets - policySnapshot.octets
      result = ( pkts, octets )
   elif policyCounter:
      result = ( policyCounter.pkts, policyCounter.octets )
   return result

# ----------------------------------------------------------------
# show traffic-engineering segment-routing policy active ...
# ----------------------------------------------------------------
def getSourceFilter( args ):
   if 'static' in args:
      return [ tacProtocol.staticProtocol ]
   if 'bgp' in args:
      return [ tacProtocol.bgpProtocol ]
   else:
      return [ tacProtocol.staticProtocol, tacProtocol.bgpProtocol ]

class ShowSrTePolicyActiveCmd( ShowCommand.ShowCliCommandClass ):
   if toggleDynamicSrTeEnabled():
      syntax = '''show traffic-engineering segment-routing policy
                  active [ source ( bgp | static ) ] [ endpoint ( ADDR | dynamic )
                  color COLOR ]'''
   else:
      syntax = '''show traffic-engineering segment-routing policy
                  active [ source ( bgp | static ) ] [ endpoint ADDR color COLOR ]'''
   data = {
      'traffic-engineering': matcherTrafficEngineering,
      'segment-routing': matcherSegmentRoutingForShow,
      'policy': matcherPolicy,
      'active': 'Show active Segment Routing Traffic Engineering policies',
      'source': matcherSource,
      'bgp': matcherBgp,
      'static': matcherStatic,
      'endpoint': matcherEndpoint,
      'ADDR': IpGenAddrMatcher( 'IPv4 or IPv6 address' ),
      'dynamic': matcherDynamic,
      'color': matcherColor,
      'COLOR': matcherColorValue,
   }
   cliModel = SrTePolicyActiveVrfModel

   @staticmethod
   def handler( mode, args ):
      ep = args.get( 'ADDR' )
      color = args.get( 'COLOR' )
      sourceFilter = getSourceFilter( args )
      vrfModel = SrTePolicyActiveVrfModel()
      active = SrTePolicyActiveModel()
      active.enlpGlobal = srConfig.enlp
      dynamicEpFilter = args.get( 'dynamic' )
      if dynamicEpFilter and tacProtocol.bgpProtocol in sourceFilter:
         # dynamic endpoint filter only applies to static source policies
         sourceFilter.remove( tacProtocol.bgpProtocol )
      active.policies = populatePolicies( ep, color, [ 'active' ], sourceFilter,
                                          dynamicEpFilter )
      vrfModel.vrfs[ DEFAULT_VRF ] = active
      return vrfModel

BasicCli.addShowCommandClass( ShowSrTePolicyActiveCmd )

# -----------------------------------------------------------------------
# show traffic-engineering segment-routing policy [ summary ]
# -----------------------------------------------------------------------
def populatePolicyStatistics():
   sourceDict = { tacProtocol.staticProtocol : 0,
                  tacProtocol.bgpProtocol : 0 }

   # Terminology:
   # We are obtaining counters for:
   # Each ( endpoint, color ) -> Per endpoint, color statistics
   # each ( endpoint, color, source ) -> Per endpoint, color, source statistics
   # and each candidate path -> Candidate Path Statistics
   # Note that for 'Per endpoint, color statistics' we don't care about the source
   # etc. we are only counting how many (E, C) are in a given state. Similarly,
   # for 'Per Source' Statistics, we are only counting per state counters of
   # policies within a given source and don't care about preferences etc.

   # PathGroup counters.
   activePathGroupCnt = { tacProtocol.staticProtocol : 0,
                          tacProtocol.bgpProtocol : 0 }
   validPathGroupCnt = { tacProtocol.staticProtocol : 0,
                            tacProtocol.bgpProtocol : 0 }
   invalidPathGroupCnt = { tacProtocol.staticProtocol : 0,
                           tacProtocol.bgpProtocol : 0 }

   # Static Source policy sub group counters per endpoint provisioning and
   # path computation types
   def getInitialPathGroupStaticSrcSubCnt():
      return { "static": { "configured": 0, "isisSr": 0, },
               "dynamic": { "isisSr": 0, }
             }
   activePathGroupStaticSrcSubCnt = getInitialPathGroupStaticSrcSubCnt()
   validPathGroupStaticSrcSubCnt = getInitialPathGroupStaticSrcSubCnt()
   invalidPathGroupStaticSrcSubCnt = getInitialPathGroupStaticSrcSubCnt()

   # Per-Source Policy counters
   activePolicyCnt = { tacProtocol.staticProtocol : 0,
                       tacProtocol.bgpProtocol : 0 }
   validPolicyCnt = { tacProtocol.staticProtocol : 0,
                         tacProtocol.bgpProtocol : 0 }
   invalidPolicyCnt = { tacProtocol.staticProtocol : 0,
                        tacProtocol.bgpProtocol : 0 }

   # Policy ( E,C ) counters
   totalValidPolicies = 0
   totalInValidPolicies = 0
   totalActivePolicies = 0

   # pylint: disable=too-many-nested-blocks
   # remove pylint warning when cleaning up the toggle
   if not toggleOcInvalidSrTeEnabled():
      # Walk through each policy (endpoint, color) in policyKeyToIdMap
      for key, policy in policyKeyToIdMap.policy.items():
         # While we iterate through the policies, for each endpoint, color (E,C)
         # Mark the ones seen as valid or invalid using the following dicts
         sourceInvalid = { }
         sourceValid = { tacProtocol.staticProtocol: False,
                         tacProtocol.bgpProtocol: False }
         active = policyStatus.status.get( key )
         validCandidates = ecPolicy.policy.get( key )
         # if policy is invalid and toggleOcInvalidSrTe is disabled then
         # validCandidates will be None otherwise if toggleOcInvalidSrTe is enabled
         # then validCandidates will be present and will have isValidPolicy flag set
         # to False or True based on policy state
         if validCandidates is None or not validCandidates.isValidPolicy:
            totalInValidPolicies += 1
         elif not active:
            totalValidPolicies += 1

         # For each Endpoint, Color, now walk through the composite candidate paths.
         for candidateId in policy.candidate:
            # For any protocol present in list of candidates for this policy,
            # we start with assuming it is an invalid policy unless proven wrong
            # by finding a match in validCandidates
            if not sourceInvalid.get( candidateId.protocol ):
               # We start with an assumption that this candidate path is invalid
               # and wait for it to be proven False later on.
               sourceInvalid.update( { candidateId.protocol: True } )

            pie = allPolicy.policyIdEntry.get( tacPolicyId( key, candidateId ) )
            if validCandidates and pie:
               priority = tacPriority( True, pie.preference,
                                       pie.id.candidateId.protocol,
                                       pie.originator,
                                       pie.id.candidateId.discriminator )
               if priority in validCandidates.candidate:
                  # This is a valid candidate path for this policy
                  sourceInvalid[ pie.id.candidateId.protocol ] = False
                  if not active or \
                     active.candidateId.protocol != pie.id.candidateId.protocol:
                     # Either this policy does not have an active candidate path Or,
                     # the active candidate path is from a different source compared
                     # to the source of the candidate path we have at hand now.
                     sourceValid[ pie.id.candidateId.protocol ] = True
                  if active and active.candidateId == pie.id.candidateId:
                     # This candidate path is the active one.
                     activePathGroupCnt[ active.candidateId.protocol ] += 1
                     activePolicyCnt[ active.candidateId.protocol ] += 1
                     totalActivePolicies += 1
                  else:
                     validPathGroupCnt[ pie.id.candidateId.protocol ] += 1
                  continue

            # This is an invalid candidate path
            invalidPathGroupCnt[ candidateId.protocol ] += 1

         for source, val in sourceInvalid.items():
            if val:
               invalidPolicyCnt[ source ] += 1
         for source, val in sourceValid.items():
            if val:
               validPolicyCnt[ source ] += 1
   else:
      # Walk through each policy (endpoint, color) in ecPolicy
      for key, policy in ecPolicy.policy.items():
         # While we iterate through the policies, for each endpoint, color (E,C)
         # Mark the ones seen as valid or invalid using the following dicts
         sourceInvalid = { }
         sourceValid = { tacProtocol.staticProtocol: False,
                         tacProtocol.bgpProtocol: False }
         active = policyStatus.status.get( key )
         staticEpPolicy = None
         dynamicEpPolicy = None
         if toggleDynamicSrTeEnabled():
            staticEpPolicy = srConfig.policy.get( key )
            dynamicEpPolicy = srConfig.dynamicEpPolicy.get( key.color )
         if not policy.isValidPolicy:
            totalInValidPolicies += 1
         elif not active:
            # then policy is valid, but not active
            totalValidPolicies += 1

         # For each Endpoint, Color, now walk through the composite candidate paths.
         for priority in policy.candidate:
            # For any protocol present in list of candidates for this policy,
            # we start with assuming it is an invalid policy unless proven wrong
            # by checking the isValidCandidate flag for the candidate
            if not sourceInvalid.get( priority.protocol ):
               # We start with an assumption that this candidate path is invalid
               # and wait for it to be proven False later on.
               sourceInvalid.update( { priority.protocol: True } )

            endpointProvisioning = None
            pathComputation = None
            if toggleDynamicSrTeEnabled() and\
               priority.protocol == tacProtocol.staticProtocol:
               if priority.discriminator >> discriminatorMSB:
                  endpointProvisioning = "static"
                  policyParams = staticEpPolicy.policyParams if staticEpPolicy\
                     is not None else None
               else:
                  endpointProvisioning = "dynamic"
                  policyParams = dynamicEpPolicy.policyParams if dynamicEpPolicy\
                     is not None else None
               if policyParams is not None:
                  candidatePath = policyParams.candidatePath.get(
                     priority.preference )
                  if candidatePath is not None:
                     pathComputation =\
                        candidatePathTypeToPathComputationDict[
                           candidatePath.candidatePathType ]
                     if candidatePath.candidatePathType ==\
                        CandidatePathType.segmentRoutingFlexAlgoLabelResolution:
                        pathComputation += "-" + candidatePath.flexAlgoName

            if priority.isValidCandidate:
               # This is a valid candidate path for this policy
               sourceInvalid[ priority.protocol ] = False
               if not active or \
                  active.candidateId.protocol != priority.protocol:
                  # Either this policy does not have an active candidate path Or,
                  # the active candidate path is from a different source compared to
                  # the source of the candidate path we have at hand now.
                  sourceValid[ priority.protocol ] = True
               candidateId = tacCandidateId( priority.protocol,
                                             priority.discriminator )
               if active and active.candidateId == candidateId:
                  # This candidate path is the active one.
                  if toggleDynamicSrTeEnabled():
                     if active.candidateId.protocol == tacProtocol.staticProtocol\
                        and pathComputation is not None:
                        endpointProvisioningCnt =\
                           activePathGroupStaticSrcSubCnt[ endpointProvisioning ]
                        if pathComputation not in endpointProvisioningCnt:
                           endpointProvisioningCnt[ pathComputation ] = 0
                        endpointProvisioningCnt[ pathComputation ] += 1
                     if active.candidateId.protocol == tacProtocol.bgpProtocol or\
                        ( active.candidateId.protocol is tacProtocol.staticProtocol
                          and pathComputation is not None ):
                        # for static source do not increment count if the path group
                        # is not present in config
                        activePathGroupCnt[ active.candidateId.protocol ] += 1
                        activePolicyCnt[ active.candidateId.protocol ] += 1
                        totalActivePolicies += 1
                  else:
                     activePathGroupCnt[ active.candidateId.protocol ] += 1
                     activePolicyCnt[ active.candidateId.protocol ] += 1
                     totalActivePolicies += 1
               else:
                  if toggleDynamicSrTeEnabled():
                     if priority.protocol == tacProtocol.staticProtocol\
                        and pathComputation is not None:
                        endpointProvisioningCnt =\
                           validPathGroupStaticSrcSubCnt[ endpointProvisioning ]
                        if pathComputation not in endpointProvisioningCnt:
                           endpointProvisioningCnt[ pathComputation ] = 0
                        endpointProvisioningCnt[ pathComputation ] += 1
                     if priority.protocol == tacProtocol.bgpProtocol or\
                        ( priority.protocol is tacProtocol.staticProtocol and
                          pathComputation is not None ):
                        # for static source do not increment count if the path group
                        # is not present in config
                        validPathGroupCnt[ priority.protocol ] += 1
                  else:
                     validPathGroupCnt[ priority.protocol ] += 1
               continue

            # This is an invalid candidate path
            if toggleDynamicSrTeEnabled():
               if priority.protocol == tacProtocol.staticProtocol and\
                  pathComputation is not None:
                  endpointProvisioningCnt =\
                     invalidPathGroupStaticSrcSubCnt[ endpointProvisioning ]
                  if pathComputation not in endpointProvisioningCnt:
                     endpointProvisioningCnt[ pathComputation ] = 0
                  endpointProvisioningCnt[ pathComputation ] += 1
               if priority.protocol == tacProtocol.bgpProtocol or\
                  ( priority.protocol is tacProtocol.staticProtocol and
                    pathComputation is not None ):
                  # for static source do not increment count if the path group
                  # is not present in config
                  invalidPathGroupCnt[ priority.protocol ] += 1
            else:
               invalidPathGroupCnt[ priority.protocol ] += 1

         for source, val in sourceInvalid.items():
            if val:
               invalidPolicyCnt[ source ] += 1
         for source, val in sourceValid.items():
            if val:
               validPolicyCnt[ source ] += 1

   pathGroupNumbers = SrTePolicyPathGroupStatisticsModel()
   pgTotalEntry = SrTePolicyPathGroupStatisticsEntryModel()
   policyNumbers = SrTePolicyStatisticsModel()
   globalPolicyEntry = SrTePolicyStatisticsEntryModel()
   totalPolicyEntry = SrTePolicyStatisticsEntryModel()
   for source in sourceDict:
      pgEntry = SrTePolicyPathGroupStatisticsEntryModel()
      policyEntry = SrTePolicyStatisticsEntryModel()

      # ACTIVE
      pgEntry.activePathGroups = activePathGroupCnt[ source ]
      pgTotalEntry.activePathGroups += pgEntry.activePathGroups
      policyEntry.activePolicies = activePolicyCnt[ source ]

      # VALID
      pgEntry.validPathGroups = validPathGroupCnt[ source ]
      pgTotalEntry.validPathGroups += pgEntry.validPathGroups
      policyEntry.validPolicies = validPolicyCnt[ source ]

      # INVALID
      pgEntry.invalidPathGroups = invalidPathGroupCnt[ source ]
      pgTotalEntry.invalidPathGroups += pgEntry.invalidPathGroups
      policyEntry.invalidPolicies = invalidPolicyCnt[ source ]

      # Sub Counters for Static Src
      if toggleDynamicSrTeEnabled():
         if source == tacProtocol.staticProtocol:
            for endpointProvisioning in [ "static", "dynamic" ]:
               pgEpProvModel = SrTePolicyPathGroupEpProvisioningStatisticsModel()
               # pylint: disable=unsupported-assignment-operation
               pgEntry.endpointProvisioning[ endpointProvisioning ] = \
                  pgEpProvModel
               activePathGroupPathCompCnt =\
                  activePathGroupStaticSrcSubCnt[ endpointProvisioning ]
               validPathGroupPathCompCnt =\
                  validPathGroupStaticSrcSubCnt[ endpointProvisioning ]
               invalidPathGroupPathCompCnt =\
                  invalidPathGroupStaticSrcSubCnt[ endpointProvisioning ]
               pathComputationTypes = set(
                  activePathGroupPathCompCnt.keys() |
                  validPathGroupPathCompCnt.keys() |
                  invalidPathGroupPathCompCnt.keys() )
               for pathComputation in pathComputationTypes:
                  pgPathCompModel =\
                     SrTePolicyPathGroupComputationStatisticsEntryModel()
                  pgEpProvModel.pathComputation[ pathComputation ] = pgPathCompModel

                  pgPathCompModel.activePathGroups =\
                     activePathGroupPathCompCnt.get( pathComputation, 0 )
                  pgPathCompModel.validPathGroups =\
                     validPathGroupPathCompCnt.get( pathComputation, 0 )
                  pgPathCompModel.invalidPathGroups =\
                     invalidPathGroupPathCompCnt.get( pathComputation, 0 )
                  pgPathCompModel.totalPathGroups =\
                     pgPathCompModel.activePathGroups + \
                     pgPathCompModel.validPathGroups + \
                     pgPathCompModel.invalidPathGroups
         else:
            # endpoint provisioning does not apply for policies from BGP src
            pgEntry.endpointProvisioning = None
         # endpoint provisioning is not filled for Total entry
         pgTotalEntry.endpointProvisioning = None

      # TOTAL per Source
      pgEntry.totalPathGroups = pgEntry.activePathGroups + \
                                pgEntry.validPathGroups + \
                                pgEntry.invalidPathGroups
      policyEntry.totalPolicies = policyEntry.activePolicies + \
                                  policyEntry.validPolicies + \
                                  policyEntry.invalidPolicies
      totalPolicyEntry.activePolicies += policyEntry.activePolicies
      totalPolicyEntry.validPolicies += policyEntry.validPolicies
      totalPolicyEntry.invalidPolicies += policyEntry.invalidPolicies
      totalPolicyEntry.totalPolicies += policyEntry.totalPolicies

      pgTotalEntry.totalPathGroups += pgEntry.totalPathGroups

      pathGroupNumbers.sources[ policySrcToString( source ) ] = pgEntry
      policyNumbers.sources[ policySrcToString( source ) ] = policyEntry

   globalPolicyEntry.activePolicies = totalActivePolicies
   globalPolicyEntry.validPolicies = totalValidPolicies
   globalPolicyEntry.invalidPolicies = totalInValidPolicies
   globalPolicyEntry.totalPolicies = totalActivePolicies + \
                                    totalValidPolicies + totalInValidPolicies
   policyNumbers.total = totalPolicyEntry
   pathGroupNumbers.total = pgTotalEntry

   return ( pathGroupNumbers, policyNumbers, globalPolicyEntry )

def getSbfdPeerStatus( slId, sbfdConfig ):
   peerStatus = None
   tunnelId = tacSegmentListId( slId ).getTunnelId()
   if sbfdConfig and sbfdConfig.sbfdRemoteDiscriminator:
      remoteDisc = sbfdConfig.sbfdRemoteDiscriminator
      peerAddr = IpGenAddr( IpAddress( remoteDisc ).stringValue )
      sbfdPeer = Tac.Value( 'Bfd::Peer', peerAddr, DEFAULT_VRF )
      sbfdPeer.tunnelId = tunnelId
      sbfdPeer.type = ConfigSessionType.sbfdInitiator
      peerStatus = bfdStatusPeer.peerStatus.get( sbfdPeer )
   return peerStatus

def populateSegmentListStatistics():
   slStats = SrTePolicySegmentListStatisticsModel()
   mayValidSegmentLists = len( slStatus.status )
   for slId in slStatus.status:
      key = None
      allSlEntry = allSl.segmentList.get( slId )
      if allSlEntry:
         key = allSlEntry.policyId.policyKey
      policy = srConfig.policy.get( key ) if key else None
      sbfdConfig = None
      if policy:
         sbfdConfig = policy.policyParams.sbfdConfig if toggleDynamicSrTeEnabled()\
            else policy.sbfdConfig
      if ( sbfdConfig is None or sbfdConfig.sbfdRemoteDiscriminator == 0 ) and \
         srConfig.globalSBFDEcho:
         sbfdConfig = PolicySbfdConfigType()
         sbfdConfig.sbfdRemoteDiscriminator = 0xFFFFFFFF
      peerStatus = getSbfdPeerStatus( slId, sbfdConfig )
      if ( peerStatus and
           peerStatus.clientStatus.status == BfdOperState.down ):
         mayValidSegmentLists = mayValidSegmentLists - 1
   slStats.validSegmentLists = mayValidSegmentLists
   slStats.invalidSegmentLists = len( allSl.segmentList ) - slStats.validSegmentLists
   slIds = list( allSl.segmentList )
   slStats.validSegmentListsWithBackup = 0
   for slId in slIds:
      sl = allSl.segmentList.get( slId )
      if sl is None:
         # The requested SL doesn't exist, so skip it.
         continue
      valid, tunnelEntry = isSlValidAndGetTunEntry( slId, sl )
      backupVias = []
      if valid:
         _, backupVias = getSegmentListVias( tunnelEntry )
      if backupVias:
         slStats.validSegmentListsWithBackup += 1
   return slStats

def countBindingSidConflictInvalidPathGroups():
   totalConflictedPathGroups = 0
   for allCandidates in policyKeyToIdMap.policy.values():
      for cInfo in allCandidates.candidate.values():
         bsidReason = cInfo.reason
         if bsidReason == 'policyBsidInConflictWithOtherApp':
            totalConflictedPathGroups += 1
   return totalConflictedPathGroups

def countBindingSidConflictInactivePolicies():
   totalInactivePolicies = 0
   for key, policy in ecPolicy.policy.items():
      active = policyStatus.status.get( key )
      # if policy is valid but not active then that means that there is BSID conflict
      # with some other policy
      if not active and policy.isValidPolicy:
         totalInactivePolicies += 1
   return totalInactivePolicies

def getDynamicBsidRanges():
   rangeList = []
   if srConfig.dynamicallyAllocateBsid and not srConfig.specifiedBsidOnly:
      for name, lba in labelManagerStatus.labelBlockAllocation.items():
         if name != 'bsid':
            continue
         for blockId, rangeVal in sorted( lba.block.items() ):
            rangeModel = SrTePolicyDynBsidLabelRangeModel()
            rangeModel.base = rangeVal.blockStart
            rangeModel.size = rangeVal.blockSize
            rangeModel.inUse = dynamicBsidStats.inUse.get( blockId )
            rangeList.append( rangeModel )
   return rangeList if rangeList else None

class ShowSrTePolicySummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show traffic-engineering segment-routing policy summary'
   data = {
      'traffic-engineering': matcherTrafficEngineering,
      'segment-routing': matcherSegmentRoutingForShow,
      'policy': matcherPolicy,
      'summary': 'Show summary of Segment Routing Traffic Engineering policies',
   }
   cliModel = SrTePolicySummaryVrfModel

   @staticmethod
   def handler( mode, args ):
      if not isSrTePolicyAgentRunning( mode ):
         mode.addError( 'SR TE Policy Agent is not running' )
         return None

      if srConfig.segmentListSharing == SegmentListSharing.withAll:
         mode.addError( 'The statistics for segment lists is counted based on '
                        'config SegmentListSharing.withinPolicy while current '
                        'config is SegmentListSharing.withAll' )

      vrfModel = SrTePolicySummaryVrfModel()
      summary = SrTePolicySummaryModel()
      pathGroupStats, policyStats, globalPolicyStats = populatePolicyStatistics()
      summary.perEndpointColorStatistics = globalPolicyStats
      summary.policyStatistics = policyStats
      summary.pathgroupStatistics = pathGroupStats
      summary.segmentListStatistics = populateSegmentListStatistics()
      summary.bindingSidConflictInvalidPathGroupsCount = \
                                    countBindingSidConflictInvalidPathGroups()
      summary.bindingSidConflictInactivePoliciesCount = \
                                    countBindingSidConflictInactivePolicies()
      summary.fecOptimizationsSupported = routingHwStatus.mplsExternalFecSupported
      summary.maximumSidDepth = routingHwStatus.nexthopGroupMplsLabelStackSize
      summary.reserveServiceLabel = mplsConfig.reserveServiceLabel
      if toggleDynamicallyAssignBsidEnabled():
         summary.dynamicallyAssignBsid = srConfig.dynamicallyAllocateBsid
         summary.dynamicBsidLabelRanges = getDynamicBsidRanges()
      if toggleOptionalBsidEnabled():
         summary.specifiedBsidOnly = srConfig.specifiedBsidOnly
      vrfModel.vrfs[ DEFAULT_VRF ] = summary
      return vrfModel

BasicCli.addShowCommandClass( ShowSrTePolicySummaryCmd )

# -----------------------------------------------------------------------
# show traffic-engineering segment-routing policy [ valid | invalid ] ...
# -----------------------------------------------------------------------

policyNamesCompletion = [ CliParser.Completion( 'WORD', 'PolicyName',
   literal=False ) ]

def policyNames( mode ):
   return { PolicyName: "" for PolicyName in policyNameToKeyMap.policy }

class ShowSrTePolicyCmd( ShowCommand.ShowCliCommandClass ):
   if toggleDynamicSrTeEnabled():
      syntax = '''show traffic-engineering segment-routing policy
               [ valid | invalid ]
               [ source ( bgp | static ) ]
               [ endpoint ( ADDR | dynamic ) color COLOR ]
               [ name NAME ]
               '''
   else:
      syntax = '''show traffic-engineering segment-routing policy
               [ valid | invalid ]
               [ source ( bgp | static ) ]
               [ endpoint ADDR color COLOR ]
               [ name NAME ]
               '''
   data = {
      'traffic-engineering': matcherTrafficEngineering,
      'segment-routing': matcherSegmentRoutingForShow,
      'policy': matcherPolicy,
      'valid': 'Show valid Segment Routing Traffic Engineering policies',
      'invalid': 'Show invalid Segment Routing Traffic Engineering policies',
      'source': matcherSource,
      'bgp': matcherBgp,
      'static': matcherStatic,
      'name': CliCommand.Node(
                  matcher=CliMatcher.KeywordMatcher(
                     'name',
                     helpdesc='Show policy for a given name' ) ),
      'NAME': CliMatcher.DynamicKeywordMatcher( policyNames,
         emptyTokenCompletion=policyNamesCompletion ),
      'endpoint': matcherEndpoint,
      'ADDR': IpGenAddrMatcher( 'IPv4 or IPv6 address' ),
      'dynamic': matcherDynamic,
      'color': matcherColor,
      'COLOR': matcherColorValue,
   }
   cliModel = SrTePolicyCandidatesVrfModel

   @staticmethod
   def handler( mode, args ):
      name = args.get( 'NAME' )
      ep = args.get( 'ADDR' )
      color = args.get( 'COLOR' )
      if 'valid' in args:
         stateFilter = [ 'valid' ]
      elif 'invalid' in args:
         stateFilter = [ 'invalid' ]
      else:
         stateFilter = [ 'active', 'valid', 'invalid' ]
      sourceFilter = getSourceFilter( args )
      vrfModel = SrTePolicyCandidatesVrfModel()
      candidates = SrTePolicyCandidatesModel()
      candidates.enlpGlobal = srConfig.enlp
      dynamicEpFilter = args.get( 'dynamic' )
      if dynamicEpFilter and tacProtocol.bgpProtocol in sourceFilter:
         # dynamic endpoint filter only applies to static source policies
         sourceFilter.remove( tacProtocol.bgpProtocol )
      candidates.policies = populatePolicies(
            ep, color, stateFilter, sourceFilter, dynamicEpFilter, name )
      vrfModel.vrfs[ DEFAULT_VRF ] = candidates
      return vrfModel

BasicCli.addShowCommandClass( ShowSrTePolicyCmd )

class ShowSrTePolicyFecCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show traffic-engineering segment-routing policy fec
               [ FECID
               | ( color COLOR )
               | ( endpoint ENDPOINT [ color COLOR ] ) ]
            '''
   data = {
      'traffic-engineering': matcherTrafficEngineering,
      'segment-routing': matcherSegmentRoutingForShow,
      'policy': matcherPolicy,
      'fec': 'Show Segment Routing Traffic Engineering FEC',
      'FECID': CliMatcher.IntegerMatcher( 1, 0xFFFFFFFFFFFFFFFF,
                                          helpdesc='FEC identifier' ),
      'endpoint': matcherEndpoint,
      'ENDPOINT': IpGenAddrMatcher( 'IPv4 or IPv6 address' ),
      'color': matcherColor,
      'COLOR': matcherColorValue,
   }
   cliModel = SrTePolicyFecTable

   @staticmethod
   def handler( mode, args ):
      return populatePolicyFec( args.get( 'FECID' ), args.get( 'ENDPOINT' ),
                                 args.get( 'COLOR' ) )

BasicCli.addShowCommandClass( ShowSrTePolicyFecCmd )

def populatePolicyFec( fecId, endpoint, color ):
   fecTable = SrTePolicyFecTable()
   keys = []
   if fecId is not None:
      keys.append( fecId )
   elif endpoint is not None and color is not None:
      key = SrTePolicyCommonLib.PolicyKey( endpoint, color )
      policy = policyStatus.status.get( key )
      if policy is not None:
         for policyFecId in [ policy.labelFecId, policy.v4FecId, policy.v6FecId ]:
            keys.append( policyFecId )
         # Remove possible duplicates
         keys = set( keys )
      else:
         return fecTable
   else:
      keys = list( srteForwardingStatus.fec )

   for key in sorted( keys ):
      fec = srteForwardingStatus.fec.get( key )
      if fec is None:
         continue
      fecInfo = SrTePolicyFecInfo()
      policy = SrTePolicyBase()
      if endpoint is not None and color is not None:
         policy.endpoint = endpoint
         policy.color = color
         fecInfo.policy = policy
      else:
         fecIdToPolicyKeyInfo = policyStatus.fecIdToPolicyKey.get( key )
         if fecIdToPolicyKeyInfo is not None:
            policy.endpoint = fecIdToPolicyKeyInfo.policyKey.endpoint
            policy.color = fecIdToPolicyKeyInfo.policyKey.color
            fecInfo.policy = policy
      if ( endpoint is not None and policy.endpoint != endpoint ) \
         or ( color is not None and policy.color != color ):
         continue
      for via in fec.via.values():
         fecVia = SrTePolicyFecVia()
         if not DynTunnelIntfId.isDynamicTunnelIntfId( via.intfId ):
            continue
         fecVia.segmentListId = \
            TunnelId( DynTunnelIntfId.tunnelId( via.intfId ) ).tunnelIndex()
         fecVia.weight = via.weight
         fecInfo.vias.append( fecVia )
      fecInfo.vias = sorted( fecInfo.vias, key=attrgetter( 'segmentListId' ) )
      fecTable.fecs[ key ] = fecInfo
   return fecTable

def RouteKey( labelStack, const=True ):
   assert isinstance( const, bool ), ( f"const is {const}" )
   boundedMplsLabelStack = None
   if isinstance( labelStack, Tac.Type( 'Arnet::DynamicMplsLabelStack' ) ):
      boundedMplsLabelStack = ConvertLabelStack( labelStack )
   elif isinstance( labelStack, Tac.Type( 'Arnet::BoundedMplsLabelStack' ) ):
      boundedMplsLabelStack = labelStack
   else:
      boundedMplsLabelStack = BoundedMplsLabelStack()
      for label in labelStack:
         boundedMplsLabelStack.prepend( 100 )
         boundedMplsLabelStack.labelIs( 0, label )
   key = TacLazyType( 'Mpls::RouteKey' )( boundedMplsLabelStack )
   constFunc = Tac.const if const else Tac.nonConst
   return constFunc( key )

def getUnprogrammedBackupVias( slId ):
   '''
   Get the resolved segment list which couldnot be used
   because it crosses MSD limit
   '''
   segList = allSl.segmentList.get( slId )
   if not segList:
      return None
   labels = [ segList.segment[ idx ].mplsLabel for idx in
         range( segList.size ) ]
   topLabel = segList.segment[ 0 ].mplsLabel
   route = lfibStatus.lfibRoute.get( RouteKey( [ topLabel ] ) )
   if not route:
      return None
   viaSet = lfibStatus.viaSet.get( route.viaSetKey )
   if not viaSet:
      return None
   viaKey = viaSet.viaKey.get( 0 )
   if not viaKey:
      return None
   tilfaTunnelId = None
   if viaKey.viaType == 'viaTypeMplsIp':
      via = lfibStatus.mplsVia.get( viaKey )
      if not via:
         return None
      if DynTunnelIntfId.isDynamicTunnelIntfId( via.intf ):
         tilfaTunnelId = DynTunnelIntfId.tunnelId(
            via.intf )
      else:
         return None
   elif viaKey.viaType == 'viaTypeExternalFec':
      via = lfibStatus.extFecVia.get( viaKey )
      if not via:
         return None
      tilfaTunnelId = FecId.fecIdToTunnelId( via.fecId )
      if not tilfaTunnelId:
         return None
   else:
      return None

   tilfaTunnelEntry = tilfaTunnelTable.entry.get( tilfaTunnelId, None )
   if tilfaTunnelEntry is None:
      return None
   # Adding the labels obtained after resolution of top label
   # Labels are stored in order, top label is at the left
   backupLabels = cliDisplayOrderLabelList(
                     tilfaTunnelEntry.backupVia.labels.boundedLabelStack )
   backupLabels = [ x for x in backupLabels if x != MplsLabel.implicitNull ]
   # Adding rest of the labels excluding top label
   backupLabels.extend( labels[ 1 : ] )
   backupVias = []
   backupVias.append( SrTeSegmentListVia(
                         nexthop=IpGenAddr(),
                         interface=IntfId( '' ),
                         mplsLabels=backupLabels ) )
   return backupVias

def getSbfdHoldDownTime( holdDownTime ):
   return int( Tac.utcOffset() +
                  holdDownTime ) if holdDownTime != 0 else 0

def populatePathGroupSegmentList( wsl, allSlArg, policyId, updateReason=False,
                                  sbfdConfig=None ):
   sl = SrTePolicyPathGroupSegmentList()
   sl.segmentListId = wsl.segmentListId
   sl.weight = wsl.weight
   sl.vias = None
   sl.backupVias = None
   slId = sl.segmentListId
   sl.protectionState = None

   segList = allSlArg.segmentList.get( slId )
   if segList is not None:
      for i in range( segList.size ):
         sl.segments.append( SrTeSegment(
            mplsLabelSid=segList.segment[ i ].mplsLabel ) )
      for i in range( segList.sbfdReturnPath.stackSize ):
         sl.sbfdReturnPathLabelStack.append( SrTeSegment(
             mplsLabelSid=segList.sbfdReturnPath.labelStack( i ) ) )

   if toggleSrTeTunnelHoldDownTimerEnabled():
      sl.sbfdHoldDownTimeConfigured = int( srConfig.sbfdHoldDownTime )

   tunnelId = tacSegmentListId( slId ).getTunnelId()
   peerStatus = getSbfdPeerStatus( slId, sbfdConfig )
   if peerStatus:
      sl.sbfdState = peerStatus.clientStatus.status

   if segList is None or segList.size == 0:
      sl.segmentListValid = False
      if updateReason:
         sl.invalidReason = "segmentListMissing"
      return sl

   if slStatus.statusWithSbfd:
      # if Sbfd toggle is turned on, read segListStatus from statusWithSbfd
      # otherwise read from status
      sbfdKey = Tac.Value( "SrTePolicy::Sbfd::SbfdKey", slId, policyId )
      segListStatus = slStatus.statusWithSbfd.get( sbfdKey, None )
      if toggleSrTeTunnelHoldDownTimerEnabled():
         if segListStatus is not None:
            sl.sbfdHoldDownStartTime = getSbfdHoldDownTime(
                                          segListStatus.holdDownTimes.start )
            sl.sbfdHoldDownStopTime = getSbfdHoldDownTime(
                                          segListStatus.holdDownTimes.stop )
            sl.sbfdHoldDownExpiryTime = int(
                  segListStatus.holdDownTimes.expiry )
   else:
      segListStatus = slStatus.status.get( slId, None )
   if ( segListStatus is None ) or ( segListStatus.resolveError != 'success' ):
      sl.segmentListValid = False
      if updateReason:
         if segListStatus is None:
            sl.invalidReason = "unresolvedTopLabel"
         elif segListStatus.resolveError == 'zeroLabels':
            sl.invalidReason = 'noResolvedLabels'
         elif segListStatus.resolveError == 'platformLabelStackDepthExceeded':
            sl.invalidReason = 'resolvedLabelsExceedPlatformMsd'
         elif segListStatus.resolveError == 'sbfdDown':
            sl.invalidReason = 'sbfdDown'
      return sl

   tunnelEntry = slTunnelTable.entry.get( tunnelId, None )
   if tunnelEntry is None:
      sl.segmentListValid = False
      if updateReason:
         sl.invalidReason = "unresolvedTopLabel"
      return sl

   if not slStatus.statusWithSbfd:
      if sl.sbfdState and sl.sbfdState == BfdOperState.down:
         sl.segmentListValid = False
         if updateReason:
            sl.invalidReason = 'sbfdDown'
         return sl

   sl.segmentListValid = True
   vias, backupVias = getSegmentListVias( tunnelEntry )
   sl.vias = vias
   sl.backupVias = backupVias
   sl.txPackets, sl.txBytes, sl.counterInUse = \
         getCountersForTunnelEntry( tunnelEntry )

   # Protection status and reason
   if toggleInstallUnprotectedSLTilfaBackupExceedsMSDEnabled():
      slResolutionStatus = slStatus.status.get( slId )
      if slResolutionStatus is not None and\
         slResolutionStatus.resolveError == 'success':
         sl.protectionState = slResolutionStatus.protectionStatus
         if sl.protectionState == 'unprotectedTilfaBackupExceedMSD':
            sl.backupVias = getUnprogrammedBackupVias( slId )
   return sl

# pylint: disable=protected-access
# Disable protected access pylint in order to use variable
# hidden from the capi json
def populatePolicyPathGroup( pie, state, lastModified,
                             endpointAf, activeModified=None,
                             updateReason=False, sbfdConfig=None,
                             pCostMetric=None, staticEpPolicy=None,
                             dynamicEpPolicy=None, bindingSid=None,
                             bindingSidDynamic=False ):
   pathGroup = SrTePolicyPathGroup()
   pathGroup.state = state
   pathGroup.protocol = policySrcToString( pie.id.candidateId.protocol )
   policyParams = None
   if toggleDynamicSrTeEnabled() and pathGroup.protocol == 'static':
      # Endpoint provisioning currently does not apply for BGP source
      # The MSB of the discriminator is set only for Static Endpoint Policies
      if pie.id.candidateId.discriminator >> discriminatorMSB:
         pathGroup.endpointProvisioning = 'static'
         policyParams = staticEpPolicy.policyParams if staticEpPolicy else None
      else:
         pathGroup.endpointProvisioning = 'dynamic'
         policyParams = dynamicEpPolicy.policyParams if dynamicEpPolicy else None
   pathGroup.preference = pie.preference
   if state == 'active':
      pathGroup.metric = pCostMetric.metric
      if pCostMetric.dynamicMetric:
         if pCostMetric.metric == pCostMetric.dynamicMetricUnreachable:
            pathGroup.metricType = 'dynamicUnresolved'
         else:
            pathGroup.metricType = 'dynamicResolved'
      else:
         pathGroup.metricType = 'static'
   pathGroup.enlp = pie.enlp
   pathGroup._endpointAf = endpointAf
   pathGroup.inheritedGlobalEnlp = pie.inheritedEnlpGlobal
   pathGroup.lastChangeTime = int( Tac.utcNow() - ( Tac.now() - lastModified ) )
   if state == 'active':
      pathGroup.activeTime = int( Tac.utcNow() - ( Tac.now() - activeModified ) )
   if toggleDynamicSrTeEnabled():
      pathGroup.discriminator = pie.id.candidateId.discriminator
   else:
      if pathGroup.protocol != 'static':
         pathGroup.discriminator = pie.id.candidateId.discriminator
   pathGroup.originator = SrTePolicyOriginator(
      nodeAddress=pie.originator.nodeAddress,
      asn=bgpFormatAsn( pie.originator.asn ) )
   if toggleDynamicallyAssignBsidEnabled():
      if bindingSid and bindingSid != MplsLabel.null:
         pathGroup.bindingSid = SrTePolicyBindingSid( mplsLabelSid=bindingSid,
                                                      dynamic=bindingSidDynamic )
      elif pie.bindingSid != MplsLabel.null:
         pathGroup.bindingSid = SrTePolicyBindingSid( mplsLabelSid=pie.bindingSid )
   else:
      if pie.bindingSid != MplsLabel.null:
         pathGroup.bindingSid = SrTePolicyBindingSid( mplsLabelSid=pie.bindingSid )
   candidatePath = None
   if policyParams:
      candidatePath = policyParams.candidatePath.get( pie.preference )
   if candidatePath and pathGroup.protocol == 'static':
      # Path computation currently does not apply for BGP source
      pathGroup.pathComputation =\
         candidatePathTypeToComputationTypeDict[ candidatePath.candidatePathType ]
      if candidatePath.candidatePathType ==\
         CandidatePathType.segmentRoutingFlexAlgoLabelResolution:
         pathGroup.flexAlgoName = candidatePath.flexAlgoName
   for index in sorted( pie.wSegmentList ):
      wsl = pie.wSegmentList.get( index )
      if wsl is not None:
         pathGroup.segmentLists.append(
            populatePathGroupSegmentList( wsl, allSl, pie.id, updateReason,
                                          sbfdConfig=sbfdConfig ) )
   return pathGroup

def checkForDynamicEpFilter( dynamicEpFilter, discriminator ):
   if not dynamicEpFilter:
      # if the filter is not enabled, we print candidates from both static and
      # dynamic endpoint policies
      return True
   elif ( discriminator >> discriminatorMSB ) == 0:
      # if dynamicEpFilter is enabled then ensure that the candidate is from a
      # Dynamic endpoint policy by verifying that the discriminator MSB is not set
      # as we set it only for Static Endpoint policies
      return True
   return False

def pathGroupExists( key, stateFilter, sourceFilter, dynamicEpFilter ):
   '''
   Returns True for the given PolicyKey when there is a PathGroup
   that can be displayed for the given constraints of stateFilter,
   sourceFilter and dynamicEpFilter
   '''
   ap = policyStatus.status.get( key )
   if 'active' in stateFilter and ap is not None:
      if ap.candidateId.protocol in sourceFilter and\
         checkForDynamicEpFilter( dynamicEpFilter, ap.candidateId.discriminator ):
         return True
   policy = ecPolicy.policy.get( key )
   if 'valid' in stateFilter and policy is not None:
      for priority in policy.candidate:
         if priority.isValidCandidate and priority.protocol in sourceFilter and\
            checkForDynamicEpFilter( dynamicEpFilter, priority.discriminator ):
            candId = tacCandidateId( priority.protocol, priority.discriminator )
            if ap is None or ap.candidateId != candId:
               return True
   if 'invalid' in stateFilter:
      allCandidates = policyKeyToIdMap.policy.get( key )
      if allCandidates is None:
         return False
      if policy is None:
         return bool( allCandidates.candidate )
      else:
         validCnt = 0
         for priority in policy.candidate:
            if priority.protocol in sourceFilter and priority.isValidCandidate and\
               checkForDynamicEpFilter( dynamicEpFilter, priority.discriminator ):
               validCnt += 1
         allCnt = 0
         for candidateId in allCandidates.candidate:
            if candidateId.protocol in sourceFilter and\
               checkForDynamicEpFilter( dynamicEpFilter, candidateId.discriminator ):
               allCnt += 1
         return allCnt > validCnt
   return False

def updatePathGroupInvalidReason( pathGroup, bsidReason=None ):
   # Update the reason based on the binding-sid
   if bsidReason is not None:
      if bsidReason == tacPolicyInvalidReason.policyBsidOutOfSrlb:
         pathGroup.notActiveReason = "bindingSidOutOfSrlbRange"
      elif bsidReason == tacPolicyInvalidReason.policyBsidNotConfigured:
         pathGroup.notActiveReason = "bindingSidNotConfigured"
      elif bsidReason == tacPolicyInvalidReason.policyBsidInConflictWithOtherApp:
         pathGroup.notActiveReason = "bindingSidInUseByOtherApplication"
      elif bsidReason == tacPolicyInvalidReason.policyNoSegmentLists:
         pathGroup.notActiveReason = "segmentListsMissing"
   else:
      # Update the reason based on the segment-list status
      resolvedSl = False
      for segmentList in pathGroup.segmentLists:
         if segmentList.segmentListValid:
            resolvedSl = True
            break
      if not resolvedSl:
         pathGroup.notActiveReason = "segmentListsUnresolved"

def getPathGroupInactiveReason( pathPriority, activePriority ):

   if pathPriority == activePriority:
      # If this is the best path and not in status, it must be a BSID conflict
      return 'bindingSidConflict'

   compareResult = activePriority.compare( pathPriority )

   if compareResult == 'isHigherPreference':
      return 'higherPreferenceAvailable'
   elif compareResult == 'isBetterProtocol':
      return 'betterProtocolAvailable'
   elif compareResult == 'isLowerOrigin':
      return 'lowerOriginAvailable'
   if srConfig.useDiscriminatorInTieBreaking and \
      compareResult == 'isHigherDiscriminator':
      return 'higherDiscriminatorAvailable'
   if activePriority.sameExceptDiscriminator( pathPriority ):
      return 'samePriorityActive'
   return None

def policyInputTableForProtocol( protocol ):
   if protocol == tacProtocol.staticProtocol:
      return staticPolicyInput
   if protocol == tacProtocol.bgpProtocol:
      return bgpPolicyInput
   return None

def populatePolicyInvalidPathGroup( pId, lastModified, endpointAf, staticEpPolicy,
                                    dynamicEpPolicy ):
   table = policyInputTableForProtocol( pId.candidateId.protocol )
   if table is None:
      return None
   policy = table.policyInputEntry.get( pId )
   if policy is None:
      return None
   pathGroup = SrTePolicyPathGroup()
   pathGroup.state = 'invalid'
   pathGroup.protocol = policySrcToString( pId.candidateId.protocol )
   policyParams = None
   if toggleDynamicSrTeEnabled() and pathGroup.protocol == 'static':
      # Endpoint provisioning currently does not apply for BGP source
      # The MSB of the discriminator is set only for Static Endpoint Policies
      if pId.candidateId.discriminator >> discriminatorMSB and staticEpPolicy:
         pathGroup.endpointProvisioning = 'static'
         policyParams = staticEpPolicy.policyParams
      elif dynamicEpPolicy:
         pathGroup.endpointProvisioning = 'dynamic'
         policyParams = dynamicEpPolicy.policyParams
   pathGroup.preference = policy.preference
   pathGroup.enlp = policy.enlp
   pathGroup._endpointAf = endpointAf
   pathGroup.inheritedGlobalEnlp = ( policy.enlp == 'localPolicy' )
   pathGroup.lastChangeTime = int( Tac.utcNow() - ( Tac.now() - lastModified ) )
   if toggleDynamicSrTeEnabled():
      pathGroup.discriminator = pId.candidateId.discriminator
   else:
      if pathGroup.protocol != "static":
         pathGroup.discriminator = pId.candidateId.discriminator
   pathGroup.originator = SrTePolicyOriginator(
                                       nodeAddress=policy.originator.nodeAddress,
                                       asn=bgpFormatAsn( policy.originator.asn ) )
   if policy.bindingSid != MplsLabel.null:
      pathGroup.bindingSid = SrTePolicyBindingSid( mplsLabelSid=policy.bindingSid )
   if policyParams and pathGroup.protocol == 'static':
      # Path computation currently does not apply for BGP source
      candidatePath = policyParams.candidatePath.get( policy.preference )
      if candidatePath is not None:
         pathGroup.pathComputation =\
            candidatePathTypeToComputationTypeDict[ candidatePath.candidatePathType ]
         if candidatePath.candidatePathType ==\
            CandidatePathType.segmentRoutingFlexAlgoLabelResolution:
            pathGroup.flexAlgoName = candidatePath.flexAlgoName
   for index in sorted( policy.segmentList ):
      sl = populateInvalidPathGroupSegmentList( policy, index )
      pathGroup.segmentLists.append( sl )
   return pathGroup

def populateInvalidPathGroupSegmentList( policy, index ):
   sl = SrTePolicyPathGroupSegmentList()
   sl.vias = None
   sl.backupVias = None
   policySl = policy.segmentList[ index ].segList
   sbfdReturnPath = policy.segmentList[ index ].sbfdReturnPath
   sl.weight = policy.segmentList[ index ].weight
   for i in range( policySl.stackSize ):
      sl.segments.append( SrTeSegment( mplsLabelSid=policySl.labelStack( i ) ) )
   for i in range( sbfdReturnPath.stackSize ):
      sl.sbfdReturnPathLabelStack.append(
         SrTeSegment( mplsLabelSid=sbfdReturnPath.labelStack( i ) ) )
   if toggleSrTeTunnelHoldDownTimerEnabled():
      sl.sbfdHoldDownTimeConfigured = int( srConfig.sbfdHoldDownTime )
   return sl

def populatePathGroups( key, stateFilter, sourceFilter, dynamicEpFilter,
                        updateReason ):
   allCandidates = policyKeyToIdMap.policy.get( key )
   if allCandidates is None:
      return
   ecCandidates = ecPolicy.policy.get( key )
   active = policyStatus.status.get( key )
   policy = srConfig.policy.get( key )
   dynamicEpPolicy = srConfig.dynamicEpPolicy.get( key.color )\
      if toggleDynamicSrTeEnabled() else None
   sbfdConfig = None
   if policy:
      sbfdConfig = policy.policyParams.sbfdConfig if toggleDynamicSrTeEnabled()\
         else policy.sbfdConfig
   if ( sbfdConfig is None or sbfdConfig.sbfdRemoteDiscriminator == 0 ) and \
      srConfig.globalSBFDEcho:
      sbfdConfig = PolicySbfdConfigType()
      sbfdConfig.sbfdRemoteDiscriminator = 0xFFFFFFFF
   # Populate the active path first
   if 'active' in stateFilter and active is not None:
      pie = allPolicy.policyIdEntry.get( tacPolicyId( key, active.candidateId ) )
      cInfo = allCandidates.candidate.get( active.candidateId )
      bindingSid = None
      bindingSidDynamic = None
      if toggleDynamicallyAssignBsidEnabled():
         if ecCandidates and ecCandidates.active:
            bindingSid = ecCandidates.active.bindingSid
            bindingSidDynamic = ecCandidates.active.bindingSidDynamic
      if pie is not None and cInfo is not None and \
            pie.id.candidateId.protocol in sourceFilter and\
            checkForDynamicEpFilter( dynamicEpFilter,
                                     pie.id.candidateId.discriminator ):
         pathGroup = populatePolicyPathGroup( pie, "active", cInfo.lastModified,
                                              key.endpoint.af,
                                              active.lastModified, updateReason,
                                              sbfdConfig=sbfdConfig,
                                              pCostMetric=active.costMetric,
                                              staticEpPolicy=policy,
                                              dynamicEpPolicy=dynamicEpPolicy,
                                              bindingSid=bindingSid,
                                              bindingSidDynamic=bindingSidDynamic )
         yield pathGroup
   # Populate the valid paths next
   if 'valid' in stateFilter and ecCandidates is not None:
      for path in ecCandidates.candidate:
         if not path.isValidCandidate or path.protocol not in sourceFilter:
            continue
         pathId = tacCandidateId( path.protocol, path.discriminator )
         cInfo = allCandidates.candidate.get( pathId )
         if cInfo is None:
            continue
         if active is not None and active.candidateId == pathId:
            continue
         pie = allPolicy.policyIdEntry.get( tacPolicyId( key, pathId ) )
         if pie is not None:
            # if dynamicEndpoint filter is enabled, then skip the ones which have the
            # MSB of the discriminator set as the MSB is only set for static policies
            if not checkForDynamicEpFilter( dynamicEpFilter,
                                            pie.id.candidateId.discriminator ):
               continue
            pathGroup = populatePolicyPathGroup( pie, "valid",
                                                 cInfo.lastModified,
                                                 key.endpoint.af,
                                                 updateReason=updateReason,
                                                 sbfdConfig=sbfdConfig,
                                                 staticEpPolicy=policy,
                                                 dynamicEpPolicy=dynamicEpPolicy )
            if updateReason and ecCandidates.active:
               pathGroup.notActiveReason = getPathGroupInactiveReason(
                  path, ecCandidates.active.priority )
            yield pathGroup
   # Populate the invalid paths
   if 'invalid' in stateFilter:
      for candidateId, cInfo in allCandidates.candidate.items():
         if candidateId.protocol not in sourceFilter:
            continue
         pId = tacPolicyId( key, candidateId )
         pie = allPolicy.policyIdEntry.get( pId )
         if pie is None or not pie.isValidConfig:
            # if dynamicEndpoint filter is enabled, then skip the ones which have the
            # MSB of the discriminator set as the MSB is only set for static policies
            if not checkForDynamicEpFilter( dynamicEpFilter,
                                            pId.candidateId.discriminator ):
               continue
            pathGroup = populatePolicyInvalidPathGroup( pId, cInfo.lastModified,
                                                        key.endpoint.af, policy,
                                                        dynamicEpPolicy )
            if pathGroup is not None:
               if updateReason:
                  updatePathGroupInvalidReason( pathGroup, bsidReason=cInfo.reason )
               yield pathGroup
            continue
         if ecCandidates is not None:
            # check if candidate path is valid, if so then skip
            priority = tacPriority( True,
               pie.preference, pie.id.candidateId.protocol,
               pie.originator, pie.id.candidateId.discriminator )
            if priority in ecCandidates.candidate:
               continue
         # if dynamicEndpoint filter is enabled, then skip the ones which have the
         # MSB of the discriminator set as the MSB is only set for static policies
         if not checkForDynamicEpFilter( dynamicEpFilter,
                                         pie.id.candidateId.discriminator ):
            continue
         pathGroup = populatePolicyPathGroup( pie, "invalid", cInfo.lastModified,
                                              key.endpoint.af,
                                              updateReason=updateReason,
                                              sbfdConfig=sbfdConfig,
                                              staticEpPolicy=policy,
                                              dynamicEpPolicy=dynamicEpPolicy )
         if updateReason:
            updatePathGroupInvalidReason( pathGroup )
         yield pathGroup

def populatePolicies( ep, color, stateFilter, sourceFilter, dynamicEpFilter,
                      name=None ):
   keys = []
   if name is not None:
      tacPolicyName = Tac.newInstance( "SrTePolicy::PolicyName",
                                       str( name ) )
      if tacPolicyName in policyNameToKeyMap.policy:
         policyKey = policyNameToKeyMap.policy[ tacPolicyName ]
         keys.append( SrTePolicyCommonLib.PolicyKey(
            policyKey.endpoint, policyKey.color ) )
         updateReason = True
   elif ep is not None and color is not None:
      keys.append( SrTePolicyCommonLib.PolicyKey( ep, color ) )
      updateReason = True
   elif dynamicEpFilter is not None and color is not None:
      keys = [ key for key in policyKeyToIdMap.policy if key.color == color ]
      updateReason = True
   else:
      keys = list( policyKeyToIdMap.policy )
      updateReason = False

   for key in sorted( keys ):
      # policyEntry.pathGroups uses a GeneratorList. When there is no
      # pathgroup exist matching the criteria for filter, the policyEntry
      # generation should also be skipped. Hence check for the pathGroup
      # existence first.
      if not pathGroupExists( key, stateFilter, sourceFilter, dynamicEpFilter ):
         continue
      # pylint: disable=stop-iteration-return
      if [ 'active' ] == stateFilter:
         pathGroup = next( populatePathGroups( key, stateFilter, sourceFilter,
                                               dynamicEpFilter,
                                               updateReason=updateReason ) )
         if pathGroup is None:
            continue
         policyEntry = SrTePolicyActive()
         policyEntry.pathGroup = pathGroup
      else:
         policyEntry = SrTePolicyCandidates()
         policyEntry.pathGroups = populatePathGroups( key, stateFilter,
                                                      sourceFilter,
                                                      dynamicEpFilter,
                                                      updateReason=updateReason )
      policyEntry.endpoint = key.endpoint
      policyEntry.color = key.color
      policyEntry.name = None
      policyEntry.description = None
      policy = srConfig.policy.get( key )
      if policy is not None:
         policyParams = policy.policyParams if toggleDynamicSrTeEnabled() else policy
         if policyParams.policyName:
            policyEntry.name = policyParams.policyName
         if policyParams.policyDescription:
            policyEntry.description = policyParams.policyDescription
      ( pkts, octets ) = getCountersForPolicy( key )
      policyEntry.txPackets = pkts
      policyEntry.txBytes = octets
      yield policyEntry

#-----------------------------------------------------------------
# clear traffic-engineering segment-routing policy ... counters
#-----------------------------------------------------------------
class ClearPolicyCountersCmd( CliCommand.CliCommandClass ):
   syntax = '''clear traffic-engineering segment-routing policy
               [ endpoint ADDR color COLOR ] counters
            '''
   data = {
      'clear': clearKwNode,
      'traffic-engineering': matcherTrafficEngineering,
      'segment-routing': matcherSegmentRoutingForShow,
      'policy': matcherPolicy,
      'endpoint': matcherEndpoint,
      'ADDR': IpGenAddrMatcher( 'IPv4 or IPv6 address' ),
      'color': matcherColor,
      'COLOR': matcherColorValue,
      'counters': 'Show packet & byte egress counters for segment lists',
   }

   @staticmethod
   def handler( mode, args ):
      policyEntriesFound = 0
      ep = args.get( 'ADDR' )
      color = args.get( 'COLOR' )

      if ep is None and color is None:
         # clear all policies by toggling the clearAllPolicyCountersRequest flag
         policyEntriesFound = len( policyCounterTable.policyCounter )
         policyCounterClear.clearAllPolicyCountersRequest = False
         policyCounterClear.clearAllPolicyCountersRequest = True
      else:
         # clear a single policy
         policyKey = SrTePolicyCommonLib.PolicyKey( ep, color )
         if policyKey in policyCounterTable.policyCounter:
            policyEntriesFound += 1
            policyCounterClear.clearPolicyCountersRequest.clear()
            policyCounterClear.clearPolicyCountersRequest[ policyKey ] = True

      if policyEntriesFound:
         entryStr = 'entry' if policyEntriesFound == 1 else 'entries'
         print( f"{policyEntriesFound} policy counter {entryStr} cleared "
                "successfully" )
      else:
         mode.addWarning( "No policy counter entries found" )

BasicCli.EnableMode.addCommandClass( ClearPolicyCountersCmd )

# ----------------------------------------------------------------
# show traffic-engineering segment-routing policy segment-list ...
# ----------------------------------------------------------------
slIdValueMatcher = CliMatcher.IntegerMatcher( tacSegmentListId.min,
                                              tacSegmentListId.max,
                                              helpdesc='Segment list ID value' )
def buildSegmentListsModel( validityFilter, slIdFilter=None, counters=False ):

   slModelDict = {}
   if slIdFilter is None:
      slIds = list( allSl.segmentList )
   else:
      slIds = [ slIdFilter ]

   for slId in slIds:
      sl = allSl.segmentList.get( slId )
      if sl is None:
         # The requested SL doesn't exist, so skip it.
         continue

      valid, tunnelEntry = isSlValidAndGetTunEntry( slId, sl )

      if validityFilter is not None and validityFilter != valid:
         # We're filtering this entry out, so don't bother constructing it.
         continue

      vias = None
      backupVias = None
      if valid:
         vias, backupVias = getSegmentListVias( tunnelEntry )

      if counters:
         txPackets, txBytes, counterInUse = getCountersForTunnelEntry( tunnelEntry )
         slModel = SrTePolicySegmentListCounters(
            vias=vias,
            txPackets=txPackets,
            txBytes=txBytes,
            counterInUse=counterInUse,
         )
      else:
         slModel = SrTePolicySegmentList(
            segments=[ SrTeSegment( mplsLabelSid=sl.segment[ i ].mplsLabel )
                       for i in range( sl.size ) ],
            valid=valid,
            vias=vias,
            backupVias=backupVias
         )

      slModelDict[ slId ] = slModel

   if counters:
      return SrTePolicySegmentListsCounters( segmentLists=slModelDict )
   else:
      return SrTePolicySegmentLists( segmentLists=slModelDict )

class ShowSrTePolicySegmentListCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show traffic-engineering segment-routing policy segment-list
               [ valid | invalid | ( id ID ) ]
            '''
   data = {
      'traffic-engineering': matcherTrafficEngineering,
      'segment-routing': matcherSegmentRoutingForShow,
      'policy': matcherPolicy,
      'segment-list': matcherSegmentList,
      'valid': 'Show valid Segment Routing Traffic Engineering policy segment lists',
      'invalid': 'Show invalid Segment Routing Traffic Engineering policy '
                 'segment lists',
      'id': matcherId,
      'ID': slIdValueMatcher,
   }
   cliModel = VrfSrTePolicySegmentLists

   @staticmethod
   def handler( mode, args ):
      validityFilter = None
      if 'valid' in args:
         validityFilter = True
      elif 'invalid' in args:
         validityFilter = False

      slId = args.get( 'ID' )
      segmentLists = buildSegmentListsModel( validityFilter, slIdFilter=slId )
      vrfSls = VrfSrTePolicySegmentLists(
         vrfs={ DEFAULT_VRF: segmentLists },
         _validityFilter=validityFilter )
      return vrfSls

BasicCli.addShowCommandClass( ShowSrTePolicySegmentListCmd )

class ShowSrTePolicySegmentListCountersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show traffic-engineering segment-routing policy segment-list counters
               [ id ID ]
            '''
   data = {
      'traffic-engineering': matcherTrafficEngineering,
      'segment-routing': matcherSegmentRoutingForShow,
      'policy': matcherPolicy,
      'segment-list': matcherSegmentList,
      'counters': 'Show packet & byte egress counters for segment lists',
      'id': matcherId,
      'ID': slIdValueMatcher,
   }
   cliModel = VrfSrTePolicySegmentListsCounters

   @staticmethod
   def handler( mode, args ):
      slId = args.get( 'ID' )
      vrfSls = VrfSrTePolicySegmentListsCounters( vrfs={
         DEFAULT_VRF: buildSegmentListsModel( True, slIdFilter=slId,
                                              counters=True ) } )
      return vrfSls

BasicCli.addShowCommandClass( ShowSrTePolicySegmentListCountersCmd )

#--------------------------------------------------------------------------------
# SrTePolicy commands in 'show tech-support'
#--------------------------------------------------------------------------------
TechSupportCli.registerShowTechSupportCmd(
   '2017-09-18 11:35:37',
   cmds=[ 'show traffic-engineering segment-routing policy',
          'show traffic-engineering segment-routing policy segment-list',
          'show traffic-engineering segment-routing policy fec' ],
   summaryCmds=[ 'show traffic-engineering segment-routing policy summary' ] )

#--------------------------------------------------------------------------------
# show tech-support extended srtepolicy
#-------------------------------------------------------------------------------
class ShowSrTeTechSupportCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show tech-support extended srtepolicy'
   data = {
         'tech-support': techSupportKwMatcher,
         'extended': extendedKwMatcher,
         'srtepolicy': 'Show detailed state of SrTePolicy Agent',
   }
   privileged = True

   @staticmethod
   def handler( mode, args ):
      AgentCommandRequest.runSocketCommand( mode.entityManager,
                                             'SrTePolicy',
                                             'SrTePolicyDump',
                                             'DUMP_SRTEPOLICYSTATE' )

BasicCli.addShowCommandClass( ShowSrTeTechSupportCmd )

def Plugin( entityManager ):
   global srConfig
   global repSegConfig
   global mplsConfig
   global mplsHwCapability
   global routingHwStatusCommon
   global routingHwStatus
   global policyStatus
   global ecPolicy
   global policyKeyToIdMap
   global allPolicy
   global allSl
   global slStatus
   global slTunnelTable
   global srteForwardingStatus
   global bgpPolicyInput
   global staticPolicyInput
   global policyCounterClear
   global policyCounterTable
   global bfdStatusPeer
   global policyNameToKeyMap
   global lfibStatus
   global tilfaTunnelTable
   global flexAlgoConfig
   global labelManagerStatus
   global dynamicBsidStats

   srConfig = ConfigMount.mount( entityManager,
                                 "te/segmentrouting/srtepolicy/staticconfig",
                                 "SrTePolicy::Config", "w" )
   repSegConfig = ConfigMount.mount( entityManager,
                                     "te/segmentrouting/replicationsegment/config",
                                     "SrTePolicy::SrP2mp::Config", "w" )
   policyCounterClear = ConfigMount.mount(
      entityManager,
      "te/segmentrouting/srtepolicy/policycounterclear",
      "SrTePolicy::Counters::PolicyCliClearConfig",
      "w" )
   policyNameToKeyMap = ConfigMount.mount(
      entityManager,
      "te/segmentrouting/srtepolicy/policynametokeymap",
      "SrTePolicy::PolicyNameToKeyMap", "w" )
   mplsConfig = LazyMount.mount( entityManager,
                                 "routing/mpls/config",
                                 "Mpls::Config", "r" )
   mplsHwCapability = LazyMount.mount( entityManager,
                                       "routing/hardware/mpls/capability",
                                       "Mpls::Hardware::Capability",
                                       "r" )
   routingHwStatusCommon = LazyMount.mount(
      entityManager,
      "routing/hardware/statuscommon",
      "Routing::Hardware::StatusCommon", "r" )
   routingHwStatus = LazyMount.mount( entityManager,
                                      "routing/hardware/status",
                                      "Routing::Hardware::Status", "r" )
   ecPolicy = LazyMount.mount( entityManager,
                               "te/segmentrouting/srtepolicy/policyvalid",
                               "SrTePolicy::EcPolicy", "r" )
   policyKeyToIdMap = LazyMount.mount( entityManager,
                                      "te/segmentrouting/srtepolicy/policykeyidmap",
                                      "SrTePolicy::PolicyKeyToIdMap", "r" )
   allPolicy = LazyMount.mount( entityManager,
                                "te/segmentrouting/srtepolicy/allpolicy",
                                "SrTePolicy::PolicySelectionInput", "r" )
   allSl = LazyMount.mount( entityManager,
                            "te/segmentrouting/srtepolicy/allsegmentlist",
                            "SrTePolicy::Export::SegmentListCollection",
                            "r" )
   slStatus = LazyMount.mount( entityManager,
                               "te/segmentrouting/srtepolicy/segmentliststatus",
                               "SrTePolicy::SegmentListStatus",
                               "r" )
   flexAlgoConfig = LazyMount.mount( entityManager, 'te/flexalgo/config',
                                     'FlexAlgo::Config', 'r' )
   slTunnelTable = readMountTunnelTable(
      TunnelTableIdentifier.srTeSegmentListTunnelTable, entityManager )
   TeCli.TeModeDependents.registerDependentClass( TeSrSubMode, priority=20 )

   smashEm = SharedMem.entityManager( sysdbEm=entityManager )
   readerInfo = Smash.mountInfo( 'reader' )
   srteForwardingStatus = smashEm.doMount( "forwarding/srte/status",
                                           "Smash::Fib::ForwardingStatus",
                                           readerInfo )
   bgpPolicyInput = smashEm.doMount( "srTePolicy/Bgp",
                                     "SrTePolicy::PolicyInput",
                                     readerInfo )
   staticPolicyInput = smashEm.doMount( "srTePolicy/Static",
                                        "SrTePolicy::PolicyInput",
                                        readerInfo )
   policyCounterTable = smashEm.doMount( "srTePolicy/policyCounterTable",
                                         "SrTePolicy::Counters::PolicyCounterTable",
                                         readerInfo )
   policyStatus = smashEm.doMount( srTePolicyStatusPath(),
                                   "SrTePolicy::PolicyStatus", readerInfo )
   bfdStatusPeer = LazyMount.mount( entityManager, 'bfd/status/peer',
                                    'Bfd::StatusPeer', 'r' )
   if toggleDynamicallyAssignBsidEnabled():
      labelManagerStatus = LazyMount.mount( entityManager, "mpls/labelManagerStatus",
                                          "Mpls::Api::LabelManagerStatus", "r" )
      dynamicBsidStats = LazyMount.mount( entityManager,
                                        "te/segmentrouting/srtepolicy/dynBsidStats",
                                        "SrTePolicy::SrTePolicyDynamicBsidStats",
                                        "r" )
   lfibStatus = smashEm.doMount( "mpls/transitLfib", "Mpls::LfibStatus", readerInfo )
   tilfaTunnelTable = readMountTunnelTable(
      TunnelTableIdentifier.tiLfaTunnelTable, entityManager )
