#!/usr/bin/env python3
# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet
import Tac
import BasicCli
import CliCommand
import CliGlobal
import CliMatcher
from CliMode.Srv6 import ( RoutingSrv6Mode,
                           RoutingSrv6VrfMode,
                           RoutingSrv6VrfUsidDomainMode,
                           RoutingSrv6VrfLocatorMode )
from CliPlugin.Srv6Model import ( LOCATOR_STATE_MAP,
                                  LOCATOR_TYPE_MAP,
                                  SID_FUNCTION_RANGE_DYNAMIC,
                                  SID_FUNCTION_RANGE_STATIC,
                                  SID_FUNCTION_RANGE_UNASSIGNED,
                                  SID_SOURCE_MAP,
                                  Srv6LocatorEntry,
                                  Srv6ConfigModel,
                                  Srv6ConfigVrfModel,
                                  Srv6RouteEntry,
                                  Srv6RoutesModel,
                                  Srv6CapabilitiesModel,
                                  Srv6EndBehaviorsModel,
                                  Srv6EndFlavorsModel,
                                  Srv6HeadendBehaviorsModel,
                                  Srv6MsdModel,
                                  Srv6SidFunctionRange,
                                  Srv6SidFunctionAllocator,
                                  Srv6SidFunctionAllocators )

from CliPlugin.Srv6Models import ( Srv6Sid,
                                   Srv6SidFlavorFlags )
from CliPlugin.VrfCli import vrfMatcherDefault
from CliPlugin import IpAddrMatcher, Ip6AddrMatcher
from CliToken.Ipv6 import ipv6MatcherForShow
from CliToken.Router import routerMatcherForConfig as routerKw
from CliToken.SegmentRoutingToken import matcherSegmentRoutingForShow
from ArnetLib import U32_MAX_VALUE
import ConfigMount
import MultiRangeRule
from IpLibConsts import DEFAULT_VRF
import LazyMount
import ShowCommand
import SharedMem
import Smash
import SmashLazyMount
from Toggles import Srv6LibToggleLib
import Tracing
import operator

traceHandle = Tracing.Handle( 'Srv6Cli' )
t1 = traceHandle.trace1 # Info

gv = CliGlobal.CliGlobal( dict( srv6Config=None, srv6Status=None, srv6Fib=None,
                                vrfIdMap=None, srv6Capability=None,
                                srv6SidFuncAllocStatus=None ) )

def getOrCreateVrfConfig( vrfName ):
   if not gv.srv6Config.defaultVrfConfig:
      gv.srv6Config.defaultVrfConfig = ( "default", )
   return gv.srv6Config.defaultVrfConfig

def getOrCreateUsidDomainConfig( vrfConfig, domainName ):
   return vrfConfig.usidDomainConfig.newMember( domainName )

def getOrCreateLocatorConfig( vrfConfig, locatorName ):
   return vrfConfig.locatorConfig.newMember( locatorName )

def getUsidDomainNames( mode ):
   if gv.srv6Config.defaultVrfConfig:
      return gv.srv6Config.defaultVrfConfig.usidDomainConfig
   return []

def srv6DisabledAddWarning( handler ):
   '''CLI handler decorator to add warning when SRv6 is disabled'''
   def inner( mode, args ):
      srv6Status = gv.srv6Status.defaultVrfStatus
      if not srv6Status or not srv6Status.enabled:
         mode.addWarning( 'Segment Routing over IPv6 is not enabled' )
      return handler( mode, args )
   return inner

# Configuration mode for router segment-routing ipv6
class Srv6ConfigMode( RoutingSrv6Mode, BasicCli.ConfigModeBase ):
   name = 'Srv6 instance configuration'

   def __init__( self, parent, session ):
      RoutingSrv6Mode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def gotoSrv6VrfConfigMode( self, args ):
      vrfName = args[ 'VRF' ]
      vrfConfig = getOrCreateVrfConfig( vrfName=vrfName )
      childMode = self.childMode( Srv6VrfConfigMode, vrfConfig=vrfConfig,
                                  vrfName=vrfName )
      self.session_.gotoChildMode( childMode )

   def delSrv6VrfConfigMode( self, args ):
      gv.srv6Config.defaultVrfConfig = None

# Configuration mode for vrf <> sub-mode
class Srv6VrfConfigMode( RoutingSrv6VrfMode, BasicCli.ConfigModeBase ):
   name = 'Srv6 Vrf instance configuration'

   def __init__( self, parent, session, vrfConfig, vrfName ):
      self.vrfConfig = vrfConfig
      RoutingSrv6VrfMode.__init__( self, vrfName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def gotoSrv6VrfUsidDomainConfigMode( self, args ):
      domainName = args[ 'DOMAIN' ]
      domainConfig = getOrCreateUsidDomainConfig( vrfConfig=self.vrfConfig,
                                                  domainName=domainName )
      childMode = self.childMode( Srv6VrfUsidDomainConfigMode,
                                  domainConfig=domainConfig,
                                  vrfName=self.vrfName,
                                  domainName=domainName )
      self.session_.gotoChildMode( childMode )

   def delSrv6VrfUsidDomainConfigMode( self, args ):
      domainName = args[ 'DOMAIN' ]
      vrfConfig = gv.srv6Config.defaultVrfConfig
      if vrfConfig is None:
         return
      del vrfConfig.usidDomainConfig[ domainName ]

   def gotoSrv6VrfLocatorConfigMode( self, args ):
      locatorName = args[ 'LOCATOR' ]
      locatorConfig = getOrCreateLocatorConfig( vrfConfig=self.vrfConfig,
                                                locatorName=locatorName )
      childMode = self.childMode( Srv6VrfLocatorConfigMode,
                                  locatorConfig=locatorConfig,
                                  vrfName=self.vrfName, locatorName=locatorName )
      self.session_.gotoChildMode( childMode )

   def delSrv6VrfLocatorConfigMode( self, args ):
      locatorName = args[ 'LOCATOR' ]
      vrfConfig = gv.srv6Config.defaultVrfConfig
      if vrfConfig is None:
         return
      del vrfConfig.locatorConfig[ locatorName ]

# Configuration mode for micro-segment domain <> submode
class Srv6VrfUsidDomainConfigMode( RoutingSrv6VrfUsidDomainMode,
                                   BasicCli.ConfigModeBase ):
   name = 'SRv6 Micro Segment configuration'

   def __init__( self, parent, session, domainConfig, vrfName, domainName ):
      self.usidDomainConfig = domainConfig
      RoutingSrv6VrfUsidDomainMode.__init__( self, ( vrfName, domainName ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

# Configuration mode for locator <> submode
class Srv6VrfLocatorConfigMode( RoutingSrv6VrfLocatorMode, BasicCli.ConfigModeBase ):
   name = 'SRv6 Locator configuration'

   def __init__( self, parent, session, locatorConfig, vrfName, locatorName ):
      self.locatorConfig = locatorConfig
      RoutingSrv6VrfLocatorMode.__init__( self, ( vrfName, locatorName ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

ipv6Kw = CliMatcher.KeywordMatcher( 'ipv6', helpdesc='IPv6 data plane' )

# ------------------------------------------------------------------------------
# The "[no | default] router segment-routing ipv6" in global configuration mode
# ------------------------------------------------------------------------------
class RouterSrv6ModeCmd( CliCommand.CliCommandClass ):
   syntax = 'router segment-routing ipv6'
   noOrDefaultSyntax = syntax
   data = {
      'router': routerKw,
      'segment-routing': 'Segment Routing configuration',
      'ipv6': ipv6Kw
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( Srv6ConfigMode )
      mode.session_.gotoChildMode( childMode )
      gv.srv6Config.created = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      gv.srv6Config.defaultVrfConfig = None
      gv.srv6Config.created = False

if Srv6LibToggleLib.toggleSrv6CoreEnabled():
   BasicCli.GlobalConfigMode.addCommandClass( RouterSrv6ModeCmd )

# -----------------------------------------------------------------
# The "[no | default] vrf VRF" command, in SRv6 configuration mode
# -----------------------------------------------------------------
class RouterSrv6VrfModeCmd( CliCommand.CliCommandClass ):
   syntax = 'vrf VRF'
   noOrDefaultSyntax = syntax
   data = {
      'vrf': 'Configured VRF',
      'VRF': vrfMatcherDefault
   }

   handler = Srv6ConfigMode.gotoSrv6VrfConfigMode
   noOrDefaultHandler = Srv6ConfigMode.delSrv6VrfConfigMode

Srv6ConfigMode.addCommandClass( RouterSrv6VrfModeCmd )

# -----------------------------------------
# Srv6 VRF sub mode configuration commands
# -----------------------------------------
#
# ------------------------------------------------------------------------------
# The "[no | default] micro-segment domain DOMAIN" command, in SRv6 vrf cfg mode
# ------------------------------------------------------------------------------
class RouterSrv6VrfUsidDomainModeCmd( CliCommand.CliCommandClass ):
   syntax = 'micro-segment domain DOMAIN'
   noOrDefaultSyntax = syntax
   data = {
      'micro-segment': 'SRv6 micro segment configuration',
      'domain': 'Segment Routing domain configuration',
      'DOMAIN': CliMatcher.DynamicNameMatcher( getUsidDomainNames,
                                               helpdesc='Domain Name' )
   }

   handler = Srv6VrfConfigMode.gotoSrv6VrfUsidDomainConfigMode
   noOrDefaultHandler = Srv6VrfConfigMode.delSrv6VrfUsidDomainConfigMode

if Srv6LibToggleLib.toggleSrv6MicroSidEnabled():
   Srv6VrfConfigMode.addCommandClass( RouterSrv6VrfUsidDomainModeCmd )

# -----------------------------------------------------------------------------
# The "[no | default] locator LOCATOR" command, in SRv6 vrf configuration mode
# -----------------------------------------------------------------------------
class RouterSrv6VrfLocatorModeCmd( CliCommand.CliCommandClass ):
   syntax = 'locator LOCATOR'
   noOrDefaultSyntax = syntax
   data = {
      'locator': 'SRv6 Locator configuration',
      'LOCATOR': CliMatcher.DynamicNameMatcher(
         ( lambda mode: mode.vrfConfig.locatorConfig ),
         helpdesc='Locator Name' )
   }

   handler = Srv6VrfConfigMode.gotoSrv6VrfLocatorConfigMode
   noOrDefaultHandler = Srv6VrfConfigMode.delSrv6VrfLocatorConfigMode

Srv6VrfConfigMode.addCommandClass( RouterSrv6VrfLocatorModeCmd )

# ----------------------------------------------------------------------
# The "[no | default] disabled" command, in SRv6 vrf configuration mode
# ----------------------------------------------------------------------
class RouterSrv6VrfDisabledCmd( CliCommand.CliCommandClass ):
   syntax = 'disabled'
   noOrDefaultSyntax = syntax
   data = {
      'disabled': 'Disable SRv6 for VRF instance'
      }

   @staticmethod
   def handler( mode, args ):
      mode.vrfConfig.enabled = False

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.vrfConfig.enabled = mode.vrfConfig.enabledDefault

Srv6VrfConfigMode.addCommandClass( RouterSrv6VrfDisabledCmd )

# --------------------------------------------------------------------------------
# The "[no | default] local address ADDR" command, in SRv6 vrf configuration mode
# --------------------------------------------------------------------------------
class RouterSrv6LocalAddressCmd( CliCommand.CliCommandClass ):
   syntax = 'local address ADDR'
   noOrDefaultSyntax = 'local address ...'
   data = {
      'local': 'Configure IPv6 source address for SRv6 tunnels',
      'address': 'Local IPv6 source Address',
      'ADDR': Ip6AddrMatcher.Ip6AddrMatcher( 'IPv6 Address' )
      }

   @staticmethod
   def handler( mode, args ):
      ip6Addr = args[ 'ADDR' ]
      mode.vrfConfig.sourceIp6Addr = ip6Addr

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.vrfConfig.sourceIp6Addr = mode.vrfConfig.sourceIp6AddrDefault

Srv6VrfConfigMode.addCommandClass( RouterSrv6LocalAddressCmd )

# ----------------------------------------------------
# Micro Segment Domain sub mode configuration commands
# ----------------------------------------------------
# ---------------------------------------------------------------------------
# The "[no | default] block PREFIX" command, in uSID domain configuration mode
# ---------------------------------------------------------------------------
class RouterSrv6VrfUsidDomainBlockCmd( CliCommand.CliCommandClass ):
   syntax = 'block PREFIX'
   noOrDefaultSyntax = 'block ...'
   data = {
      'block': 'Segment Routing domain block',
      'PREFIX': Ip6AddrMatcher.Ip6PrefixMatcher(
         'IPv6 address prefix',
         overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO )
      }

   @staticmethod
   def handler( mode, args ):
      prefixAddress = args[ 'PREFIX' ]
      mode.usidDomainConfig.block = Arnet.Ip6Prefix( prefixAddress )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.usidDomainConfig.block = mode.usidDomainConfig.blockDefault

Srv6VrfUsidDomainConfigMode.addCommandClass( RouterSrv6VrfUsidDomainBlockCmd )

# ---------------------------------------------------------------------------
# The "[no | default] usid length" command, in uSID domain configuration mode
# ---------------------------------------------------------------------------
class RouterSrv6VrfUsidDomainUsidLengthCmd( CliCommand.CliCommandClass ):
   syntax = 'usid length ( 16 | 32 )'
   noOrDefaultSyntax = 'usid length ...'
   data = {
      'usid': 'SRv6 micro segment related configuration',
      'length': 'micro segment length',
      '16': 'uSID length of 16 bits',
      '32': 'uSID length of 32 bits'
   }

   @staticmethod
   def handler( mode, args ):
      usidLength = 16 if '16' in args else 32
      mode.usidDomainConfig.usidLength = usidLength

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.usidDomainConfig.usidLength = mode.usidDomainConfig.usidLengthDefault

Srv6VrfUsidDomainConfigMode.addCommandClass( RouterSrv6VrfUsidDomainUsidLengthCmd )

# -------------------------------------------------------------------------
# The "[no | default] gib range" command, in uSID domain configuration mode
# -------------------------------------------------------------------------
class RouterSrv6VrfUsidDomainGibRangeCmd( CliCommand.CliCommandClass ):
   syntax = 'gib range GIB_RANGE'
   noOrDefaultSyntax = 'gib range ...'
   data = {
      'gib': 'Global ID Block configuration',
      'range': 'block range',
      'GIB_RANGE': MultiRangeRule.MultiRangeMatcher(
         rangeFn=lambda: ( 1, U32_MAX_VALUE ),
         noSingletons=True,
         maxRanges=1,
         helpdesc='range of uSID values',
         value=lambda mode, grList: ( grList.ranges()[ 0 ][ 0 ],
                                      grList.ranges()[ 0 ][ 1 ] ) )
   }

   @staticmethod
   def handler( mode, args ):
      values = args[ 'GIB_RANGE' ]
      gibRange = Tac.Value( 'Srv6::GibRangeConfig',
                            values[ 0 ], values[ 1 ] )
      mode.usidDomainConfig.gibRange = gibRange

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.usidDomainConfig.gibRange = mode.usidDomainConfig.gibRangeDefault

Srv6VrfUsidDomainConfigMode.addCommandClass( RouterSrv6VrfUsidDomainGibRangeCmd )

# ----------------------------------------
# Locator sub mode configuration commands
# ----------------------------------------
# ----------------------------------------------------------------------------------
# The "[no | default] prefix ( PREFIX | micro-segment domain DOMAIN end usid USID )"
# command, in Locator configuration mode
# ----------------------------------------------------------------------------------
class RouterSrv6VrfLocatorPrefixCmd( CliCommand.CliCommandClass ):
   syntax = 'prefix PREFIX'
   noOrDefaultSyntax = 'prefix ...'
   data = {
      'prefix': 'Locator prefix',
      'PREFIX': Ip6AddrMatcher.Ip6PrefixMatcher(
         'IPv6 address prefix',
         overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO )
      }

   @staticmethod
   def handler( mode, args ):
      mode.locatorConfig.usidPrefixConfig = \
         mode.locatorConfig.usidPrefixConfigDefault
      prefixAddress = args[ 'PREFIX' ]
      mode.locatorConfig.prefix = Arnet.Ip6Prefix( prefixAddress )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.locatorConfig.prefix = mode.locatorConfig.prefixDefault
      mode.locatorConfig.usidPrefixConfig = \
         mode.locatorConfig.usidPrefixConfigDefault

Srv6VrfLocatorConfigMode.addCommandClass( RouterSrv6VrfLocatorPrefixCmd )

class RouterSrv6VrfLocatorUsidPrefixCmd( CliCommand.CliCommandClass ):
   syntax = 'prefix micro-segment domain DOMAIN end usid USID'
   noOrDefaultSyntax = 'prefix micro-segment ...'
   data = {
      'prefix': 'Locator prefix',
      'micro-segment': 'SRv6 micro segment',
      'domain': 'Segment Routing domain',
      'DOMAIN': CliMatcher.DynamicNameMatcher(
         getUsidDomainNames,
         helpdesc='Name of the micro-segment domain' ),
      'end': 'End behavior',
      'usid': 'Global uSID',
      'USID': CliMatcher.IntegerMatcher( 1, U32_MAX_VALUE, helpdesc='uSID value' )
   }

   @staticmethod
   def handler( mode, args ):
      mode.locatorConfig.prefix = mode.locatorConfig.prefixDefault
      prefixConfig = Tac.Value( 'Srv6::UsidPrefixConfig',
                                args[ 'DOMAIN' ], args[ 'USID' ] )
      mode.locatorConfig.usidPrefixConfig = prefixConfig

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.locatorConfig.prefix = mode.locatorConfig.prefixDefault
      mode.locatorConfig.usidPrefixConfig = \
         mode.locatorConfig.usidPrefixConfigDefault

if Srv6LibToggleLib.toggleSrv6MicroSidEnabled():
   Srv6VrfLocatorConfigMode.addCommandClass( RouterSrv6VrfLocatorUsidPrefixCmd )

# --------------------------------------------------------------------------------
# The "[no | default] function length LEN" command, in Locator configuration mode
# --------------------------------------------------------------------------------
class RouterSrv6VrfLocatorFunctionLengthCmd( CliCommand.CliCommandClass ):
   syntax = 'function length ( 16 | 32 )'
   noOrDefaultSyntax = 'function length ...'
   data = {
      'function': 'SRv6 Function related configuration',
      'length': 'Function length',
      '16': 'Function length of 16 bits',
      '32': 'Function length of 32 bits'
   }

   @staticmethod
   def handler( mode, args ):
      functionLength = 16 if '16' in args else 32
      mode.locatorConfig.functionLength = functionLength

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.locatorConfig.functionLength = mode.locatorConfig.functionLengthDefault

Srv6VrfLocatorConfigMode.addCommandClass( RouterSrv6VrfLocatorFunctionLengthCmd )

# -----------------------------------------------------------------------------
# The "[ no ] sid end function <FUNCT>" command in Locator configuration mode"
# -----------------------------------------------------------------------------
class RouterSrv6VrfLocatorEndFunctionCmd( CliCommand.CliCommandClass ):
   syntax = "sid end function FUNC"
   noOrDefaultSyntax = 'sid end function ...'
   data = {
      "sid": "Configure a SRv6 SID",
      "end": "END SID",
      "function": "Function value of SID",
      "FUNC": CliMatcher.IntegerMatcher( 1, U32_MAX_VALUE,
                                          helpdesc="Function value" )
   }

   @staticmethod
   def handler( mode, args ):
      function = args.get( "FUNC" )
      endSid = Tac.Value( "Srv6::EndSid", function )
      mode.locatorConfig.endSid = endSid

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.locatorConfig.endSid = mode.locatorConfig.endSidDefault

if Srv6LibToggleLib.toggleSrv6EndSidEnabled():
   Srv6VrfLocatorConfigMode.addCommandClass( RouterSrv6VrfLocatorEndFunctionCmd )

def getAlgorithmName( algoId ):
   if algoId == 0:
      return 'SPF'
   elif algoId >= 128 and algoId <= 255:
      return 'user-defined'

   return 'undefined'

def buildSrv6Locator( locatorStatus ):
   locator = Srv6LocatorEntry()
   if locatorStatus.state == "locatorActive":
      locator.state = "active"
   else:
      locator.state = "inactive"
      locator.inactiveReason = LOCATOR_STATE_MAP[ locatorStatus.state ]
   if locatorStatus.prefix:
      locator.prefix = str( locatorStatus.prefix )
   locator.locatorType = LOCATOR_TYPE_MAP[ locatorStatus.locatorType ]
   if locatorStatus.locatorType == 'microSid':
      locator.locatorUsidDomain = locatorStatus.usidDomain
   locator.algorithm = getAlgorithmName( locatorStatus.algorithm )
   locator.algorithmId = locatorStatus.algorithm
   locator.blockLength = locatorStatus.blockLength
   locator.functionLength = locatorStatus.functionLength
   locator.functionPool = locatorStatus.functionPool
   locator.functionAllocator = locatorStatus.functionAllocator.stringValue()
   return locator

def populateLocator( sysdbVrfStatus, locatorName=None ):
   if locatorName:
      locatorStatus = sysdbVrfStatus.locatorStatus.get( locatorName )
      if not locatorStatus:
         return
      else:
         yield locatorName, buildSrv6Locator( locatorStatus )
   else:
      for locatorStatus in sysdbVrfStatus.locatorStatus.values():
         yield locatorStatus.name, buildSrv6Locator( locatorStatus )

def populateSrv6VrfConfig( locatorName, vrfName=DEFAULT_VRF ):
   sysdbVrfStatus = gv.srv6Status.defaultVrfStatus
   vrfStatus = Srv6ConfigVrfModel()
   if not sysdbVrfStatus:
      return None
   vrfStatus.enabled = sysdbVrfStatus.enabled
   vrfStatus.operational = sysdbVrfStatus.operational
   if sysdbVrfStatus.sourceIp6Addr:
      vrfStatus.sourceAddress = sysdbVrfStatus.sourceIp6Addr
   vrfStatus.locators = populateLocator( sysdbVrfStatus, locatorName )
   # pylint: disable=protected-access
   vrfStatus._numLocators = len( sysdbVrfStatus.locatorStatus )
   return vrfStatus

def getLocatorNameCompletion( mode ):
   if gv.srv6Config.defaultVrfConfig:
      return gv.srv6Config.defaultVrfConfig.locatorConfig
   else:
      return []

# -----------------------------------------------
# show segment-routing ipv6 locator [ LOCATOR ]
# -----------------------------------------------
class ShowSrv6LocatorCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show segment-routing ipv6 locator [ LOCATOR ]'''

   data = {
      'segment-routing': matcherSegmentRoutingForShow,
      'ipv6': ipv6MatcherForShow,
      'locator': 'Show status of configured locators',
      'LOCATOR': CliMatcher.DynamicNameMatcher(
         getLocatorNameCompletion,
         helpdesc='Show details for a given locator' )
   }

   cliModel = Srv6ConfigModel
   @staticmethod
   def handler( mode, args ):
      locatorName = args.get( 'LOCATOR' )

      srv6Config = Srv6ConfigModel()
      srv6VrfConfig = populateSrv6VrfConfig( locatorName )
      if srv6VrfConfig:
         srv6Config.vrfs[ DEFAULT_VRF ] = srv6VrfConfig
      return srv6Config

if Srv6LibToggleLib.toggleSrv6CoreEnabled():
   BasicCli.addShowCommandClass( ShowSrv6LocatorCmd )

cliBehaviorToEnumBehavior = { 'end-dt4': 'endDt4', 'end-dt6': 'endDt6',
                              'end': 'end' }
cliProtocolToEnumProtocol = { 'bgp': 'sfibSourceBgpL3Vpn',
                              'static': 'sfibSourceStatic' }

def routeFilter( route, prefix, filterLocator, filterBehavior, filterProtocol ):
   if prefix:
      tacPrefix = Arnet.Ip6Prefix( prefix )
      if not tacPrefix.contains( route.sid.prefix.address ):
         return False
   if filterLocator and filterLocator != route.locatorName:
      return False
   if filterBehavior:
      behavior = cliBehaviorToEnumBehavior[ filterBehavior ]
      if behavior != route.sid.behavior:
         return False
   if filterProtocol:
      protocol = cliProtocolToEnumProtocol[ filterProtocol ]
      if protocol != route.source:
         return False
   return True

def populateSrv6Routes( prefix, filterLocator, filterBehavior, filterProtocol ):

   routes = Srv6RoutesModel()
   for route in gv.srv6Fib.sfibRoute.values():
      if not routeFilter( route, prefix, filterLocator, filterBehavior,
                          filterProtocol ):
         continue
      routeEntry = Srv6RouteEntry()
      routeEntry.source = SID_SOURCE_MAP[ route.source ]
      routeEntry.locator = route.locatorName

      srv6Sid = Srv6Sid()
      srv6Sid.sid = Arnet.IpGenPrefix( str( route.key ) )
      srv6Sid.behavior = route.sid.behavior
      srv6SidFlavors = Srv6SidFlavorFlags()
      flavorList = Srv6SidFlavorFlags.__attributes__
      if any( getattr( route.sid.flavors, f ) for f in flavorList ):
         for flavor in flavorList:
            if getattr( route.sid.flavors, flavor ):
               setattr( srv6SidFlavors, flavor, True )
         srv6Sid.flavors = srv6SidFlavors
      routeEntry.sidInformation = srv6Sid

      vrfEntry = gv.vrfIdMap.vrfIdToName.get( route.lookupVrfId )
      if vrfEntry:
         routeEntry.lookupVrf = vrfEntry.vrfName

      routes.routes[ str( route.key ) ] = routeEntry

   return routes

endBehaviorNames = { 'end': 'Show only End endpoints',
                     'end-dt4': 'Show only End.DT4 endpoints',
                     'end-dt6': 'Show only End.DT6 endpoints' }
protoNames = { 'bgp': 'Show only BGP SIDs',
               'static': 'Show only statically configured SIDs' }
addr6Matcher = Ip6AddrMatcher.Ip6AddrMatcher( 'SID address' )
prefix6Matcher = Ip6AddrMatcher.ip6PrefixExpr(
   'IPv6 address', 'Subnet mask', 'IPv6 prefix',
   overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO )
# ------------------------------------------------------------------
# show segment-routing ipv6 route [ locator LOCNAME ]
#                                 [ protocol bgp|static ]
#                                 [ end behavior end | end-dt4 | end-dt6 ]
#                                 [ PREFIX6 | ADDR6 ]
# ------------------------------------------------------------------
class ShowSrv6RouteCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show segment-routing ipv6 route [ locator LOCNAME ]
                                               [ protocol PROTONAME ]
                                               [ end behavior BEHAVNAME ]
                                               [ PREFIX6 | ADDR6 ]'''

   data = {
      'segment-routing': matcherSegmentRoutingForShow,
      'ipv6': ipv6MatcherForShow,
      'route': 'Show routes to locally configured Endpoints',
      'PREFIX6': prefix6Matcher,
      'ADDR6': addr6Matcher,
      'locator': 'Show routes only from a specific locator',
      'LOCNAME': CliMatcher.DynamicNameMatcher(
         getLocatorNameCompletion,
         helpdesc='Locator Name' ),
      'end': 'Filter based on endpoint types',
      'behavior': 'Filter based on endpoint behavior',
      'BEHAVNAME': CliMatcher.DynamicKeywordMatcher(
         lambda mode: endBehaviorNames ),
      'protocol': 'Filter based on source protocol',
      'PROTONAME': CliMatcher.DynamicKeywordMatcher(
         lambda mode: protoNames )
   }

   cliModel = Srv6RoutesModel

   @staticmethod
   @srv6DisabledAddWarning
   def handler( mode, args ):
      filterBehavior = args.get( 'BEHAVNAME' )
      filterLocator = args.get( 'LOCNAME' )
      filterProtocol = args.get( 'PROTONAME' )
      prefix = args.get( 'PREFIX6', args.get( 'ADDR6' ) )

      srv6Routes = populateSrv6Routes( prefix, filterLocator, filterBehavior,
                                       filterProtocol )
      return srv6Routes

if Srv6LibToggleLib.toggleSrv6CoreEnabled():
   BasicCli.addShowCommandClass( ShowSrv6RouteCmd )

class ShowSrv6CapabilitiesCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show segment-routing ipv6 capabilities'''

   data = {
      'segment-routing': matcherSegmentRoutingForShow,
      'ipv6': ipv6MatcherForShow,
      'capabilities': 'SRv6 capabilities',
   }

   cliModel = Srv6CapabilitiesModel
   @staticmethod
   def handler( mode, args ):
      srv6Capabilities = populateSrv6Capabilities()
      return srv6Capabilities

if Srv6LibToggleLib.toggleSrv6CoreEnabled():
   BasicCli.addShowCommandClass( ShowSrv6CapabilitiesCmd )

def populateSrv6Capabilities():
   capabilities = Srv6CapabilitiesModel()
   srv6Supported = gv.srv6Capability.srv6Supported
   # pylint: disable=protected-access
   capabilities._srv6Supported = srv6Supported
   if srv6Supported:
      eb = Srv6EndBehaviorsModel()
      eb.end = gv.srv6Capability.srv6SupportedEndBehaviors.end
      eb.endX = gv.srv6Capability.srv6SupportedEndBehaviors.endX
      eb.endDx4 = gv.srv6Capability.srv6SupportedEndBehaviors.endDx4
      eb.endDx6 = gv.srv6Capability.srv6SupportedEndBehaviors.endDx6
      eb.endDt4 = gv.srv6Capability.srv6SupportedEndBehaviors.endDt4
      eb.endDt6 = gv.srv6Capability.srv6SupportedEndBehaviors.endDt6
      capabilities.supportedEndBehaviors = eb
      ef = Srv6EndFlavorsModel()
      ef.psp = gv.srv6Capability.srv6SupportedEndBehaviorFlavors.psp
      ef.usp = gv.srv6Capability.srv6SupportedEndBehaviorFlavors.usp
      ef.usd = gv.srv6Capability.srv6SupportedEndBehaviorFlavors.usd
      ef.nextCsid = gv.srv6Capability.srv6SupportedEndBehaviorFlavors.nextCsid
      capabilities.supportedEndBehaviorFlavors = ef
      hb = Srv6HeadendBehaviorsModel()
      hb.hEncaps = gv.srv6Capability.srv6SupportedHeadendBehaviors.encaps
      hb.hEncapsRed = gv.srv6Capability.srv6SupportedHeadendBehaviors.encapsRed
      hb.hInsert = gv.srv6Capability.srv6SupportedHeadendBehaviors.insert
      hb.hInsertRed = gv.srv6Capability.srv6SupportedHeadendBehaviors.insertRed
      capabilities.supportedHeadendBehaviors = hb
      msd = Srv6MsdModel()
      msd.maxSegmentsLeft = gv.srv6Capability.maxSegmentsLeft
      msd.maxSidsPopped = gv.srv6Capability.maxSidsPopped
      msd.maxSidsPushed = gv.srv6Capability.maxSidsPushed
      msd.maxSidsDecapsulated = gv.srv6Capability.maxSidsDecap
      capabilities.supportedMsd = msd

   return capabilities

def getAllocatorNameCompletion( mode ):
   if gv.srv6Status.defaultVrfStatus:
      return gv.srv6Status.defaultVrfStatus.sidFunctionAllocatorStatus
   return []

def buildSidFunctionAllocator( allocatorStatus, sidFunctionRangeInfo ):
   allocator = Srv6SidFunctionAllocator(
      functionLength=allocatorStatus.functionLength,
      locators=list( allocatorStatus.locatorUser )
   )
   # Build dynamic used ranges by iterating over the collection of dynamic
   # allocations indexed by client, adding client blocks with locator assigned to
   # this allocator
   for cltAlloc in gv.srv6SidFuncAllocStatus.sidFunctionBlockAllocation.values():
      for block in cltAlloc.block.values():
         if block.locator in allocatorStatus.locatorUser:
            dr = Srv6SidFunctionRange()
            dr.blockStart = block.start
            dr.blockSize = block.size
            dr.protocol = cltAlloc.name
            dr.rangeType = SID_FUNCTION_RANGE_DYNAMIC
            allocator.functionRanges.append( dr )
   # Add unused dynamic ranges (free spaces between allocated ranges)
   aDynRange = allocatorStatus.dynamicFunctionRange
   nextDrStart = aDynRange.start
   sortKey = operator.attrgetter( 'blockStart' )
   for used in sorted( allocator.functionRanges, key=sortKey ):
      if nextDrStart < used.blockStart:
         unused = Srv6SidFunctionRange( rangeType=SID_FUNCTION_RANGE_DYNAMIC )
         unused.blockStart = nextDrStart
         unused.blockSize = used.blockStart - nextDrStart
         allocator.functionRanges.append( unused )
      nextDrStart = used.blockStart + used.blockSize
   # Add unassigned range for unused functions at the end of the dynamic range
   if nextDrStart < aDynRange.start + aDynRange.size:
      unassigned = Srv6SidFunctionRange( rangeType=SID_FUNCTION_RANGE_UNASSIGNED )
      unassigned.blockStart = nextDrStart
      unassigned.blockSize = aDynRange.start + aDynRange.size - nextDrStart
      allocator.functionRanges.append( unassigned )
   # Add the static range. Eventually each allocator will track its own static and
   # reserved range configuration, but currently only the default static range
   # is defined
   sr = Srv6SidFunctionRange()
   sr.blockStart = sidFunctionRangeInfo.staticDefaultRange.start
   sr.blockSize = sidFunctionRangeInfo.staticDefaultRange.size
   sr.rangeType = SID_FUNCTION_RANGE_STATIC
   allocator.functionRanges.append( sr )
   return allocator

def populateSidFunctionAllocator( sysdbAllocatorStatus, allocatorName=None ):
   rangeInfo = Tac.newInstance( "Srv6::SidFunctionRangeInfo", "" )
   if allocatorName:
      allocatorStatus = sysdbAllocatorStatus.get( allocatorName )
      if allocatorStatus:
         yield allocatorName, buildSidFunctionAllocator( allocatorStatus, rangeInfo )
   else:
      for allocStatus in sysdbAllocatorStatus.values():
         yield allocStatus.name, buildSidFunctionAllocator( allocStatus, rangeInfo )

def populateSrv6SidFunctionAllocators( sysdbVrfStatus, allocatorName=None ):
   functionAllocators = Srv6SidFunctionAllocators()
   if sysdbVrfStatus and sysdbVrfStatus.enabled:
      sidFuncAllocStatus = sysdbVrfStatus.sidFunctionAllocatorStatus
      allocators = populateSidFunctionAllocator( sidFuncAllocStatus, allocatorName )
      functionAllocators.allocators = allocators
   return functionAllocators

class ShowSrv6FunctionAllocatorCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show segment-routing ipv6 sid function allocators
               [ allocator ALLOCATOR ]'''

   data = {
      'segment-routing': matcherSegmentRoutingForShow,
      'ipv6': ipv6MatcherForShow,
      'sid': 'SID information',
      'function': 'SID function information',
      'allocators': 'Show status of configured SID function allocators',
      'allocator': 'Show status for a given SID function allocator',
      'ALLOCATOR': CliMatcher.DynamicNameMatcher(
         getAllocatorNameCompletion,
         helpdesc='SID function allocators keyed by Allocator name' ),
   }

   cliModel = Srv6SidFunctionAllocators

   @staticmethod
   @srv6DisabledAddWarning
   def handler( mode, args ):
      sysdbVrfStatus = gv.srv6Status.defaultVrfStatus
      allocName = args.get( 'ALLOCATOR' )
      funcAlloc = populateSrv6SidFunctionAllocators( sysdbVrfStatus, allocName )
      return funcAlloc

if Srv6LibToggleLib.toggleSrv6CoreEnabled():
   BasicCli.addShowCommandClass( ShowSrv6FunctionAllocatorCmd )

def Plugin( entMan ):
   shmemEm = SharedMem.entityManager( sysdbEm=entMan )
   smi = Smash.mountInfo( 'reader' )

   gv.srv6Config = ConfigMount.mount( entMan, "routing/srv6/config",
                                      "Srv6::Config", "w" )

   gv.srv6Status = LazyMount.mount( entMan, "routing/srv6/status",
                                    "Srv6::Status", "r" )

   gv.srv6Capability = LazyMount.mount( entMan, "routing/hardware/srv6/capability",
                                        "Srv6::Hardware::Capability", "r" )

   gv.srv6Fib = shmemEm.doMount( "srv6/sfib/status",
                                 "Srv6::SfibStatus", smi )
   gv.vrfIdMap = SmashLazyMount.mount( entMan, "vrf/vrfIdMapStatus",
                                       "Vrf::VrfIdMap::Status",
                                       SmashLazyMount.mountInfo( 'reader' ),
                                       autoUnmount=True )

   sidFuncAllocStatus = "Srv6::ClientSidFunctionBlockStatus"
   gv.srv6SidFuncAllocStatus = LazyMount.mount( entMan,
                                                "srv6/sidFunctionAlloc/status",
                                                sidFuncAllocStatus, "r" )
