#!/usr/bin/env python3
# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import Model, Submodel
from CliModel import ( Bool,
                       Dict,
                       Enum,
                       GeneratorDict,
                       Int,
                       List,
                       Str )
from ArnetModel import Ip6Address, IpGenericPrefix
from CliPlugin.Srv6Models import Srv6Sid
from TableOutput import createTable, Format
import operator

LOCATOR_TYPE_MAP = { "classic": "classic",
                     "microSid": "micro-SID" }


LOCATOR_STATE_MAP = {
   "locatorInactiveVrfInoperational": "SRv6 not operational in VRF",
   "locatorInactivePrefixConflict": "conflicting locator prefix",
   "locatorInactivePrefixNotConfigured": "prefix not configured",
   "locatorInactivePrefixLengthExceeded": "incompatible prefix "
   "length and function length",
   "locatorInactiveUsidDomainNotConfigured": "micro segment domain has not "
   "been fully configured",
   "locatorInactiveUsidDomainBlockLengthInvalid": "micro segment domain's block "
   "length is invalid",
   "locatorInactiveUsidDomainBlockConflict": "conflicting micro segment domain "
   "block prefix",
   "locatorInactiveEndUsidNotInRange": "End uSID value is not within GIB range",
   "locatorInactiveEndUsidConflict": "conflicting locator prefix End uSID value",
}

SID_FUNCTION_RANGE_STATIC = 'static'
SID_FUNCTION_RANGE_DYNAMIC = 'dynamic'
SID_FUNCTION_RANGE_UNASSIGNED = 'unassigned'

class Srv6LocatorEntry( Model ):
   state = Enum( values=( "active", "inactive" ), help="Locator state" )
   inactiveReason = Enum( values=list( LOCATOR_STATE_MAP.values() ), optional=True,
                          help="Reason why locator is not active" )
   prefix = IpGenericPrefix( help="Prefix", optional=True )
   locatorType = Enum( values=list( LOCATOR_TYPE_MAP.values() ),
                       help="Locator type" )
   locatorUsidDomain = Str( help="Micro Segment Domain name", optional=True )
   algorithm = Str( help="IGP Algorithm" )
   algorithmId = Int( help="IGP Algorithm ID" )
   functionLength = Int( help="Function Length" )
   blockLength = Int( help="Block length" )
   functionAllocator = Str( help="Function allocator for SID allocation" )
   functionPool = Str( help="Function pool for SID allocation" )

   def render( self ):
      if self.state == 'active':
         print( f"State: {self.state}" )
      else:
         print( f"State: {self.state} ({self.inactiveReason})" )
      print( f"Type: {self.locatorType}" )
      if self.locatorType == 'micro-SID' and self.locatorUsidDomain:
         print( f"Micro-SID domain: {self.locatorUsidDomain}" )
      if self.prefix:
         print( f"Prefix: {self.prefix}" )
      print( f"IGP algorithm: {self.algorithm}" )
      print( f"Block length: {self.blockLength}" )
      print( f"Function length: {self.functionLength}" )
      print( f"Function allocator: {self.functionAllocator}" )
      print( f"SID function value pools: {self.functionPool}" )

class Srv6ConfigVrfModel( Model ):
   enabled = Bool( help="SRv6 enabled" )
   operational = Bool( help="SRv6 active" )
   sourceAddress = Ip6Address( optional=True,
                               help="Source encapsulation address for SRv6 Tunnels" )
   locators = GeneratorDict( keyType=str, valueType=Srv6LocatorEntry,
                             help="Locator details" )
   _numLocators = Int( help="Number of configured locators" )

   def render( self ):
      enabledStr = "enabled" if self.enabled else "disabled"
      operationalStr = "enabled" if self.operational else "disabled"
      print( f"Administrative status: {enabledStr}" )
      print( f"Operational status: {operationalStr}" )
      if self.sourceAddress:
         print( f"Encapsulation source address: {self.sourceAddress}" )
      print( f"Number of configured locators: {self._numLocators}" )
      for name, locator in self.locators:
         print( f"\nLocator {name}" )
         locator.render()

class Srv6ConfigModel( Model ):
   vrfs = Dict( keyType=str, valueType=Srv6ConfigVrfModel,
                help="Per VRF SRv6 configurations" )

   def render( self ):
      for vrf in self.vrfs:
         print( f"SRv6 configuration for VRF {vrf}" )
         self.vrfs[ vrf ].render()

sourceStrMap = { "bgpL3Vpn": "B3", "static": "S" }
behaviorStrMap = { "endDt4": "End.DT4", "endDt6": "End.DT6", "end": "End" }
SID_SOURCE_MAP = { "sfibSourceBgpL3Vpn": "bgpL3Vpn",
                   "sfibSourceStatic": "static" }

class Srv6RouteEntry( Model ):
   sidInformation = Submodel( valueType=Srv6Sid, help="SID for the route" )
   source = Enum( values=list( SID_SOURCE_MAP.values() ),
                  help="Source protocol" )
   locator = Str( help="Locator for the SID" )
   lookupVrf = Str( help="Lookup VRF name for SID behaviors", optional=True )

   def render( self ):
      print( "{}  {}, locator {}".format( sourceStrMap[ self.source ],
                                          self.sidInformation.formattedString(),
                                          self.locator ) )
      leftIndent = " " * 4
      fwdInfoStr = self.buildFwdInfo()
      print( f"{leftIndent}via {fwdInfoStr}" )

   def buildFwdInfo( self ):
      if self.sidInformation.behavior == "endDt4":
         return f"IPv4 lookup, VRF {self.lookupVrf}"
      elif self.sidInformation.behavior in [ "endDt6", "end" ]:
         return f"IPv6 lookup, VRF {self.lookupVrf}"
      else:
         return ""

class Srv6RoutesModel( Model ):
   routes = Dict( keyType=str, valueType=Srv6RouteEntry,
                  help="SRv6 Routes keyed by SID" )

   def render( self ):
      print( "VRF default" )
      print( "Source Codes: B3 - BGP L3 VPN, S - Static" )
      print( "End Behaviors: End - Endpoint" )
      print( "               End.DT4 (6) - Endpoint with decapsulation and "
             "specific IPv4 (6) table lookup\n" )
      sortedRoutes = sorted( self.routes )
      for routeKey in sortedRoutes:
         self.routes[ routeKey ].render()

supportStrMap = { True: "supported", False: "unsupported" }

class Srv6EndBehaviorsModel( Model ):
   end = Bool( help='Supports End behavior' )
   endX = Bool( help='Supports End.X behavior' )
   endDx4 = Bool( help='Supports End.DX4 behavior' )
   endDx6 = Bool( help='Supports End.DX6 behavior' )
   endDt4 = Bool( help='Supports End.DT4 behavior' )
   endDt6 = Bool( help='Supports End.DT6 behavior' )

   def render( self ):
      print( "Supported end behaviors" )
      print( f"End: {supportStrMap[ self.end ]}" )
      print( f"End.X: {supportStrMap[ self.endX ]}" )
      print( f"End.DX4: {supportStrMap[ self.endDx4 ]}" )
      print( f"End.DX6: {supportStrMap[ self.endDx6 ]}" )
      print( f"End.DT4: {supportStrMap[ self.endDt4 ]}" )
      print( f"End.DT6: {supportStrMap[ self.endDt6 ]}\n" )

class Srv6EndFlavorsModel( Model ):
   psp = Bool( help='Supports Penultimate Segment Pop (PSP) of the SRH flavor' )
   usp = Bool( help='Supports Ultimate Segment Pop (USP) of the SRH flavor' )
   usd = Bool( help='Supports Ultimate Segment Decapsulation (USD) flavor' )
   nextCsid = Bool( help='Supports Next-CSID flavor' )

   def render( self ):
      print( "Supported end behavior flavors" )
      print( f"PSP: {supportStrMap[ self.psp ]}" )
      print( f"USP: {supportStrMap[ self.usp ]}" )
      print( f"USD: {supportStrMap[ self.usd ]}" )
      print( f"Next-CSID: {supportStrMap[ self.nextCsid ]}\n" )

class Srv6HeadendBehaviorsModel( Model ):
   hEncaps = Bool( help='Supports H.Encaps headend behavior' )
   hEncapsRed = Bool( help='Supports H.Encaps with Reduced Encapsulation' )
   hInsert = Bool( help='Supports H.Insert headend behavior' )
   hInsertRed = Bool( help='Supports H.Insert with Reduced Encapsulation'
                      ' headend behavior' )

   def render( self ):
      print( "Supported headend behaviors" )
      print( f"H.Encaps: {supportStrMap[ self.hEncaps ]}" )
      print( f"H.Encaps.Red: {supportStrMap[ self.hEncapsRed ]}" )
      print( f"H.Insert: {supportStrMap[ self.hInsert ]}" )
      print( f"H.Insert.Red: {supportStrMap[ self.hInsertRed ]}\n" )

class Srv6MsdModel( Model ):
   maxSegmentsLeft = Int( help='Maximum value of the SegmentsLeft field in'
                          ' the SRH of a received packet before performing'
                          ' the end behavior' )
   maxSidsPopped = Int( help='Maximum number of SIDs in the SRH that can be '
                        ' popped' )
   maxSidsPushed = Int( help='Maximum number of SIDs that can be pushed as part of'
                        ' the H.Encaps* behaviors' )
   maxSidsDecapsulated = Int( help='Maximum number of SIDs in an SRH when'
                              ' applying End.D* behaviors' )

   def render( self ):
      print( "Supported MSD" )
      print( f"Maximum segments left: {self.maxSegmentsLeft}" )
      print( f"Maximum SIDs popped: {self.maxSidsPopped}" )
      print( f"Maximum SIDs pushed: {self.maxSidsPushed}" )
      print( f"Maximum SIDs decapsulated: {self.maxSidsDecapsulated}" )

class Srv6CapabilitiesModel( Model ):
   """ Models node's SRv6 capabilities """
   supportedEndBehaviors = Submodel( optional=True,
                                     valueType=Srv6EndBehaviorsModel,
                                     help='Supported end behaviors' )
   supportedEndBehaviorFlavors = Submodel( optional=True,
                                           valueType=Srv6EndFlavorsModel,
                                           help='Supported end behavior flavors' )
   supportedHeadendBehaviors = Submodel( optional=True,
                                         valueType=Srv6HeadendBehaviorsModel,
                                         help='Supported headend behaviors' )
   supportedMsd = Submodel( optional=True,
                            valueType=Srv6MsdModel,
                            help='Supported maximum segment depth' )
   _srv6Supported = Bool( help='SRv6 supported' )

   def render( self ):
      if self._srv6Supported:
         self.supportedEndBehaviors.render()
         self.supportedEndBehaviorFlavors.render()
         self.supportedHeadendBehaviors.render()
         self.supportedMsd.render()

class Srv6SidFunctionRange( Model ):
   blockStart = Int( help='The first function in the range' )
   blockSize = Int( help='The number of functions in the range' )
   protocol = Str( optional=True, help='The protocol which uses the range' )
   rangeType = Enum( help='Purpose of the range',
                     values=( SID_FUNCTION_RANGE_STATIC,
                              SID_FUNCTION_RANGE_DYNAMIC,
                              SID_FUNCTION_RANGE_UNASSIGNED ) )

class Srv6SidFunctionAllocator( Model ):
   functionLength = Int( optional=True, help='Length of SID function in bits' )
   locators = List( valueType=str,
                    help='Names of locators using the allocator' )
   functionRanges = List( valueType=Srv6SidFunctionRange,
                          help='List of all function ranges' )

   def render( self ):
      print( "Function length:", self.functionLength )
      print( "Locator users:", ", ".join( sorted( self.locators ) ) )

      headings = ( ( "Start", "rh" ), ( "End", "rh" ),
                   ( "Size", "rh" ), ( "Usage", "lh" ) )
      t = createTable( headings )
      fmtL = Format( justify='left' )
      fmtL.padLimitIs( True )
      fmtR = Format( justify='right' )
      fmtR.padLimitIs( True )
      t.formatColumns( fmtR, fmtR, fmtR, fmtL )

      sortKey = operator.attrgetter( 'blockStart' )
      for functionRange in sorted( self.functionRanges, key=sortKey ):
         usage = functionRange.protocol
         if functionRange.rangeType == SID_FUNCTION_RANGE_DYNAMIC:
            usage = '%s (dynamic)' % ( usage if usage else 'free' )
         elif functionRange.rangeType == SID_FUNCTION_RANGE_UNASSIGNED:
            usage = functionRange.rangeType
         elif functionRange.rangeType == SID_FUNCTION_RANGE_STATIC:
            usage = 'static'
         t.newRow( functionRange.blockStart,
                   functionRange.blockStart + functionRange.blockSize - 1,
                   functionRange.blockSize, usage )
      print( t.output() )

class Srv6SidFunctionAllocators( Model ):
   allocators = GeneratorDict( keyType=str,
                               valueType=Srv6SidFunctionAllocator,
                               help='SID function allocators keyed by name' )

   def render( self ):
      print( "SRv6 SID function allocators for VRF default\n" )
      for name, allocator in self.allocators:
         print( "Allocator", name )
         allocator.render()
