# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import LazyMount
import BasicCli
import CliCommand
import CliParser
import ConfigMount
import Tracing
from CliToken.Ip import ipMatcherForConfigIf
from CliPlugin import IraVrfCli
from CliPlugin.VrfCli import VrfExprFactory
from CliPlugin import IntfCli
from CliPlugin.UpnpConfigCli import (
      UpnpConfigMode,
      UpnpDependentBase,
      UpnpCli )
from CliMode.SsdpMode import SsdpMode, SsdpVrfMode

__defaultTraceHandle__ = Tracing.Handle( 'Ssdp' )
t0 = Tracing.trace0

ssdpConfig = None
l3IntfConfigDir = None

class SsdpIntfModelet( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      return mode.intf.routingSupported()

IntfCli.IntfConfigMode.addModelet( SsdpIntfModelet )

# -------------------------------------------------------------------------------
class SsdpIntf( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      del ssdpConfig.interface[ self.intf_.name ]

# ----------------------------------------------------------------
# [no|default] ip ssdp
# ----------------------------------------------------------------
class SsdpIntfCmd( CliCommand.CliCommandClass ):
   syntax = 'ip ssdp'
   noOrDefaultSyntax = syntax
   data = {
      'ip': ipMatcherForConfigIf,
      'ssdp': 'Enable SSDP service',
   }

   @staticmethod
   def handler( mode, args ):
      ssdpConfig.interface.add( mode.intf.name )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      del ssdpConfig.interface[ mode.intf.name ]

SsdpIntfModelet.addCommandClass( SsdpIntfCmd )

# -----------------------------------------------------------------------------------
# Ssdp VRF sub mode
# disabled
# -----------------------------------------------------------------------------------
class SsdpVrfConfigMode( SsdpVrfMode, BasicCli.ConfigModeBase ):
   name = "SSDP VRF configuration"

   def __init__( self, parent, session, vrfName ):
      cfg = ssdpConfig.vrfConfig.newMember( vrfName )
      cfg.serverState = "enabled"
      SsdpVrfMode.__init__( self, vrfName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class VrfDisableCmd( CliCommand.CliCommandClass ):
   syntax = 'disabled'
   noOrDefaultSyntax = syntax
   data = {
      'disabled': 'Disable for the VRF',
   }

   @staticmethod
   def handler( mode, args ):
      cfg = ssdpConfig.vrfConfig[ mode.vrfName ]
      cfg.serverState = "disabled"

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cfg = ssdpConfig.vrfConfig[ mode.vrfName ]
      cfg.serverState = "enabled"

SsdpVrfConfigMode.addCommandClass( VrfDisableCmd )

# -----------------------------------------------------------------------------------
# [ no | default ] ssdp
# -----------------------------------------------------------------------------------
class SsdpConfigMode( SsdpMode, BasicCli.ConfigModeBase ):
   name = "SSDP configuration"

   def __init__( self, parent, session ):
      SsdpMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

# -----------------------------------------------------------------------------------
# [ no | default ] vrf VRF
# -----------------------------------------------------------------------------------
class VrfModeCmd( CliCommand.CliCommandClass ):
   syntax = 'VRF'
   noOrDefaultSyntax = syntax
   data = {
            'VRF': VrfExprFactory( helpdesc='Enter VRF sub-mode',
                                   inclDefaultVrf=True ),
          }

   @staticmethod
   def handler( mode, args ):
      vrfName = args[ 'VRF' ]
      childMode = mode.childMode( SsdpVrfConfigMode, vrfName=vrfName )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vrfName = args[ 'VRF' ]
      del ssdpConfig.vrfConfig[ vrfName ]

SsdpConfigMode.addCommandClass( VrfModeCmd )

class SsdpCmd( CliCommand.CliCommandClass ):
   syntax = 'ssdp'
   noOrDefaultSyntax = syntax
   data = {
      'ssdp': 'Configure SSDP server'
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( SsdpConfigMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      SsdpCmd.clearConfig_()

   @staticmethod
   def clearConfig_():
      ssdpConfig.vrfConfig.clear()

UpnpConfigMode.addCommandClass( SsdpCmd )

class SsdpConfigCleaner( UpnpDependentBase ):
   def clearConfig( self ):
      SsdpCmd.clearConfig_()

UpnpCli.registerDependentClass( SsdpConfigCleaner, priority=10 )

def ssdpVrfCleanup( vrfName ):
   del ssdpConfig.vrfConfig[ vrfName ]

def Plugin( entityManager ):
   global ssdpConfig
   global l3IntfConfigDir
   ssdpConfig = ConfigMount.mount( entityManager,
         'upnp/ssdp/config', 'Ssdp::Config', 'w' )
   l3IntfConfigDir = LazyMount.mount( entityManager, 'l3/intf/config',
                                      'L3::Intf::ConfigDir', 'r' )

   IraVrfCli.deletedVrfHook.addExtension( ssdpVrfCleanup )
   IntfCli.Intf.registerDependentClass( SsdpIntf, priority=32 )
