# Copyright (c) 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os

import BasicCli, HostnameCli
import CliCommand
import CliMatcher
import CliToken.Reset
import CliToken
from CliPlugin import AclCli
from CliPlugin import AclCliModel
from CliPlugin import ConfigMgmtMode
from CliPlugin.VrfCli import VrfExprFactory
import ShowCommand
import CommonGuards
import SshCertLib
import SysMgrLib
import Tac
from Url import UrlMatcher
import LazyMount
import ConfigMount
from SshAlgorithms import HOSTKEYS, SUPPORTED_HOSTKEYS

# ------------------------------
# Ssh config commands
#
# From config mode
#    management ssh    - enter ssh config mode
#    idle-timeout x - set the idle session timeout to x minutes
#    authentication mode [ password | keyboard-interactive ]
#          - make sshd enter specified mode, mutually exclusive with the
#            authentication protocol command and deprecated in favor of it
#    authentication protocol { password | keyboard-interactive | public-key }
#          - enables the specified methods of authentication
#    ciphers SSH_CIPHER1 ... SSH_CIPHERN - set SSH exclusive cipher list
#    [ no | default ] ciphers - clear filter and use all supported ciphers
#    [ no | default ] authentication - return sshd to keyboard-interactive mode
#    [ no | default ] shutdown - disables or enables sshd
#    [ no | default ] qos dscp <0-63> - set dscp value in IP header
#    vrf VRF - enter ssh vrf config mode
#
# From ssh vrf config mode
#    [ no | default ] shutdown - disables or enables sshd in vrf
#
#    For example:
#    management ssh
#       A
#       vrf x
#          B
#
#    A\B             B default             B shutdown            B no shutdown
#    A default       enabled\enabled       enabled\disabled      enabled\enabled
#    A shutdown      disabled\disabled     disabled\disabled     disabled\enabled
#    A no shutdown   enabled\enabled       enabled\disabled      enabled\enabled

HOSTKEY_ALGORITHM_HIDDEN_ALIASES = {
   "ecdsa": "Elliptic Curve Digital Signature Algorithm NIST-P521",
}

SshConstants = Tac.Type( "Mgmt::Ssh::Constants" )

aclStatus = None
aclCheckpoint = None
aclCpConfig = None

sshShowKwMatcher = CliMatcher.KeywordMatcher( 'ssh',
      helpdesc='Show SSH status and statistics' )
showKeyKwMatcher = CliMatcher.KeywordMatcher( 'key',
      alternates=[ 'hostkey' ],
      helpdesc='Show sshd key information' )
trustedCaMatcher = CliMatcher.DynamicNameMatcher(
      lambda mode: _listdir( SshConstants.caKeysDirPath() ),
      pattern=r'[A-Za-z0-9_:{}\.\[\]-]+',
      helpdesc='CA public key filename',
      helpname='WORD' )
hostCertMatcher = CliMatcher.DynamicNameMatcher(
      lambda mode: _listdir( SshConstants.hostCertsDirPath() ),
      pattern=r'[A-Za-z0-9_:{}\.\[\]-]+',
      helpdesc='Switch\'s SSH hostkey cert filename',
      helpname='WORD' )
revokeListMatcher = CliMatcher.DynamicNameMatcher(
      lambda mode: _listdir( SshConstants.revokeListsDirPath() ),
      pattern=r'[A-Za-z0-9_:{}\.\[\]-]+',
      helpdesc='Revoked SSH user keys filename',
      helpname='WORD' )

def _listdir( path ):
   try:
      return os.listdir( path )
   except OSError:
      return []

class HostKeyAlgoExpression( CliCommand.CliExpression ):
   expression = 'KEYALGO | hidden_alias'
   data = { 'KEYALGO': CliCommand.Node(
               matcher=CliMatcher.EnumMatcher( { kw: helpstr
                                                 for kw, helpstr in HOSTKEYS.items()
                                                 if kw in SUPPORTED_HOSTKEYS } ),
               storeSharedResult=True ),
            'hidden_alias': CliCommand.Node(
               matcher=CliMatcher.EnumMatcher( HOSTKEY_ALGORITHM_HIDDEN_ALIASES ),
               hidden=True,
               alias='KEYALGO',
               storeSharedResult=True ) }

# Format strings for printing helpstrings in the CLI; this must be complete, and
# must take exactly 1 %s format arg
keyParamHelpdesc = {
   "rsa": {
      "keysize": ( "Size (bits) of the RSA key", "RSA with %s-bit key" ),
   },
}

def getKeyAlgoFromContext( context ):
   keyAlgo = None
   if "KEYALGO" in context.sharedResult:
      keyAlgo = context.sharedResult[ "KEYALGO" ]
   elif "FILENAME" in context.sharedResult:
      keyAlgo = SshCertLib.getAlgoFromKeyPath(
         context.sharedResult[ "FILENAME" ].pathname )
   return keyAlgo

def hostKeyParamNameKeywords( mode, context ):
   keyAlgo = getKeyAlgoFromContext( context )
   if not keyAlgo or keyAlgo not in SysMgrLib.keyTypeToLegalParams:
      return {}
   # Infrastructure sorts the helpstrings, so no need to sort explicitly
   legalKeyParams = SysMgrLib.keyTypeToLegalParams[ keyAlgo ].keys()
   # keyParamHelpdesc must contain entries for all legalKeyParams
   return { k: keyParamHelpdesc[ keyAlgo ][ k ][ 0 ] for k in legalKeyParams }

def hostKeyParamValueKeywords( mode, context ):
   keyAlgo = getKeyAlgoFromContext( context )
   if not keyAlgo or keyAlgo not in SysMgrLib.keyTypeToLegalParams:
      return {}
   legalKeyParams = SysMgrLib.keyTypeToLegalParams[ keyAlgo ]
   paramNames = context.sharedResult.get( 'KEY_PARAM_NAME' )
   # Since the parser allows multiple set of name - value pairs in key params,
   # we're only looking at the last entry here
   if not paramNames or paramNames[ -1 ] not in legalKeyParams:
      return {}
   # key parameters should not be repeated
   if len( set( paramNames ) ) != len( paramNames ):
      return {}
   paramName = paramNames[ -1 ]
   legalKeyParamValues = legalKeyParams[ paramName ]
   keyParamHelpdescFmt = keyParamHelpdesc[ keyAlgo ][ paramName ][ 1 ]
   return { k: keyParamHelpdescFmt % k for k in legalKeyParamValues }

class HostKeyParamExpression( CliCommand.CliExpression ):
   expression = 'KEY_PARAM_NAME KEY_PARAM_VALUE'
   data = {
      'KEY_PARAM_NAME': CliCommand.Node(
         matcher=CliMatcher.DynamicKeywordMatcher(
            keywordsFn=hostKeyParamNameKeywords,
            passContext=True,
         ),
         storeSharedResult=True,
      ),
      'KEY_PARAM_VALUE': CliCommand.Node(
         matcher=CliMatcher.DynamicKeywordMatcher(
            keywordsFn=hostKeyParamValueKeywords,
            passContext=True,
         )
      ),
   }

#-----------------------------------------------------------------------------------
# show management ssh known-hosts [ vrf VRF ] [ HOST ]
#-----------------------------------------------------------------------------------
vrfExprFactory = VrfExprFactory(
      helpdesc='Use a specific VRF' )

class ShowManagementSshKnownHosts( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show management ssh known-hosts [ VRF ] [ HOST ]' )
   data = {
            'management': ConfigMgmtMode.managementShowKwMatcher,
            'ssh': sshShowKwMatcher,
            'known-hosts': 'SSH Known Hosts public keys',
            'VRF': vrfExprFactory,
            'HOST': HostnameCli.IpAddrOrHostnameMatcher( ipv6=True ),
         }
   privileged = True
   handler = "SshHandler.showKnownHosts"

BasicCli.addShowCommandClass( ShowManagementSshKnownHosts )

#-------------------------------------------------------------------------------
# show management ssh key ( KEYALGO | FILENAME ) public
#-------------------------------------------------------------------------------
class ShowManagementSshHostKey( ShowCommand.ShowCliCommandClass ):
   syntax = 'show management ssh key ( KEYALGOEXPR | FILENAME ) public'
   data = {
            'management': ConfigMgmtMode.managementShowKwMatcher,
            'ssh': sshShowKwMatcher,
            'key': showKeyKwMatcher,
            'KEYALGOEXPR': HostKeyAlgoExpression,
            'FILENAME': UrlMatcher( fsFunc=lambda fs: fs.scheme == 'ssh-key:' and
               fs.supportsWrite(), helpdesc='SSH Private key file name',
               acceptSimpleFile=False ),
            'public': 'Show public key'
         }
   cliModel = "SshModels.SshHostKey"
   handler = "SshHandler.showPublicHostKey"

BasicCli.addShowCommandClass( ShowManagementSshHostKey )

#-------------------------------------------------------------------------------
# show management ssh trusted-ca key public [ CAKEY ]
#-------------------------------------------------------------------------------
class ShowTrustedCaKey( ShowCommand.ShowCliCommandClass ):
   syntax = 'show management ssh trusted-ca key public [ CAKEY ]'
   data = { 'management': ConfigMgmtMode.managementShowKwMatcher,
            'ssh': sshShowKwMatcher,
            'trusted-ca': 'Show configured trusted CA information',
            'key': 'Show trusted CA keys',
            'public': 'Show trusted CA public keys',
            'CAKEY': trustedCaMatcher,
          }
   cliModel = "SshModels.TrustedCaKeys"
   handler = "SshHandler.showTrustedCaKeyHandler"
   privileged = True

BasicCli.addShowCommandClass( ShowTrustedCaKey )

#-------------------------------------------------------------------------------
# show management ssh key server cert [ HOSTCERT ]
#-------------------------------------------------------------------------------
class ShowHostkeyCert( ShowCommand.ShowCliCommandClass ):
   syntax = 'show management ssh key server cert [ HOSTCERT ]'
   data = { 'management': ConfigMgmtMode.managementShowKwMatcher,
            'ssh': sshShowKwMatcher,
            'key': showKeyKwMatcher,
            'server': 'Show sshd key information',
            'cert': 'Show sshd key certificate information',
            'HOSTCERT': hostCertMatcher,
          }
   cliModel = "SshModels.HostCertificates"
   handler = "SshHandler.showHostkeyCertHandler"

BasicCli.addShowCommandClass( ShowHostkeyCert )

#-------------------------------------------------------------------------------
# show management ssh user-keys revoke-list [ REVOKELIST ]
#-------------------------------------------------------------------------------
class ShowRevokeList( ShowCommand.ShowCliCommandClass ):
   syntax = 'show management ssh user-keys revoke-list [ REVOKELIST ]'
   data = { 'management': ConfigMgmtMode.managementShowKwMatcher,
            'ssh': sshShowKwMatcher,
            'user-keys': 'Show information about configured user keys',
            'revoke-list': 'Show revoked keys',
            'REVOKELIST': revokeListMatcher,
          }
   cliModel = "SshModels.RevokedKeys"
   handler = "SshHandler.showRevokeListHandler"
   privileged = True

BasicCli.addShowCommandClass( ShowRevokeList )

#-------------------------------------------------------------------------------
# show management ssh user-keys authorized-keys [ USER ]
#-------------------------------------------------------------------------------
class ShowAuthorizedKeys( ShowCommand.ShowCliCommandClass ):
   syntax = 'show management ssh user-keys authorized-keys [ USER ]'
   data = { 'management': ConfigMgmtMode.managementShowKwMatcher,
            'ssh': sshShowKwMatcher,
            'user-keys': 'Show information about configured user keys',
            'authorized-keys': 'Show SSH authorized keys',
            'USER': CliMatcher.PatternMatcher( pattern='[A-Za-z0-9_-]+',
               helpdesc='SSH user name', helpname='WORD' )
          }
   cliModel = "SshModels.SshUsers"
   handler = "SshHandler.showAuthorizedKeysHandler"

BasicCli.addShowCommandClass( ShowAuthorizedKeys )

#-------------------------------------------------------------------------------
# show management ssh authorized-principals [ USER ]
#-------------------------------------------------------------------------------
class ShowAuthorizedPrincipals( ShowCommand.ShowCliCommandClass ):
   syntax = 'show management ssh authorized-principals [ USER ]'
   data = { 'management': ConfigMgmtMode.managementShowKwMatcher,
            'ssh': sshShowKwMatcher,
            'authorized-principals': 'Show SSH authorized principals',
            'USER': CliMatcher.PatternMatcher( pattern='[A-Za-z0-9_-]+',
               helpdesc='SSH user name', helpname='WORD' )
          }
   cliModel = "SshModels.SshUsers"
   handler = "SshHandler.showAuthorizedPrincipalsHandler"

BasicCli.addShowCommandClass( ShowAuthorizedPrincipals )

#-------------------------------------------------------------------------------
# show management ssh [ vrf VRF ]
#-------------------------------------------------------------------------------
class ShowManagementSsh( ShowCommand.ShowCliCommandClass ):
   syntax = 'show management ssh [ VRF ]'
   data = {
            'management': ConfigMgmtMode.managementShowKwMatcher,
            'ssh': sshShowKwMatcher,
            'VRF': vrfExprFactory,
         }
   handler = "SshHandler.showSshStatus"

BasicCli.addShowCommandClass( ShowManagementSsh )

#-------------------------------------------------------------------------------
# show management ssh [ip|ipv6] access-list [ACL] [summary]
#-------------------------------------------------------------------------------

# Lazify these ACL commands once AclCli is lazified
def showSshAcl( mode, args ):
   aclType = 'ip' if 'ip' in args else 'ipv6'
   name = args[ '<aclNameExpr>' ]
   return AclCli.showServiceAcl( mode,
                                 aclCpConfig,
                                 aclStatus,
                                 aclCheckpoint,
                                 aclType,
                                 name,
                                 supressVrf=False,
                                 serviceName='ssh' )

class ShowManagementSshAcl( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show management ssh'
              '('
              ' ( ip access-list [ ACL4 ] ) | '
              ' ( ipv6 access-list [ ACL6 ] ) '
              ')' )
   data = {
            'management': ConfigMgmtMode.managementShowKwMatcher,
            'ssh': sshShowKwMatcher,
            'ip': AclCli.ipKwForShowServiceAcl,
            'ipv6': AclCli.ipv6KwForShowServiceAcl,
            'access-list': AclCli.accessListKwMatcherForServiceAcl,
            'ACL4': AclCli.ipAclNameExpression,
            'ACL6': AclCli.ip6AclNameExpression
          }

   cliModel = AclCliModel.AllAclList
   privileged = True
   handler = showSshAcl

BasicCli.addShowCommandClass( ShowManagementSshAcl )

#----------------------------------------------------------------
# "clear management ssh counters ( ip | ipv6 ) access-list"
#----------------------------------------------------------------
class ClearIpAclCounters( CliCommand.CliCommandClass ):
   syntax = 'clear management ssh counters ( ip | ipv6 ) access-list'
   data = { 'clear': CliToken.Clear.clearKwNode,
            'management': ConfigMgmtMode.managementClearKwMatcher,
            'ssh': 'Clear SSH statistics',
            'counters': 'Connection Counters',
            'ip': AclCli.ipKwForClearServiceAclMatcher,
            'ipv6': AclCli.ipv6KwMatcherForClearServiceAcl,
            'access-list': AclCli.accessListKwMatcherForServiceAcl }

   @staticmethod
   def handler( mode, args ):
      if 'ip' in args:
         aclType = 'ip'
      elif 'ipv6' in args:
         aclType = 'ipv6'
      else:
         assert False
      AclCli.clearServiceAclCounters( mode, aclStatus, aclCheckpoint, aclType )

BasicCli.EnableMode.addCommandClass( ClearIpAclCounters )

#-----------------------------------------------------------------------------------
# [ no | default ] reset ssh key ( KEYALGOEXPR | FILENAME )
#-----------------------------------------------------------------------------------
class ResetSshHostKey( CliCommand.CliCommandClass ):
   syntax = 'reset ssh key ( KEYALGOEXPR | FILENAME ) [ { KEYPARAMEXPR } ]'
   data = {
            'reset': CliToken.Reset.resetKwApi,
            'ssh': 'Configure ssh',
            'key': CliCommand.Node(
                        matcher=CliMatcher.KeywordMatcher( 'key',
                           alternates=[ 'hostkey' ],
                           helpdesc='Regenerate sshd keys' ),
                        guard=CommonGuards.ssoStandbyGuard ),
            'KEYALGOEXPR': HostKeyAlgoExpression,
            'FILENAME': CliCommand.Node(
                  matcher=UrlMatcher(
                     fsFunc=lambda fs: fs.scheme == 'ssh-key:' and
                     fs.supportsWrite(), helpdesc='SSH Private key file name',
                     acceptSimpleFile=False ),
                  storeSharedResult=True ),
            'KEYPARAMEXPR': HostKeyParamExpression,
          }
   handler = "SshHandler.doResetHostKey"

BasicCli.EnableMode.addCommandClass( ResetSshHostKey )

#-----------------------------------------------------------------------------------
# [ no | default ] management ssh
#-----------------------------------------------------------------------------------
class ManagmentSsh( CliCommand.CliCommandClass ):
   syntax = 'management ssh'
   noOrDefaultSyntax = 'management ssh'
   data = {
            'management': ConfigMgmtMode.managementKwMatcher,
            'ssh': 'Configure ssh'
          }
   handler = "SshHandler.gotoSshConfigMode"
   noOrDefaultHandler = "SshHandler.defaultSshConfig"

BasicCli.GlobalConfigMode.addCommandClass( ManagmentSsh )

def Plugin( entityManager ):
   global aclCheckpoint
   global aclStatus
   global aclCpConfig
   aclCpConfig = ConfigMount.mount( entityManager, "acl/cpconfig/cli",
                                    "Acl::Input::CpConfig", "w" )
   aclStatus = LazyMount.mount( entityManager, "acl/status/all",
                                "Acl::Status", "r" )
   aclCheckpoint = LazyMount.mount( entityManager, "acl/checkpoint",
                                   "Acl::CheckpointStatus", "w" )
