#!/usr/bin/env python3
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Ark
import BasicCli
import CliModel
import LazyMount
import Cell
import ShowCommand
import CliPlugin.TechSupportCli
from StageMgr import defaultStageInstance

bootStagesLog = None

#-----------------------------------------------------------
# The 'show boot stages log' command
#-----------------------------------------------------------
class BootStagesLogModel( CliModel.Model ):
   class BootStagesLogEntryModel( CliModel.Model ):
      timestamp = CliModel.Float( help="Timestamp when the log event occured" )
      deltaTime = CliModel.Float( help="Time of the log relative to stage start" )
      msg = CliModel.Str( help="Log message" )
   logs = CliModel.List( valueType=BootStagesLogEntryModel,
                         help="List of boot stage logs",
                         optional=True )

   def render( self ):
      # pylint: disable-next=consider-using-f-string
      print( "%-19s %-11s %s" % ( "Timestamp", "Delta Begin", "Msg" ) )
      for log in self.logs:
         timestamp = Ark.timestampToStr( log.timestamp, relative=False )
         # pylint: disable-next=consider-using-f-string
         print( "%s %010.6f %s" % ( timestamp, log.deltaTime, log.msg ) )

def doShowBootStagesLog( mode, args ):
   logModel = BootStagesLogModel()
   beginTime = 0
   if defaultStageInstance in bootStagesLog.instanceLog:
      logs = bootStagesLog.instanceLog[ defaultStageInstance ]
      for logEntry in logs.log.values():
         if beginTime == 0:
            beginTime = logEntry.timestamp
         logEntryModel = BootStagesLogModel.BootStagesLogEntryModel()
         logEntryModel.timestamp = logEntry.timestamp
         logEntryModel.deltaTime = logEntry.timestamp - beginTime
         logEntryModel.msg = logEntry.msg
         logModel.logs.append( logEntryModel )
   return logModel

class ShowBootStagesLog( ShowCommand.ShowCliCommandClass ):
   syntax = 'show boot stages log'
   data = {
            'boot': 'Boot information',
            'stages': 'Stages information',
            'log': 'Stage progression logs',
          }
   cliModel = BootStagesLogModel
   handler = doShowBootStagesLog

BasicCli.addShowCommandClass( ShowBootStagesLog )

CliPlugin.TechSupportCli.registerShowTechSupportCmd(
   '2015-02-06 09:41:58',
   cmds=[ 'show boot stages log' ] )

#--------------------------------------------------
# Plugin method - Mount the objects we need from Sysdb
#--------------------------------------------------
def Plugin( entityManager ):
   global bootStagesLog
   bootStagesLog = LazyMount.mount( entityManager, Cell.path( 'stage/boot/log' ),
                                   'Stage::Log', 'r' )
