# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os
import shutil
import BasicCliUtil
from CliPlugin import BlackBox
from CliPlugin import FruCli
from CliPlugin import FruModel
from CliPlugin import ReloadCli
from CliPlugin import TechSupportCli
from CliPlugin import StorageDevicesModels
from StorageDevices import FileUtils
import EmmcCliLib
import SmartCliLib
import LazyMount
import Cell
import FileUrl
import Url
import SimpleConfigFile
import Tac
import Ark
from ArPyUtils.Threading import availableRam
from FileCliUtil import sizeFmt

# pkgdeps: rpm BlackBox-lib
storageDevicesStatus = None
pushButtonRecoveryConfig = None
SECONDS_IN_DAY = 60 * 60 * 24

# StorageDevShowInvCallback
#     fs --> hardware/entmib/fixedSystem
def StorageDevShowInvCallback ( entMib, model ):

   # This is a list of tuples of the form (device, supervisor)
   deviceList = []

   if entMib.fixedSystem:
      for dev in entMib.fixedSystem.storageDevices.values():
         deviceList.append( ( dev, None ) )

   elif entMib.chassis:
      for slot in sorted( entMib.chassis.cardSlot.values() ):
         card = slot.card
         pos = slot.relPos
         if card:

            # List of Devices of this supervisor in the modular system
            for dev in slot.card.storageDevices.values():
               deviceList.append( ( dev, pos ) )

   mntFlashKey = None
   for storageDeviceMibEnt, _ in deviceList:
      if storageDeviceMibEnt.mount == '/mnt/flash':
         mntFlashKey = storageDeviceMibEnt.modelName.strip()
         break
   suffix = 2

   # Code section that populates the StorageDevices, and emmcFlashDevices Dicts
   for storageDeviceMibEnt, supervisor in deviceList:
      modelName = storageDeviceMibEnt.modelName.strip()
      serialNum = storageDeviceMibEnt.serialNum.strip()
      firmwareRev = storageDeviceMibEnt.firmwareRev.strip()
      mount = storageDeviceMibEnt.mount
      storageSize = storageDeviceMibEnt.sizeGB
      storageType = FruModel.storageTypes.get(
         storageDeviceMibEnt.storageType.upper(), 'unknown' )

      # Code Section that populates the storageSerials Dict
      model.storageSerials[ serialNum ] = FruModel.StorageDevice(
               model=modelName,
               mount=mount,
               storageType=storageType,
               serialNum=serialNum,
               firmwareRev=firmwareRev,
               storageSize=storageSize,
               supervisor=supervisor )



def showSystemHealthStorage( mode, args ):
   storageHealth = StorageDevicesModels.Devices()

   def getDevice( name ):
      device = storageHealth.devices.get( name )
      if device is None:
         device = StorageDevicesModels.StorageHealth()
         storageHealth.devices[ name ] = device
      return device

   def gatherSmartData():
      smartHealths = SmartCliLib.smartHealthStatus( storageDevicesStatus )
      for name, info in smartHealths.items():
         device = getDevice( name )
         health = StorageDevicesModels.SmartHealth()
         health.reallocationsRemaining = info.get( SmartCliLib.REALLOCATIONS )
         health.wearLevel = info.get( SmartCliLib.WEAR_LIFE )
         health.health = info.get( SmartCliLib.HEALTH )

         if not health.isEmpty():
            device.smart = health

   def gatherEmmcData():
      if not storageDevicesStatus.emmcData:
         return
      emmcLifetimes = EmmcCliLib.emmcHealthStatus( storageDevicesStatus )
      for name, info in emmcLifetimes.items():
         device = getDevice( name )
         lifetime = StorageDevicesModels.EmmcLifetime()
         lifetime.slcWearLevel = info.get( EmmcCliLib.LIFETIMEA )
         lifetime.mlcWearLevel = info.get( EmmcCliLib.LIFETIMEB )
         lifetime.reservesLevel = info.get( EmmcCliLib.RESERVES )

         if not lifetime.isEmpty():
            device.emmc = lifetime

   gatherSmartData()
   gatherEmmcData()
   return storageHealth

def resetSystemStorage( mode, args ):
   redundancyStatus_ = mode.session_.entityManager_.redundancyStatus()

   if ( redundancyStatus_.mode == "active" and
        redundancyStatus_.peerMode == "standby" ):
      activeStr = ( "ERROR! Cannot reset system storage\r\n"
                    "on the active supervisor while another\r\n"
                    "supervisor is on standby." )
      mode.addWarning( activeStr )
      return
   if 'secure' in args:
      secureResetSystemStorage( mode, args )
   elif 'rollback' in args:
      rollbackSystemStorage( mode )
   elif 'factory' in args:
      factoryResetSystemStorage( mode )

def resetHelperFunc( requestedReset ):
   resetHelpers = { "independence": CaravanResetHelper,
                    "councilbluffs": CaravanResetHelper,
                    "prairieisland": PrairieResetHelper,
                    "raspberryisland": PrairieResetHelper,
                    "wh3p": Wolfhound3PlusResetHelper,
                  }
   # Don't try to run scd if running as btest
   if 'SIMULATION_VMID' not in os.environ:
      resetHelper = resetHelpers[ Ark.getPlatform() ]()
      if requestedReset == "factoryReset":
         resetHelper.triggerFactoryReset()
      elif requestedReset == "rollback":
         resetHelper.triggerRollback()

def factoryResetSystemStorage( mode ):
   factoryResetWarning = ( "WARNING! This will destroy all\r\n"
                              "data except for files saved \r\n"
                              "in the in the recovery partition.\r\n"
                              "Would you like to proceed? [y/N] " )
   confirm = BasicCliUtil.confirm( mode, prompt=factoryResetWarning,
                                   answerForReturn=False )
   if not confirm:
      return
   resetHelperFunc( "factoryReset" )

def rollbackSystemStorage( mode ):
   ''' rollback consists of writing to the requested restart register'''
   rollbackWarning = ( "WARNING! This will destroy all\r\n"
                       "data except for files saved \r\n"
                       "in the snapshots' directory.\r\n"
                       "Would you like to proceed? [y/N] " )
   confirm = BasicCliUtil.confirm( mode, prompt=rollbackWarning,
                                   answerForReturn=False )
   if not confirm:
      return

   swiPrompt = " Are you sure you want to rollback? [confirm]"
   if not checkSwiWarning( mode, swiPrompt ):
      return

   resetHelperFunc( "rollback" )

def secureResetSystemStorage( mode, args ):
   if "rollback" in args:
      warningStr = ( "! This will destroy all data\r\n"
                     "except for the files in snapshot:,\r\n"
                     "which will be restored to flash:.\r\n"
                     "Device will reboot, and execution\r\n"
                     "may take up to one hour.\r\n"
                     "Proceed? [y/N] " )
   else:
      warningStr = ( "WARNING! This will destroy all\r\n"
                     "data and will NOT be recoverable.\r\n"
                     "Device will reboot into Aboot, and\r\n"
                     "execution may take up to one hour.\r\n"
                     "Would you like to proceed? [y/N] " )
   confirm = BasicCliUtil.confirm( mode, prompt=warningStr, answerForReturn=False )
   if not confirm:
      return

   if "rollback" in args and not checkSecureRollbackAndConfirmed( mode ):
      return

   def devExists( baseDir, devName ):
      return os.path.exists( os.path.join( baseDir, f"{devName}.conf" ) )
   try:
      BlackBox.doEraseBlackbox()
   except ( Tac.SystemCommandError, Tac.Timeout, SystemError ):
      failEraseStr = ( "Failed to erase persistent console logs\n"
                       "Would you like to continue "
                       "the secure erase process?\n" )
      confirm = BasicCliUtil.confirm( mode, prompt=failEraseStr,
                                      answerForReturn=False )
      if not confirm:
         return

   devNames = [ 'crash', 'drive', 'flash' ]
   baseDir = ''
   # FILESYSTEM_ROOT is a temporary directory created during
   # breadth test Cli initialization
   baseDir = os.environ.get( 'FILESYSTEM_ROOT', '/mnt' )
   trigPath = os.path.join( baseDir, 'flash', '.trigResetSysStorage' )
   devStr = ''

   os.mknod( trigPath )

   if "rollback" in args:
      preserveTrigPath = os.path.join( baseDir, 'flash', '.trigPreserveSysStorage' )
      os.mknod( preserveTrigPath )

   devNames = [ dev + ':' for dev in devNames if devExists( baseDir, dev ) ]
   if len( devNames ) == 1:
      devStr = devNames[ 0 ]
   else:
      devStr += ', '.join( devNames[ :-1 ] )
      devStr += f' & {devNames[ -1 ]}'

   mode.addWarning( f"Rebooting to process {devStr}" )
   # Don't try to reboot if running as btest
   if 'A4_CHROOT' not in os.environ:
      ReloadCli.doReload( mode, power=True, now=True )

def checkSecureRollbackAndConfirmed( mode ):
   """
   Check both the swi and the memory warning.
   """
   baseDir = os.environ.get( 'FILESYSTEM_ROOT', '/mnt' )
   rollbackDirPath = os.path.join( baseDir, "flash", FileUtils.snapshotFsLocation )

   swiPrompt = "\nContinue with rollback? [confirm]"
   if not checkSwiWarning( mode, swiPrompt ):
      return False

   def getDirSize( path ):
      if not os.path.exists( path ):
         return 0
      total = 0
      with os.scandir( path ) as it:
         for entry in it:
            if entry.is_file():
               total += entry.stat().st_size
            elif entry.is_dir():
               total += getDirSize( entry.path )
      return total

   # Use 40% of total system memory as the memory threshold of snapshot:
   if 'ROLLBACK_TEST_FREE_SPACE' in os.environ:
      freeSpace = int( os.environ[ "ROLLBACK_TEST_FREE_SPACE" ] )
   else:
      freeSpace = availableRam() * 0.4

   sizeDiff = getDirSize( rollbackDirPath ) - freeSpace
   if sizeDiff > 0:
      memWarning = ( "Insufficient RAM to save from snapshot:. "
                     "The operation may abort after system reboot. "
                     f"Consider freeing { sizeFmt( sizeDiff ) } "
                     "from snapshot:.\n"
                     "Continue with rollback? [confirm]" )
      confirm = BasicCliUtil.confirm( mode, prompt=memWarning,
                                      answerForReturn=False )
      if not confirm:
         return False
   return True

def checkSwiWarning( mode, swiPrompt ):
   # check bootConfig to see if EOS.swi is in user_recovery directory. Change the
   # the resulting scheme to snapshot as it will still point to /mnt/flash
   _, softwareImage = getImagePath( mode, f"{FileUtils.snapshotFsName}/boot-config" )
   scheme = softwareImage[ : softwareImage.find( ":" ) + 1 ]
   if scheme != "":
      softwareImage = softwareImage.replace( scheme, f"{FileUtils.snapshotFsName}" )
   context = Url.Context( *Url.urlArgsFromMode( mode ) )
   imageFile = Url.parseUrl( softwareImage, context )
   swiAbsentWarningMsg = ( f"The boot image ({softwareImage}) is not present."
                           f"{swiPrompt}" )

   if not imageFile.exists():
      confirm = BasicCliUtil.confirm( mode, prompt=swiAbsentWarningMsg,
                                      answerForReturn=False )
      if not confirm:
         return False
   return True

def noOrDefaultSaveStorageSnapshot( mode, args ):
   # remove all files in directory
   baseDir = os.environ.get( 'FILESYSTEM_ROOT', '/mnt' )
   rollbackDirPath = os.path.join( baseDir, "flash", FileUtils.snapshotFsLocation )
   if os.path.exists( rollbackDirPath ):
      shutil.rmtree( rollbackDirPath )
   os.makedirs( rollbackDirPath )
   return baseDir, rollbackDirPath

def saveStorageSnapshot( mode, args ):
   """
      To mitigate against incomplete data while saving snapshots, we save files to a
      tmp directory and flush the data to disk before renaming the tmp directory to
      the actual directory, in the hope that the renaming operation is atomic( or as
      close as we can get).
   """
   baseDir = os.environ.get( 'FILESYSTEM_ROOT', '/mnt' )
   rollbackDirPath = os.path.join( baseDir, "flash", FileUtils.snapshotFsLocation )
   tmpRollbackDirPath = os.path.join( baseDir, "flash",
      FileUtils.tmpSnapshotFsLocation )
   tmpRollbackDirPath2 = os.path.join( baseDir, "flash",
      FileUtils.tmpSnapshotFsLocation2 )
   try:
      bootConfigPath = getBootConfigPath( mode, FileUrl.bootConfigFile )
      # make sure tempRollbackDir is empty
      if os.path.exists( tmpRollbackDirPath ):
         shutil.rmtree( tmpRollbackDirPath )
      os.makedirs( tmpRollbackDirPath )
      filesToSave = []
      # Find the SWI indicated in the bootconfig file
      imageFile, _ = getImagePath( mode, FileUrl.bootConfigFile )
      imageFilePath = imageFile.localFilename()
      if os.path.exists( imageFilePath ):
         filesToSave = [ bootConfigPath, imageFilePath ]
      # if the default profile is chosen add startup, zeroTouch and swixes
      if "default" in args:
         startupConfigPath = getStartupConfigPath( mode )
         if os.path.exists( startupConfigPath ):
            filesToSave.append( startupConfigPath )
         zeroTouchconfigPath = getZeroTouchConfigPath( mode )
         if os.path.exists( zeroTouchconfigPath ):
            filesToSave.append( zeroTouchconfigPath )
         # Copy swixes that are in the hidden extensions directory
         # if swi doesn't exist don't copy extensions
         if os.path.exists( imageFilePath ):
            extensionDir = os.path.join( baseDir, "flash", ".extensions" )
            rollbackExtensionDir = os.path.join( tmpRollbackDirPath, ".extensions" )
            FileUtils.copyTreeMaybePreservePerms( extensionDir, rollbackExtensionDir,
                                                  dirsExistOk=True )

      # copy image and all configs to snapshot directory
      for filePath in filesToSave:
         FileUtils.copyMaybePreservePerms( filePath, tmpRollbackDirPath )
   except Exception as e: # pylint: disable=broad-except
      # clean up the tmp rollback directory
      if os.path.exists( tmpRollbackDirPath ):
         shutil.rmtree( tmpRollbackDirPath )
      if "No space left on device" in e.args[ 1 ]:
         mode.addError( "No space left on device. "
                        "Aborting copying files to snapshots." )
         return
      else:
         raise
   Url.syncfile( tmpRollbackDirPath )
   # Since it is not allowed to rename if the dst directory is not empty, rename
   # .user_recovery to .user_recovery_tmp2 before renaming .user_recovery_tmp
   # to the .user_recovery.
   if os.path.exists( rollbackDirPath ) and os.listdir( rollbackDirPath ):
      if os.path.exists( tmpRollbackDirPath2 ):
         shutil.rmtree( tmpRollbackDirPath2 )
      os.rename( rollbackDirPath, tmpRollbackDirPath2 )
      os.rename( tmpRollbackDirPath, rollbackDirPath )
      shutil.rmtree( tmpRollbackDirPath2 )
   else:
      os.rename( tmpRollbackDirPath, rollbackDirPath )

def handlePushButtonRecoveryConfig( mode, args ):
   pushButtonRecoveryConfig.recoveryEnabled = False

def noOrDefaultHandlePushButtonRecoveryConfig( mode, args ):
   pushButtonRecoveryConfig.recoveryEnabled = True

def getBootConfigPath( mode, bootConfigFile ):
   context = Url.Context( *Url.urlArgsFromMode( mode ) )
   return Url.parseUrl( bootConfigFile, context ).localFilename()

def getStartupConfigPath( mode ):
   url = FileUrl.localStartupConfig( *Url.urlArgsFromMode( mode ) )
   return url.localFilename()

def getZeroTouchConfigPath( self ):
   filesystemRoot = os.environ.get( 'FILESYSTEM_ROOT', '/mnt' )
   zeroTouchConfigPath = os.path.join( filesystemRoot, "flash",
      "zerotouch-config" )
   return zeroTouchConfigPath

def getImagePath( mode, bootConfigFile ):
   """ return the url and the url string of the swi.
      Loosely adapted from Fru/ReloadCli.py """
   bootConfigFilename = getBootConfigPath( mode, bootConfigFile )
   simpleConfigFile = SimpleConfigFile.SimpleConfigFileDict( bootConfigFilename,
                                                             autoSync=True )
   softwareImage = simpleConfigFile.get( 'SWI', '(not set)' )
   context = Url.Context( *Url.urlArgsFromMode( mode ) )
   imageFile = Url.parseUrl( softwareImage, context )
   return imageFile, softwareImage

def _doShowSystemStatisticsStorage( measurementType, extractReadStatFn,
                                    extractWriteStatFn ):
   devicesIOStatRet = StorageDevicesModels.DevicesIOStat()

   if not storageDevicesStatus.iostat:
      return devicesIOStatRet

   for ( serial, stat ) in storageDevicesStatus.iostat.ioStatData.items():
      connectedDisk = storageDevicesStatus.connectedDevices.disk[ serial ]
      if not connectedDisk.fsName():
         continue

      ioStat = StorageDevicesModels.IOStat()
      ioStat.read = extractReadStatFn( stat )
      ioStat.write = extractWriteStatFn( stat )
      ioStat.measurementTimestamp = stat.topMeasurement().timestamp
      ioStat.measurementType = measurementType

      devicesIOStatRet.devices[ connectedDisk.fsName() ] = ioStat

   return devicesIOStatRet

def showSystemStatisticsStorage( mode, args ):
   return _doShowSystemStatisticsStorage( 'boot',
      lambda stat: stat.topMeasurement().bytesRead,
      lambda stat: stat.topMeasurement().bytesWritten )

def showStorageStatisticsRates( mode, args ):
   uptimeInSeconds = float( os.getenv( '__UPTIME_IN_SECONDS', '0' ) )
   if not uptimeInSeconds:
      with open( '/proc/uptime', 'r' ) as f:
         uptimeInSeconds = float( f.read().split()[ 0 ] )

   uptimeInDays = uptimeInSeconds / SECONDS_IN_DAY

   # If the system hasn't been up for a day, simply report the current numbers
   # since boot
   uptimeInDays = max( uptimeInDays, 1 )

   return _doShowSystemStatisticsStorage(
      'dayAverage',
      lambda stat: int( stat.topMeasurement().bytesRead // uptimeInDays ),
      lambda stat: int( stat.topMeasurement().bytesWritten // uptimeInDays )
   )

def showStorageStatisticsLastDay( mode, args ):
   return _doShowSystemStatisticsStorage(
      'lastDay',
      lambda stat: stat.topMeasurement().bytesRead -
                   stat.getMeasurement( SECONDS_IN_DAY ).bytesRead,
      lambda stat: stat.topMeasurement().bytesWritten -
                   stat.getMeasurement( SECONDS_IN_DAY ).bytesWritten
   )

class AttrWrapper:
   def id( self ):
      return None

   def name( self ):
      return None

   def value( self ):
      return None

   def status( self ):
      return "n/a"

   def normalized( self ):
      return None

   def worst( self ):
      return None

   def threshold( self ):
      return None

   def raw( self ):
      return None

class SmartAttrWrapper( AttrWrapper ):
   def __init__( self, attr ):
      self.attr = attr

   def id( self ):
      return self.attr.id

   def name( self ):
      return self.attr.name

   def value( self ):
      return self.attr.standard

   def status( self ):
      return "ok" if self.attr.isPassing() else "failed"

   def normalized( self ):
      return self.attr.value

   def worst( self ):
      return self.attr.worst

   def threshold( self ):
      return self.attr.threshold

   def raw( self ):
      return self.attr.raw

class SmartAttrKnownWrapper( SmartAttrWrapper ):
   def __init__( self, data, attrName ):
      optAttr = getattr( data, attrName )
      super().__init__( optAttr.value )
      self.present = optAttr.present

class SmartAttrUnknownWrapper( SmartAttrWrapper ):
   def __init__( self, data, id_ ):
      super().__init__( data.unknownAttr[ id_ ] )

class NvmeAttrWrapper( AttrWrapper ):
   def __init__( self, data, attrName ):
      self.attr = getattr( data, attrName )

   def name( self ):
      return self.attr.name

   def value( self ):
      return self.attr.value

   def raw( self ):
      return self.attr.value

class AttrTemplate:
   def __init__( self, index, name=None, unit="-" ):
      self.index = index
      self.name = name
      self.unit = unit

   @staticmethod
   def wrapper( data, index ):
      raise NotImplementedError()

   def extractAttr( self, data ):
      return self.wrapper( data, self.index )

   def makeSummaryAttr( self, data ):
      tacAttr = self.extractAttr( data )
      attr = StorageDevicesModels.SmartAttr()
      attr.name = self.name or tacAttr.name()
      attr.unit = self.unit
      attr.value = tacAttr.value()
      attr.status = tacAttr.status()

      return attr

   def makeEscalatedAttr( self, data ):
      tacAttr = self.extractAttr( data )
      attr = StorageDevicesModels.SmartAttr()
      attr.name = tacAttr.name()
      attr.status = tacAttr.status()
      return attr

   def makeDetailAttr( self, data ):
      tacAttr = self.extractAttr( data )
      attr = StorageDevicesModels.SmartAttrDetail()

      attr.id = tacAttr.id()
      attr.name = tacAttr.name()
      attr.normalized = tacAttr.normalized()
      attr.worst = tacAttr.worst()
      attr.threshold = tacAttr.threshold()
      attr.raw = tacAttr.raw()

      return attr

   def makeNamedDetailAttr( self, data ):
      attr = self.makeDetailAttr( data )
      if self.name is not None:
         attr.name = self.name

      return attr

class SmartAttrKnownTemplate( AttrTemplate ):
   wrapper = staticmethod( SmartAttrKnownWrapper )

class SmartAttrUnknownTemplate( AttrTemplate ):
   wrapper = staticmethod( SmartAttrUnknownWrapper )

class NvmeAttrTemplate( AttrTemplate ):
   wrapper = staticmethod( NvmeAttrWrapper )

SMART_SUMMARY_ATTRS = [
   SmartAttrKnownTemplate( "lifetimeRemaining",
                           name="Lifetime remaining", unit="%" ),
   SmartAttrKnownTemplate( "powerOnHours",
                           name="Power on time", unit="hours" ),
   SmartAttrKnownTemplate( "temperatureCelsius",
                           name="Temperature", unit="celsius" ),
]

SMART_DETAILED_ATTRS = [
   SmartAttrKnownTemplate( "rawReadErrorRate" ),
   SmartAttrKnownTemplate( "reallocatedSectorCount" ),
   SmartAttrKnownTemplate( "powerCycleCount" ),
   SmartAttrKnownTemplate( "programFailCountTotal" ),
   SmartAttrKnownTemplate( "programFailCountChip" ),
   SmartAttrKnownTemplate( "wearLevelingCount" ),
   SmartAttrKnownTemplate( "eraseFailCountTotal" ),
   SmartAttrKnownTemplate( "eraseFailCountChip" ),
   SmartAttrKnownTemplate( "usedReserveBlockCountChip" ),
   SmartAttrKnownTemplate( "reportedUncorrectableErrorCount" ),
   SmartAttrKnownTemplate( "unexpectedPowerOffCount" ),
   SmartAttrKnownTemplate( "hardwareEccRecovered" ),
   SmartAttrKnownTemplate( "reallocationEventCount" ),
   SmartAttrKnownTemplate( "currentPendingSectorCount" ),
   SmartAttrKnownTemplate( "offlineUncorrectableSectorCount" ),
   SmartAttrKnownTemplate( "udmaCrcErrorCount" ),
   SmartAttrKnownTemplate( "availableReservedSpace" ),
   SmartAttrKnownTemplate( "totalLbasWritten" ),
   SmartAttrKnownTemplate( "totalLbasRead" ),
]

NVME_SUMMARY_ATTRS = [
      NvmeAttrTemplate( "lifetimeRemaining", name="Lifetime remaining", unit="%" ),
      NvmeAttrTemplate( "powerOnHours", name="Power on time", unit="hours" ),
      NvmeAttrTemplate( "temperatureCelsius", name="Temperature", unit="celsius" ),
]

NVME_DETAILED_ATTRS = [
      NvmeAttrTemplate( "availableSpare", name="Available Spare" ),
      NvmeAttrTemplate( "availableSpareThreshold",
                        name="Available Spare Threshold" ),
      NvmeAttrTemplate( "criticalWarning", name="Critical Warning" ),
      NvmeAttrTemplate( "controllerBusyTime", name="Controller Busy Time" ),
      NvmeAttrTemplate( "dataUnitsRead", name="Data Units Read" ),
      NvmeAttrTemplate( "dataUnitsWritten", name="Data Units Written" ),
      NvmeAttrTemplate( "hostReadCommands", name="Host Read Commands" ),
      NvmeAttrTemplate( "hostWriteCommands", name="Host Write Commands" ),
      NvmeAttrTemplate( "mediaErrors", name="Media and Data Integrity Errors" ),
      NvmeAttrTemplate( "numErrLogEntries", name="Error Information Log Entries" ),
      NvmeAttrTemplate( "powerCycleCount", name="Power Cycles" ),
      NvmeAttrTemplate( "temperatureTimeWarning",
                        name="Warning Comp. Temperature Time" ),
      NvmeAttrTemplate( "temperatureTimeCritical",
                        name="Critical Comp. Temperature Time" ),
      NvmeAttrTemplate( "unsafeShutdowns", name="Unsafe Shutdowns" ),
]

class DeviceDataAttrExtractor:
   def __init__( self, parentAttrs, deviceData ):
      self.parentAttrs = parentAttrs
      self.deviceData = deviceData

   @staticmethod
   def createModel():
      raise NotImplementedError()

   @staticmethod
   def buildAttrList( data ):
      raise NotImplementedError()

   def extract( self ):
      if not self.deviceData:
         return

      for ( serial, data ) in self.deviceData.deviceData.items():
         if not storageDevicesStatus.connectedDevices:
            continue
         connectedDisk = storageDevicesStatus.connectedDevices.disk[ serial ]
         device = connectedDisk.fsName()
         if not device:
            continue

         deviceAttrs = self.buildAttrList( data )

         deviceModelAttrs = self.createModel()
         deviceModelAttrs.attributes = deviceAttrs
         self.parentAttrs.devices[ device ] = deviceModelAttrs

#pylint: disable=abstract-method
class SummaryDeviceDataAttrExtractor( DeviceDataAttrExtractor ):
   createModel = staticmethod( StorageDevicesModels.DeviceSmartAttrs )

class SmartDataAttrExtractor( SummaryDeviceDataAttrExtractor ):
   @staticmethod
   def buildAttrList( data ):
      deviceAttrs = []
      deviceAttrs.extend( template.makeSummaryAttr( data )
                          for template in SMART_SUMMARY_ATTRS
                          if template.extractAttr( data ).present )
      for id_ in data.unknownAttr:
         template = SmartAttrUnknownTemplate( id_ )
         if template.extractAttr( data ).status() != "failed":
            continue
         deviceAttrs.append( template.makeEscalatedAttr( data ) )
      deviceAttrs.sort( key=lambda attr: attr.name )
      return deviceAttrs

class NvmeDataAttrExtractor( SummaryDeviceDataAttrExtractor ):
   @staticmethod
   def buildAttrList( data ):
      deviceAttrs = []
      deviceAttrs.extend( template.makeSummaryAttr( data )
                          for template in NVME_SUMMARY_ATTRS )
      deviceAttrs.sort( key=lambda attr: attr.name )
      return deviceAttrs

#pylint: disable=abstract-method
class DetailDeviceDataAttrExtractor( DeviceDataAttrExtractor ):
   createModel = staticmethod( StorageDevicesModels.DeviceSmartAttrDetails )

class SmartDataDetailAttrExtractor( DetailDeviceDataAttrExtractor ):
   @staticmethod
   def buildAttrList( data ):
      deviceAttrs = []
      deviceAttrs.extend( template.makeNamedDetailAttr( data )
                          for template in SMART_SUMMARY_ATTRS
                          if template.extractAttr( data ).present )
      deviceAttrs.extend( template.makeNamedDetailAttr( data )
                          for template in SMART_DETAILED_ATTRS
                          if template.extractAttr( data ).present )
      deviceAttrs.extend( SmartAttrUnknownTemplate( id_ ).makeDetailAttr( data )
                          for id_ in data.unknownAttr )
      deviceAttrs.sort( key=lambda attr: attr.id )
      return deviceAttrs

class NvmeDataDetailAttrExtractor( DetailDeviceDataAttrExtractor ):
   @staticmethod
   def buildAttrList( data ):
      deviceAttrs = []
      deviceAttrs.extend( template.makeNamedDetailAttr( data )
            for template in NVME_SUMMARY_ATTRS )
      deviceAttrs.extend( template.makeNamedDetailAttr( data )
            for template in NVME_DETAILED_ATTRS )
      return deviceAttrs

def showSystemStorageHealthSmart( mode, args ):
   attrs = StorageDevicesModels.SmartAttrs()

   SmartDataAttrExtractor( attrs, storageDevicesStatus.smartData ).extract()

   NvmeDataAttrExtractor( attrs, storageDevicesStatus.nvmeData ).extract()

   return attrs

def showSystemStorageHealthSmartDetail( mode, args ):
   attrs = StorageDevicesModels.SmartAttrDetails()

   SmartDataDetailAttrExtractor( attrs, storageDevicesStatus.smartData ).extract()

   NvmeDataDetailAttrExtractor( attrs, storageDevicesStatus.nvmeData ).extract()

   return attrs

# TODO BUG:901805 Refractor code below to be fdl driven
class ResetHelperBase:
   resetRegister = None
   enableRegister = None
   factoryResetValue = None
   rollbackResetValue = None

   def triggerRollback( self ):
      raise NotImplementedError

   def triggerFactoryReset( self ):
      raise NotImplementedError

   def enableReset( self, bitPosList ):
      '''
         Enable all type of push-button resets if they are disabled.
      '''
      assert self.enableRegister
      currEnableRegVal = Tac.run( [ "scd", "read", self.enableRegister ],
                                   asRoot=True, stdout=Tac.CAPTURE )
      newEnableRegVal = int( currEnableRegVal.split()[ 1 ], base=16 )
      for bitPos in bitPosList:
         newEnableRegVal = newEnableRegVal | ( 1 << bitPos )
      Tac.run( [ "scd", "write", self.enableRegister, str( newEnableRegVal ) ],
                 asRoot=True )

class CaravanResetHelper( ResetHelperBase ):
   resetRegister = "0x900"
   factoryResetValue = "0xc0de0003"
   rollbackResetValue = "0xc0de0001"
   # TODO: BUG 900627: update enable register and bits after caravan scd is updated
   enableRegister = "0x7040"
   # Factory Reset:27, simple reboot:28 rollback:29
   bitPosDict = [ 27, 28, 29 ]

   def triggerRollback( self ):
      self.enableReset( self.bitPosDict )
      # trigger rollback
      Tac.run( [ "scd", "write", self.resetRegister, self.rollbackResetValue ],
               asRoot=True )

   def triggerFactoryReset( self ):
      self.enableReset( self.bitPosDict )
      # trigger factory Reset
      Tac.run( [ "scd", "write", self.resetRegister, self.factoryResetValue ],
               asRoot=True )

class PrairieResetHelper( ResetHelperBase ):
   resetRegister = "0x1200"
   enableRegister = "0x7040"
   rollbackResetValue = "0xc0de0004"
   # rollback:11, simple reboot:2
   bitPosDict = [ 2, 11 ]

   def triggerRollback( self ):
      self.enableReset( self.bitPosDict )
      # trigger rollback
      Tac.run( [ "scd", "write", self.resetRegister, self.rollbackResetValue ],
               asRoot=True )

class Wolfhound3PlusResetHelper( ResetHelperBase ):
   pushButtonRegister = "0x91"
   rollbackResetValue = "0xA4"
   enableRegister = "0x11"
   i2cBus = "0"
   cpldAddress = "0x66"
   # rollback:7 will also trigger reboot
   bitPosDict = [ 7 ]

   def enableReset( self, bitPosList ):
      '''
         Enable rollback push-button reset if it is disabled.
      '''
      assert self.enableRegister
      currEnableRegVal = Tac.run( [ "i2cget", "-y", self.i2cBus, self.cpldAddress,
                                   self.enableRegister ], asRoot=True,
                                   stdout=Tac.CAPTURE )
      newEnableRegVal = int( currEnableRegVal.split()[ 0 ], base=16 )
      for bitPos in bitPosList:
         newEnableRegVal = newEnableRegVal | ( 1 << bitPos )
      Tac.run( [ "i2cset", "-y", self.i2cBus, self.cpldAddress, self.enableRegister,
                 str( newEnableRegVal ) ], asRoot=True )

   def triggerRollback( self ):
      self.enableReset( self.bitPosDict )
      # trigger rollback
      Tac.run( [ "i2cset", "-y", self.i2cBus, self.cpldAddress,
               self.pushButtonRegister, self.rollbackResetValue ], asRoot=True )

# -------------------------------------------------------------------------------
# Register "show tech-support" commands
# -------------------------------------------------------------------------------
TechSupportCli.registerShowTechSupportCmd(
   '2018-10-15 16:56:01',
   cmds=[ 'show system health storage' ],
   summaryCmds=[ 'show system health storage' ] )

TechSupportCli.registerShowTechSupportCmd(
   '2021-04-30 08:37:53',
   cmds=[ 'show system storage statistics' ],
   summaryCmds=[ 'show system storage statistics' ] )

TechSupportCli.registerShowTechSupportCmd(
   '2021-05-20 13:50:24',
   cmds=[ 'show system storage health smart',
          'show system storage health smart detail' ],
   summaryCmds=[ 'show system storage health smart' ] )

# --------------------------------------------------
# Plugin method - Mount the objects we need from Sysdb
#--------------------------------------------------
def Plugin( entityManager ): #pylint: disable-msg=W0613
   global storageDevicesStatus, pushButtonRecoveryConfig
   storageDevicesStatus = LazyMount.mount(
      entityManager,
      Cell.path( "hardware/storageDevices/status" ),
      "StorageDevices::Status", "r"
   )
   pushButtonRecoveryConfig = LazyMount.mount(
      entityManager,
      "hardware/pushButton/cliConfig",
      "StorageDevices::PushButtonRecoveryCliConfig", "fw" )

   # Register Show Inventory callback for Storage Devices. Show inventory takes
   # care of mounting appropriate objects.
   FruCli.registerShowInventoryCallback( StorageDevShowInvCallback )
