#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import BasicCli
import CliMatcher
import CliPlugin.StorageDevicesCli as StorageDevicesCli
from CliPlugin.StorageDevicesModels import Devices, DevicesIOStat, SmartAttrs,      \
   SmartAttrDetails
from CliToken.System import systemMatcherForShow
import ShowCommand

storageMatcherForShow = CliMatcher.KeywordMatcher(
   'storage',
   helpdesc='Show the storage sub system'
)

storageStatisticsMatcherForShow = CliMatcher.KeywordMatcher(
   'statistics',
   helpdesc='Show storage devices related statistics',
)

storageStatisticsLastMatcherForShow = CliMatcher.KeywordMatcher(
   'last',
   helpdesc='Retrieve statistics over a period of time',
)

storageHealthMatcherForShow = CliMatcher.KeywordMatcher(
   'health',
   helpdesc='Show health information on storage devices',
)

storageHealthSmartMatcherForShow = CliMatcher.KeywordMatcher(
   'smart',
   helpdesc='Show SMART attributes for storage devices',
)

#--------------------------------------------------------------------------------
# show system health storage
#--------------------------------------------------------------------------------
class SystemHealthStorageCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show system health storage'
   data = {
      'system': systemMatcherForShow,
      'health': 'Show the health of system components',
      'storage': 'Show the health of mounted storage devices',
   }
   handler = StorageDevicesCli.showSystemHealthStorage
   cliModel = Devices

BasicCli.addShowCommandClass( SystemHealthStorageCmd )

#--------------------------------------------------------------------------------
# show system storage health (alias on show system health storage)
#--------------------------------------------------------------------------------
class SystemStorageHealthCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show system storage health'
   data = {
      'system': systemMatcherForShow,
      'storage': storageMatcherForShow,
      'health': storageHealthMatcherForShow,
   }
   handler = StorageDevicesCli.showSystemHealthStorage
   cliModel = Devices

BasicCli.addShowCommandClass( SystemStorageHealthCmd )

#--------------------------------------------------------------------------------
# show system storage health smart
#--------------------------------------------------------------------------------
class SystemStorageHealthSmartCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show system storage health smart'
   data = {
      'system': systemMatcherForShow,
      'storage': storageMatcherForShow,
      'health': storageHealthMatcherForShow,
      'smart': storageHealthSmartMatcherForShow,
   }
   handler = StorageDevicesCli.showSystemStorageHealthSmart
   cliModel = SmartAttrs

BasicCli.addShowCommandClass( SystemStorageHealthSmartCmd )

#--------------------------------------------------------------------------------
# show system storage health smart detail
#--------------------------------------------------------------------------------
class SystemStorageHealthSmartDetailCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show system storage health smart detail'
   data = {
      'system': systemMatcherForShow,
      'storage': storageMatcherForShow,
      'health': storageHealthMatcherForShow,
      'smart': storageHealthSmartMatcherForShow,
      'detail': 'Show raw SMART data for each attribute',
   }
   handler = StorageDevicesCli.showSystemStorageHealthSmartDetail
   cliModel = SmartAttrDetails

BasicCli.addShowCommandClass( SystemStorageHealthSmartDetailCmd )

#--------------------------------------------------------------------------------
# show system storage statistics
#--------------------------------------------------------------------------------
class SystemStatisticsStorageCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show system storage statistics'
   data = {
      'system': systemMatcherForShow,
      'storage': storageMatcherForShow,
      'statistics': storageStatisticsMatcherForShow,
   }
   handler = StorageDevicesCli.showSystemStatisticsStorage
   cliModel = DevicesIOStat

BasicCli.addShowCommandClass( SystemStatisticsStorageCmd )

#--------------------------------------------------------------------------------
# show system storage statistics rates
#--------------------------------------------------------------------------------
class SystemStorageStatisticsRatesCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show system storage statistics rates'
   data = {
      'system': systemMatcherForShow,
      'storage': storageMatcherForShow,
      'statistics': storageStatisticsMatcherForShow,
      'rates': 'Show average statistics for storage devices',
   }
   handler = StorageDevicesCli.showStorageStatisticsRates
   cliModel = DevicesIOStat

BasicCli.addShowCommandClass( SystemStorageStatisticsRatesCmd )

#--------------------------------------------------------------------------------
# show system storage statistics last day
#--------------------------------------------------------------------------------
class SystemStorageStatisticsLastDayCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show system storage statistics last day'
   data = {
      'system': systemMatcherForShow,
      'storage': storageMatcherForShow,
      'statistics': storageStatisticsMatcherForShow,
      'last': storageStatisticsLastMatcherForShow,
      'day': 'Show last day statistics for storage devices',
   }
   handler = StorageDevicesCli.showStorageStatisticsLastDay
   cliModel = DevicesIOStat

BasicCli.addShowCommandClass( SystemStorageStatisticsLastDayCmd )
