#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
import BasicCli
import CliCommand
import CliParser
from CliPlugin import StorageDevicesCli
import CliToken.Reset
import CliToken.System
import os
import Ark
import AbootVersion
import FirmwareRev
import Toggles.StorageDevicesToggleLib as StorageDevicesToggles

factoryResetCliEnabled = \
   StorageDevicesToggles.toggleStorageDevicesFactoryResetCliEnabled()

secureEraseRollbackEnabled = \
   StorageDevicesToggles.toggleStorageDevicesSecureEraseRollbackEnabled()

# BUG901805: tracking replacing hardcoded list of platform
pushButtonRecoverySupportedPlat = { "councilbluffs", "independence",
                                    "prairieisland", "raspberryisland", "wh3p" }

# Only Arista hardware (except certain platforms) and dropbear are supported
def platformGuard( mode, token ):
   with open( "/proc/cmdline" ) as cmdlineFile:
      cmdline = cmdlineFile.read()

   # List of platforms that do not support "reset system storage [secure]"
   #  - Redondo/Redondo2 is not supported because these platforms use a special GPT
   #    partition layout, which the command cannot handle yet. This feature is not
   #    yet required in practice.
   unsupportedPlatforms = [ "onie_platform=", "platform=baremetal",
                            "platform=dropbear", "platform=hermosa",
                            "platform=veos" ]

   for platform in unsupportedPlatforms:
      if platform in cmdline:
         return CliParser.guardNotThisPlatform
   return None

def rollbackPlatformGuard( mode, token ):
   # BUG901805: tracking replacing hardcoded list of platform
   supportedPlatforms = { "councilbluffs", "independence", "prairieisland",
   "raspberryisland", "wh3p" }
   platform = Ark.getPlatform()
   if platform in supportedPlatforms:
      if 'SIMULATION_VMID' not in os.environ:
         abootVersionStr = FirmwareRev.abootFirmwareRev()
         abootVersion = AbootVersion.parseVersion( abootVersionStr )
         if not abootVersion.isRollbackSupported():
            return CliParser.guardNotThisAbootVersion
      return None
   return CliParser.guardNotThisPlatform

def factoryResetPlatformGuard( mode, token ):
   # BUG901805: tracking replacing hardcoded list of platform
   supportedPlatforms = { "councilbluffs", "independence" }
   platform = Ark.getPlatform()
   if platform in supportedPlatforms:
      if 'SIMULATION_VMID' not in os.environ:
         abootVersionStr = FirmwareRev.abootFirmwareRev()
         abootVersion = AbootVersion.parseVersion( abootVersionStr )
         if not abootVersion.isFactoryResetSupported():
            return CliParser.guardNotThisAbootVersion
      return None
   return CliParser.guardNotThisPlatform

def pushButtonRecoveryPlatformGuard( mode, token ):
   usupportedSids = { "sid=BcmWh3pReference" }

   with open( "/proc/cmdline" ) as cmdlineFile:
      cmdline = cmdlineFile.read()

   for sid in usupportedSids:
      if sid in cmdline:
         return CliParser.guardNotThisPlatform

   platform = Ark.getPlatform()
   if platform in pushButtonRecoverySupportedPlat:
      return None
   return CliParser.guardNotThisPlatform

def snapshotPlatformGuard( mode, token ):
   """
   Return the union of platformGuard and rollbackPlatformGuard.
   """
   return platformGuard( mode, token ) and rollbackPlatformGuard( mode, token )

storageMatcher = CliCommand.guardedKeyword( 'storage',
                           helpdesc="reset storage devices",
                           guard=platformGuard )

# --------------------------------------------------------------------------------
# reset system storage secure [rollback]
# --------------------------------------------------------------------------------
class ResetStorage( CliCommand.CliCommandClass ):
   syntax = "reset system storage secure"
   data = {
         'reset': CliToken.Reset.resetKwApi,
         'system': CliToken.System.systemMatcherForResetSystem,
         'storage': storageMatcher,
         'secure': "Secure erase, then repartition and "
                    "reformat crash:, drive:, and flash: "
                    "(as applicable)",
   }
   if secureEraseRollbackEnabled:
      syntax = "reset system storage secure [ rollback ]"
      data = {
            'reset': CliToken.Reset.resetKwApi,
            'system': CliToken.System.systemMatcherForResetSystem,
            'storage': storageMatcher,
            'secure': "Secure erase, then repartition and "
                       "reformat crash:, drive:, and flash: "
                       "(as applicable)",
            'rollback': "Restore EOS.swi and other files saved "
                        "in snapshot: after secure erase",
      }
   handler = StorageDevicesCli.resetSystemStorage

# --------------------------------------------------------------------------------
# reset system storage ( rollback | factory )
# --------------------------------------------------------------------------------

class ResetStorageExtended( CliCommand.CliCommandClass ):
   syntax = "reset system storage ( rollback | factory )"
   data = {
         'reset': CliToken.Reset.resetKwApi,
         'system': CliToken.System.systemMatcherForResetSystem,
         'storage': storageMatcher,
         'rollback': CliCommand.guardedKeyword( 'rollback',
                           helpdesc='Reset system with user-saved files',
                           guard=rollbackPlatformGuard ),
         'factory': CliCommand.guardedKeyword( 'factory',
                           helpdesc='Reset system with factory files',
                           guard=factoryResetPlatformGuard ),
   }

   handler = StorageDevicesCli.resetSystemStorage

if factoryResetCliEnabled:
   BasicCli.EnableMode.addCommandClass( ResetStorageExtended )
BasicCli.EnableMode.addCommandClass( ResetStorage )

# --------------------------------------------------------------------------------
# [no | default] reset system snapshot ( default | ztp-bootstrap )
# --------------------------------------------------------------------------------
class StorageSnapshot( CliCommand.CliCommandClass ):
   syntax = "reset system snapshot ( default | ztp-bootstrap )"
   noOrDefaultSyntax = "reset system snapshot ..."
   data = {
         'reset': CliToken.Reset.resetKwApi,
         'system': CliToken.System.systemMatcherForResetSystem,
         'snapshot': CliCommand.guardedKeyword( 'snapshot',
                           helpdesc="Configure which system files to save in a "
                                    "snapshot",
                           guard=snapshotPlatformGuard ),
         'default': "Save EOS.swi, boot-config, startup-config, swix and "
                    "zerotouch-config files",
         'ztp-bootstrap': "Save EOS.swi, boot-config files"

   }
   handler = StorageDevicesCli.saveStorageSnapshot
   noOrDefaultHandler = StorageDevicesCli.noOrDefaultSaveStorageSnapshot

if factoryResetCliEnabled:
   BasicCli.EnableMode.addCommandClass( StorageSnapshot )

# --------------------------------------------------------------------------------
# [no | default] reset system storage push-button disabled
# --------------------------------------------------------------------------------
class PushButtonRecovery( CliCommand.CliCommandClass ):
   syntax = "reset system storage push-button disabled"
   noOrDefaultSyntax = "reset system storage push-button ..."
   data = {
         'reset': CliToken.Reset.resetKwApi,
         'system': CliToken.System.systemMatcherForResetSystem,
         'storage': storageMatcher,
         'push-button': CliCommand.guardedKeyword( 'push-button',
                           helpdesc='Allow reset through button press',
                           guard=pushButtonRecoveryPlatformGuard ),
         'disabled': "Disable reset through button press"
   }
   handler = StorageDevicesCli.handlePushButtonRecoveryConfig
   noOrDefaultHandler = StorageDevicesCli.noOrDefaultHandlePushButtonRecoveryConfig

if factoryResetCliEnabled:
   BasicCli.GlobalConfigMode.addCommandClass( PushButtonRecovery )
