#!/usr/bin/env python3
# Copyright (c) 2009, 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# Implements no CLI commands.  Contains code to reset portConfig state to
# default values when interfaces are deleted.
#-------------------------------------------------------------------------------
from CliPlugin import IntfCli
import Tac, ConfigMount, Tracing
from StpConst import * # pylint: disable=wildcard-import

__defaultTraceHandle__ = Tracing.Handle( 'StpCli' )
t0 = Tracing.trace0
t1 = Tracing.trace0

stpInputConfig = None

class StpIntfJanitor( IntfCli.IntfDependentBase ):
   """My job is to reset attributes in Stp port configs to default values when
   the interfaces associated with those port configs go away."""
   def setDefault( self ):
      self.removePortConfigs()
      self.removeBackupIntf()

   def removePortConfigs( self ):
      t0( "StpIntfJanitor: interface", self.intf_.name, "going away" )
      pc = stpInputConfig.portConfig.get( self.intf_.name )
      if pc is not None:
         t0( "StpIntfJanitor resetting PortConfig", self.intf_.name, "to defaults" )
         pc.portPriority = PortPriorityDefault
         pc.networkPort = NetworkPortDefault
         pc.adminEdgePort = AdminEdgePortDefault
         pc.autoEdgePort = AutoEdgePortDefault
         pc.adminPointToPointMac = pc.defaultAdminPointToPoint
         pc.bpduguard = 'bpduguardDefault'
         pc.guard = 'guardDefault'
         pc.rateLimitInterval = 0
         pc.rateLimitMaxCount = 0
         pc.rateLimitEnabled = 'rateLimitDefault'
         pc.bpdufilter = 'bpdufilterDefault' 
         pc.extPathCost = PathCostUnknown
         pc.spanTreeLogging = pc.spanTreeLoggingDefault
      for stpiConfig in stpInputConfig.stpiConfig.values():
         pc = stpiConfig.stpiPortConfig.get( self.intf_.name )
         if pc is not None:
            t1( "StpIntfJanitor resetting StpiPortConfig", stpiConfig.name,
                "/", self.intf_.name, "to defaults" )
            pc.extPathCost = PathCostUnknown
         for mstiConfig in stpiConfig.mstiConfig.values():
            pc = mstiConfig.mstiPortConfig.get( self.intf_.name )
            if pc is not None:
               t1( "StpIntfJanitor resetting MstiPortConfig", stpiConfig.name,
                   "/", mstiConfig.name, "/", self.intf_.name, "to defaults" )
               pc.intPathCost = PathCostUnknown
               pc.portPriority = PortPriorityUnconfigured

   def removeBackupIntf( self ):
      bic = stpInputConfig.backupIntfConfig.get( self.intf_.name )
      if bic is not None:
         t1( "StpIntfJanitor removing backupToBackupIntfConfig",
             bic.backupIntfName )
         del stpInputConfig.backupToBackupIntfConfig[ bic.backupIntfName ]
         t1( "StpIntfJanitor removing backupIntfConfig", self.intf_.name )
         del stpInputConfig.backupIntfConfig[ self.intf_.name ]
      b2bic = stpInputConfig.backupToBackupIntfConfig.get( self.intf_.name )
      if b2bic is not None:
         t1( "StpIntfJanitor removing backupToBackupIntfConfig", 
             self.intf_.name )
         del stpInputConfig.backupToBackupIntfConfig[ self.intf_.name ]
         t1( "StpIntfJanitor removing backupIntfConfig", b2bic.name )
         del stpInputConfig.backupIntfConfig[ b2bic.name ]
         
 
def Plugin( entityManager ):
   global stpInputConfig
   stpInputConfig = ConfigMount.mount( entityManager, "stp/input/config/cli",
                                       "Stp::Input::Config", "w" )
   priority = 10 # arbitrary; I don't think this matters... -nschrenk
   IntfCli.Intf.registerDependentClass( StpIntfJanitor, priority )   
