#!/usr/bin/env python3
# Copyright (c) 2007-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

#-------------------------------------------------------------------------------
# This module implements Stp show commands
#
# show spanning-tree [ detail ]
# show spanning-tree blockedports
# show spanning-tree instance [ detail ] 
# legacy: show spanning-tree bridge [ detail ]
# show spanning-tree transmit active [ detail ]
# legacy: show spanning-tree bridge assurance [ detail ]
# show spanning-tree inconsistentports
# show spanning-tree interface <interface> [ detail ]
# show spanning-tree root [ detail ]
# show spanning-tree summary [ totals ]
# show spanning-tree vlan <vlan-id>
#
# show spanning-tree mst configuration [ digest ]
# show spanning-tree mst <instance-id> [ interface <interface> ] [ detail ]
#
#-------------------------------------------------------------------------------
'''Show commands supported for SpanningTree'''

import Tac
import Arnet
import ArPyUtils
import LazyMount
import Tracing
import CliPlugin.TechSupportCli
from CliPlugin import VlanCli
from CliPlugin.StpModels import (
   BridgeIntfInfo,
   BridgeMstiInfo,
   BridgeMstiIntfInfo,
   BridgeStpiInfo,
   PortChannelPortIdAlloc,
   SpanningTreeBlockedPorts,
   SpanningTreeBlockedPortsList,
   SpanningTreeBridge,
   SpanningTreeBridgeAssurance,
   SpanningTreeBridgeAssuranceInterfaces,
   SpanningTreeBridgeAssuranceValue,
   SpanningTreeBridgeBrief,
   SpanningTreeBridgeDetail,
   SpanningTreeBridgeMstTestInfoModel,
   SpanningTreeBridges,
   SpanningTreeBridgesDetail,
   SpanningTreeCounters,
   SpanningTreeFeatureStatus,
   SpanningTreeInconsistentFeatures,
   SpanningTreeInstance,
   SpanningTreeInstanceDetail,
   SpanningTreeInstances,
   SpanningTreeInterface,
   SpanningTreeInterfaceAllDetail,
   SpanningTreeInterfaceCounters,
   SpanningTreeInterfaceMstDetail,
   SpanningTreeInterfaces,
   SpanningTreeMstConfiguration,
   SpanningTreeMstInstance,
   SpanningTreeMstInstances,
   SpanningTreeMstInterface,
   SpanningTreeMstInterfaces,
   SpanningTreePathCostCist,
   SpanningTreePathCostDefault,
   SpanningTreePortId,
   SpanningTreeRoot,
   SpanningTreeRootPort,
   SpanningTreeRoots,
   SpanningTreeVlanInstance,
   SpanningTreeVlanInstances,
   VlanList,
)
from StpStableUtil import stableFileExists

# Import Stp TAC object accessors, which create TAC objects if needed.
from StpCliUtil import (
   allMstiPortStatusPairs,
   bridgingConfigIs,
   isCist,
   isPvstInstName,
   MstConfig,
   mstiStatusCliName,
   MstStpiName,
   stpCliInfo,
   stpCliInfoIs,
   stpConfig,
   stpConfigIs,
   stpCounterName,
   stpInputConfigReq,
   stpMstiInstName,
   stpPortCounterDir,
   stpPortCounterDirIs,
   stpStatus,
   stpStatusIs,
   stpTopoStatus,
   stpTopoStatusIs,
   stpTxRxInputConfig,
   stpVersionCliString,
   stpVlanMstiConfig,
   stpVlanStpiConfigStatus,
   ValidVlanIdMax,
   ValidVlanIdMin,
   vlanPortConfig,
)

__defaultTraceHandle__ = Tracing.Handle( 'StpCli' )

def updatePortCounters( mode ):
   configReq = stpInputConfigReq()
   status = stpStatus()
   reqTime = Tac.now() # monotonic time
   configReq.counterUpdateRequestTime = reqTime

   # this is being done for Mlag backward compatibility
   stpTxRxInputConfig().counterUpdateRequestTime = reqTime

   def countersUpdated():
      # wait until the time has been updated, indicating all the counters have been
      return reqTime <= status.counterUpdateTime
   try:
      Tac.waitFor( countersUpdated, description="counter update", maxDelay=0.1,
                   sleep=True, timeout=30.0 )
   except Tac.Timeout:
      mode.addWarning( "displaying stale counters" )

def portRateLimiterInfo( mode, stpiName, portStatus ):
   values = ( portStatus.rateLimitEnabled,
              portStatus.rateLimitInterval,
              portStatus.rateLimitMaxCount )
   return values
   
def portCounterValues( mode, stpiName, portName, portStatus ):
   name = stpCounterName( stpiName, portName )
   Tracing.trace8( 'PortStatus creation time', portStatus.creationTime,
                   'counter value for', name )

   # get session snapshot
   sessionSnapshot = None
   sessionPortCounter = mode.session.sessionData( 'Stp.sessionPortCounter', None )
   if sessionPortCounter is not None:
      sessionSnapshot = sessionPortCounter.get( name )

   # get global snapshot
   globalSnapshot = None
   portCounterDir = stpPortCounterDir()
   if portCounterDir is not None:
      globalSnapshot = portCounterDir.portCounters.get( name )

   if sessionSnapshot:
      Tracing.trace8( 'session snapshot was at', sessionSnapshot.timestamp )

   if globalSnapshot:
      Tracing.trace8( 'global snapshot was at', globalSnapshot.timestamp )
      
   # compare session and global snapshots, return newest
   zero = Tac.Value( "Stp::PortCounters",
                     bpduRx=0, bpduTx=0,
                     bpduTaggedError=0, bpduOtherError=0, timestamp=0 )
   if ( sessionSnapshot is None ) and ( globalSnapshot is None ):
      snapshot = zero
   elif sessionSnapshot is None:
      if portStatus.creationTime < globalSnapshot.timestamp:
         snapshot = globalSnapshot
      else:
         snapshot = zero
   elif globalSnapshot is None:
      if portStatus.creationTime < sessionSnapshot.timestamp:
         snapshot = sessionSnapshot
      else:
         snapshot = zero
   elif ( portStatus.creationTime > sessionSnapshot.timestamp ) and (
      portStatus.creationTime > globalSnapshot.timestamp):
      snapshot = zero
   elif globalSnapshot.timestamp > sessionSnapshot.timestamp:
      snapshot = globalSnapshot
   else:
      snapshot = sessionSnapshot

   assert portStatus.counters.bpduRx >= snapshot.bpduRx
   assert portStatus.counters.bpduTx >= snapshot.bpduTx
   assert portStatus.counters.bpduTaggedError >= snapshot.bpduTaggedError
   assert portStatus.counters.bpduOtherError >= snapshot.bpduOtherError

   values = ( portStatus.counters.bpduTx - snapshot.bpduTx,
              portStatus.counters.bpduRx - snapshot.bpduRx,
              portStatus.counters.bpduTaggedError -
              snapshot.bpduTaggedError,
              portStatus.counters.bpduOtherError -
              snapshot.bpduOtherError,
              portStatus.bpduCount)
   return values

def portStatusBoundaryStringCapi( config, portStatus ):
   if not portStatus.sendRstp:
      # We're sending STP bpdus out this port, so must be a legacy STP
      # box out there.
      if ( portStatus.mstpRegionBorder and
           config.forceProtocolVersion == 'mstp' and
           portStatus.mstpRegionPvstBorder ):
         return 'mstpRegionPvst'
      else:
         return 'stp'
   elif portStatus.mstpRegionBorder:
      if config.forceProtocolVersion == 'mstp':
         return 'mstpRegion'
      else:
         return 'none'
   else:
      return 'none'

def getBridgeCapi( config, mstiStatus, detail ):
   bridgeCapi = SpanningTreeBridge()
   bridgeId = mstiStatus.bridgeId

   bridgeCapi.priority = bridgeId.priority
   bridgeCapi.systemIdExtension = bridgeId.systemId
   bridgeCapi.macAddress = bridgeId.address
   bridgeCapi.helloTime = config.bridgeHelloTime/1000.0
   bridgeCapi.maxAge = config.maxAge
   bridgeCapi.forwardDelay = config.bridgeForwardDelay

   if detail:
      bridgeCapi.detail = SpanningTreeBridgeDetail()
      bridgeCapi.detail.transmitHoldCount = config.txHoldCount #int

   return bridgeCapi

def getBridgeBriefCapi( obj, field ):
   bridgeIdCapi = SpanningTreeBridgeBrief()
   fieldObject = getattr( obj, field )
   bridgeIdCapi.priority = fieldObject.priority
   bridgeIdCapi.macAddress = fieldObject.address
   bridgeIdCapi.systemIdExtension = fieldObject.systemId
   return bridgeIdCapi

def getPortIdCapi( portId ):
   portIdCapi = SpanningTreePortId()
   portIdCapi.priority = portId.portPriority
   portIdCapi.number = portId.portNumber
   return portIdCapi

def getRootPortCapi( stpiStatus, mstiStatus, mstiRootPort ):
   rootPortCapi = SpanningTreeRootPort()
   cist = isCist( mstiStatus )
   rootPortCapi.interface = mstiRootPort.name
   rootPortCapi.portNumber = mstiRootPort.portId.portNumber

   if cist:
      rootPortCapi.cost = SpanningTreePathCostCist()
      rootPortCapi.cost.internalCost = mstiStatus.rootPathCost
      rootPortCapi.cost.externalCost = stpiStatus.cistPathCost
   else:
      rootPortCapi.cost = SpanningTreePathCostDefault()
      rootPortCapi.cost.cost = mstiStatus.rootPathCost
   return rootPortCapi

def getRegionalRootBridgeCapi( stpiStatus, mstiStatus ):
   rootBridgeCapi = None
   cist = isCist( mstiStatus )

   if cist:
      rootBridgeCapi = SpanningTreeBridgeBrief()
      rootId = mstiStatus.designatedRoot
      rootBridgeCapi.priority = rootId.priority
      rootBridgeCapi.systemIdExtension = rootId.systemId
      rootBridgeCapi.macAddress = rootId.address
   return rootBridgeCapi

def getRootBridgeCapi( stpiStatus, mstiStatus, detail ):
   cist = isCist( mstiStatus )

   if cist:
      rootId = stpiStatus.cistRoot
   else:
      rootId = mstiStatus.designatedRoot
   rootTimes = mstiStatus.rootTimes

   if detail == 'mst':
      rootBridgeCapi = SpanningTreeBridgeBrief()
      rootBridgeCapi.priority = rootId.priority
      rootBridgeCapi.systemIdExtension = rootId.systemId
      rootBridgeCapi.macAddress = rootId.address
   else: #None or detail
      rootBridgeCapi = SpanningTreeBridge()
      rootBridgeCapi.priority = rootId.priority
      rootBridgeCapi.systemIdExtension = rootId.systemId
      rootBridgeCapi.macAddress = rootId.address
      # Pre CAPI, only the seconds part was shown
      rootBridgeCapi.helloTime = ( float( rootTimes.helloTime.seconds ) +
                                   float( rootTimes.helloTime.fraction ) / 256 )
      rootBridgeCapi.maxAge = rootTimes.maxAge.seconds
      rootBridgeCapi.forwardDelay = rootTimes.fwdDelay.seconds
   return rootBridgeCapi


# There is some amount of code duplication here which needs to be fixed.
def getInterfaceCapi( mode, stpiStatus, mstiStatus, mstiPortStatus,
                        portStatus, portConfig,
                        detail, mstIntfModel=False ):

   interfaceCapi = SpanningTreeInterface()
   cist = isCist( mstiStatus )
   config = stpConfig()
   portId = mstiPortStatus.portId

   loopGuard = mstiPortStatus.loopGuardInconsistent
   rootGuard = mstiPortStatus.rootGuardInconsistent
   bridgeAssurance = mstiPortStatus.bridgeAssureInconsistent
   mstPvstBorder = portStatus.mstPvstBorderIncon
   portPriority = mstiPortStatus.portPriority
   
   if cist:
      pathCost = portStatus.operExtPathCost
   else:
      pathCost = mstiPortStatus.operIntPathCost

   if cist:
      rootBridgeId = portPriority.cistRootBridgeId
      regionalRootBridgeId = portPriority.regionalRootBridgeId
   else:
      rootBridgeId = portPriority.regionalRootBridgeId
   bridgeId = portPriority.bridgeId
   portTimes = mstiPortStatus.portTimes

   interfaceCapi.isEdgePort = bool( portStatus.operEdgePort )

   if portConfig.adminEdgePort:
      interfaceCapi.edgePortType = 'portfast'
   else:
      interfaceCapi.edgePortType = 'auto'

   if not detail or detail == 'mst':
      if portStatus.operPointToPointMac:
         interfaceCapi.linkType = 'p2p'
      else:
         interfaceCapi.linkType = 'shared'

      interfaceCapi.inconsistentFeatures = SpanningTreeInconsistentFeatures()
      if bridgeAssurance:
         interfaceCapi.inconsistentFeatures.bridgeAssurance = True
      if loopGuard:
         interfaceCapi.inconsistentFeatures.loopGuard = True
      if rootGuard:
         interfaceCapi.inconsistentFeatures.rootGuard = True
      if mstPvstBorder:
         interfaceCapi.inconsistentFeatures.mstPvstBorder = True

      interfaceCapi.boundaryType = portStatusBoundaryStringCapi( config, 
                                                             portStatus )

      interfaceCapi.role = mstiPortStatus.role
      interfaceCapi.state = mstiPortStatus.state
      interfaceCapi.cost = pathCost
      interfaceCapi.priority = portId.portPriority
      interfaceCapi.portNumber = portId.portNumber

      if mstIntfModel:
         interfaceCapi.mappedVlans = VlanList( vlans=vlansMappedToInstanceCapi(
               config, mstiStatus.instanceId ) )

      if detail == 'mst':
         interfaceCapi.detail = SpanningTreeInterfaceMstDetail()

         interfaceCapi.detail.designatedRootPriority = \
                                    rootBridgeId.priority
         interfaceCapi.detail.designatedRootAddress = rootBridgeId.address

         interfaceCapi.detail.designatedBridgePriority = \
                                    bridgeId.priority
         interfaceCapi.detail.designatedBridgeAddress = bridgeId.address

         interfaceCapi.detail.designatedPortPriority = \
                                    portPriority.portId.portPriority
         interfaceCapi.detail.designatedPortNumber = portPriority.portId.portNumber
         if cist:
            interfaceCapi.detail.designatedPathCost = SpanningTreePathCostCist()
            interfaceCapi.detail.designatedPathCost.externalCost = \
                                    portPriority.extRootPathCost
            interfaceCapi.detail.designatedPathCost.internalCost = \
                                    portPriority.intRootPathCost
            interfaceCapi.detail.regionalRootAddress = \
                                    regionalRootBridgeId.address
            interfaceCapi.detail.regionalRootPriority = \
                                    regionalRootBridgeId.priority
         else:
            interfaceCapi.detail.designatedPathCost = SpanningTreePathCostDefault()
            interfaceCapi.detail.designatedPathCost.cost = \
                                    portPriority.intRootPathCost

      return interfaceCapi
   elif detail == 'mstIntf':
      interfaceCapi = SpanningTreeMstInterface()
      interfaceCapi.isEdgePort = bool( portStatus.operEdgePort )

      interfaceCapi.features[ 'linkType' ] = SpanningTreeFeatureStatus()
      if portConfig.adminPointToPointMac == 'forceTrue':
         interfaceCapi.features[ 'linkType' ].value = 'p2p'
         interfaceCapi.features[ 'linkType' ].default = False
      elif portConfig.adminPointToPointMac == 'forceFalse':
         interfaceCapi.features[ 'linkType' ].value = 'shared'
         interfaceCapi.features[ 'linkType' ].default = True
      else:
         # We're auto-detect, so look at oper status
         if portStatus.operPointToPointMac:
            interfaceCapi.features[ 'linkType' ].value = 'p2p'
            interfaceCapi.features[ 'linkType' ].default = True
         else:
            interfaceCapi.features[ 'linkType' ].value = 'shared'
            interfaceCapi.features[ 'linkType' ].default = True

      (bpduguard, default) = intfBpduguardStatus( config, portConfig, portStatus )
      interfaceCapi.features[ 'bpduGuard' ] = SpanningTreeFeatureStatus()
      if bpduguard == 'enabled':
         interfaceCapi.features[ 'bpduGuard' ].value = 'enabled'
      else:
         interfaceCapi.features[ 'bpduGuard' ].value = 'disabled'
      interfaceCapi.features[ 'bpduGuard' ].default = bool( default )

      ( guard, default ) = intfguardStatus( config, portConfig, mstiPortStatus )
      if guard != 'disabled':
         interfaceCapi.features[ 'interfaceGuard' ] = \
                                       SpanningTreeFeatureStatus()
         interfaceCapi.features[ 'interfaceGuard' ].value = guard
         interfaceCapi.features[ 'interfaceGuard' ].default = bool( default )

      ( networkPort, baEnabled ) = \
          intfNetworkPortStatus( config, portConfig, portStatus )
      if networkPort:
         interfaceCapi.features[ 'networkPortBridgeAssurance' ] = \
                                                      SpanningTreeFeatureStatus()
         if baEnabled:
            interfaceCapi.features[ 'networkPortBridgeAssurance' ].value = \
                                                      'enabled'
         else:
            interfaceCapi.features[ 'networkPortBridgeAssurance' ].value = \
                                                      'disabled'
         interfaceCapi.features[ 'networkPortBridgeAssurance' ].default = \
                                                      True

      (bpdufilter, default) = intfBpdufilterStatus( config, portConfig, 
                                                            portStatus )
      interfaceCapi.features[ 'bpduFilter' ] = SpanningTreeFeatureStatus()
      if bpdufilter:
         interfaceCapi.features[ 'bpduFilter' ].value = 'enabled'
      else:
         interfaceCapi.features[ 'bpduFilter' ].value = 'disabled'
      interfaceCapi.features[ 'bpduFilter' ].default = bool( default )

      interfaceCapi.counters = SpanningTreeInterfaceCounters()
      ( interfaceCapi.counters.bpduSent,
        interfaceCapi.counters.bpduReceived,
        interfaceCapi.counters.bpduTaggedError,
        interfaceCapi.counters.bpduOtherError,
        interfaceCapi.counters.bpduRateLimitCount ) = \
              portCounterValues( mode, stpiStatus.name, portStatus.name, portStatus )

      return interfaceCapi
   else:
      interfaceCapi.detail = SpanningTreeInterfaceAllDetail()

      interfaceCapi.detail.designatedRootPriority = \
                                 rootBridgeId.priority + rootBridgeId.systemId
      interfaceCapi.detail.designatedRootAddress = rootBridgeId.address

      interfaceCapi.detail.designatedBridgePriority = \
                                 bridgeId.priority + bridgeId.systemId
      interfaceCapi.detail.designatedBridgeAddress = bridgeId.address

      interfaceCapi.detail.designatedPortPriority = \
                                 portPriority.portId.portPriority
      interfaceCapi.detail.designatedPortNumber = portPriority.portId.portNumber
      if cist:
         interfaceCapi.detail.designatedPathCost = SpanningTreePathCostCist()
         interfaceCapi.detail.designatedPathCost.externalCost = \
                                 portPriority.extRootPathCost
         interfaceCapi.detail.designatedPathCost.internalCost = \
                                 portPriority.intRootPathCost
      else:
         interfaceCapi.detail.designatedPathCost = SpanningTreePathCostDefault()
         interfaceCapi.detail.designatedPathCost.cost = \
                                 portPriority.intRootPathCost

      interfaceCapi.detail.features[ 'linkType' ] = SpanningTreeFeatureStatus()
      if portConfig.adminPointToPointMac == 'forceTrue':
         interfaceCapi.linkType = 'p2p'
         interfaceCapi.detail.features[ 'linkType' ].value = 'p2p'
         interfaceCapi.detail.features[ 'linkType' ].default = False
      elif portConfig.adminPointToPointMac == 'forceFalse':
         interfaceCapi.linkType = 'shared'
         interfaceCapi.detail.features[ 'linkType' ].value = 'shared'
         interfaceCapi.detail.features[ 'linkType' ].default = True
      else:
         # We're auto-detect, so look at oper status
         if portStatus.operPointToPointMac:
            interfaceCapi.linkType = 'p2p'
            interfaceCapi.detail.features[ 'linkType' ].value = 'p2p'
            interfaceCapi.detail.features[ 'linkType' ].default = True
         else:
            interfaceCapi.linkType = 'shared'
            interfaceCapi.detail.features[ 'linkType' ].value = 'shared'
            interfaceCapi.detail.features[ 'linkType' ].default = True

      interfaceCapi.inconsistentFeatures = SpanningTreeInconsistentFeatures()
      if bridgeAssurance:
         interfaceCapi.inconsistentFeatures.bridgeAssurance = True
      if loopGuard:
         interfaceCapi.inconsistentFeatures.loopGuard = True
      if rootGuard:
         interfaceCapi.inconsistentFeatures.rootGuard = True
      if mstPvstBorder:
         interfaceCapi.inconsistentFeatures.mstPvstBorder = True

      interfaceCapi.boundaryType = portStatusBoundaryStringCapi( config, 
                                                             portStatus )

      interfaceCapi.role = mstiPortStatus.role
      interfaceCapi.state = mstiPortStatus.state
      interfaceCapi.cost = pathCost
      interfaceCapi.priority = portId.portPriority
      interfaceCapi.portNumber = portId.portNumber

      interfaceCapi.detail.messageAge = portTimes.messageAge.seconds
      interfaceCapi.detail.forwardDelay = portTimes.fwdDelay.seconds
      interfaceCapi.detail.maxAge = portTimes.maxAge.seconds

      interfaceCapi.detail.transitionsToForwarding = \
                                       mstiPortStatus.transitionsToForwardingCount

      (bpduguard, default) = intfBpduguardStatus( config, portConfig, portStatus )
      interfaceCapi.detail.features[ 'bpduGuard' ] = SpanningTreeFeatureStatus()
      if bpduguard == 'enabled':
         interfaceCapi.detail.features[ 'bpduGuard' ].value = 'enabled'
      else:
         interfaceCapi.detail.features[ 'bpduGuard' ].value = 'disabled'
      interfaceCapi.detail.features[ 'bpduGuard' ].default = bool( default )

      ( guard, default ) = intfguardStatus( config, portConfig, mstiPortStatus )
      if guard != 'disabled':
         interfaceCapi.detail.features[ 'interfaceGuard' ] = \
                                       SpanningTreeFeatureStatus()
         interfaceCapi.detail.features[ 'interfaceGuard' ].value = guard
         interfaceCapi.detail.features[ 'interfaceGuard' ].default = bool( default )

      ( networkPort, baEnabled ) = \
          intfNetworkPortStatus( config, portConfig, portStatus )
      if networkPort:
         interfaceCapi.detail.features[ 'networkPortBridgeAssurance' ] = \
                                                      SpanningTreeFeatureStatus()
         if baEnabled:
            interfaceCapi.detail.features[ 'networkPortBridgeAssurance' ].value = \
                                                      'enabled'
         else:
            interfaceCapi.detail.features[ 'networkPortBridgeAssurance' ].value = \
                                                      'disabled'
         interfaceCapi.detail.features[ 'networkPortBridgeAssurance' ].default = \
                                                      True

      (bpdufilter, default) = intfBpdufilterStatus( config, portConfig, 
                                                            portStatus )
      interfaceCapi.detail.features[ 'bpduFilter' ] = SpanningTreeFeatureStatus()
      if bpdufilter:
         interfaceCapi.detail.features[ 'bpduFilter' ].value = 'enabled'
      else:
         interfaceCapi.detail.features[ 'bpduFilter' ].value = 'disabled'
      interfaceCapi.detail.features[ 'bpduFilter' ].default = bool( default )

      interfaceCapi.detail.counters = SpanningTreeInterfaceCounters()
      ( interfaceCapi.detail.counters.bpduSent,
        interfaceCapi.detail.counters.bpduReceived,
        interfaceCapi.detail.counters.bpduTaggedError,
        interfaceCapi.detail.counters.bpduOtherError,
        interfaceCapi.detail.counters.bpduRateLimitCount ) = \
              portCounterValues( mode, stpiStatus.name, portStatus.name, portStatus )

      ( interfaceCapi.detail.rateLimitEnabled, 
        interfaceCapi.detail.rateLimitInterval,
        interfaceCapi.detail.rateLimitMaxCount ) = \
              portRateLimiterInfo( mode, stpiStatus.name, portStatus )

      return interfaceCapi

def getSpanningTreeInstance( mode, config, stpiStatus, mstiStatus, detail,
                             vlan=None ):

   instance = SpanningTreeInstance()

   protocol = stpVersionCliString( config )
   if protocol == 'none':
      return None

   if protocol == 'rapid-pvst':
      protocol = 'rapidPvst'
   instance.protocol = protocol
   
   mstiRootPort = mstiStatus.rootPort
   if mstiRootPort:
      instance.rootPort = getRootPortCapi( stpiStatus, mstiStatus, 
                                           mstiRootPort )

   if not detail:
      instance.bridge = getBridgeCapi( config, mstiStatus, None )
      if mstiRootPort:
         instance.rootBridge = getRootBridgeCapi( stpiStatus, mstiStatus, None )

   else:
      instance.detail = SpanningTreeInstanceDetail()
      instance.bridge = getBridgeCapi( config, mstiStatus, detail )
      if mstiRootPort:
         instance.rootBridge = getRootBridgeCapi( stpiStatus, mstiStatus, detail )
      instance.detail.topologyChanges = mstiStatus.tcnCount
      if mstiStatus.tcnCount:
         instance.detail.lastTopologyChangeTime = int( Tac.now() - \
                                           mstiStatus.lastTopologyChange )
         instance.detail.lastTopologyChangePort = mstiStatus.lastTopologyChangePort

   for portKey in mstiStatus.mstiPortStatus:
      if vlan is not None and vlanPortConfig( vlan, portKey ) is None:
         # We're filtering by vlan and the filtering vlan is not active on
         # the current port, so skip it.
         continue
      mstiPortStatus = mstiStatus.mstiPortStatus.get( portKey )
      portStatus = stpiStatus.stpiPortStatus.get( portKey )
      portConfig = config.portConfig.get( portKey )
      if mstiPortStatus and portStatus and portConfig:
         instance[ 'interfaces' ][ mstiPortStatus.name ] = \
               getInterfaceCapi( mode, stpiStatus, mstiStatus, mstiPortStatus,
                                 portStatus, portConfig, detail )
   return instance

def inverseVlanMapCapi( vlanMap, includeInst0=True ):
   # Figure out which vlans are mapped to which msti.  Instance 0 is
   # special in that it gets all unmapped vlans.  We figure out which
   # ones those are by adding all vlans to the id0Set and then removing
   # them as we find them mapped to some other msti.
   idMap = {}
   id0List = VlanList( vlans=list( range( ValidVlanIdMin, ValidVlanIdMax + 1 ) ) )
   for ( vlanId, instId ) in vlanMap.items():
      vlanList = idMap.get( instId )
      if not vlanList:
         vlanList = VlanList()
         idMap[ instId ] = vlanList
      vlanList.vlans.append( vlanId ) # pylint: disable-msg=E1103
      id0List.vlans.remove( vlanId )
   if includeInst0:
      idMap[ 0 ] = id0List
   return idMap

def getMstInfoCapi( mstConfig, digest=None, message=None ):
   regionId = mstConfig.regionId()
   rev = mstConfig.configRevision()
   vlanMap = mstConfig.vlanMapDict()
   idMap = inverseVlanMapCapi( vlanMap )
   instCount = len( idMap )
   if digest:
      digestStr = "0x"
      for i in range( 16 ):
         val = getattr( digest, 'configDigest%d' % i )
         digestStr += "%02X" % val
      return SpanningTreeMstConfiguration( mstRegionName=regionId,
                                           mstConfigurationRevision=rev,
                                           configuredMstInstances=instCount,
                                           mstConfigurationDigest=digestStr,
                                           mstInstances=idMap,
                                           _message=message )
   else:
      return SpanningTreeMstConfiguration( mstRegionName=regionId,
                                           mstConfigurationRevision=rev,
                                           configuredMstInstances=instCount,
                                           mstInstances=idMap,
                                           _message=message )

def vlansMappedToInstanceCapi( config, instId ):
   mstConfig = MstConfig( config )
   mstConfig.revert()
   mapped = []

   vlanMap = mstConfig.vlanMapDict()
   if instId == 0:
      mapped = list( range( ValidVlanIdMin, ValidVlanIdMax + 1 ) )
      for v in vlanMap:
         mapped.remove( v )
   else:
      for v, i in vlanMap.items():
         if i == instId:
            mapped.append( v )
   return mapped
#-------------------------------------------------------------------------------
# CAPI helper functions end here, for now.
#-------------------------------------------------------------------------------

def intfBpduguardStatus( config, portConfig, portStatus ):
   # Return a 2 entry tuple.  The first entry is "enabled" if bpduguard is active
   # on the port and "disabled" otherwise.  The second is "default" if this is due
   # to the global default configuration and "" otherwise.
   
   
   if not portConfig or (portConfig.bpduguard == 'bpduguardDefault'):
      if( config.portfastBpduguard and portConfig and portConfig.adminEdgePort and
          portStatus.operEdgePort ):
         bpduguard = ("enabled", "default")
      else:
         bpduguard = ("disabled", "")
   elif portConfig.bpduguard == 'bpduguardEnabled':
      bpduguard = ("enabled", "")
   else:
      bpduguard = ("disable", "")
   return bpduguard

def intfBpdufilterStatus( config, portConfig, portStatus ):
   # Return a 2 entry tuple.  The first entry is True if bpdufilter is 
   # active on the port and False otherwise.  The second is True if
   # bpdufilter is active due to the global default configuration 
   # and False otherwise.
   
   
   if not portConfig or (portConfig.bpdufilter == 'bpdufilterDefault'):
      if ( config.portfastBpdufilter and portConfig and portConfig.adminEdgePort
         and portStatus.operEdgePort ):
         bpdufilter = (True, True)
      else:
         bpdufilter = (False, False)
   elif portConfig.bpdufilter == 'bpdufilterEnabled':
      bpdufilter = (True, False)
   else:
      bpdufilter = (False, False)
   return bpdufilter

def intfguardStatus( config, portConfig, mstiPortStatus ):
   # Return a 2 entry tuple.  The first entry is "loop" if loopguard is active
   # on the port, "root" if rootguard is active on the port,  and "disabled" 
   # otherwise.  The second is True if this is due to the global default 
   # configuration and False otherwise.
   
   if mstiPortStatus.guardStatus == 'loopGuardEnabled':
      if portConfig and portConfig.guard == 'loopguardEnabled':
         guard = ( "Loop", False )
      else:
         guard = ( "Loop", True )
   elif mstiPortStatus.guardStatus == 'rootGuardEnabled':
      guard = ( "Root", False )
   else:
      if portConfig and portConfig.guard == 'loopguardDisabled':
         guard = ( "disabled", False )
      else:
         guard = ( "disabled", True )
   return guard

def intfNetworkPortStatus( config, portConfig, portStatus ):
   # determines whether the port is a network port, and whether
   # bridge assurance is enabled
   
   networkPort = portConfig and portConfig.networkPort
   baEnabled = config.bridgeAssurance
   return ( networkPort, baEnabled )

#-------------------------------------------------------------------------------
# show spanning-tree [ detail ]
#-------------------------------------------------------------------------------
def showStp( mode, args ):
   detail = 'detail' in args
   config = stpConfig()
   status = stpStatus()

   if config.version == 'none':
      return SpanningTreeInstances( _message='Spanning-tree has been disabled in'
                                             ' the configuration.' )
   if config.version == 'backup':
      return SpanningTreeInstances( _message='Spanning-tree running in backup-mode' )

   elif detail and status.stpiStatus:
      updatePortCounters( mode )

   instances = {}
   for stpiName in status.stpiStatus:
      stpiStatus = status.stpiStatus.get( stpiName )
      if not stpiStatus:
         continue
      for key in stpiStatus.mstiStatus:
         mstiStatus = stpiStatus.mstiStatus.get( key )
         if mstiStatus:
            spanningTreeInstance = \
               getSpanningTreeInstance( mode, config, stpiStatus,
                                             mstiStatus, detail )
            if spanningTreeInstance:
               instances[ mstiStatusCliName( mstiStatus ) ] = \
                  spanningTreeInstance

   return SpanningTreeInstances( spanningTreeInstances=instances )
         
#-------------------------------------------------------------------------------
# show spanning-tree blockedports
#-------------------------------------------------------------------------------
def showStpBlocked( mode, args ):
   instances = {}
   for (mstiStatus, mstiPortStatus) in allMstiPortStatusPairs( stpStatus() ):
      mstiName = mstiStatusCliName( mstiStatus )
      if mstiPortStatus.state == 'discarding':
         if mstiName not in instances:
            instances[ mstiName ] = SpanningTreeBlockedPortsList()
         instances[ mstiName ].spanningTreeBlockedPorts.append( mstiPortStatus.name )

   return SpanningTreeBlockedPorts( spanningTreeInstances=instances )
         
#-------------------------------------------------------------------------------
# show spanning-tree instance [ detail ]
#
# legacy:
# show spanning-tree bridge [ detail ]
# Dumps out a table, one row per spanning tree instance
#-------------------------------------------------------------------------------
def showBridge( mode, args ):
   bridges = {}
   detail = 'detail' in args
   detailed = None
   config = stpConfig()
   status = stpStatus()
   cliInfo = stpCliInfo()
   topoStatus = stpTopoStatus()
   if not status.stpiStatus:
      return SpanningTreeBridges( spanningTreeBridges=bridges )

   if detail:
      detailed = SpanningTreeBridgesDetail()
      detailed.restartable = \
          stableFileExists( config.bridgeName ) and not status.recentlyRestarted
      detailed.mstPvstBorderEffective = status.mstPvstBorderEffective
      detailed.peerHeartbeatWhileEnabled = status.peerHeartbeatWhileEnabled
      detailed.overriddenBy = status.overriddenBy
      detailed.heartbeatTimeout = cliInfo.heartbeatTimeout
      detailed.lastLocalHeartbeatTimeout = status.lastLocalHeartbeatTimeout
      detailed.lastPeerHeartbeatTimeout = status.lastPeerHeartbeatTimeout
      detailed.localHeartbeatTimeoutsSinceReboot = \
            status.localHeartbeatTimeoutsSinceReboot
      detailed.peerHeartbeatTimeoutsSinceReboot = \
            status.peerHeartbeatTimeoutsSinceReboot
      detailed.superRoot = status.superRoot
      detailed.portChannelPortIdAllocationRange = PortChannelPortIdAlloc() 
      ( detailed.portChannelPortIdAllocationRange.reservedRangeMin,
        detailed.portChannelPortIdAllocationRange.reservedRangeMax ) = \
      ( config.portChannelPortIdAlloc.reservedRangeMin, 
        config.portChannelPortIdAlloc.reservedRangeMax )
      detailed.preserveOriginal2p5GAnd5GCost = \
            topoStatus.preserveOriginal2p5GAnd5GCost
      detailed.mismatchedBpduMode = config.mismatchedBpduMode
   for stpiName in status.stpiStatus:
      stpiStatus = status.stpiStatus.get( stpiName )
      stpiConfig = config.stpiConfig.get( stpiName )
      if (not stpiStatus) or (not stpiConfig):
         continue
      for key in stpiStatus.mstiStatus:
      
         mstiConfig = stpiConfig.mstiConfig.get( key )
         mstiStatus = stpiStatus.mstiStatus.get( key )
         if (not mstiConfig) or (not mstiStatus):
            continue

         bridges[ mstiStatusCliName( mstiStatus ) ] = \
               getBridgeCapi( config, mstiStatus, None )

   return SpanningTreeBridges( spanningTreeBridges=bridges, detail=detailed )

#-------------------------------------------------------------------------------
# show spanning-tree transmit active [ detail ]
#
# legacy:
# show spanning-tree bridge assurance [detail]
# Dumps configured and operating state of bridge assurance for all network
# ports
#-------------------------------------------------------------------------------
def showBridgeAssurance( mode, args ):
   detail = 'detail' in args
   instances = {}
   config = stpConfig()
   if not config.bridgeAssurance:
      return SpanningTreeBridgeAssurance( spanningTreeInstances=instances,
                                    bridgeAssuranceEnabled=config.bridgeAssurance )

   for ( mstiStatus, mstiPortStatus ) in allMstiPortStatusPairs( stpStatus() ):

      instanceName = mstiStatusCliName( mstiStatus )
      intfName = mstiPortStatus.name

      if mstiPortStatus.bridgeAssuranceStatus:
         # This must be a network port
         assert config.portConfig[ mstiPortStatus.name ].networkPort
         if instanceName not in instances:
            instances[ instanceName ] = SpanningTreeBridgeAssuranceInterfaces()
         if mstiPortStatus.bridgeAssureInconsistent:
            instances[ instanceName ].interfaces[ intfName
                  ] = SpanningTreeBridgeAssuranceValue(
                        bridgeAssuranceValue='inconsistent' )
         else:
            instances[ instanceName ].interfaces[ intfName
                  ] = SpanningTreeBridgeAssuranceValue(
                        bridgeAssuranceValue='consistent' )
      elif detail:
         if instanceName not in instances:
            instances[ instanceName ] = SpanningTreeBridgeAssuranceInterfaces()
         instances[ instanceName ].interfaces[ intfName
               ] = SpanningTreeBridgeAssuranceValue(
                     bridgeAssuranceValue='notNetworkPort' )

   return SpanningTreeBridgeAssurance( spanningTreeInstances=instances,
                                     bridgeAssuranceEnabled=config.bridgeAssurance )
      
#-------------------------------------------------------------------------------
# show spanning-tree counters
# Print the sum of all StpiPortStatus counters for each interface.
#-------------------------------------------------------------------------------
def showCounters( mode, args):
   # we should check the configuration in the case where STP is disabled to
   # prevent excessive wait times.
   config = stpConfig()

   if config.version == 'none':
      return SpanningTreeCounters( _message='Spanning-tree has been disabled in'
                                   ' the configuration.' )

   updatePortCounters( mode )
   status = stpStatus()
   counters = {}
   interfaces = {}

   for stpiName in ArPyUtils.naturalsorted( status.stpiStatus ):
      stpiStatus = status.stpiStatus.get( stpiName )
      if not stpiStatus:
         continue
      portKeys = stpiStatus.stpiPortStatus
      for key in Arnet.sortIntf( portKeys ):
         portStatus = stpiStatus.stpiPortStatus.get( key )
         if not portStatus:
            continue
         current = counters.get( key, (0, 0, 0, 0, 0) )
         new = portCounterValues( mode, stpiName, portStatus.name, portStatus )
         # This returns 5 columns. Of which first 4 are per port per vlan combination
         # RateLimiterCount (last column) is per port. It's already a multiple of
         # number of VLANS. So, summing up the values of first 4 columns to get
         # per port value. As the last column is already summed up, we are displaying
         # the max of the current and the new value. Current and the new will most
         # likely be the same. In some cases all the vlans don't receive same
         # number of bpdus hence displaying max.
         latest = list( map( sum, zip( current[ : - 1 ], new[ : - 1 ] ) ) )
         latest.append( max( current[ -1 ], new[ -1 ] ) )
         counters[ key ] = latest

   for key in counters: # pylint: disable=consider-using-dict-items
      interfaces[ key ] = SpanningTreeInterfaceCounters()
      interfaces[ key ].bpduSent = counters[key][0]
      interfaces[ key ].bpduReceived = counters[key][1]
      interfaces[ key ].bpduTaggedError = counters[key][2]
      interfaces[ key ].bpduOtherError = counters[key][3]
      interfaces[ key ].bpduRateLimitCount = counters[key][4]

   return SpanningTreeCounters( interfaces=interfaces )

#-------------------------------------------------------------------------------
# show spanning-tree inconsistentports
#-------------------------------------------------------------------------------
def showInconsistentPorts( mode ):
   status = stpStatus()
   total = 0

   print( 'Name                 Interface              Inconsistency' )
   print( '-------------------- ---------------------- ------------------' )

   for stpiName in ArPyUtils.naturalsorted( status.stpiStatus ):
      stpiStatus = status.stpiStatus.get( stpiName )
      if not stpiStatus:
         continue
      for key in ArPyUtils.naturalsorted( stpiStatus.mstiStatuses ):
         mstiStatus = stpiStatus.mstiStatuses.get( key )
         if not mstiStatus:
            continue
      
         for portKey in Arnet.sortIntf( mstiStatus.mstiPortStatus ):
            mstiPortStatus = mstiStatus.mstiPortStatus.get( portKey )
            if not mstiPortStatus:
               continue
            #TBD - When we implement detecting inconsistencies, like duplex.
            if mstiPortStatus.disputed:
               print( '%-20.20s %-22.22s %s' % ( mstiStatusCliName( mstiStatus ),
                                                mstiPortStatus.name,
                                                'inconsistent' ) )
               total += 1
      
   print()
   print( 'Number of inconsistent ports in the system : %d' % (
      total ) )

#-------------------------------------------------------------------------------
# show spanning-tree interface <interface> [ detail ]
#-------------------------------------------------------------------------------
def showInterface( mode, args ):
   intf = args[ 'ETHINTF' ] 
   detail = 'detail' in args
   Tracing.trace9( 'Name is', intf.name, 'Shortname is', intf.shortname )
   instances = {}
   status = stpStatus()
   config = stpConfig()
   if not intf.name in config.portConfig:
      return SpanningTreeInterfaces( interface=intf.name,
                                     spanningTreeInstances=instances,
                                     _interfaceExists=False )
   portConfig = config.portConfig[ intf.name ]

   if detail:
      updatePortCounters( mode )
      
   for stpiName in ArPyUtils.naturalsorted( status.stpiStatus ):
      stpiStatus = status.stpiStatus.get( stpiName )
      if not stpiStatus:
         continue
      for key in ArPyUtils.naturalsorted( stpiStatus.mstiStatus ):
         mstiStatus = stpiStatus.mstiStatus.get( key )
         if not mstiStatus:
            continue
         mstiPortStatus = mstiStatus.mstiPortStatus.get( intf.name )
         portStatus = stpiStatus.stpiPortStatus.get( intf.name )
         if mstiPortStatus and portStatus:
            # I've found an instance which contains my port
            instances[ mstiStatusCliName( mstiStatus ) ] = \
               getInterfaceCapi( mode, stpiStatus, mstiStatus, mstiPortStatus,
                                portStatus, portConfig, detail )

   return SpanningTreeInterfaces( interface=intf.name,
                                  spanningTreeInstances=instances,
                                  _detail=detail )
         
#-------------------------------------------------------------------------------
# show spanning-tree mst configuration [ digest ]
#-------------------------------------------------------------------------------
def showMstConfiguration( mode, args ):
   digest = 'digest' in args
   if digest:
      digest = stpStatus().mstpConfigDigest
   # Make a new MstConfig object that we can revert to the current config
   # and extract the info from that.
   currentMstConfig = MstConfig( stpConfig() )
   currentMstConfig.revert()
   return getMstInfoCapi( currentMstConfig, digest )

#-------------------------------------------------------------------------------
# show spanning-tree mst [ <instance-id> ] [ detail ]
#-------------------------------------------------------------------------------
def showMst( mode, args ):
   instId = args.get( 'MST' )
   detail = 'detail' in args

   status = stpStatus()
   stpiStatus = status.stpiStatus.get( MstStpiName )
   config = stpConfig()
   stpiConfig = config.stpiConfig.get( MstStpiName )

   instName = stpMstiInstName( instId ) if instId is not None else None

   detailRender = False

   if detail:
      detail = 'mst'
      detailRender = True

   if ( ( config.forceProtocolVersion != 'mstp' ) or ( stpiConfig is None ) or
      ( stpiStatus is None ) ):
      return SpanningTreeMstInstances( _message='MST is not running' )

   instances = {}
   for key in ArPyUtils.naturalsorted( stpiStatus.mstiStatus ):
      mstiStatus = stpiStatus.mstiStatus.get( key )
      if mstiStatus:
         if instName and (instName != mstiStatus.name):
            continue
         mstInstance = SpanningTreeMstInstance()
         mstInstance.mappedVlans = VlanList( vlans=vlansMappedToInstanceCapi(
            config, mstiStatus.instanceId ) )
         mstInstance.bridge = getBridgeBriefCapi( mstiStatus, 'bridgeId' )
         mstInstance.rootBridge = getRootBridgeCapi( stpiStatus, mstiStatus, 'mst' )
         mstInstance.regionalRootBridge = getRegionalRootBridgeCapi( stpiStatus, 
                                                                     mstiStatus )
         for portKey in mstiStatus.mstiPortStatus:
            mstiPortStatus = mstiStatus.mstiPortStatus.get( portKey )
            portStatus = stpiStatus.stpiPortStatus.get( portKey )
            portConfig = config.portConfig.get( portKey )
            if mstiPortStatus and portStatus and portConfig:
               mstInstance.interfaces[ mstiPortStatus.name ] = getInterfaceCapi( 
                                                      mode, stpiStatus, mstiStatus, 
                                                      mstiPortStatus, portStatus, 
                                                      portConfig, detail )
         instances[ mstiStatusCliName( mstiStatus ) ] = mstInstance
   return SpanningTreeMstInstances( spanningTreeMstInstances=instances,
                                    _detail=detailRender )

#-------------------------------------------------------------------------------
# show spanning-tree mst [ <instance-id> ] interface ETHINTF [ detail ]
#-------------------------------------------------------------------------------

def showMstInterface( mode, args ):
   instId = args.get( 'MST' )
   intf = args[ 'ETHINTF' ] 
   detail = 'detail' in args
   
   status = stpStatus()
   stpiStatus = status.stpiStatus.get( MstStpiName )
   config = stpConfig()
   stpiConfig = config.stpiConfig.get( MstStpiName )

   instName = stpMstiInstName( instId ) if instId is not None else None
   
   # Pre CAPI, the port counters were not being updated when this command was run
   updatePortCounters( mode )

   if detail:
      detail = 'mst'

   if ( ( config.forceProtocolVersion != 'mstp' ) or ( stpiConfig is None ) or
      ( stpiStatus is None ) ):
      return SpanningTreeMstInterfaces( _message='MST is not running' )

   baseInstId = instId
   if baseInstId is None:
      baseInstId = 0
   baseInstName = stpMstiInstName( baseInstId )
   mstiStatus = stpiStatus.mstiStatus.get( baseInstName )
   if not mstiStatus:
      return SpanningTreeMstInterfaces( _message='No mst information '
            'available for instance(s) %s on interface %s' % (baseInstId,
            intf.name) )
   mstiPortStatus = mstiStatus.mstiPortStatus.get( intf.name )
   portStatus = stpiStatus.stpiPortStatus.get( intf.name )   
   if not portStatus and detail:
      return SpanningTreeMstInterfaces( _message='No mst information '
            'available for interface %s' % ( intf.name ) )
   if not portStatus or not mstiPortStatus:
      return SpanningTreeMstInterfaces( _message='No mst information '
            'available for instance(s) %s on interface %s' % (baseInstId,
            intf.name) )
   portConfig = config.portConfig.get( intf.name )

   mstInterfaceCapi = getInterfaceCapi( mode, stpiStatus, mstiStatus,
                                        mstiPortStatus, portStatus, portConfig,
                                        'mstIntf' )
   mstInterfaceCapi.interfaceName = intf.name
   instName = stpMstiInstName( instId ) if instId is not None else None
   for key in stpiStatus.mstiStatus:
      mstiStatus = stpiStatus.mstiStatus.get( key )
      if mstiStatus:
         if instName and (instName != mstiStatus.name):
            continue
         mstiPortStatus = mstiStatus.mstiPortStatus.get( intf.name )
         if not mstiPortStatus:
            continue
         mstInterfaceCapi.mstInstances[ mstiStatusCliName( mstiStatus ) ] = \
                                                   getInterfaceCapi( 
                                                   mode, stpiStatus, mstiStatus, 
                                                   mstiPortStatus, portStatus, 
                                                   portConfig, detail, True )
   return SpanningTreeMstInterfaces( spanningTreeMstInterface=mstInterfaceCapi )

#-------------------------------------------------------------------------------
# show spanning-tree mst test information
#
# This is a command used for automated testing.  It prints out the current
# protocol state as python code.  This makes it easy to parse. You just take
# the output and feed it to python 'exec'.
#-------------------------------------------------------------------------------
def bridgeIdFwkValueString( bridgeId ):
   return 'Tac.Value( "Stp::BridgeId", priority=%r, systemId=%r, address=%r )' % \
         (bridgeId.priority, bridgeId.systemId, bridgeId.address)

def showMstTestInfo( mode, args ):
   config = stpConfig()
   inputConfig = stpConfig()
   status = stpStatus()
   stpiStatus = status.stpiStatus.get( MstStpiName )

   print( 'import MstInfo' )
   print( 'import Tac' )
   print( 'bi = MstInfo.BridgeInfo( "dut" )' )
   print( 'bi.stpVersion = "%s"' % inputConfig.forceProtocolVersion )
   mstConfigSpec = inputConfig.mstConfigSpec
   if mstConfigSpec.regionId:
      regionId = mstConfigSpec.regionId
   else:
      regionId = ''
   print( 'bi.mstpRegionId = "%s"' % regionId )
   print( 'bi.bridgeAddr = "%s"' % config.bridgeAddr )

   # For CAPI
   mstTestInfo = SpanningTreeBridgeMstTestInfoModel()
   mstTestInfo.name = "dut"
   mstTestInfo.stpVersion = inputConfig.forceProtocolVersion
   mstTestInfo.mstpRegionId = regionId
   mstTestInfo.bridgeAddr = config.bridgeAddr

   for ( stpiName, stpiStatus ) in status.stpiStatus.items():
      print( 'si = MstInfo.BridgeStpiInfo( "%s" )' % stpiName )
      print( 'bi.stpiInfoIs( "%s", si )' % stpiName )
      print( 'si.cistRoot = %s' % bridgeIdFwkValueString( stpiStatus.cistRoot ) )
      print( 'si.cistPathCost = %d' % stpiStatus.cistPathCost )
      
      # For CAPI
      stpiInfoInstance = BridgeStpiInfo()
      stpiInfoInstance.cistRoot = getBridgeBriefCapi( stpiStatus, 'cistRoot' )
      stpiInfoInstance.cistPathCost = stpiStatus.cistPathCost

      for key in ArPyUtils.naturalsorted( stpiStatus.mstiStatus ):
         mstiStatus = stpiStatus.mstiStatus.get( key )
         if not mstiStatus:
            continue
         if( isCist( mstiStatus ) and not isPvstInstName( mstiStatus.name ) ):
            bmiName = 'Mst0'
         else:
            bmiName = mstiStatus.name 
         print( 'bmi = MstInfo.BridgeMstiInfo( "%s" )' % bmiName )
         print( 'bmi.bridgeId = %s' % bridgeIdFwkValueString( mstiStatus.bridgeId ) )
         print( 'bmi.designatedRoot = %s' %
                bridgeIdFwkValueString( mstiStatus.designatedRoot ) )
         print( 'si.mstiInfoIs( "%s", bmi )' % bmiName )
         
         # For CAPI
         mstiInfoInstance = BridgeMstiInfo()
         mstiInfoInstance.bridgeId = getBridgeBriefCapi( mstiStatus, 'bridgeId' )
         mstiInfoInstance.designatedRoot = getBridgeBriefCapi( mstiStatus, 
                                                               'designatedRoot' )
         stpiInfoInstance.mstpInstances[ bmiName ] = mstiInfoInstance

         for mps in mstiStatus.mstiPortStatus.values():
            print( 'bmii = MstInfo.BridgeMstiIntfInfo( "%s", "%s" )' %
                   ( bmiName, mps.name ) )
            print( 'bmii.portId = Tac.Value( "Stp::PortId", ' )
            print( '                         portPriority=%d, portNumber=%d )' %
                   ( mps.portId.portPriority, mps.portId.portNumber ) )
            print( 'bmii.role = "%s"' % mps.role )
            print( 'bmii.operIntPathCost = %d' % mps.operIntPathCost )
            print( 'bmii.fdbFlush = %d' % mps.fdbFlush )
            print( 'bmi.mstiIntfInfoIs( "%s", bmii )' % mps.name )

            # For CAPI
            mstiIntfInfoInstance = BridgeMstiIntfInfo()
            mstiIntfInfoInstance.portId = getPortIdCapi( mps.portId )
            mstiIntfInfoInstance.role = mps.role
            mstiIntfInfoInstance.internalPathCost = mps.operIntPathCost
            mstiIntfInfoInstance.fdbFlushes = mps.fdbFlush
            mstiInfoInstance.interfaces[ mps.name ] = mstiIntfInfoInstance
            
            if( (bmiName == 'Mst0') or isPvstInstName( bmiName ) ):
               print( 'bii = MstInfo.BridgeIntfInfo( "%s" )' % mps.name )
               portStatus = stpiStatus.stpiPortStatus.get( mps.name )
               cost = portStatus.operExtPathCost if portStatus else 0
               print( 'bii.operExtPathCost = %d' % cost )
               print( 'si.intfInfoIs( "%s", bii )' % mps.name )
               
               # For CAPI
               intfInfoInstance = BridgeIntfInfo()
               intfInfoInstance.externalPathCost = cost
               stpiInfoInstance.interfaces[ mps.name ] = intfInfoInstance
         # For CAPI
         mstTestInfo.stpInstances[ stpiName ] = stpiInfoInstance
   return mstTestInfo

#-------------------------------------------------------------------------------
# show spanning-tree root [ detail ]
#-------------------------------------------------------------------------------
def showRoot( mode, args ):
   detail = 'detail' in args
   status = stpStatus()

   roots = {} 
   for stpiName in ArPyUtils.naturalsorted( status.stpiStatus ):
      stpiStatus = status.stpiStatus.get( stpiName )
      if not stpiStatus:
         continue
      for key in ArPyUtils.naturalsorted( stpiStatus.mstiStatus ):
         mstiStatus = stpiStatus.mstiStatus.get( key )
         if not mstiStatus:
            continue

         root = SpanningTreeRoot()
         mstiRootPort = mstiStatus.rootPort
         if mstiRootPort:
            root.rootPort = getRootPortCapi( stpiStatus, mstiStatus, 
                                             mstiRootPort )
         root.rootBridge = getRootBridgeCapi( stpiStatus, mstiStatus, None )

         roots[ mstiStatusCliName( mstiStatus ) ] = root

   return SpanningTreeRoots( instances=roots, _detail=detail )

#-------------------------------------------------------------------------------
# show spanning-tree summary [ totals ]
# "totals" has the same output, except it only prints the totals line, and
# suppresses the lines for each instance (Vlan for PVST).
#-------------------------------------------------------------------------------
def showSummary( mode, totals ):
   pass

# Not implementing this for now. 02/19/2008
# BasicCli.registerShowCommand( tokenShowStp, tokenShowSummary,
#                               [ '>>totals', tokenShowTotals ], showSummary )

#-------------------------------------------------------------------------------
# show spanning-tree vlan <vlan-id>
#-------------------------------------------------------------------------------
def showSpanVlan( mode, args ):
   vlanSet = args.get( 'VLANSET' )
   detail = 'detail' in args
   config = stpConfig()
   instances = {}

   vlans = VlanCli.Vlan.getAll( mode )
   vlansInternal = []
   if vlanSet is not None:
      vlans = [ v for v in vlans if v.id in vlanSet ]
      vlansInternal = VlanCli.Vlan.getAll( mode, internal=True )
      vlansInternal = [ v for v in vlansInternal if v.id in vlanSet ]
      if not vlans and not vlansInternal:
         return SpanningTreeVlanInstances( spanningTreeVlanInstances=instances,
                  _message='No VLANs matching %s found in current vlan database' 
                                                                     % vlanSet )
      vlans.extend( vlansInternal )

   vlans.sort()              

   for v in vlans:
      if v.id in config.disabledVlan and config.disabledVlan[ v.id ]:
         # Child?
         instances[ v.id ] = SpanningTreeVlanInstance(
                           _message='Spanning tree is disabled for vlan %d' % v.id )
         continue

      if v in vlansInternal:
         instances[ v.id ] = SpanningTreeVlanInstance(
            _message='No spanning tree information available for vlan %d' % v.id )
         continue

      stpiConfig, stpiStatus = stpVlanStpiConfigStatus( v.id )
      if (not stpiConfig) or (not stpiStatus):
         instances[ v.id ] = SpanningTreeVlanInstance(
            _message='No spanning tree information available for vlan %d\n' % v.id )
         continue
        
      mstiConfig = stpVlanMstiConfig( stpiConfig, v.id )
      mstiStatus = stpiStatus.mstiStatus.get( mstiConfig.name )
      if (not mstiConfig) or (not mstiStatus):
         instances[ v.id ] = SpanningTreeVlanInstance(
            _message='No spanning tree information available for vlan %d\n' % v.id )
         continue
   
      if detail:
         updatePortCounters( mode )
          
      Tracing.trace2( 'MstiConfig is', mstiConfig.name )
      spanningTreeInstance = getSpanningTreeInstance( mode, config, stpiStatus, 
                                                      mstiStatus, detail, v.id )
      if spanningTreeInstance:
         instances[ v.id ] = SpanningTreeVlanInstance(
                                 spanningTreeVlanInstance=spanningTreeInstance )
         instances[ v.id ].spanningTreeVlanInstance.instanceName = \
                                 mstiStatusCliName( mstiStatus )

   return SpanningTreeVlanInstances( spanningTreeVlanInstances=instances )
   
#------------------------------------------------------
# Register Stp show commands into "show tech-support".
#------------------------------------------------------
# Timestamps are made up to maintain historical order within show tech-support
CliPlugin.TechSupportCli.registerShowTechSupportCmd( 
   '2010-01-01 00:10:00',
   cmds=[ 'show spanning-tree detail',
          'show spanning-tree topology status detail',
          'show spanning-tree instance detail' ] )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   status = LazyMount.mount( entityManager, "stp/status", "Stp::Status", "r" )
   stpStatusIs( status )
   cliInfo = LazyMount.mount( entityManager, "stp/txrx/cliInfo",
                              "StpTxRx::CliInfo", "r" )
   stpCliInfoIs( cliInfo )
   config = LazyMount.mount( entityManager, "stp/config", "Stp::Config", "r" )
   stpConfigIs( config )
   counterDir = LazyMount.mount(
      entityManager, "stp/counter", "Stp::PortCounterDir", "w" )
   stpPortCounterDirIs( counterDir )
   bridgingCfg = LazyMount.mount(
      entityManager, "bridging/config", "Bridging::Config", "r" )
   bridgingConfigIs( bridgingCfg )
   topoStatus = LazyMount.mount( entityManager, "stp/topology/status",
                              "Stp::Topology::Status", "r" )
   stpTopoStatusIs( topoStatus )
