#!/usr/bin/env python3
# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import  Tac
import TableOutput

from ArnetModel import IpAddrAndPort, IpGenericAddress
from CliModel import Bool
from CliModel import Dict
from CliModel import Enum
from CliModel import Int
from CliModel import GeneratorDict
from CliModel import List
from CliModel import Model
from CliModel import Str
from CliModel import Submodel
from datetime import timedelta
from operator import attrgetter
from IntfModels import Interface
from time import ctime

showTlvFmt = "{:<16} {:>6} {:<16}"

def Fmt( justify='left', minWidth=5, maxWidth=10, wrap=False ):
   _fmt = TableOutput.Format( justify=justify, minWidth=minWidth,
                              maxWidth=maxWidth, wrap=wrap )
   _fmt.noPadLeftIs( True )
   _fmt.padLimitIs( True )
   return _fmt

agentFmt = Fmt()
transactionIdFmt = Fmt( minWidth=24, maxWidth=24, justify="right" )
addressFmt = Fmt( minWidth=21, maxWidth=21 )
tlvFmt = Fmt( maxWidth=10, justify="right" )
timeoutFmt = Fmt( minWidth=8, maxWidth=12, wrap=True )

showServerHeadings = ( "Transaction ID", "Public Address", "Attributes", "Timeout" )
showClientHeadings = ( "Agent", "Transaction ID", "Source Address", "Public Address",
                       "Last Refreshed" )

class StunTlvModel( Model ):
   tlvType = Str( help='STUN TLV type' )
   tlvLen = Int( help='STUN TLV length' )
   tlvValue = Str( help='STUN TLV value' )

   def render( self ):
      print( showTlvFmt.format( self.tlvType, self.tlvLen,
                                self.tlvValue ) )

class StunServerBindingModel( Model ):
   publicAddress = Submodel( valueType=IpAddrAndPort,
                             help='Translated IP address and port' )
   numTlvs = Int( help='Number of STUN Attributes' )
   timeout = Int( help='Seconds remaining until timeout', optional=True )
   timeoutInterval = Int( help='Bindings timeout interval in seconds' )
   tlvs = List( valueType=StunTlvModel, help="STUN Attributes" )

   def getTimeout( self ):
      if not self.timeout:
         return "n/a"
      return str( timedelta( seconds=self.timeout ) )

   def renderTlvs( self ):
      print( showTlvFmt.format( "Attribute Type", "Length",
                                "Value" ) )
      print( showTlvFmt.format( '-' * 16, '-' * 6, '-' * 16 ) )
      for tlv in sorted( self.tlvs, key=attrgetter( "tlvType" ) ):
         tlv.render()

   def renderBinding( self, tid ):
      timeoutInterval = str( timedelta( seconds=self.timeoutInterval ) )
      print( "Transaction ID", tid )
      print( "Public Address:", self.publicAddress.formatStr() )
      print( "Number of Attributes:", str( self.numTlvs ) )
      print( "Timeout Interval:", timeoutInterval )
      print( "Timeout:", self.getTimeout() )
      if self.numTlvs:
         self.renderTlvs()
      print( "" )

   def values( self, tid ):
      return [ tid, self.publicAddress.formatStr(), self.numTlvs,
               self.getTimeout() ]

class StunServerBindingsModel( Model ):
   bindings = GeneratorDict( keyType=str, valueType=StunServerBindingModel,
                             help="STUN binding bindings keyed by transaction ID" )
   _detailedView = Bool( help="Detail view" )

   def renderTable( self ):
      table = TableOutput.createTable( showServerHeadings )
      table.formatColumns( transactionIdFmt, addressFmt, tlvFmt, timeoutFmt )
      for tid, binding in sorted( self.bindings ) :
         table.newRow( *( binding.values( tid ) ) )
      print( table.output() )

   def renderDetail( self ):
      for tid, binding in sorted( self.bindings ):
         binding.renderBinding( tid )

   def render( self ):
      print( 'Current System Time:', ctime( Tac.utcNow() ) )
      if self._detailedView:
         self.renderDetail()
      else:
         self.renderTable()

class StunClientBindingModel( Model ):
   agentName = Str( help="Agent Name" )
   sourceAddress = Submodel( valueType=IpAddrAndPort,
                             help="Source IP address and port",
                             optional=True )

   publicAddress = Submodel( valueType=IpAddrAndPort,
                             help='Translated IP address and port',
                             optional=True )
   lastRefreshed = Int( help='Seconds elapsed since last response', optional=True )
   timeoutInterval = Int( help='STUN bindings timeout interval in seconds' )
   tlvs = List( valueType=StunTlvModel, help="STUN Attributes" )

   def getLastRefreshedTime( self ):
      if self.lastRefreshed is None :
         return "n/a"
      return str( timedelta( seconds=self.lastRefreshed ) ) + " ago"

   def getPublicAddress( self ):
      if not self.publicAddress:
         return "n/a"
      return self.publicAddress.formatStr()

   def getSourceAddress( self ):
      if not self.sourceAddress:
         return "n/a"
      return self.sourceAddress.formatStr()

   def renderTlv( self ):
      print( showTlvFmt.format( "Attribute Type", "Length", "Value" ) )
      print( showTlvFmt.format( '-' *16, '-' * 6, '-' * 16 ) )
      for tlv in sorted( self.tlvs, key=attrgetter( "tlvType" ) ):
         tlv.render()

   def renderBinding( self, tid ):
      numTlvs = len( self.tlvs ) if self.tlvs else 0
      print( "Transaction ID", tid )
      print( "Agent Name:", self.agentName )
      print( "Source Address:", self.getSourceAddress() )
      print( "Public Address:", self.getPublicAddress() )
      print( "Number of Attributes:", numTlvs )
      print( "Timeout Interval:", str( timedelta( seconds=self.timeoutInterval ) ) )
      print( "Last Refreshed:", self.getLastRefreshedTime() )
      if self.tlvs:
         self.renderTlv()
      print( "" )

   def values( self, tid ):
      return [ self.agentName, tid, self.getSourceAddress(),
               self.getPublicAddress(), self.getLastRefreshedTime() ]

class StunClientBindingsModel( Model ):
   bindings = GeneratorDict( keyType=str, valueType=StunClientBindingModel,
                             help="STUN bindings keyed by transaction ID" )
   _detailedView = Bool( help="Detail view" )

   def renderTable( self ):
      table = TableOutput.createTable( showClientHeadings, tableWidth=90 )
      table.formatColumns( agentFmt, transactionIdFmt, addressFmt,
                           addressFmt, timeoutFmt )
      for tid, binding in sorted( self.bindings ):
         table.newRow( *( binding.values( tid ) ) )
      print( table.output() )

   def renderDetail( self ):
      for tid, binding in sorted( self.bindings ):
         binding.renderBinding( tid )

   def render( self ):
      print( 'Current System Time:', ctime( Tac.utcNow() ) )
      if self._detailedView:
         self.renderDetail()
      else:
         self.renderTable()

class StunServerStatusModel( Model ):
   enabled = Bool( help="Service is enabled" )
   pid = Int( help="Process ID of the service", optional=True )
   timeOfLastRcvdBindingReq = Int( help="Seconds elapsed since the last binding "
                                        "response was received" )
   authMode = Enum( values=( "none", "ssl", "stunLongTermCredentials" ),
                    help="Authentication mode" )
   listeningIps = Dict( keyType=IpGenericAddress, valueType=Interface,
                        help="Listening IP address and interface of the service" )
   listeningPort = Int( help="Listening port of the service" )
   bindingTimeout = Int( help="Bindings timeout interval in seconds" )
   sslConnectionLifetime = Int( help="SSL connection lifetime in seconds" )

   def getLastRcvdBindingTime( self ):
      if not self.timeOfLastRcvdBindingReq:
         return "n/a"
      return str( timedelta( seconds=self.timeOfLastRcvdBindingReq ) ) + " ago"

   def render( self ):
      print( "Service status:", "enabled" if self.enabled else "disabled" )
      if self.pid:
         print( f"Service running: yes (process ID {self.pid})" )
      else:
         print( "Service running: no" )
      print( "Last binding received:", self.getLastRcvdBindingTime() )
      authStr = 'none'
      if self.authMode == "ssl":
         authStr = "SSL"
      elif self.authMode == "stunLongTermCredentials":
         authStr= "STUN long term credentials"
      print( "Authentication mode:", authStr )
      ipAndIntfStr = ", ".join( sorted( f"{addr} ({intf.shortName})"
                                      for addr, intf in self.listeningIps.items() ) )
      print( "Listening IP:", ipAndIntfStr or "n/a" )
      print( "Listening port:", self.listeningPort )
      print( "Binding timeout:", self.bindingTimeout, "seconds" )
      print( "SSL connection lifetime:", self.sslConnectionLifetime, "seconds" )

class StunTransportCountersModel( Model ):
   transport = Str( help="SSL or UDP" )
   successfulInitializations = Int( help="No of successful initializations" )
   initializationFailures = Int( help="No of times the initialization failed" )
   readFailures = Int( help="No of read failures" )
   writeFailures = Int( help="No of write failures" )
   connectionErrors = Int( help="Connection errors" )

   def render( self ):
      print( "{} successful initializations: {}".format( self.transport,
                                                   self.successfulInitializations ) )
      print( "{} failed initializations: {}".format( self.transport,
                                                    self.initializationFailures ) )
      print( f"{self.transport} read failures: {self.readFailures}" )
      print( f"{self.transport} write failures: {self.writeFailures}" )
      print( "{} connection errors: {}".format( self.transport,
                                               self.connectionErrors ) )

class StunServerProfileCountersModel( Model ):
   tid = Str( help="Transaction ID" )
   name = Str( help="Server profile name" )
   refreshRequests = Int( help="No of refresh requests sent" )
   numRetries = Int( help="No of retries carried out for failed requests" )
   retriesExhausted = Int( help="No of times the entire retry process failed" )
   successfulResponses = Int( help="No of successful responses received" )
   invalidResponses = Int( help="No of invalid responses recevied" )
   responseProcessingFailures = Int( help="No of times an error was encountered"
                                          " during the processing of a received"
                                          " response" )
   sslTransportCounters = Submodel( valueType=StunTransportCountersModel,
                                      help="SSL transport counters" )
   udpTransportCounters = Submodel( valueType=StunTransportCountersModel,
                                      help="UDP transport counters" )

   def render( self ):
      print( "\nTransaction ID: {}, server profile: {}".format( self.tid,
                                                                self.name ) )
      print( f"Refresh requests: {self.refreshRequests}" )
      print( f"Number of retries: {self.numRetries}" )
      print( f"Retry process failures: {self.retriesExhausted}" )
      print( f"Successful responses: {self.successfulResponses}" )
      print( f"Invalid responses: {self.invalidResponses}" )
      print( "Failure to process the response: {}".format(
                                                self.responseProcessingFailures ) )
      self.sslTransportCounters.render()
      self.udpTransportCounters.render()

class StunClientCountersModel( Model ):
   cleanStarts = Int( help="No of times the agent started with proper cleanup done"
                           " by the previous run" )
   nonCleanStarts = Int( help="No of times the agent started without proper cleanup"
                              " done by the previous run")
   stunServerProfileCounters = List( valueType=StunServerProfileCountersModel,
                                     help="Counters associated with a server profile"
                                   )

   def render( self ):
      if not self.stunServerProfileCounters:
         return
      print( f"Clean starts of the agent: {self.cleanStarts}" )
      print( f"Non clean starts of the agent: {self.nonCleanStarts}" )
      for spCntrs in self.stunServerProfileCounters:
         spCntrs.render()
