#!/usr/bin/env python3
# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from binascii import  hexlify
from ipaddress import IPv4Address, AddressValueError
from struct import unpack, error
import Tac

class TlvBaseParser:
   def __init__( self, value ):
      self.value = value

   def typeStr( self ):
      raise NotImplementedError

   def valueStr( self ):
      raise NotImplementedError

class WanIdParser( TlvBaseParser ):
   def __init__( self, value ):
      TlvBaseParser.__init__( self, value )

   def typeStr( self ):
      return "WAN_ID"

   def valueStr( self ):
      try:
         # Note: Value is in Network byte order
         val = unpack( "!I", self.value )[ 0 ]
         return str( val )
      except error:
         return hexlify( self.value )

class PathGroupParser( TlvBaseParser ):
   def __init__( self, value ):
      TlvBaseParser.__init__( self, value )

   def typeStr( self ):
      return "PATH_GROUP"

   def valueStr( self ):
      return self.value.decode()

class VtepIpParser( TlvBaseParser ):
   def __init__( self, value ):
      TlvBaseParser.__init__( self, value )

   def typeStr( self ):
      return "VTEP_IP"

   def valueStr( self ):
      try:
         # Note: Value is in network byte order
         val = unpack( "!I", self.value )[ 0 ]
         return str( IPv4Address( val ) )
      except ( AddressValueError, error ):
         return hexlify( self.value )

tlvTypeToParser = {
   # Type : Parser Class
   Tac.enumValue( 'Stun::DpsStunAttribute', 'StunWanId' ) : WanIdParser,
   Tac.enumValue( 'Stun::DpsStunAttribute', 'StunPathGroup' ) : PathGroupParser,
   Tac.enumValue( 'Stun::DpsStunAttribute', 'StunVtepIp' ) : VtepIpParser,
}
