# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements the Switch interface type.  In particular, it provides
# the SwitchIntf class.
#-------------------------------------------------------------------------------

from os import getenv

import CliParser
# pylint: disable-next=consider-using-from-import
import CliPlugin.EthIntfCli as EthIntfCli
import CliPlugin.IntfCli as IntfCli # pylint: disable=consider-using-from-import
# pylint: disable-next=consider-using-from-import
import CliPlugin.IntfStatusMatcher as IntfStatusMatcher
import Intf.IntfRange as IntfRange # pylint: disable=consider-using-from-import
from IntfRangePlugin.SwitchIntf import SwitchAutoIntfType
import LazyMount
import QuickTrace
import Tac

qv = QuickTrace.Var
qt0 = QuickTrace.trace0
qt1 = QuickTrace.trace1
qt8 = QuickTrace.trace8

switchPhyIntfStatusDir = None
globalIntfConfig = None

switchPhyIntfStatusPath = 'interface/status/eth/phy/all'

switchIntfGuards_ = []

def switchIntfGuard( mode, token ):
   if getenv( "SIMULATION_MAKO_PRODUCT" ):
      return None
   for fn in switchIntfGuards_:
      if fn( mode, token ) is None:
         return None
   return CliParser.guardNotThisPlatform

# If Fru has not started yet or entity mib root has not yet fully been created,
# each element in limits is the virtual limit for each part of a Switch interface
limits = [ ( 1, 7 ), ( 1, 65 ), ( 1, 8 ) ]

defaultMatcher = IntfStatusMatcher.DefaultIntfNumberMatcher(
   limits, minParts=2, maxParts=3 )
defaultSubMatcher = IntfStatusMatcher.DefaultIntfNumberMatcher(
   limits, minParts=2, maxParts=3, subIntf=True )

def valueFunc_( mode, intf ):
   return SwitchIntf( intf, mode )

def allSwitchIntfStatus( mode ):
   intfs = [ intf for intf in switchPhyIntfStatusDir.intfStatus
             if intf.startswith( 'Sw' ) ]
   return intfs

#-------------------------------------------------------------------------------
# A subclass of the base EthIntfCli.EthIntf class for Switch interfaces.
#-------------------------------------------------------------------------------
class SwitchIntf( EthIntfCli.EthPhyIntf ):

   #----------------------------------------------------------------------------
   # Creates a new SwitchIntf instance of the specified name.
   #----------------------------------------------------------------------------
   def __init__( self, name, mode ):
      EthIntfCli.EthPhyIntf.__init__( self, name, mode )

   matcher = IntfStatusMatcher.DefaultIntfMatcher( 'Switch',
                                                   defaultMatcher,
                                                   helpdesc='Switch interface',
                                                   guard=switchIntfGuard,
                                                   value=valueFunc_ )

   #----------------------------------------------------------------------------
   # Returns the EthIntfXcvrStatus object for this interface.
   #----------------------------------------------------------------------------
   def intfXcvrStatus( self ):
      dummyEixs = Tac.newInstance( 'Interface::EthIntfXcvrStatus', '' )
      if self.status():
         dummyEixs.xcvrPresence = 'xcvrPresent'
      return dummyEixs

   #----------------------------------------------------------------------------
   # Outputs information about this interface in an interface type-specific
   # manner.
   #----------------------------------------------------------------------------
   def hardware( self ):
      return "switch"

   def countersSupported( self ):
      return True

   def routingSupported( self ):
      return True

   def switchportEligible( self ):
      return True

   def vrfSupported( self ):
      return True

   def xcvrTypeStr( self ):
      return "N/A"

   def internal( self ):
      # TODO BUG587806: Switch interfaces should eventually be treated as internal
      #                 interfaces.
      return globalIntfConfig.peerAliasedLinksEnabled

   def linkModeUnsupportedErrorMsg( self ):
      # pylint: disable-next=consider-using-f-string
      return ( 'Speed and duplex settings are not available on interface %s.'
                % self.name )

   #----------------------------------------------------------------------------
   # Returns an unsorted list of SwitchIntf objects representing all the existing
   # Interface::SwitchIntfStatus objects.
   #----------------------------------------------------------------------------
   @staticmethod
   def getAllPhysical( mode ):
      intfs = []
      for name in switchPhyIntfStatusDir.intfStatus:
         if not name.startswith( 'Switch' ):
            continue
         intf = SwitchIntf( name, mode )
         if intf.lookupPhysical():
            intfs.append( intf )
      return intfs

#-------------------------------------------------------------------------------
# Register the SwitchIntf class as a type of physical interface.
#-------------------------------------------------------------------------------
IntfCli.Intf.addPhysicalIntfType( SwitchIntf, SwitchAutoIntfType,
                                  withIpSupport=True,
                                  withRoutingProtoSupport=True,
                                  matcherWithRoutingProtoSupport=SwitchIntf.matcher )
SwitchAutoIntfType.registerEthPhyIntfClass( SwitchIntf )

IntfRange.registerIntfTypeGuard( SwitchAutoIntfType, switchIntfGuard )

#-------------------------------------------------------------------------------
# Adds SwitchIntf specific CLI commands to the "config-if" mode.
#-------------------------------------------------------------------------------
class SwitchIntfModelet( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      return isinstance( mode.intf, SwitchIntf )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global switchPhyIntfStatusDir
   global globalIntfConfig

   switchPhyIntfStatusDir = LazyMount.mount( entityManager,
      switchPhyIntfStatusPath, "Interface::AllEthPhyIntfStatusDir", "r" )
   globalIntfConfig = LazyMount.mount( entityManager,
      'interface/config/global', "Interface::GlobalIntfConfig", "r" )
