# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import (
   absolute_import,
   division,
   print_function,
)

import functools

import AgentDirectory
from Ark import LazyModule
from BasicCli import addShowCommandClass
from CliCommand import singleKeyword
from CliGlobal import CliGlobal
import LazyMount
from ShowCommand import ShowCliCommandClass
from SynceTypesFuture import Synce
from TypeFuture import TacLazyType
import six

from CliPlugin.PhysicalIntfRule import PhysicalIntfMatcher
from CliPlugin.SynceModel import (
   SynceEsmcCounters,
   SynceEsmcCountersInfo,
   SynceEsmcCountersRxInfo,
   SynceEsmcCountersTxInfo,
   SynceEsmcDetail,
   SynceEsmcDetailInfo,
   SynceEsmcDetailRxInfo,
   SynceEsmcDetailTxInfo,
   SynceSelectionStatus,
   SynceSelectionStatuses,
   SynceStatus,
   SynceStatusInfo,
)

IntfId = TacLazyType( 'Arnet::IntfId' )

Tac = LazyModule( 'Tac' )

gv = CliGlobal(
   dict(
      clockTimerStatus=None,
      eecStatus=None,
      effectiveBestClockStatus=None,
      esmcRxStats=None,
      esmcTxStats=None,
      instanceConfig=None,
      hwCapability=None,
      hwConfig=None,
      nominatedClockStatus=None,
   ) )

synceKeyword = singleKeyword( 'sync-e', 'Synchronous Ethernet' )
esmcKeyword = singleKeyword( 'esmc', 'Show ESMC message statistics' )
phyIntfMatcher = PhysicalIntfMatcher( 'Ethernet' )

def showSynceWarnings( mode ):
   if not gv.instanceConfig.enable:
      mode.addWarning( "Agent 'Synce' is not enabled" )
   elif not AgentDirectory.agentIsRunning( mode.entityManager.sysname(), 'Synce' ):
      mode.addWarning( "Agent 'Synce' is not running" )

class ShowSynceStatusCmd( ShowCliCommandClass ):
   syntax = 'show SYNCE'
   data = {
      'SYNCE': synceKeyword,
   }
   cliModel = SynceStatus

   @staticmethod
   def handler( mode, args ):
      showSynceWarnings( mode )
      # We only support QL-enabled, so we always pass in qualityLevel
      qlEnabled = True
      option = ( gv.instanceConfig.networkOption
                 if gv.instanceConfig.networkOption != Synce.NetworkOption.invalid
                 else None )
      clockSource = ( gv.eecStatus.selectedClock.clockSource if
                      gv.eecStatus.selectedClock.clockSource != Synce.ClockSource()
                      else None )
      localQl = gv.hwCapability.localQualityLevel( option ) if option else None
      clockIdentity = None
      numEeecHop = None
      numEecHop = None
      extendedFlag = None
      if gv.eecStatus.selectedClock.clockIdentity != '00:00:00:00:00:00:00:00':
         clockIdentity = Tac.Value(
            'Arnet::Eui64', stringValue=gv.eecStatus.selectedClock.clockIdentity )
         numEeecHop = gv.eecStatus.selectedClock.numEeecHop
         numEecHop = gv.eecStatus.selectedClock.numEecHop
         extendedFlag = gv.eecStatus.selectedClock.extendedFlag
      infoModel = None
      if gv.instanceConfig.enable:
         infoModel = SynceStatusInfo.factory_(
            qlEnabled,
            gv.eecStatus.state,
            gv.instanceConfig.holdoff.seconds(),
            float( gv.instanceConfig.waitToRestore.seconds ),
            networkOption=option,
            clock=clockSource,
            ql=gv.eecStatus.qualityLevel,
            localQl=localQl,
            clockIdentity=clockIdentity,
            cascadedEeecCount=numEeecHop,
            cascadedEecCount=numEecHop,
            etlvFlag=extendedFlag )
      return SynceStatus( info=infoModel )

addShowCommandClass( ShowSynceStatusCmd )

class ShowSynceSelectionCmd( ShowCliCommandClass ):
   syntax = 'show SYNCE selection'
   data = {
      'SYNCE':
      synceKeyword,
      'selection': ( 'Show candidate clock sources participating in '
                     'the selection process' ),
   }
   cliModel = SynceSelectionStatuses

   @staticmethod
   def handler( mode, args ):
      showSynceWarnings( mode )

      def getModelExpTime( timerStatus ):
         ''' Returns a tuple( holdoffExpTime, waitToRestoreExpTime ) in
             the format expected by the CAPI model.

             Specifically, If the expiry time == Tac.endOfTime, we return
             None instead (since it's optional in the Model)
         '''
         def expiryTime( expTime ):
            return expTime if expTime != Tac.endOfTime else None

         if timerStatus is not None:
            return ( expiryTime( timerStatus.holdoffExpTime ),
                     expiryTime( timerStatus.waitToRestoreExpTime ) )
         else:
            return ( None, None )

      entries = []

      def preferred( lhs, rhs ):
         if Synce.QualityLevelHelper.preferred( lhs.qualityLevel, rhs.qualityLevel ):
            return -1
         return 1

      option = ( gv.instanceConfig.networkOption
                 if gv.instanceConfig.networkOption != Synce.NetworkOption.invalid
                 else None )
      if option is None:
         # Synce runnability is dependent on sync-e submode, not network option,
         # even though it's a mandatory config. If it's not yet configured, return
         # an empty output
         return SynceSelectionStatuses( clockSources=entries )

      localQl = gv.hwCapability.localQualityLevel( option )
      for qlClocks in sorted(
            gv.nominatedClockStatus.clockStatusByQualityLevel.values(),
            key=functools.cmp_to_key( preferred ) ):
         for prioClocks in six.itervalues( qlClocks.clockStatusByPriority ):
            for clock, clockStatus in six.iteritems( prioClocks.clock ):
               timerStatus = gv.clockTimerStatus.timerStatus.get( clock )
               holdoffExpTime, waitToRestoreExpTime = getModelExpTime( timerStatus )
               # The mount must be forced prior to invoking C++.
               gv.eecStatus.force()
               gv.effectiveBestClockStatus.force()
               status = Synce.CliHelper.selectionStatusCapiValue(
                  clockStatus, gv.eecStatus, gv.effectiveBestClockStatus, localQl )
               entries.append(
                  SynceSelectionStatus.factory_(
                     clock,
                     status,
                     clockStatus.priority,
                     ql=qlClocks.qualityLevel,
                     holdoff=holdoffExpTime,
                     waitToRestore=waitToRestoreExpTime ) )
      return SynceSelectionStatuses( clockSources=entries )

addShowCommandClass( ShowSynceSelectionCmd )

def getEsmcSsmAndQl( ssmTupleFromEsmc, option ):
   return ( ssmTupleFromEsmc,
            Synce.QualityLevelHelper.fromSsm( ssmTupleFromEsmc.ssmTuple(), option ) )

class ShowSynceEsmcCountersCmd( ShowCliCommandClass ):
   syntax = 'show SYNCE ESMC counters'
   data = {
      'SYNCE': synceKeyword,
      'ESMC': esmcKeyword,
      'counters': 'Show ESMC RX and TX counters',
   }
   cliModel = SynceEsmcCounters

   @staticmethod
   def handler( mode, args ):
      showSynceWarnings( mode )
      option = gv.instanceConfig.networkOption
      intfs = set( gv.esmcRxStats.esmcRxStat ) | set( gv.esmcTxStats.esmcTxStat )
      entries = {}
      for intf in intfs:
         rxInfo = gv.esmcRxStats.esmcRxStat.get( intf )
         rxModel = None
         if rxInfo is not None:
            ssmTuple, ql = ( getEsmcSsmAndQl( rxInfo.ssmTuple, option )
                             if rxInfo.isLastValid() else ( None, None ) )
            rxModel = SynceEsmcCountersRxInfo( validCount=rxInfo.validCount,
                                               invalidCount=rxInfo.invalidCount )
            if ql:
               rxModel.lastQualityLevel = Synce.QualityLevelHelper.toStr( ql )
            if ssmTuple:
               rxModel.lastSsm = ssmTuple.ssm
               if ssmTuple.essmSet:
                  rxModel.lastEssm = ssmTuple.essm
                  rxModel.lastCascadedEeecCount = rxInfo.numEeecHop
                  rxModel.lastCascadedEecCount = rxInfo.numEecHop

         txInfo = gv.esmcTxStats.esmcTxStat.get( intf )
         txModel = None
         if txInfo is not None:
            ssmTuple, ql = ( getEsmcSsmAndQl( txInfo.ssmTuple, option )
                             if txInfo.totalCount else ( None, None ) )
            txModel = SynceEsmcCountersTxInfo( sentCount=txInfo.totalCount )
            if ql:
               txModel.lastQualityLevel = Synce.QualityLevelHelper.toStr( ql )
            if ssmTuple:
               txModel.lastSsm = ssmTuple.ssm
               if ssmTuple.essmSet:
                  txModel.lastEssm = ssmTuple.essm
                  txModel.lastCascadedEeecCount = txInfo.numEeecHop
                  txModel.lastCascadedEecCount = txInfo.numEecHop
         entries[ intf ] = SynceEsmcCountersInfo( rxInfo=rxModel, txInfo=txModel )
      return SynceEsmcCounters( clockSourceCounters=entries )

addShowCommandClass( ShowSynceEsmcCountersCmd )

class ShowSynceEsmcDetailCmd( ShowCliCommandClass ):
   syntax = 'show SYNCE ESMC detail [ETH]'
   data = {
      'SYNCE': synceKeyword,
      'ESMC': esmcKeyword,
      'detail': 'Show detailed ESMC RX and TX information',
      'ETH': phyIntfMatcher,
   }
   cliModel = SynceEsmcDetail

   @staticmethod
   def handler( mode, args ):
      showSynceWarnings( mode )
      option = gv.instanceConfig.networkOption
      intf = args.get( 'ETH' )
      if intf is not None:
         intfs = { Tac.const( IntfId( intf ) ) }
      else:
         intfs = set( gv.esmcRxStats.esmcRxStat ) | set( gv.esmcTxStats.esmcTxStat )
      entries = {}
      for intf in intfs:
         rxInfo = gv.esmcRxStats.esmcRxStat.get( intf )
         rxModel = None
         if rxInfo is not None:
            ssmTuple = None
            ql = None
            smac = None
            esmcType = None
            clockIdentity = None
            cascadedEeecCount = None
            cascadedEecCount = None
            etlvFlag = None
            if rxInfo.isLastValid():
               ssmTuple, ql = getEsmcSsmAndQl( rxInfo.ssmTuple, option )
               smac = rxInfo.src
               esmcType = rxInfo.eventFlag
               if ssmTuple.essmSet:
                  clockIdentity = Tac.Value( 'Arnet::Eui64',
                                             stringValue=rxInfo.clockIdentity )
                  cascadedEeecCount = rxInfo.numEeecHop
                  cascadedEecCount = rxInfo.numEecHop
                  etlvFlag = rxInfo.extendedFlag
            rxModel = SynceEsmcDetailRxInfo.factory_(
               option, rxInfo.validCount, rxInfo.invalidCount, smac,
               rxInfo.lastValidUpdated, rxInfo.lastInvalidUpdated, esmcType, ql,
               ssmTuple, rxInfo.eventFlagCount, rxInfo.ssmTupleCount, clockIdentity,
               cascadedEeecCount, cascadedEecCount, etlvFlag )

         txInfo = gv.esmcTxStats.esmcTxStat.get( intf )
         txModel = None
         if txInfo is not None:
            ssmTuple = None
            ql = None
            smac = None
            esmcType = None
            clockIdentity = None
            cascadedEeecCount = None
            cascadedEecCount = None
            etlvFlag = None
            if txInfo.totalCount:
               ssmTuple, ql = getEsmcSsmAndQl( txInfo.ssmTuple, option )
               smac = txInfo.src
               esmcType = txInfo.eventFlag
               if ssmTuple.essmSet:
                  clockIdentity = Tac.Value( 'Arnet::Eui64',
                                             stringValue=txInfo.clockIdentity )
                  cascadedEeecCount = txInfo.numEeecHop
                  cascadedEecCount = txInfo.numEecHop
                  etlvFlag = txInfo.extendedFlag
            txModel = SynceEsmcDetailTxInfo.factory_(
               option, txInfo.totalCount, smac, txInfo.lastUpdated, esmcType, ql,
               ssmTuple, txInfo.eventFlagCount, txInfo.ssmTupleCount, clockIdentity,
               cascadedEeecCount, cascadedEecCount, etlvFlag )
         entries[ intf ] = SynceEsmcDetailInfo( rxInfo=rxModel, txInfo=txModel )
      return SynceEsmcDetail( clockSourceDetail=entries )

addShowCommandClass( ShowSynceEsmcDetailCmd )

def Plugin( entityManager ):
   gv.clockTimerStatus = LazyMount.mount( entityManager, 'synce/selection/timers',
                                          'Synce::ClockTimerStatus', 'r' )
   gv.eecStatus = LazyMount.mount( entityManager, 'synce/eec/status',
                                   'Synce::EecStatus', 'r' )
   gv.esmcRxStats = LazyMount.mount( entityManager, 'synce/stats/esmc/rx',
                                     'Synce::AllEsmcRxStat', 'r' )
   gv.esmcTxStats = LazyMount.mount( entityManager, 'synce/stats/esmc/tx',
                                     'Synce::AllEsmcTxStat', 'r' )
   gv.hwCapability = LazyMount.mount( entityManager, 'synce/hw/capability',
                                      'Synce::Hardware::Capability', 'r' )
   gv.hwConfig = LazyMount.mount( entityManager, 'synce/hw/config',
                                  'Synce::Hardware::HwConfig', 'r' )
   gv.instanceConfig = LazyMount.mount( entityManager, 'synce/config/instance',
                                        'Synce::CliConfig', 'r' )
   gv.nominatedClockStatus = LazyMount.mount( entityManager,
                                              'synce/selection/nominated',
                                              'Synce::NominatedClockStatus', 'r' )
   gv.effectiveBestClockStatus = LazyMount.mount( entityManager,
                                                  'synce/selection/bestClockStatus',
                                                  'Synce::BestClockStatus', 'r' )
