# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from Ark import ReversibleDict
from CliModel import Dict, Int, Model, Str, Enum, Bool

# pylint: disable=consider-using-f-string

class SshHostKey( Model ):
   hostKey = Str( help="The public host key of the switch" )
   metadata = Dict(
      help="Algorithm-specific metadata for ssh host key",
      keyType=str, valueType=str, optional=True )

   def render( self ):
      out = ""
      if self.hostKey:
         out = self.hostKey
      if self.hostKey:
         if self.metadata:
            for paramName, paramValue in self.metadata.items():
               out += f" {paramName}={paramValue}"
      print( out )

class LoginBanner( Model ):
   loginBanner = Str( help="Login banner for the switch" )

   def render( self ):
      print( self.loginBanner )

class Motd( Model ):
   motd = Str( help="Message of the day for the switch" )

   def render( self ):
      print( self.motd )

class BootConfig( Model ):
   softwareImage = Str( help="Location of software image" )
   personalityOnReboot = Enum( help='The personality on reboot',
                               values=( 'cpu', 'bmc' ),
                               optional=True )
   consoleSpeed = Int( help="Baud rate of console", optional=True )
   abootPassword = Str( help="Encrypted aboot password" )
   memTestIterations = Int( help="Iterations of Memtest to perform at boot time" )
   firstInstructionVerifiedbootSupported = Bool(
      help="Verified boot is supported with First Instruction Security" )
   firstInstructionVerifiedbootEnabled = Bool(
      help="Verified boot is enabled with First Instruction Security" )
   firstInstructionMeasuredbootSupported = Bool(
      help="Measured boot is supported with First Instruction Security" )
   firstInstructionMeasuredbootEnabled = Bool(
      help="Measured boot is enabled with First Instruction Security" )
   securebootSupported = Bool( help="Secure boot is supported" )
   securebootEnabled = Bool( help="Secure boot is enabled" )
   spiFlashWriteProtected = Bool( help="SPI flash is physically protected against "
                                       "writes" )
   tpmPassword = Bool( help="Aboot password is stored in the TPM" )
   aristaCertEnabled = Bool( help="SWIs signed by Arista Networks are authorized "
                                  "to boot" )
   spiUpdateEnabled = Bool( help="Updating the SPI flash from Aboot is supported "
                                  "and enabled" )
   measuredbootEnabled = Bool( help="Measuring boot components via TPM PCRs is "
                                    "enabled" )
   certsLoaded = Bool( help="Certificates were successfully loaded" )
   aristaCert = Str( help="Arista certificate" )
   aristaCert2 = Str( help="Arista certificate #2" )
   aristaCert3 = Str( help="Arista certificate #3" )
   userCert = Str( help="User certficate" )
   upgradeCert = Str( help="SPI upgrade certificate" )
   fileChecksum = Str( help="Checksum of the underlying boot-config file" )
   fileChecksumAlg = Str( help="Algorithm used to hash the underlying boot-config "
                               "file" )

   def render( self ):
      def _boolState( value ):
         return 'enabled' if value else 'disabled'

      def _printAttr( description, attr ):
         attr = attr if attr and attr != 0 else "(not set)"
         print( "%s: %s" % ( description, attr ) )

      _printAttr( "Software image", self.softwareImage )

      if self.personalityOnReboot:
         cpuOrBmc = self.personalityOnReboot.upper()
         print( "Personality on reboot (CPU or BMC): %s" % cpuOrBmc )

      _printAttr( "Console speed", self.consoleSpeed )
      _printAttr( "Aboot password (encrypted)", self.abootPassword )
      _printAttr( "Memory test iterations", self.memTestIterations )
      _printAttr( "Checksum", self.fileChecksum )
      _printAttr( "Checksum algorithm", self.fileChecksumAlg )

      # If we don't support secureboot, don't bother to print anything about it
      if not self.securebootSupported:
         return

      print()
      _printAttr( "Secure boot", _boolState( self.securebootEnabled ) )
      _printAttr( "SPI flash protection", _boolState( self.spiFlashWriteProtected ) )
      _printAttr( "SPI flash update", _boolState( self.spiUpdateEnabled ) )
      _printAttr( "Password storage", 'tpm' if self.tpmPassword else 'boot-config' )
      _printAttr( "Measured boot", _boolState( self.measuredbootEnabled ) )
      if self.firstInstructionVerifiedbootSupported:
         _printAttr( "First instruction verified boot",
                     _boolState( self.firstInstructionVerifiedbootEnabled ) )
         _printAttr( "First instruction measured boot",
                     _boolState( self.firstInstructionMeasuredbootEnabled ) )
      _printAttr( "Arista certificate", _boolState( self.aristaCertEnabled ) )

      print( "Upgrade cert:" )
      print( self.upgradeCert )

      # Don't print anything if the certificate is empty
      if not self.certsLoaded or not self.userCert:
         return

      print( "Extra certificate:" )
      print( self.userCert )

class ManagementTelnet( Model ):
   serverState = Enum( values=( "enabled", "disabled" ),
                  help="State of the telnet session either enabled or disabled" )
   vrfName = Str( help="Configured VRF for telnet session", default="default" )
   maxTelnetSessions = Int( help="Maximum telnet sessions", default=20 )
   maxTelnetSessionsPerHost = Int( help="Maximum telnet sessions from one host",
                                   default=20 )
   globalDefault = Bool( help="VRF service state is dependent on default \
                         telnet state", optional=True )

   def render( self ):
      vrf = ( 'Default VRF' if self.vrfName == 'default' else
            'VRF %s' % self.vrfName )
      if not self.serverState:
         return
      print( 'Telnet status for', vrf, 'is', self.serverState, end=' ' )
      print( '( globalDefault )' if self.globalDefault else '' )
      print( 'Telnet session limit is', self.maxTelnetSessions )
      print( 'Telnet session limit per host is', self.maxTelnetSessionsPerHost )
      print()

CapiStringToCliString = ReversibleDict( {
      'default' : 'default',
      'qsfpExpanded' : 'qsfp_expanded',
      'qsfpDense' : 'qsfp_dense',
      'qsfpSparse' : 'qsfp_sparse',
      'OCTAL4' : 'OCTAL_4',
      'notAvailable' : 'notAvailable',
      'unknown' : 'Unknown' } )

class PortNumbering( Model ):
   numberingScheme = Enum( values=list( CapiStringToCliString ),
                           help='Port numbering scheme' )

   def render( self ):
      print( "\nPort numbering scheme: %r\n" % (
                              CapiStringToCliString[ self.numberingScheme ] ) )
