#
# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from datetime import datetime
from CliModel import Model, Enum
from CliModel import Str, Int
from CliModel import Dict, Float

class SystemdUnitState( Model ):
   # from systemd source. systemd/src/basic/unit-name.c
   unitActiveState = Enum( values=[
                           'undefined',
                           # mount unit state
                           'dead', 'mounting', 'mounting-done', 'mounted',
                           'remounting', 'unmounting', 'mounting-sigterm',
                           'mounting-sigkill', 'remounting-sigterm',
                           'remounting-sigkill', 'unmounting-sigterm',
                           'unmounting-sigkill', 'failed',

                           # service unit state
                           'dead', 'start-pre', 'start', 'start-post',
                           'running', 'exited', 'reload', 'stop', 'stop-sigabrt',
                           'stop-sigterm', 'stop-sigkill', 'stop-post',
                           'final-sigterm', 'final-sigkill',
                                  # 'failed' => already in mount state
                           'auto-restart',

                           # socket unit state
                                  # 'dead' => present in service states
                                  # 'start-pre',
                           'start-chown', 'listening',
                                  # 'start-post', 'stop-post', 'running'
                           'stop-pre-sigterm', 'stop-pre-sigkill',
                                  #'final-sigterm', 'final-sigkill'
                           ],
                           help="Active state of a systemd unit" )

class SystemdServiceData( Model ):
   svcDesc = Str( help='Systemd service short description' )
   svcActvState = SystemdUnitState.unitActiveState
   timestamp = Float( help="Timestamp of when unit reached current state",
                      optional=True )
   svcPid = Int( help='Process id. Not shown for reloaded SYS-V service' )
   svcState = Str( help='Running/Exited status of service with status code' )
   cgrpName = Str( help='Name of with CGroup for system service unit' )
   cgrpTasks = Dict( keyType=int, valueType=str,
                     help='Mapping of a cgroup process id to its command line.' )

def timeToStr( timestamp ):
   s = None
   if timestamp is not None:
      s = datetime.fromtimestamp( timestamp ).strftime( '%Y-%m-%d %H:%M:%S' )
   return '' if timestamp is None else 'since ' + s

class SystemdServicesDetail( Model ):
   systemdServicesInfo = Dict( keyType=str, valueType=SystemdServiceData,
                          help='Mapping of a systemd service unit to detailed'
                               'description of its state.' )

   def _renderCgroup( self, cgrp ):
      keys = sorted( cgrp.keys() )
      for pid in keys:
         print( f'{pid:>10d} {cgrp[ pid ]:s}' )

   def render( self ):
      keys = sorted( self.systemdServicesInfo.keys() )
      for e in keys:
         v = self.systemdServicesInfo[ e ]
         print( "    Service:", e )
         print( "Description:", v.svcDesc )
         print( "     Active:", end=' ' )
         print( f"{v.svcActvState:<15s} {timeToStr( v.timestamp ):s}" )
         if v.svcPid:
            print( "        PID:", v.svcPid )
         print( "     Status:", v.svcState )
         if len( v.cgrpName ) > 0:
            print( "     CGroup:", v.cgrpName )
            self._renderCgroup( v.cgrpTasks )
         print()

class SystemdSocketInfo( Model ):
   sockDesc = Str( help='Description of socket' )
   sockState = SystemdUnitState.unitActiveState
   sockType = Str( help='Type of socket ie., stream/datagram etc' )
   timestamp = Float( help="Timestamp of when unit reached current state",
                      optional=True )

class SystemdSocketsDetail( Model ):
   systemdSocketsInfo = Dict( keyType=str, valueType=SystemdSocketInfo,
                              help='Mapping of a systemd socket to detailed'
                                   'description of its state' )

   def render( self ):
      keys = sorted( self.systemdSocketsInfo.keys() )
      for s in keys:
         v = self.systemdSocketsInfo[ s ]
         print( "     Socket:", s )
         print( "Description:", v.sockDesc )
         print( "     Active:", end=' ' )
         print( f"{v.sockState:<15s} {timeToStr( v.timestamp ):s}" )
         print( "     Listen:", v.sockType )
         print()
