# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliPlugin import TapAggPmapCliLib
from TypeFuture import TacLazyType
import CliGlobal
import ConfigMount
import Tracing

t0 = Tracing.trace0
gv = CliGlobal.CliGlobal( dict( tapAggPmapConfig=None ) )

# TAC Types
EthAddr = TacLazyType( 'Arnet::EthAddr' )
EthAddrTuple = TacLazyType( 'Arnet::EthAddrTuple' )
UniqueId = TacLazyType( 'Ark::UniqueId' )
IdentityTuple = TacLazyType( 'Bridging::IdentityTuple' )

class ActionContext( object ): # pylint: disable=useless-object-inheritance
   def __init__( self, pmapClassContext, actionName, action ):
      # PolicyMap::PolicyMapClassContext
      self.pmapClassContext_ = pmapClassContext
      self.mapType_ = pmapClassContext.mapType()
      self.pmapName_ = pmapClassContext.pmapName()
      self.cmapName_ = pmapClassContext.policyClassName()
      self.actionName_ = actionName
      # PolicyMap::ActionSet
      self.action_ = action
      # PolicyMap::NamedActionSet
      self.namedActionSet_ = action.namedActionSet.newMember( actionName )
      self.namedActionSet_.internalGroupName = self.internalGroupName()
      self.actionType_ = TapAggPmapCliLib.actActionSet

   def mapType( self ):
      return self.mapType_

   def pmapName( self ):
      return self.pmapName_

   def cmapName( self ):
      return self.cmapName_

   def actionName( self ):
      return self.actionName_

   def actionSetEmpty( self ):
      return ( not self.namedActionSet_.setAggGroup and
               not self.namedActionSet_.stripHeaderBytes and
               not self.namedActionSet_.setIdentityTag and
               not self.namedActionSet_.setMacAddress )

   def internalGroupName( self ):
      # Format of internal group name is
      # graset%<policy-map-name>%<class-map-name>%<action-set-name>
      return "graset%%%s%%%s%%%s" % ( # pylint: disable=consider-using-f-string
            self.pmapName_, self.cmapName_, self.actionName_ )

   def removeActionSetChanges( self ):
      del gv.tapAggPmapConfig.tapAggActions.actionSet[ self.action_.id ]

   def identicalActionSet( self ):
      '''
      Determine whether or not the modified action set differs what's
      currently configured in Sysdb.
      '''
      currentNamedActionSet = TapAggPmapCliLib.getNamedActionSetObj(
            self.pmapClassContext_,
            self.pmapName_,
            self.cmapName_,
            self.actionName_ )
      if not currentNamedActionSet:
         return False
      return currentNamedActionSet.equalTo( self.namedActionSet_ )

   def commit( self ):
      # Update PolicyMap::PolicyClassRuleAction
      policyRuleAction = self.pmapClassContext_.policyRuleAction
      policyRuleAction.delAction( self.actionType_ )
      # Cleanup ActionSet if NamedActionSet collection is empty
      if self.action_.namedActionSet:
         if not self.identicalActionSet():
            self.action_.updatedActionSet[ self.actionName_ ] = True
         else:
            del self.action_.updatedActionSet[ self.actionName_ ]
         policyRuleAction.addAction( self.actionType_, self.action_ )
      else:
         self.removeActionSetChanges()

   def removeAction( self, actionType ):
      if actionType == TapAggPmapCliLib.actSetAggregationGroup:
         self.namedActionSet_.setAggGroup = None
      elif actionType == TapAggPmapCliLib.actSetIdentityTag:
         self.namedActionSet_.setIdentityTag = None
      elif actionType == TapAggPmapCliLib.actStripHdrBytes:
         self.namedActionSet_.stripHeaderBytes = None
      elif actionType == TapAggPmapCliLib.actSetMacAddress:
         self.namedActionSet_.setMacAddress = None

   def actionExists( self, actionType ):
      if actionType == TapAggPmapCliLib.actSetAggregationGroup:
         return self.namedActionSet_.setAggGroup is not None
      elif actionType == TapAggPmapCliLib.actSetIdentityTag:
         return self.namedActionSet_.setIdTag is not None
      elif actionType == TapAggPmapCliLib.actStripHdrBytes:
         return self.namedActionSet_.stripHeaderBytes is not None
      elif actionType == TapAggPmapCliLib.actSetMacAddress:
         return self.namedActionSet_.setMacAddress is not None
      return False

   def getAction( self, actionType ):
      if actionType == TapAggPmapCliLib.actSetAggregationGroup:
         if not self.namedActionSet_.setAggGroup:
            self.namedActionSet_.setAggGroup = \
                  ( self.cmapName_, self.action_.id, )
         return self.namedActionSet_.setAggGroup
      return None

   def setIdTag( self, vid, inner ):
      self.removeAction( TapAggPmapCliLib.actSetIdentityTag )
      idTuple = IdentityTuple( vid, inner )
      self.namedActionSet_.setIdentityTag = \
            ( self.cmapName_, self.action_.id, idTuple, )

   def removeDot1Q( self, hdrInfo ):
      self.removeAction( TapAggPmapCliLib.actStripHdrBytes )
      hdrType = TapAggPmapCliLib.stripHdrDot1q
      self.namedActionSet_.stripHeaderBytes = \
            ( self.cmapName_, self.action_.id, )
      self.namedActionSet_.stripHeaderBytes.stripHdrBytes = \
            TapAggPmapCliLib.stripHdrAction( hdrType, hdrInfo )

   def setMacAddress( self, ethAddrTuple ):
      self.removeAction( TapAggPmapCliLib.actSetMacAddress )
      self.namedActionSet_.setMacAddress = \
            ( self.cmapName_, self.action_.id, ethAddrTuple, )

def Plugin( entityManager ):
   gv.tapAggPmapConfig = ConfigMount.mount( entityManager, 'tapagg/pmapconfig',
                                            'TapAgg::PmapConfig', 'w' )
