#!/usr/bin/env python3
# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import Dict
from CliModel import List
from CliModel import Model
from CliModel import Submodel
from CliModel import Str
from CliModel import Int
from CliModel import Enum
from CliModel import Bool
from IntfModels import Interface
from CliMode.Intf import IntfMode
import Arnet
from ArnetModel import MacAddress
from CliPlugin import IntfModel
from TableOutput import TableFormatter, Headings, Format
import six
from TypeFuture import TacLazyType

EthAddr = TacLazyType( 'Arnet::EthAddr' )

switchportModes = ( 'access', 'trunk', 'dot1qTunnel',
                    'tap', 'tool', 'tap-tool', 'routed' )
intfLinkStatus = IntfModel.InterfaceStatus.interfaceStatus.values.\
    difference( [ 'connected' ] ).\
    union( [ 'tap', 'tool', 'non-tap', 'non-tool' ] )

def mapPropertyListKey( mpl ):
   '''We sort by policy name, group name, then interface.
   '''
   return ( mpl.policyName, mpl.groupName,
            Arnet.intfNameKey( mpl.mappedIntfName.stringValue ) )

def actionLabel( flag, supported=True, actionStr="strip" ):
   action = "none"
   if not supported:
      action = "unsupported"
   elif flag:
      action = actionStr
   return action

class AclsAppliedPerType( Model ):
   aclsApplied = List( valueType=str, help="List of ACL names applied on a port" )

class TapAggProperties( Model ):
   class TapProperty( Model ):
      configuredMode = Enum( values=switchportModes,
                             help="The configured switchport mode of an interface" )
      linkStatus = Enum( values=intfLinkStatus,
                         help="The link status of an interface" )
      portIdentity = Int( help="The port identity of a tap port" )
      innerPortIdentity = Int( help="The inner port identity of a tap port" )
      allowedVlans = Str( help="The set of allowed vlans on a tap port "
                               "in the range 1-4094" )
      nativeVlan = Int( help="The native vlan associated with a tap port" )
      truncationSize = Int( help="The size to which packet is to be truncated,"
                                 " in bytes" )

      class Details( Model ):
         # Now there could be multiple default groups. keeping default group for
         # backward compatibility reasons
         defaultGroup = Str( help="The default group associated with a tap port" )
         #list for group and interfaces added
         groups = List( valueType=str,
                        help="List of default groups for the tap port",
                        optional=True)
         toolPorts = List( valueType=Interface,
                           help="List of tool ports for the tap port",
                           optional=True)
         aclsAppliedPerType = Dict( valueType=AclsAppliedPerType,
                                    help="The list of ACLs applied on a port keyed "
                                         "by ACL type" )
         vxlanStrip = Bool( help="Indicates whether VXLAN strip is enabled on a "
                                 "tap port" )
         mplsPop = Bool( help="Indicates whether MPLS pop is enabled on a tap port" )
         destMac = MacAddress( help='Destination MAC address' )
         srcMac = MacAddress( help='Source MAC address' )

      class Tunnel( Model ):
         vnTagStrip = Bool( help="Indicates whether VN Tag strip is enabled on a "
                                 "tap port" )
         brTagStrip = Bool( help="Indicates whether 802.1BR Tag strip is enabled on "
                                 "a tap port" )
         vxlanStrip = Bool( help="Indicates whether VXLAN strip is enabled on a "
                                 "tap port" )
         mplsPop = Bool( help="Indicates whether MPLS pop is enabled on a tap port" )

      details = Submodel( valueType=Details, help="Detailed tap properties",
                          optional=True )
      tunnel = Submodel( valueType=Tunnel, help="Tunnel tap actions",
                          optional=True )

   class ToolProperty( Model ):
      configuredMode = Enum( values=switchportModes,
                             help="The configured switchport mode of an interface" )
      linkStatus = Enum( values=intfLinkStatus,
                         help="The link status of an interface" )
      identificationTag = Enum( values=( 'Off', 'dot1q', 'qinq' ),
                                help="The identity tag requirement for packets "
                                     "recieved on a tool port" )
      allowedVlans = Str( help="The set of allowed vlans on a tool port "
                               "in the range 1-4094" )
      timestampMode = Enum( values=( 'NA', 'None', 'before-fcs', 'replace-fcs',
                                     'header' ),
                            help="The timestamping mode on a packet recieved "
                                 "on a tool port" )

      class Details( Model ):
         #list for groups added
         groups = List( valueType=str, help="List of groups of the tool port",
                        optional=True)
         removedVlans = Str( help="The set of 802.1Q tags to remove on a tool port",
                             optional=True)
         truncationSize = Int( help="The size to which packet is to be truncated,"
                                    " in bytes" )
         aclsAppliedPerType = Dict( valueType=AclsAppliedPerType,
                                    help="The dictionary of ACLs applied on a port"
                                         "keyed by ACL type",
                                    optional=True )
      class Tunnel( Model ):
         vnTagStrip = Bool(
               help="Indicates whether VN Tag strip is enabled on a tool port" )
         brTagStrip = Bool(
               help="Indicates whether 802.1BR Tag strip is enabled on a tool port" )
         mplsPop = Bool(
               help="Indicates whether MPLS pop is enabled on a tool port" )

      details = Submodel( valueType=Details, help="Detailed tool properties",
                          optional=True )
      tunnel = Submodel( valueType=Tunnel, help="Tunnel tool actions",
                          optional=True )

   tapProperties = Dict( keyType=Interface, valueType=TapProperty,
                         help="Dictionary of tap port "
                              "properties keyed by interface name" )
   toolProperties = Dict( keyType=Interface, valueType=ToolProperty,
                          help="Dictionary of tool port "
                               "properties keyed by interface name" )
   _detail = Bool( help="Flag for details on tap-agg group interfaces" )
   _tunnel = Bool( help="Flag for details on tap-agg tunnel stripping" )
   _mplsPopSupported = Bool( help="Whether MPLS Pop is supported on tap ports")
   _mplsPopToolSupported = Bool( help="Whether MPLS Pop is supported on tool ports" )
   _vxlanStripSupported = Bool( help="Whether VxLAN Stripping is supported" )
   _brVnTagStripToolSupported = Bool(
         help="Whether VN/BR tag stripping is supported on tool ports" )
   _brVnTagStripSupported = Bool(
         help="Whether VN/BR tag stripping is supported on tap ports" )

   def render( self ):
      tf = TableFormatter()
      portFormat = Format( justify="left", wrap=True, maxWidth=10 )
      configModeFormat = Format( justify="left", wrap=True, maxWidth=12 )
      statusFormat = Format( justify="left", wrap=True, maxWidth=12 )
      aVlansFormat = Format( justify="left", wrap=True, maxWidth=14 )

      if self.tapProperties:
         portIdFormat = Format( justify="left", wrap=True, maxWidth=9 )
         nVlansFormat = Format( justify="left", wrap=True, maxWidth=7 )
         truncSizeFormat = Format( justify="left", wrap=True, maxWidth=11 )
         heading = ( 'Port',  'Configured Mode', 'Status', 'Port Identity',
                     'Allowed Vlans', 'Native Vlan', 'Truncation Size' )
         header = Headings( heading )
         header.doApplyHeaders( tf )
         tf.formatColumns( portFormat, configModeFormat, statusFormat, portIdFormat,
                           aVlansFormat, nVlansFormat, truncSizeFormat )
         tapKeys = Arnet.sortIntf( self.tapProperties )
         for tapKey in tapKeys:
            tapProperty = self.tapProperties[ tapKey ]
            portIdentity = tapProperty.portIdentity
            if tapProperty.innerPortIdentity:
               # pylint: disable-next=consider-using-f-string
               portIdentity = '%d,%d' % ( tapProperty.portIdentity,
                                          tapProperty.innerPortIdentity )
            tf.newRow( IntfMode.getShortname( tapKey ),
                       tapProperty.configuredMode,
                       tapProperty.linkStatus, portIdentity,
                       tapProperty.allowedVlans, tapProperty.nativeVlan,
                       tapProperty.truncationSize )
         print( tf.output() )
         if self._detail:
            print( '\n\n' )
            t = TableFormatter()
            f1 = Format( justify="left", wrap=True, maxWidth=10, minWidth=10 )
            f2 = Format( justify="left", wrap=True, maxWidth=15, minWidth=15 )
            f3 = Format( justify="left", wrap=True, maxWidth=12, minWidth=12 )
            f4 = Format( justify="left", wrap=True, maxWidth=18, minWidth=12 )
            headings = ( "Port", "ACLs Applied", "Groups", "Tool Ports",
                         "VXLAN Strip", "MPLS Pop", "Dest MAC", "Src MAC" )
            th = Headings( headings )
            th.doApplyHeaders( t )

            for tapKey in tapKeys:
               tapPort = IntfMode.getShortname( tapKey )
               tapProperty = self.tapProperties[ tapKey ]
               aclsApplied = []
               for aclType in sorted( tapProperty.details.aclsAppliedPerType ):
                  aclsApplied.extend( tapProperty.details. \
                                         aclsAppliedPerType[ aclType ].aclsApplied )
               if aclsApplied:
                  aclsAppliedStr = ', '.join( aclsApplied )
               else:
                  aclsAppliedStr = '---'
               tapGroups = sorted( list(tapProperty.details.groups) )
               if tapGroups:
                  tapGroupsStr = ', '.join( tapGroups )
               else:
                  tapGroupsStr = '---'
               tapIntfs = [ IntfMode.getShortname( intf ) for \
                              intf in sorted( tapProperty.details.toolPorts ) ]
               if not tapIntfs:
                  tapIntfs.append( "None" )
               tapIntfsStr = tapIntfs[ 0 ]
               for intf in tapIntfs[ 1: ]:
                  tapIntfsStr += ', '
                  tapIntfsStr += intf
               vxlanStrip = tapProperty.details.vxlanStrip
               mplsPop = tapProperty.details.mplsPop
               destMac = tapProperty.details.destMac.displayString
               if str( tapProperty.details.destMac ) == EthAddr.ethAddrZero:
                  destMac = 'None'
               srcMac = tapProperty.details.srcMac.displayString
               if str( tapProperty.details.srcMac ) == EthAddr.ethAddrZero:
                  srcMac = 'None'
               t.newRow( tapPort, aclsAppliedStr, tapGroupsStr, tapIntfsStr,
                         vxlanStrip, mplsPop, destMac, srcMac )
            t.formatColumns( f1, f2, f3, f3, f3, f3, f4, f4 )
            print( t.output() )

         elif self._tunnel:
            print( '\n\n' )

            t = TableFormatter()
            f1 = Format( justify="left", wrap=True, maxWidth=10 )
            f2 = Format( justify="left", wrap=True, maxWidth=15 )

            headings = ( 'Port', 'VN Tag Action', 'BR Tag Action',
                         'VxLAN Action', 'MPLS Action' )
            th = Headings( headings )
            th.doApplyHeaders( t )

            for tapKey in tapKeys:
               tapPort = IntfMode.getShortname( tapKey )
               tapProperty = self.tapProperties[ tapKey ]
               t.newRow( tapPort,
                         actionLabel( tapProperty.tunnel.vnTagStrip,
                                      supported=self._brVnTagStripSupported ),
                         actionLabel( tapProperty.tunnel.brTagStrip,
                                      supported=self._brVnTagStripSupported ),
                         actionLabel( tapProperty.tunnel.vxlanStrip,
                                      supported=self._vxlanStripSupported ),
                         actionLabel( tapProperty.tunnel.mplsPop,
                                      supported=self._mplsPopSupported,
                                      actionStr='pop' ) )
            t.formatColumns( f1, f2, f2, f2, f2 )
            print( t.output() )

      elif self.toolProperties:
         idTagFormat = Format( justify="left", wrap=True, maxWidth=9 )
         timeStampFormat = Format( justify="left", wrap=True, maxWidth=11 )
         heading = ( 'Port', 'Configured Mode', 'Status', 'Id Tag', 'Allowed Vlans',
                     'Timestamp Mode' )
         header = Headings( heading )
         header.doApplyHeaders( tf )
         tf.formatColumns( portFormat, configModeFormat, statusFormat,
                           idTagFormat, aVlansFormat, timeStampFormat )
         toolKeys = Arnet.sortIntf( self.toolProperties )
         for toolKey in toolKeys:
            toolProperty = self.toolProperties[ toolKey ]
            tf.newRow( IntfMode.getShortname( toolKey ),
                       toolProperty.configuredMode,
                       toolProperty.linkStatus,
                       toolProperty.identificationTag,
                       toolProperty.allowedVlans,
                       toolProperty.timestampMode )
         print( tf.output() )
         if self._detail:
            print( '\n\n' )
            t = TableFormatter()
            f1 = Format( justify="left", wrap=True, maxWidth=10 )
            f2 = Format( justify="left", wrap=True, maxWidth=15 )
            f3 = Format( justify="left", wrap=True, maxWidth=12 )
            headings = ( 'Port', 'Truncation Size', 'Remove Outer 802.1Q',
                         'ACLs Applied', 'Groups' )
            th = Headings( headings )
            th.doApplyHeaders( t )

            for toolKey in toolKeys:
               toolPort = IntfMode.getShortname( toolKey )
               toolProperty = self.toolProperties[ toolKey ]
               aclsApplied = []
               for aclType in sorted( toolProperty.details.aclsAppliedPerType ):
                  aclsApplied.extend( toolProperty.details. \
                                         aclsAppliedPerType[ aclType ].aclsApplied )
               if aclsApplied:
                  aclsAppliedStr = ', '.join( aclsApplied )
               else:
                  aclsAppliedStr = '---'
               toolGroups = sorted( list(toolProperty.details.groups) )
               if toolGroups:
                  toolGroupsStr = ', '.join( toolGroups )
               else:
                  toolGroupsStr = '---'
               t.newRow( toolPort, toolProperty.details.truncationSize,
                         toolProperty.details.removedVlans,
                         aclsAppliedStr, toolGroupsStr )
            t.formatColumns( f1, f2, f2, f2, f3 )
            print( t.output() )
         elif self._tunnel:
            print( '\n\n' )
            t = TableFormatter()
            f1 = Format( justify="left", wrap=True, maxWidth=10 )
            f2 = Format( justify="left", wrap=True, maxWidth=15 )

            headings = ( 'Port', 'VN Tag Action', 'BR Tag Action', 'MPLS Action' )

            th = Headings( headings )
            th.doApplyHeaders( t )

            for toolKey in toolKeys:
               toolPort = IntfMode.getShortname( toolKey )
               toolProperty = self.toolProperties[ toolKey ]
               t.newRow( toolPort,
                         actionLabel( toolProperty.tunnel.vnTagStrip,
                                      supported=self._brVnTagStripToolSupported ),
                         actionLabel( toolProperty.tunnel.brTagStrip,
                                      supported=self._brVnTagStripToolSupported ),
                         actionLabel( toolProperty.tunnel.mplsPop,
                                      supported=self._mplsPopToolSupported,
                                      actionStr='pop' ) )
            t.formatColumns( f1, f2, f2, f2 )
            print( t.output() )

class MapProperty( Model ):
   mappedIntfName = Interface( help="Tap/Tool Port Name" )
   groupName = Str( help="Group Name" )
   policyName = Str( help="PolicyMap/ClassMap Name" )

class MapPropertyList( Model ):
   mapPropertyList = List( valueType=MapProperty,
                           help='List of tap/tool map properties' )

def renderMapping( headings, mapping ):
   t = TableFormatter()
   f1 = Format( justify="left", wrap=True, maxWidth=15, minWidth=1 )
   f2 = Format( justify="left", wrap=True, maxWidth=25, minWidth=1 )
   th = Headings( headings )
   th.doApplyHeaders( t )
   for port in Arnet.sortIntf( mapping ):
      mapPropertyList = sorted( mapping[ port ].mapPropertyList,
                                key=mapPropertyListKey )
      for mapProperty in mapPropertyList:
         policyName = mapProperty.policyName
         t.newRow( port, mapProperty.mappedIntfName.stringValue,
                   mapProperty.groupName, policyName )

   t.formatColumns( f1, f1, f1, f2 )
   print( t.output() )

class TapToToolMap( Model ):
   tapPortsMap = Dict( keyType=Interface, valueType=MapPropertyList,
                       help="A mapping from tap to tool ports" )
   def render( self ):
      headings = ( 'Tap Port', 'Tool Ports', 'Groups', 'PolicyMap/ClassMap' )
      renderMapping( headings, self.tapPortsMap )

class ToolToTapMap( Model ):
   toolPortsMap = Dict( keyType=Interface, valueType=MapPropertyList,
                        help="A mapping from tool to tap ports" )
   def render( self ):
      headings = ( 'Tool Port', 'Tap Ports', 'Via Groups', 'Via PolicyMap/ClassMap' )
      renderMapping( headings, self.toolPortsMap )

def formatPolicy( intf, intfStatusMap ):
   # return a string including the shortname of the port
   # and, if there is, the policy that the port applies
   portInfo = IntfMode.getShortname( intf )
   intfStatus = intfStatusMap.get( intf )
   if not intfStatus:
      return portInfo
   if not intfStatus.policies:
      return portInfo
   if not intfStatus.policies[ 1 : ] and intfStatus.policies[ 0 ] == '*':
      # if a tap port has only default setting and no configured policies
      return portInfo

   if intfStatus.policies[ 0 ] == '*':
      # if a tap port has both default setting and configured policies
      policyList = intfStatus.policies[ 1 : ]
      policyList.sort()
      portInfo += ', ' + portInfo + ' ('
   else:
      policyList = intfStatus.policies[ 0 : ]
      policyList.sort()
      portInfo += ' ('
   portInfo += ', '.join( policyList )
   portInfo += ')'
   return portInfo

class TapAggGroups( Model ):

   class TapAggGroup( Model ):
      """ Represents a TapAggGroup """

      class PortStatus( Model ):
         statuses = List( valueType=str, help="List statuses of the interface. "
                                              "Active, Interface status not found, "
                                              "Link down, Link unknown, Lag member, "
                                              "Mirror destination" )
         policies = List( valueType=str, help="List policies by which"
                                              "the port is added to the group" )

      tapPortStatuses = Dict( keyType=Interface, valueType=PortStatus,
                              help="Mapping from interface name to a list of their"
                                   "statuses and configured policies" )
      toolPortStatuses = Dict( keyType=Interface, valueType=PortStatus,
                               help="Mapping from interface name to a list of their"
                                    "statuses" )

   _detail = Bool( help="Flag for details on tap-agg group interfaces" )
   groups = Dict( valueType=TapAggGroup, help="Tap-agg groups keyed by group name" )

   def _getActiveIntfs( self, intfStatusMap ):
      activePorts = []
      for intf in intfStatusMap:
         intfStatus = intfStatusMap.get( intf )
         if not intfStatus:
            continue
         for status in intfStatus.statuses:
            if status == "Active":
               activePorts.append( intf )

      if not activePorts:
         activePorts.append( "None" )
      return activePorts

   def _getInActiveIntfs( self, intfStatusMap ):
      inactivePorts = {}
      for intf in intfStatusMap:
         intfStatus = intfStatusMap.get( intf )
         if not intfStatus:
            continue
         for status in intfStatusMap[ intf ].statuses:
            if status == "Active":
               continue

            if status not in inactivePorts:
               inactivePorts[ status ] = []

            inactivePorts[ status ].append( intf )

      return inactivePorts

   def formatOutput( self, inputList, portStatuses, fieldWidth, prefixFields,
                     ports=True ):
      # Print the list of ports. If the list is
      # too long to fit on one line, wrap it onto subsequent lines.
      if ports:
         inputList = Arnet.sortIntf( inputList )
      else:
         if not inputList:
            inputList = ['---']
         inputList.sort()
      prefixLine = ' ' * prefixFields
      firstLine = True
      inputLine = formatPolicy( inputList[ 0 ], portStatuses )
      outputLine = ''
      for intf in inputList[ 1: ]:
         inputLine += ', ' + formatPolicy( intf, portStatuses )
         while len( inputLine ) > fieldWidth:
            outputLine = inputLine[ :fieldWidth ]
            # pylint: disable-next=redefined-builtin
            breakpoint = outputLine.rfind( ', ' ) + 2
            outputLine = inputLine[ :breakpoint ]
            inputLine = inputLine[ breakpoint: ]
            if not firstLine and prefixFields:
               print( prefixLine, end=' ' )
            print( outputLine )
            firstLine = False
      if not firstLine and prefixFields:
         print( prefixLine, end=' ' )
      print( inputLine )

   def renderDetailGroup( self, groupName, group ):
      print()
      print( 'Group %s' % groupName ) # pylint: disable=consider-using-f-string
      print( '---------------------------------------------------------------' )
      self.renderPortStatus( group.tapPortStatuses, 'Source' )
      print()
      self.renderPortStatus( group.toolPortStatuses, 'Destination' )

   def renderPortStatus( self, portStatuses, portType ):
      print()
      print( '%s Port:' % portType ) # pylint: disable=consider-using-f-string
      print( '  Active:', end=' ' )
      self.formatOutput( self._getActiveIntfs( portStatuses ),
                         portStatuses, fieldWidth=70, prefixFields=9 )
      print()
      self.printInactivePorts( self._getInActiveIntfs( portStatuses ),
                               portStatuses )

   def printInactivePorts( self, inactivePorts, portStatuses ):
      """ print inactive ports if present """
      if not inactivePorts:
         return

      prefixFields = 29
      detailedFormat = "%-5s%-25s%-s"
      briefFormat = "%-5s%-24s"
      print( '  Configured, but inactive ports:' )
      print( detailedFormat % ( '', 'Reason inactive', 'Ports' ) )
      print( detailedFormat % ( '', '--------------------', '----------' ) )
      for status in inactivePorts:
         print( briefFormat % ( '', status ), end=' ' )
         self.formatOutput( inactivePorts[ status ], portStatuses,
                            fieldWidth=80-prefixFields,
                            prefixFields=prefixFields )

   def renderDetail( self ):
      groupKeys = sorted( self.groups.keys() )
      for groupName in groupKeys:
         group = self.groups.get( groupName )
         if not group:
            continue
         self.renderDetailGroup( groupName, group )

   def renderNoDetail( self ):
      t = TableFormatter()
      width = t.width_ - 15
      width = width // 2 - 1
      # Max width of columns are restricted so that table does not split into two
      f1 = Format( justify="left", wrap=True, maxWidth=15, minWidth=10 )
      # Remove padding to avoid the last column to be wrapped.
      f1.noPadLeftIs( True )
      f1.noTrailingSpaceIs( True )
      # If f2's columns does not exceed 20, fix the max to the min value to
      # allow columns to be wrapped.
      if width < 20: # pylint: disable=consider-using-max-builtin
         width = 20
      f2 = Format( justify="left", wrap=True, maxWidth=width, minWidth=20 )
      f2.noPadLeftIs( True )
      f2.noTrailingSpaceIs( True )
      headings = ( ("Group Name","l"), ("Tool Members","l"), ("Tap Members","l") )
      th = Headings( headings)
      th.doApplyHeaders( t )
      groupKeys = sorted( self.groups.keys() )
      for groupName in groupKeys:
         group = self.groups.get( groupName )
         if not group:
            continue
         tools = Arnet.sortIntf( self._getActiveIntfs( group.toolPortStatuses ) )
         taps = Arnet.sortIntf( self._getActiveIntfs( group.tapPortStatuses ) )
         tools = [ IntfMode.getShortname( intf ) for intf in tools ]
         toolsList = tools[ 0 ]
         for intf in tools[ 1: ]:
            toolsList += ', '
            toolsList += intf
         tapsList = formatPolicy( taps[ 0 ], group.tapPortStatuses )
         for intf in taps[ 1: ]:
            tapsList += ', '
            tapsList += formatPolicy( intf, group.tapPortStatuses )
         t.newRow( groupName, toolsList, tapsList )
      t.formatColumns( f1, f2, f2 )
      print( t.output() )

   def render( self ):
      if not self.groups:
         return
      if self._detail:
         self.renderDetail()
      else:
         self.renderNoDetail()

class TapAggNexthopGroups( Model ):

   class FecNexthopGroup( Model ):
      class Source( Model ):
         tap = Str( help="Interface which sends traffic to the nexthop group" )
         policy = Str( help="Policy by which traffic is sent, if available" )

         def toString( self ):
            asStr = IntfMode.getShortname( self.tap )
            if self.policy:
               asStr += ' (' + self.policy + ')'
            return asStr

      taps = List( valueType=Source, help="Tap interfaces" )
      members = List( valueType=str, help="Nexthop interfaces" )

   class McgNexthopGroup( Model ):
      class Source( Model ):
         tap = Str( help="Interface which sends traffic to the nexthop group" )
         policy = Str( help="Policy by which traffic is sent, if available" )

         def toString( self ):
            asStr = IntfMode.getShortname( self.tap )
            if self.policy:
               asStr += ' (' + self.policy + ')'
            return asStr

      class Destination( Model ):
         nhg = Str( help="Nexthop group member of the multicast" )
         chosenIntf = Str( help="Interface selected among its members" )

         def toString( self ):
            return self.nhg + ':' + IntfMode.getShortname( self.chosenIntf )

      taps = List( valueType=Source, help="Tap interfaces" )
      destinations = List( valueType=Destination, help="Destinations" )

   fecNexthopGroups = Dict( keyType=str, valueType=FecNexthopGroup,
                            help="Load balanced nexthop groups keyed by name" )
   mcgNexthopGroups = List( valueType=McgNexthopGroup,
                            help="Multicast nexthop groups" )

   def addFecEntry( self, tap, policy, nhg, members ):
      if nhg in self.fecNexthopGroups:
         fecNhg = self.fecNexthopGroups[ nhg ]
      else:
         fecNhg = self.FecNexthopGroup()
         fecNhg.members = members
         self.fecNexthopGroups[ nhg ] = fecNhg
      source = self.FecNexthopGroup.Source()
      source.tap = tap
      source.policy = policy
      fecNhg.taps.append( source )

   def addMcgEntry( self, tap, policy, nhgs ):
      for mcgNhg in self.mcgNexthopGroups:
         if all( any( dest.nhg == nhg
            for dest in mcgNhg.destinations ) for nhg in nhgs ):
            break
      else:
         mcgNhg = self.McgNexthopGroup()
         for nhg, chosenIntf in nhgs.items():
            dest = self.McgNexthopGroup.Destination()
            dest.nhg = nhg
            dest.chosenIntf = chosenIntf
            mcgNhg.destinations.append( dest )
         self.mcgNexthopGroups.append( mcgNhg )
      source = self.McgNexthopGroup.Source()
      source.tap = tap
      source.policy = policy
      mcgNhg.taps.append( source )

   def render( self ):
      self.renderFecTable()
      self.renderMcgTable()

   def renderFecTable( self ):
      if not self.fecNexthopGroups:
         return
      t = TableFormatter()
      width = min( ( t.width_ - 15 ) // 2 - 1, 20 )
      f1 = Format( justify='left', wrap=True, maxWidth=width, minWidth=20 )
      f1.noPadLeftIs( True )
      f1.noTrailingSpaceIs( True )
      f2 = Format( justify='left', wrap=True, maxWidth=15, minWidth=10 )
      f2.noPadLeftIs( True )
      f2.noTrailingSpaceIs( True )
      headings = ( ( 'Tap Interfaces', 'l' ),
                   ( 'Group Name', 'l' ),
                   ( 'Nexthop Interfaces', 'l' ) )
      th = Headings( headings )
      th.doApplyHeaders( t )
      fecNhgs = sorted( self.fecNexthopGroups.keys() )
      for nhg in fecNhgs:
         fecNhg = self.fecNexthopGroups[ nhg ]
         tapStr = ', '.join( sorted( tap.toString() for tap in fecNhg.taps ) )
         memberStr = ', '.join(
            IntfMode.getShortname( member ) for member in fecNhg.members )
         t.newRow( tapStr, nhg, memberStr )
      t.formatColumns( f1, f2, f1 )
      print( 'Load Balanced Nexthop Groups:' )
      print( t.output() )

   def renderMcgTable( self ):
      if not self.mcgNexthopGroups:
         return
      t = TableFormatter()
      fmt = Format( justify='left', wrap=True, maxWidth=30, minWidth=15 )
      fmt.noPadLeftIs( True )
      fmt.noTrailingSpaceIs( True )
      headings = ( ( 'Tap Interfaces', 'l' ),
                   ( 'Destinations (nhg:chosen_intf)', 'l' ) )
      th = Headings( headings )
      th.doApplyHeaders( t )
      for mcgNhg in self.mcgNexthopGroups:
         tapStr = ', '.join( sorted( tap.toString() for tap in mcgNhg.taps ) )
         destStr = ', '.join(
            sorted( dest.toString() for dest in mcgNhg.destinations ) )
         t.newRow( tapStr, destStr )
      t.formatColumns( fmt, fmt )
      print( 'Multicast Nexthop Groups:' )
      print( t.output() )

class CounterValue( Model ):
   """Wraps the value of a Hardware::Sand::CounterValue for TapAgg CLI."""
   packetCount = Int( help="The number of packets that have been counted" )
   byteCount = Int( help="The number of bytes that have been counted" )

class IntfCounterModel( Model ):
   """A mapping of generic interfaces to counter values.

   Rendering presents the interfaces as TAP ports only by hard-coded help text."""
   interfaces = Dict(
      keyType=Interface,
      valueType=CounterValue,
      help="A mapping from interfaces to counter values" )

   def render( self ):
      if not self.interfaces:
         print( "No tap ports supporting default-forwarded counters "
                "are configured." )
         return

      t = TableFormatter()
      colFormat = Format( justify="left", wrap=True, minWidth=1, padding=1 )
      colFormat.noPadLeftIs( True )
      th = Headings( ( ( "TAP Port", "l" ), ( "Packets", "l" ),
                       ( "Bytes", "l" ) ) )
      th.doApplyHeaders( t )
      for intf, counterValue in sorted( six.iteritems( self.interfaces ) ):
         t.newRow( intf, counterValue.packetCount, counterValue.byteCount )
      t.formatColumns( colFormat, colFormat, colFormat )
      print( t.output() )

class TunnelCounterModel( Model ):
   """A mapping of TAP tunnels to counter values."""
   tunnels = Dict(
      keyType=str,
      valueType=CounterValue,
      help="A mapping from TAP tunnels to counter values" )

   def render( self ):
      if not self.tunnels:
         print( "No GRE tunnel termination is configured" )
         return

      t = TableFormatter()
      colFormat = Format( justify="left", wrap=True, minWidth=1, padding=1 )
      th = Headings( ( ( "Tunnel (SIP/DIP)", "l" ), ( "Rx Packets", "l" ),
                       ( "Rx Bytes", "l" ) ) )
      th.doApplyHeaders( t )
      for tunnel, counterValue in sorted( self.tunnels.items() ):
         t.newRow( tunnel, counterValue.packetCount, counterValue.byteCount )
      t.formatColumns( colFormat, colFormat, colFormat )
      print( t.output() )
