#!/usr/bin/env python3
# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-from-import

from SysConstants.in_h import IPPROTO_IP

import Cell
import CliParser
import CliCommand
import CliMatcher
import ConfigMount
import CliPlugin.AclCli as AclCli
import CliPlugin.AclCliRules as AclCliRules
from CliPlugin.EthIntfCli import EthPhyAutoIntfType
import CliPlugin.IntfCli as IntfCli
from CliPlugin.LagIntfCli import LagAutoIntfType
from CliPlugin.SwitchIntfCli import SwitchAutoIntfType
from CliPlugin.InternalRecircIntfCli import InternalRecircAutoIntfType
from CliPlugin.MacAddr import macAddrMatcher
from CliPlugin.MirroringPolicyMapClassModeTapAgg import (
      getIntfOpAndList,
      guardPmapDot1qRemove,
      guardPmapRemoveDot1qRange,
      guardTapAggHeaderRemove,
      PolicyMapClassModeTapAgg,
      TimestampExprFactory,
      AggNexthopGroupExprFactory,
      validateHeaderRemoveConfig,
)
import CliPlugin.PolicyMapCliLib as PolicyMapCliLib
import CliPlugin.PolicyMapModel as PolicyMapModel
from CliPlugin.PolicyMapModelImpl import (
      PolicyMapModelContainer,
      clearPolicyMapCounters,
      clearPolicyMapCheckpoint,
      registerCmapStatusFilter
)
import CliPlugin.TapAggPmapCliLib as TapAggPmapCliLib
from CliPlugin.TapAggIntfCli import (
      qinqIdentityTaggingGuard,
      matcherTapAggGroupName,
      guardMacAcls,
      matcherMacAddress,
      matcherMacAddressDest,
      matcherMacAddressSrc,
)
import Intf.IntfRange
import LazyMount
import MultiRangeRule
import PolicyMap
import Tac
import Tracing
import six
from Arnet import EthAddr as EthAddrStr

t0 = Tracing.trace0

aclConfig = None
tapAggHwStatus = None
tapAggHwConfig = None
tapAggIntfConfig = None
tapAggPmapConfig = None
tapAggStatusRequestDir = None
tapAggPmapStatus = None
aclStatus = None
policyOpChkr = None
tapAggPmapCheckpoint = None

EthAddr = Tac.Type( 'Arnet::EthAddr' )
EthAddrTuple = Tac.Type( 'Arnet::EthAddrTuple' )
UniqueId = Tac.Type( 'Ark::UniqueId' )
IdentityTuple = Tac.Type( 'Bridging::IdentityTuple' )

def _tapAggPmapSessionCheckpoint( mode ):
   tapAggPmapSessionCheckpoint = mode.session.sessionData(
                                          'TapAggPmapCli.sessionCheckpoint', None )
   if tapAggPmapSessionCheckpoint is None:
      tapAggPmapSessionCheckpoint = Tac.newInstance( 'TapAgg::PmapCheckpointStatus',
                                                    'tapagg-pmapcounter-checkpoint' )
      mode.session.sessionDataIs( 'TapAggPmapCli.sessionCheckpoint',
                                  tapAggPmapSessionCheckpoint )
   return tapAggPmapSessionCheckpoint

def guardPMapTapAgg( mode, token ):
   if not tapAggHwStatus.modeSupported:
      return CliParser.guardNotThisPlatform
   elif not tapAggHwStatus.pmapSupported:
      return CliParser.guardNotThisPlatform
   else:
      return None

def guardIp6Acls( mode, token ):
   if not aclStatus.dpIp6AclSupported:
      return CliParser.guardNotThisPlatform
   elif not tapAggHwStatus.pmapIpv6Supported:
      return CliParser.guardNotThisPlatform
   else:
      return None

#------------------------------------------------------------------------
#                         POLICY MAP CLI
#------------------------------------------------------------------------
class RemoveBytesExpr( CliCommand.CliExpression ):
   expression = 'dot1q outer REMOVE_VLANS'
   data = {
      'dot1q' : 'Remove dot1q tag',
      'outer' : 'Remove outer tag',
      'REMOVE_VLANS' : CliCommand.Node(
         matcher=MultiRangeRule.MultiRangeMatcher(
            rangeFn=lambda: ( 1, tapAggHwStatus.dot1qRemoveMaxIndex ),
            noSingletons=False,
            helpdesc='Specify indices of vlan tags to be removed',
            value=lambda mode, grList: str( grList ) ),
         guard=guardPmapRemoveDot1qRange )
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      if not args.pop( 'dot1q', None ):
         return
      args.pop( 'outer' )
      args[ 'removeBytesActType' ] = (
         TapAggPmapCliLib.actStripHdrBytes,
         TapAggPmapCliLib.stripHdrAction( TapAggPmapCliLib.stripHdrDot1q,
                                          args.pop( 'REMOVE_VLANS' ) ) )

class HeaderRemoveExprFactory( CliCommand.CliExpressionFactory ):
   def generate( self, name ):
      headerName = name + "_header"
      sizeName = name + "_size"
      preserveName = name + "_preserve"
      ethernetName = name + "_ethernet"
      keywords = ( headerName, sizeName, preserveName, ethernetName )
      class HeaderRemoveExpr( CliCommand.CliExpression ):
         # pylint: disable-next=consider-using-f-string
         expression = "%s %s SIZE [ %s %s ]" % keywords
         data = {
            headerName : CliCommand.guardedKeyword(
               'header',
               helpdesc='Remove a configurable number of bytes after the Ethernet '
               'header, including any VLAN tags',
               guard=guardTapAggHeaderRemove ),
            sizeName : CliMatcher.KeywordMatcher(
               'size',
               helpdesc='Set the number of bytes to remove' ),
            'SIZE' : CliMatcher.IntegerMatcher(
               2, 62, helpdesc='The size in bytes (must be even)' ),
            preserveName : CliMatcher.KeywordMatcher(
               'preserve',
               helpdesc='Preserve the original header' ),
            ethernetName : CliMatcher.KeywordMatcher(
               'ethernet',
               helpdesc='Preserve the original Ethernet header, including VLAN tags '
               'if any' )
         }
         @staticmethod
         def adapter( mode, args, argsList ):
            if headerName not in args:
               return
            ethernet = args.pop( ethernetName, None )
            if ethernet:
               args[ 'ethernet' ] = True
            headerRemove, err = validateHeaderRemoveConfig( args )
            if err:
               mode.addError( err )
               return
            args[ name ] = headerRemove

      return HeaderRemoveExpr

class RemoveCmdExpr( CliCommand.CliExpression ):
   expression = 'remove ( ( REMOVE_BYTES [ REMOVE_HEADER ] ) | REMOVE_HEADER )'
   data = {
      'remove' : CliCommand.guardedKeyword( 'remove', helpdesc='Remove a header',
                                            guard=guardPmapDot1qRemove ),
      'REMOVE_BYTES' : RemoveBytesExpr,
      'REMOVE_HEADER' : HeaderRemoveExprFactory()
   }

class AggGroupExpr( CliCommand.CliExpression ):
   expression = ( 'aggregation-group ( AGG_SINGLE_GROUP | '
                                      '{ AGG_GROUP_KW AGG_GROUPS } )' )
   data = {
      'aggregation-group' : 'List of groups to aggregate flow',
      'AGG_SINGLE_GROUP' : matcherTapAggGroupName,
      'AGG_GROUP_KW' : CliMatcher.KeywordMatcher( 'group',
         helpdesc='Set tap group for the interface' ),
      'AGG_GROUPS' : matcherTapAggGroupName,
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      if not args.pop( 'aggregation-group', None ):
         return

      if args.pop( 'AGG_GROUP_KW', None ):
         args[ 'AGG_GROUP' ] = args.pop( 'AGG_GROUPS' )
      else:
         args[ 'AGG_GROUP' ] = [ args.pop( 'AGG_SINGLE_GROUP' ) ]

class AggIdTagExpr( CliCommand.CliExpression ):
   expression= 'id-tag AGG_PORT_ID [ AGG_INNER_PORT_KW AGG_INNER_PORT_ID ]'
   data = {
      'id-tag' : 'Port identity',
      'AGG_PORT_ID' : CliMatcher.IntegerMatcher( 1, 4094,
         helpdesc='Tap port id tag' ),
      'AGG_INNER_PORT_KW' : CliCommand.guardedKeyword( 'inner',
         helpdesc='Inner port identity',
         guard=qinqIdentityTaggingGuard ),
      'AGG_INNER_PORT_ID' : CliMatcher.IntegerMatcher( 1, 4094,
         helpdesc='Inner tap port id tag' )
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      if not args.pop( 'id-tag', None ):
         return
      args.pop( 'AGG_INNER_PORT_KW', None )
      args[ 'AGG_ID_TAG' ] = IdentityTuple( args.pop( 'AGG_PORT_ID' ),
            args.pop( 'AGG_INNER_PORT_ID', 0 ) )

class AggMacAddressExpr( CliCommand.CliExpression ):
   expression= 'mac-address dest DMAC [ src SMAC ]'
   data = {
      'mac-address' : matcherMacAddress,
      'dest' : matcherMacAddressDest,
      'DMAC' : macAddrMatcher,
      'src' : matcherMacAddressSrc,
      'SMAC' : macAddrMatcher
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      if not args.pop( 'mac-address', None ):
         return
      args.pop( 'dest', None )
      args.pop( 'src', None )
      if EthAddrStr( args[ 'DMAC' ] ) == EthAddr():
         mode.addError( TapAggPmapCliLib.TapAggCliError[ 'setMacAddress' ] )
         return
      elif 'SMAC' in args and EthAddrStr( args[ 'SMAC' ] ) == EthAddr():
         mode.addError( TapAggPmapCliLib.TapAggCliError[ 'setMacAddress' ] )
         return
      args[ 'MAC_ADDRESSES' ] = EthAddrTuple( args.pop( 'DMAC' ),
            args.pop( 'SMAC', EthAddr() ) )

class AggIntfExpr( CliCommand.CliExpression ):
   expression = 'AGG_INTF_KW { AGG_ET_INTFS | AGG_LAG_INTFS | AGG_SW_INTFS | ' \
                              'AGG_INTERNAL_RECIRC_INTFS }'
   data = {
      'AGG_INTF_KW' : CliMatcher.KeywordMatcher( 'interface',
         helpdesc='List of interfaces to aggregate flow' ),
      'AGG_ET_INTFS' : CliCommand.singleNode(
         matcher=Intf.IntfRange.IntfRangeMatcher(
            explicitIntfTypes=( EthPhyAutoIntfType, ) ) ),
      'AGG_LAG_INTFS' : CliCommand.singleNode(
         matcher=Intf.IntfRange.IntfRangeMatcher(
            explicitIntfTypes=( LagAutoIntfType, ) ) ),
      'AGG_SW_INTFS' : CliCommand.singleNode(
         matcher=Intf.IntfRange.IntfRangeMatcher(
            explicitIntfTypes=( SwitchAutoIntfType, ) ) ),
      'AGG_INTERNAL_RECIRC_INTFS' : CliCommand.singleNode(
         matcher=Intf.IntfRange.IntfRangeMatcher(
            explicitIntfTypes=( InternalRecircAutoIntfType, ) ) ),
   }

   @classmethod
   def adapter( cls, mode, args, argsList ):
      if not args.pop( 'AGG_INTF_KW', None ):
         return

      # Make a flat list of all interfaces.
      args[ 'AGG_INTFS' ] = [ intf
                              for arg in cls.data
                              for intf in args.pop( arg, () ) ]

class PmapActTypeExpr( CliCommand.CliExpression ):
   expression = '''
      drop | REMOVE |
      ( set ( ( NEXTHOP_GROUPS [ TIMESTAMP ] )
              | ( AGG_GROUP [ REMOVE ] )
              | ( [ AGG_GROUP ] (
                      ( TIMESTAMP
                      | ( MAC_ADDRESS [ TIMESTAMP ] )
                      | ( ID_TAG [ MAC_ADDRESS ] [ TIMESTAMP ] )
                      | ( AGG_INTFS [ ID_TAG ] [ MAC_ADDRESS ] [ TIMESTAMP ] ) )
                   [ REMOVE ] ) ) ) )'''
   data = {
      'drop': PolicyMapCliLib.matcherDrop,
      'REMOVE': RemoveCmdExpr,
      'set' : PolicyMapCliLib.matcherActionSet,
      'AGG_GROUP' : AggGroupExpr,
      'NEXTHOP_GROUPS' : AggNexthopGroupExprFactory(),
      'TIMESTAMP' : TimestampExprFactory(),
      'ID_TAG' : AggIdTagExpr,
      'AGG_INTFS': AggIntfExpr,
      'MAC_ADDRESS': AggMacAddressExpr,
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      setArg = args.pop( 'set', None )
      dropArg = args.pop( 'drop', None )
      if not ( setArg or dropArg ):
         return
      group = args.pop( 'AGG_GROUP', None )
      nexthopGroups = args.pop( 'NEXTHOP_GROUPS', None )
      intfs = args.pop( 'AGG_INTFS', None )
      idTag = args.pop( 'AGG_ID_TAG', None )
      macAddress = args.pop( 'MAC_ADDRESSES', None )
      timestamp = args.pop( 'TIMESTAMP', None )
      headerRemove = args.pop( 'REMOVE_HEADER', None )
      actions = []
      data = []
      if group or intfs:
         aggGroupData = {}
         if group:
            aggGroupData[ 'group' ] = ( 'set', group )
         if intfs:
            aggGroupData[ 'intf' ] = ( 'set', intfs )
         actions.append( TapAggPmapCliLib.actSetAggregationGroup )
         data.append( aggGroupData )
      if idTag:
         actions.append( TapAggPmapCliLib.actSetIdentityTag )
         data.append( idTag )
      if nexthopGroups:
         actions.append( TapAggPmapCliLib.actSetNexthopGroup )
         data.append( nexthopGroups )
      if dropArg:
         actions.append( TapAggPmapCliLib.actDrop )
         data.append( None )
      if macAddress:
         actions.append( TapAggPmapCliLib.actSetMacAddress )
         data.append( macAddress )
      if timestamp:
         actions.append( TapAggPmapCliLib.actSetTimestampHeader )
         data.append( None )
      if headerRemove:
         actions.append( TapAggPmapCliLib.actSetHeaderRemove )
         data.append( headerRemove )
      args[ 'removeBytesActType' ] = args.get( 'removeBytesActType' )
      args[ 'rawRuleActType' ] = ( tuple( actions ), tuple( data ) )

#-------------------------------------------------------------------------------
# Raw match statements in policy map mode
#-------------------------------------------------------------------------------
class TapAggCfgCmdBinder( PolicyMapCliLib.PolicyCfgCmdBinder ):

   def __init__( self ):
      PolicyMapCliLib.PolicyCfgCmdBinder.__init__( self )
      cls = self.__class__

      ########################################################
      #                       IP filters
      ########################################################
      class IpAclCommand( CliCommand.CliCommandClass ):
         syntax = ( '[ SEQ_NUM ] ACTION [ VLAN_EXPR ] ip IP_FILTER [ log ] '
                                                '[ RAW_RULE ]' )
         noOrDefaultSyntax = ( 'ACTION [ VLAN_EXPR ] ip IP_FILTER [ log ] '
                                                '[ RAW_RULE ]' )
         data = {
            'SEQ_NUM': AclCli.seqnumMatcher,
            'ACTION': self.actionMatcher,
            'VLAN_EXPR': self.vlanExpression,
            'ip' : PolicyMapCliLib.matcherIpAccessGroupType,
            'IP_FILTER': self.ipFilterExpression,
            'log': AclCliRules.logMatcher,
            'RAW_RULE' : PmapActTypeExpr,
         }

         @staticmethod
         def adapter( mode, args, argsList ):
            cls.adaptRuleCommonArgs( mode, args )
            args[ 'mirrorSession' ] = None
            args[ 'rawRuleActType' ] = args.get( 'rawRuleActType' )
            # We explicitly set protoMask as 0xff to make it consistent with
            # how class-map handles it
            if 'protocol' in args:
               args[ 'protoMask' ] = 0xff
            args[ 'protocol' ] = args.get( 'protocol', IPPROTO_IP )
            args.pop( 'ip' )
            cls._handleIpArgs( mode, args ) # pylint: disable-msg=protected-access

         @staticmethod
         def handler( mode, args ):
            if CliCommand.isNoOrDefaultCmd( args ):
               cls.adaptNoOrDefaultRuleCommonArgs( mode, args )

            cls.handleIpRule( mode, **args )

         noOrDefaultHandler = handler

      PolicyMapModeTapAgg.addCommandClass( IpAclCommand )

      ########################################################
      #                    IPv6 filters
      ########################################################
      class Ip6AclCommand( CliCommand.CliCommandClass ):
         syntax = ( '[ SEQ_NUM ] ACTION [ VLAN_EXPR ] ipv6 IP_FILTER [ log ] '
                                                   '[ RAW_RULE ]' )
         noOrDefaultSyntax = ( 'ACTION [ VLAN_EXPR ] ipv6 IP_FILTER [ log ] '
                                                   '[ RAW_RULE ]' )
         data = {
            'SEQ_NUM': AclCli.seqnumMatcher,
            'ACTION': self.actionMatcher,
            'VLAN_EXPR': self.vlanExpression,
            'ipv6' : CliCommand.guardedKeyword( 'ipv6',
               helpdesc='Specify Ipv6 Access-groups',
               guard=guardIp6Acls ),
            'IP_FILTER': self.ip6FilterExpression,
            'log': AclCliRules.logMatcher,
            'RAW_RULE' : PmapActTypeExpr,
         }

         @staticmethod
         def adapter( mode, args, argsList ):
            cls.adaptRuleCommonArgs( mode, args )
            args[ 'mirrorSession' ] = None
            args[ 'rawRuleActType' ] = args.get( 'rawRuleActType' )
            args[ 'removeBytesActType' ] = args.get( 'removeBytesActType' )
            # We explicitly set protoMask as 0xff to make it consistent with
            # how class-map handles it
            if 'protocol' in args:
               args[ 'protoMask' ] = 0xff
            args[ 'protocol' ] = args.get( 'protocol', IPPROTO_IP )
            args.pop( 'ipv6' )
            self._handleIp6Args( mode, args ) # pylint: disable-msg=protected-access

         @staticmethod
         def handler( mode, args ):
            if CliCommand.isNoOrDefaultCmd( args ):
               cls.adaptNoOrDefaultRuleCommonArgs( mode, args )

            cls.handleIp6Rule( mode, **args )

         noOrDefaultHandler = handler

      PolicyMapModeTapAgg.addCommandClass( Ip6AclCommand )

      ########################################################
      #                    Mac filters
      ########################################################
      class MacAclCommand( CliCommand.CliCommandClass ):
         syntax = ( '[ SEQ_NUM ] ACTION [ VLAN_EXPR ] mac MAC_FILTER [ log ] '
                                                   '[ RAW_RULE ]' )
         noOrDefaultSyntax = ( 'ACTION [ VLAN_EXPR ] mac MAC_FILTER [ log ] '
                                                   '[ RAW_RULE ]' )
         data = {
            'SEQ_NUM': AclCli.seqnumMatcher,
            'ACTION': self.actionMatcher,
            'VLAN_EXPR': self.vlanExpression,
            'mac' : CliCommand.guardedKeyword( 'mac',
               helpdesc='Specify Mac Access-groups',
               guard=guardMacAcls ),
            'MAC_FILTER': self.macFilterExpression,
            'log': AclCliRules.logMatcher,
            'RAW_RULE' : PmapActTypeExpr,
         }
         @staticmethod
         def adapter( mode, args, argsList ):
            cls.adaptRuleCommonArgs( mode, args )
            args[ 'mirrorSession' ] = None
            args[ 'rawRuleActType' ] = args.get( 'rawRuleActType' )
            args.pop( 'mac' )

         @staticmethod
         def handler( mode, args ):
            if CliCommand.isNoOrDefaultCmd( args ):
               cls.adaptNoOrDefaultRuleCommonArgs( mode, args )
            cls.handleMacRule( mode, **args )

         noOrDefaultHandler = handler

      PolicyMapModeTapAgg.addCommandClass( MacAclCommand )

      #-----------------------------------------------------------------
      #              no sequence-number
      #-----------------------------------------------------------------
      class NoSequenceCmd( CliCommand.CliCommandClass ):
         noOrDefaultSyntax = 'SEQUENCES'
         data = {
            'SEQUENCES': AclCliRules.seqRangeMatcher
         }
         noOrDefaultHandler = cls.handleNoSeq

      PolicyMapModeTapAgg.addCommandClass( NoSequenceCmd )

      #-----------------------------------------------------------------
      #              resequence
      #-----------------------------------------------------------------
      class ResequenceCmd( CliCommand.CliCommandClass ):
         syntax = 'resequence [ START [ INC ] ]'
         data = {
            'resequence' : PolicyMapCliLib.matcherResequence,
            'START' : self.startSeqMatcher,
            'INC' : self.incSeqMatcher,
         }

         @staticmethod
         def handler( mode, args ):
            start = args.get( 'START', self.defaultSeqStart )
            inc = args.get( 'INC', self.defaultSeqInc )
            cls.handleResequence( mode, start, inc )

      PolicyMapModeTapAgg.addCommandClass( ResequenceCmd )

def getPMapNameRuleTapAgg( mode ):
   return sorted( tapAggPmapConfig.pmapType.pmap.keys() )

def getCMapNameRuleTapAgg( mode ):
   return sorted( tapAggPmapConfig.cmapType.cmap.keys() )

#-------------------------------------------------------------------------------
# [no] service-policy type tapagg input <name>
#-------------------------------------------------------------------------------
def handleServicePolicy( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   pmapName = args.get( 'PMAP_NAME' )

   if no is False:
      intfName = mode.intf.name
      if not intfName in tapAggHwConfig.tapPort:
         mode.addWarning(
            "Service policy will not be deployed until the interface"
            " is in tap mode and state is up." )

   PolicyMapCliLib.handleServicePolicy( mode, no, pmapName, tapAggPmapConfig,
                                        tapAggStatusRequestDir, tapAggPmapStatus,
                                        intfConfig=tapAggIntfConfig )

class PolicyMapModeTapAgg( PolicyMapCliLib.PolicyMapMode ):
   name = 'Policy Map TapAgg Configuration'

   def __init__( self, parent, session, context ):
      PolicyMapCliLib.PolicyMapMode.__init__( self, parent, session, 
                                               context, 'tapagg' )

   def timestampActionSupported( self ):
      return True

   def switchToPmapClassMode( self, context ):
      childMode = self.childMode( PolicyMapClassModeTapAgg, context=context )
      self.session_.gotoChildMode( childMode )

   def defaultActionTypeAndData( self ):
      return ( tuple(), tuple() )

   def actionDataToClassAction( self, cmapName, actType, actData ):
      actions = tapAggPmapConfig.tapAggActions
      if actType == TapAggPmapCliLib.actSetAggregationGroup:
         groupData = actData.get( 'group' )
         intfData = actData.get( 'intf' )
         tapAggAct = actions.aggGroup.newMember( cmapName, UniqueId() )
         pmapGroup = tapAggPmapConfig.group.newMember( self.pmapName + '_' + \
                                                          cmapName )
         pmapGroup.rawAclMatch = True
         if groupData:
            groupOp, groupList = groupData
            assert groupOp == 'set'
            for group in groupList:
               pmapGroup.groupName[ group ] = True
               tapAggAct.aggGroup[ group ] = True
         if intfData:
            intfOp, intfList = getIntfOpAndList( intfData )
            assert intfOp == 'set'
            for intf in intfList:
               tapAggAct.aggIntf[ intf ] = True
      elif actType == TapAggPmapCliLib.actSetIdentityTag:
         tapAggAct = actions.idTag.newMember( cmapName, UniqueId(), actData )
      elif actType == TapAggPmapCliLib.actStripHdrBytes:
         tapAggAct = actions.stripHdrBytes.newMember( cmapName, UniqueId() )
         tapAggAct.stripHdrBytes = actData
      elif actType == TapAggPmapCliLib.actSetNexthopGroup:
         tapAggAct = actions.nexthopGroup.newMember( cmapName, UniqueId() )
         for nexthopGroup in actData:
            tapAggAct.nexthopGroup[ nexthopGroup ] = True
      elif actType == TapAggPmapCliLib.actDrop:
         tapAggAct = actions.drop.newMember( cmapName, UniqueId() )
      elif actType == TapAggPmapCliLib.actSetMacAddress:
         tapAggAct = actions.macAddress.newMember(
               cmapName, UniqueId(), actData )
      elif actType == TapAggPmapCliLib.actSetTimestampHeader:
         tapAggAct = actions.timestampHeader.newMember( cmapName, UniqueId() )
      elif actType == TapAggPmapCliLib.actSetHeaderRemove:
         tapAggAct = actions.headerRemove.newMember(
               cmapName, UniqueId(), actData )
      else:
         tapAggAct = None
      return tapAggAct

   def actionArgsToActionData( self, rawRuleActType=None, **kwargs ):
      actType, actData = self.defaultActionTypeAndData()

      if rawRuleActType:
         rawActType, rawActData = rawRuleActType
         if isinstance( rawActType, str ):
            actType += ( rawActType, )
            actData += ( rawActData, )
         else:
            actType += rawActType
            actData += rawActData

      if kwargs.get( 'removeBytesActType' ):
         removeBytesType, removeBytesData = kwargs.pop( 'removeBytesActType' )
         actType += ( removeBytesType, )
         actData += ( removeBytesData, )

      return ( actType, actData )

   @staticmethod
   def showPolicyMap( mode, args ):
      pmapName = args.get( 'PMAPNAME' )
      summary = 'summary' in args
      pmapAllModel = PolicyMapModel.PolicyMapAllModel()
      pmapAllModel.mapType = 'mapTapAgg'
      emapType = PolicyMap.mapTypeToEnum( 'tapagg' )
      intfConfig = tapAggIntfConfig

      # PolicyMapModelContainer requires both cli input config and merged config.
      # In case of tapAgg, merged config is the same as cli input config since
      # there is no configMergeSm.
      pmapsContainer = PolicyMapModelContainer( tapAggPmapConfig, intfConfig,
                                                intfConfig, tapAggPmapConfig,
                                                tapAggPmapStatus, aclConfig,
                                                None, None, None, None, emapType,
                                                None, pmapAllModel )
      checkpoints = ( tapAggPmapCheckpoint, _tapAggPmapSessionCheckpoint( mode ) )
      if pmapName is None:
         pmapsContainer.populateAll( mode, summary, checkpoints )
      else:
         pmapsContainer.populatePolicyMap( mode, pmapName, summary, checkpoints )
      return pmapAllModel

TapAggCfgCmdBinder()

def gotoPolicyMapMode( mode, args ):
   pmapName = args.get( 'PMAPNAME' )
   mapType = 'tapagg'
   context = None
   t0( 'gotoPolicyMapMode of policy', pmapName )
   emapType = PolicyMap.mapTypeToEnum( mapType )
   if context is None:
      context = TapAggPmapCliLib.TapAggPolicyMapContext( tapAggPmapConfig,
                                                   tapAggStatusRequestDir,
                                                   tapAggPmapStatus, pmapName,
                                                   emapType )

   if pmapName in tapAggPmapConfig.pmapType.pmap:
      context.copyEditPmap()
   else:
      context.newEditPmap()

   mode.pmapContext = context
   childMode = mode.childMode( PolicyMapModeTapAgg, context=context )
   mode.session_.gotoChildMode( childMode )

def clearPolicyMapCheckpointStatus( mode, pmapName ):
   clearPolicyMapCheckpoint( pmapName, _tapAggPmapSessionCheckpoint( mode ) )
   clearPolicyMapCheckpoint( pmapName, tapAggPmapCheckpoint )

def deletePolicyMap( mode, args ):
   pmapName = args.get( 'PMAPNAME' )
   mapType = 'tapagg'
   lastMode = mode.session_.modeOfLastPrompt()
   if isinstance( lastMode, PolicyMapModeTapAgg ):
      if pmapName == lastMode.pmapName:
         lastMode.pmapContext = None
   if pmapName not in tapAggPmapConfig.pmapType.pmap:
      return

   for className in \
          tapAggPmapConfig.pmapType.pmap[ pmapName ].currCfg.classAction:
      groupKey = pmapName + '_' + className
      if groupKey in tapAggPmapConfig.group:
         del tapAggPmapConfig.group[ groupKey ]

   clearPolicyMapCheckpointStatus( mode, pmapName )

   emapType = PolicyMap.mapTypeToEnum( mapType )
   dummyContext = TapAggPmapCliLib.TapAggPolicyMapContext( tapAggPmapConfig, 
                                                tapAggStatusRequestDir,
                                                tapAggPmapStatus, pmapName, 
                                                emapType )
   dummyContext.delPmap( mode, pmapName )

#-------------------------------------------------------------------------------
# [ no | default ] [ SEQ_NUM ] class CLASS
#-------------------------------------------------------------------------------
class SetClassCmd( CliCommand.CliCommandClass ):
   syntax = '[ SEQ_NUM ] class CLASS'
   noOrDefaultSyntax = syntax
   data = {
      'SEQ_NUM': PolicyMapCliLib.matcherSeqNum,
      'class': PolicyMapCliLib.matcherClass,
      'CLASS': CliMatcher.DynamicNameMatcher( getCMapNameRuleTapAgg,
         helpdesc='Class Map Name' )
   }

   @staticmethod
   def handler( mode, args ):
      mode.setClass( False, args.get( 'SEQ_NUM' ), args[ 'CLASS' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.setClass( True, args.get( 'SEQ_NUM' ), args[ 'CLASS' ] )

PolicyMapModeTapAgg.addCommandClass( SetClassCmd )

#-------------------------------------------------------------------------------
# Support for default interface command.
#-------------------------------------------------------------------------------
class TapAggPmapIntfJanitor( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      t0( "TapAggPmapIntfJanitor: interface", self.intf_.name, "going away" )
      if self.intf_.name in tapAggIntfConfig.intf:
         del tapAggIntfConfig.intf[ self.intf_.name ]

def clearPolicyMapCountersAll( mode, ckptStatus ):
   for policyName in tapAggPmapConfig.pmapType.pmap:
      clearOnePolicyMapCounters( mode, policyName, ckptStatus )

def clearOnePolicyMapCounters( mode, policyName, ckptStatus ):
   pmapConfig = PolicyMapCliLib.getPmapSubConfig( tapAggPmapConfig, policyName )
   pmapStatuses = [ status.status.get( policyName ) for status in
                    six.itervalues( tapAggPmapStatus ) ]
   pmapStatuses = [ _f for _f in pmapStatuses if _f ]
   pmapStatus = pmapStatuses[ 0 ]

   if not pmapConfig or not pmapStatus:
      return

   clearPolicyMapCounters( mode, policyName, tapAggPmapConfig, tapAggPmapConfig,
                           tapAggPmapStatus, ckptStatus )

def clearTapAggPMapCounters( mode, args ):
   policyName = args.get( 'PMAPNAME' )
   session = args.get( 'session' )

   if session:
      ckptStatus = _tapAggPmapSessionCheckpoint( mode )
      logAction = False
   else:
      ckptStatus = tapAggPmapCheckpoint
      logAction = True

   if policyName:
      if logAction:
         # pylint: disable-next=consider-using-f-string
         Intf.Log.logClearCounters( "tapagg policy", "policy map %s" % policyName )
      clearOnePolicyMapCounters( mode, policyName, ckptStatus )
   else:
      if logAction:
         Intf.Log.logClearCounters( "tapagg policy", "policy map all" )
      clearPolicyMapCountersAll( mode, ckptStatus )

def Plugin( entityManager ):
   global aclConfig, tapAggHwStatus, tapAggHwConfig
   global tapAggPmapConfig, tapAggStatusRequestDir, tapAggPmapStatus, aclStatus
   global policyOpChkr, tapAggPmapCheckpoint, tapAggIntfConfig

   #
   # Acl change status callback that returns the impact of an Acl change on
   # routing policy.
   def isPolicyHwStatusOk( mode, aclName, aclType, aclUpdateType ):
      rc = True
      err = {}
      err[ 'error' ] = ''
      if not mode.session_.startupConfig() and AclCli.allHwStatusEnabled():
         rc, result = policyOpChkr.verify( 'acl', aclName )
         if not rc:
            err[ 'error' ] = ( result.error if result else 
                               'unknown TagAgg policy status' )
         else:
            if result and result.warning:
               # pylint: disable-next=consider-using-f-string
               err[ 'warning' ] = "Warning: ACL %s %s" % ( aclName, result.warning )
      return rc, err

   # pylint: disable-next=consider-using-f-string
   tapAggCellRootNode = 'cell/%d/tapagg' % Cell.cellId()
   tapAggStatusPath = tapAggCellRootNode + '/pmapstatus'
   tapAggPmapStatus = LazyMount.mount( entityManager, tapAggStatusPath, 
                                       "Tac::Dir", 'ri' )

   aclConfig = ConfigMount.mount( entityManager, "acl/config/cli",
                                  "Acl::Input::Config", "w" )
   tapAggHwStatus = LazyMount.mount( entityManager, 'tapagg/hwstatus',
                                     'TapAgg::HwStatus', 'r' )
   tapAggHwConfig = LazyMount.mount( entityManager, 'tapagg/hwconfig',
                                     'TapAgg::HwConfig', 'r' )
   tapAggStatusRequestDir = LazyMount.mount( entityManager,
                                             'tapagg/pmapStatusRequestDir/cli',
                                             'PolicyMap::PolicyMapStatusRequestDir',
                                             'wc' )
   tapAggIntfConfig = ConfigMount.mount( entityManager, 'tapagg/intfconfig',
                                         'PolicyMap::IntfConfig', 'w' )
   tapAggPmapConfig = ConfigMount.mount( entityManager, 'tapagg/pmapconfig',
                                         'TapAgg::PmapConfig', 'w' )

   aclStatus = LazyMount.mount( entityManager, "acl/status/all",
                                "Acl::Status", "r" )

   tapAggPmapCheckpoint = LazyMount.mount( entityManager,
                                           "tapagg/pmapcheckpoint",
                                           "TapAgg::PmapCheckpointStatus", "w" )

   # Intall a callback to verify impact of acl changes on policy
   policyOpChkr = PolicyMapCliLib.PolicyOpChkr( tapAggStatusRequestDir,
                                                tapAggPmapStatus )
   AclCli.registerAclConfigValidCallback( isPolicyHwStatusOk )
   IntfCli.Intf.registerDependentClass( TapAggPmapIntfJanitor )
   registerCmapStatusFilter( PolicyMap.tapaggMapType, ( lambda x: x ) )
