#!/usr/bin/env python3
# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import Tracing
from CliPlugin import PolicyMapCliLib

__defaultTraceHandle__ = Tracing.Handle( 'TapAggPmapCliLib' )
t0 = Tracing.trace0
t1 = Tracing.trace1
t9 = Tracing.trace9

tapAggIpAcls = 4096
tapAggMacAcls = 4096
tapAggMaxRules = tapAggIpAcls + tapAggMacAcls
tapAggMaxSequence = 0xFFFFFFFF
tapAggDefaultSeqInc = 10

tacActionType = Tac.Type( 'PolicyMap::ActionType' )
actSetAggregationGroup = tacActionType.setAggregationGroup
actSetIdentityTag = tacActionType.setIdentityTag
actStripHdrBytes = tacActionType.stripHeaderBytes
actSetNexthopGroup = tacActionType.setNexthopGroup
actSetMacAddress = tacActionType.setMacAddress
actSetTimestampHeader = tacActionType.setTimestampHeader
actSetHeaderRemove = tacActionType.setHeaderRemove
actDrop = tacActionType.deny
actActionSet = tacActionType.actionSet

tacStripHdrType = Tac.Type( 'PolicyMap::StripHeaderType' )
stripHdrNone = tacStripHdrType.noStrip
stripHdrDot1q = tacStripHdrType.dot1q

UniqueId = Tac.Type( 'Ark::UniqueId' )

aclConfig = None
ethLagIntfStatusDir = None
allIntfStatusDir = None

TapAggCliError = {
   'setMacAddress' : 'MAC address 0000.0000.0000 is not supported',
}

# Retrieve PolicyMap::NamedActionSet from current policy-map
# config using PolicyMapClassContext object
def getNamedActionSetObj( policyMapClassContext, pmapName, cmapName, aName ):
   pmaps = policyMapClassContext.pmapContext.config().pmapType.pmap
   pmapCfg = pmaps.get( pmapName )
   if not pmapCfg or not pmapCfg.currCfg:
      return None
   classAction = pmapCfg.currCfg.classAction.get( cmapName )
   if not classAction:
      return None 
   policyAction = classAction.policyAction.get( actActionSet )
   if not policyAction:
      return None 
   namedActionSet = policyAction.namedActionSet.get( aName )
   return namedActionSet

def stripHdrAction( hdrType, hdrInfo ):
   stripHdr = Tac.Value( 'PolicyMap::StripHeaderInfo' )
   stripHdr.hdrType = hdrType
   if hdrType == stripHdrDot1q:
      stripHdr.dot1qRemoveVlans = hdrInfo
   # Add new strippable hdr-types here.
   return stripHdr

def removeActionFromTapAggActions( config, action ):
   actType = action.actionType
   actions = config.tapAggActions
   if actType == actSetAggregationGroup:
      del actions.aggGroup[ action.id ]
   elif actType == actSetIdentityTag:
      del actions.idTag[ action.id ]
   elif actType == actStripHdrBytes:
      del actions.stripHdrBytes[ action.id ]
   elif actType == actSetNexthopGroup:
      del actions.nexthopGroup[ action.id ]
   elif actType == actSetMacAddress:
      del actions.macAddress[ action.id ]
   elif actType == actSetTimestampHeader:
      del actions.timestampHeader[ action.id ]
   elif actType == actSetHeaderRemove:
      del actions.headerRemove[ action.id ]
   elif actType == actActionSet:
      del actions.actionSet[ action.id ]
   elif actType == actDrop:
      del actions.drop[ action.id ]
   else:
      # pylint: disable-next=consider-using-f-string
      assert False, 'Invalid actType %s' % actType

class TapAggPolicyMapContext( PolicyMapCliLib.PolicyMapContext ):
   def __init__( self, config, statusReqDir, status, pmapName, pmapType ):
      PolicyMapCliLib.PolicyMapContext.__init__( self, config, statusReqDir, \
                     status, pmapName, pmapType )

   def defaultSequenceInc( self ):
      return tapAggDefaultSeqInc

   def maxRules( self ):
      return tapAggMaxRules 

   def removeAction( self, action ):
      removeActionFromTapAggActions( self.config(), action )

   # Function to remove groups from tapAggPmapConfig::group 
   # These groups are only those specified in Raw acl match in policy Maps
   def removeGroupsFromClass( self, seq ):
      #seq = self.lookupRuleToSeq( action )
      if not seq:
         return
      cmapName = self.npmap.classPrio.get( seq, None )
      groupKey = self.pmapName() + '_' + cmapName
      if groupKey in self.config().group:
         del self.config().group[ groupKey ]
   
   def delRuleAtSeqnum( self, seqnum, clearActions=False ):
      self.removeGroupsFromClass( seqnum )
      PolicyMapCliLib.PolicyMapContext.delRuleAtSeqnum( self, seqnum,
                                                        clearActions=clearActions )

   def copyAction( self, src ):
      actionType = src.actionType
      actions = self.config().tapAggActions
      if actionType == actSetAggregationGroup:
         dst = actions.aggGroup.newMember( src.className, UniqueId() )
         dst.copyAction( src )
         return dst 
      elif actionType == actSetIdentityTag:
         return actions.idTag.newMember( src.className, UniqueId(), src.idTag )
      elif actionType == actStripHdrBytes:
         dst = actions.stripHdrBytes.newMember( src.className, UniqueId() )
         dst.copyAction( src )
         return dst 
      elif actionType == actSetNexthopGroup:
         dst = actions.nexthopGroup.newMember( src.className, UniqueId() )
         dst.copyAction( src )
         return dst
      elif actionType == actSetMacAddress:
         return actions.macAddress.newMember(
               src.className, UniqueId(), src.macAddress )
      elif actionType == actSetTimestampHeader:
         return actions.timestampHeader.newMember(
               src.className, UniqueId() )
      elif actionType == actSetHeaderRemove:
         return actions.headerRemove.newMember(
               src.className, UniqueId(), src.headerRemove )
      elif actionType == actActionSet:
         dst = actions.actionSet.newMember( src.className, UniqueId() )
         dst.copyAction( src )
         return dst
      elif actionType == actDrop:
         return actions.drop.newMember( src.className, UniqueId() )
      else:
         # pylint: disable-next=consider-using-f-string
         assert False, "Invalid actionType %s" % actionType
         return None

   def identicalClassActions( self, c1Action, c2Action ):
      c1ActTypes = list( c1Action.policyAction )
      c2ActTypes = list( c2Action.policyAction )
      if c1ActTypes != c2ActTypes:
         return PolicyMapCliLib.CHANGED
      for actType in c1ActTypes:
         if not c1Action.policyAction[ actType ].equalTo(
               c2Action.policyAction[ actType ] ):
            return PolicyMapCliLib.CHANGED
      return PolicyMapCliLib.IDENTICAL
