# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import re
import sys

import BasicCli
import BasicCliModes
import CliCommand
import CliMatcher
import CliModel
import CliParser
import CliPlugin.IntfCli as IntfCli # pylint: disable=consider-using-from-import
import CliPlugin.VrfCli as VrfCli # pylint: disable=consider-using-from-import
import Cell
import CmdExtension
import ConfigMount
from IpLibConsts import DEFAULT_VRF
import LazyMount
import ShowCommand
import TableOutput
import Tac
import TcpdumpLib
import Url

# pkgdeps: import UrlPlugin.FlashUrl (forced dependency, see AID 10)

#------------------------------------------------------------------------------------
# Enable mode commands:
# tcpdump [<options>] [filter <filter>] 
#
# Config mode commands:
# [no] tcpdump session <name> [<options>] [filter <filter>]
# [no] tcpdump session <name> shutdown
#
# Show commands:
# show tcpdump [<name>]
#------------------------------------------------------------------------------------

tcpdumpSessionConfigDir = None
tcpdumpSessionStatusDir = None
intfStatusAll = None
ipAddressDir = None
ip6AddressDir = None
mirroringStatus = None
mirroringHwCapability = None
intfStatusLocal = None
allVrfStatusLocal = None

def tcpdumpMirroringSupportGuard( mode, token ):
   if mirroringHwCapability.cpuDestSupported:
      return None
   return CliParser.guardNotThisPlatform

tcpdumpKwMatcher = CliMatcher.KeywordMatcher( 'tcpdump', 
   helpdesc= 'Monitor packets with tcpdump' )
sessionKwMatcher = CliMatcher.KeywordMatcher( 'session', 
   helpdesc='Define the name of a session' )
sessionNameMatcher = CliMatcher.DynamicNameMatcher(
   lambda mode: tcpdumpSessionStatusDir.session,
   'The name of the session' )
filterKwMatcher = CliMatcher.KeywordMatcher( 'filter', 
   helpdesc='Set the filtering expression' )
filterMatcher = CliMatcher.StringMatcher(
   helpname='FILTER',
   helpdesc='Selects which packets will be dumped' )
verboseKwMatcher = CliMatcher.KeywordMatcher( 'verbose',
   helpdesc='Enable verbose mode' )
lookupNamesKwMatcher = CliMatcher.KeywordMatcher( 'lookup-names',
   helpdesc='Enable reverse DNS lookups' )
fileMatcher = Url.UrlMatcher(
   lambda fs: fs.realFileSystem() and fs.supportsWrite(), 'Output file',
   acceptSimpleFile=False )
filePatternMatcher = CliMatcher.PatternMatcher( '[^:]+', helpname='WORD',
   helpdesc='Output file' )

def filenameToUrl( mode, filename ):
   url = filename
   if isinstance( filename, str ):
      # force it to be file: for error checking
      context = Url.Context( *Url.urlArgsFromMode( mode ) )
      url = Url.parseUrl( "file:" + filename, context )
   
   if isinstance( mode, BasicCliModes.EnableMode ):
      # Under enable mode, the fs.allowedPath check is skipped
      # tcpdump command runs in nsCmdExt.subprocessPopen with admin permission
      # and will return permission denied error for files with no write access
      url.setAllowAllPaths( True )
   return url

def formatTcpdumpOption( mode, name, value ):
   '''Resolve EOS to UNIX path for file'''
   if name == 'file' and value:
      url = filenameToUrl( mode, value )
      return url.localFilename() # pylint: disable-msg=E1103
   elif name == 'filter' and value is None:
      return ''
   elif name == 'device':
      if 'intf' in value:
         return str( value[ 'intf' ] )
      elif 'intfAddr' in value:
         return str( value[ 'intfAddr' ] )
      elif 'mirrorName' in value:
         return value[ 'mirrorName' ]
      elif 'lanz' in value:
         return 'lanz'
      else:
         return None
   else:
      return value

def parseTcpdumpOptions( mode, options ):
   # Parses the arguments given to the Cli and produces a dictionary of option
   # settings which exhaustively define a configuration, except for the
   # 'enabled' parameter. There is a different set of default options for a
   # tcpdump session (i.e. when sessionName is not null) than for a tcpdump cli
   # command
   tcpdumpDefaultValue = { 'device': '',
                           'size': 0,
                           'maxFileSize': 0,
                           'file': '',
                           'packetCount': 0,
                           'fileCount': 0,
                           'duration': 0,
                           'verbose' : False,
                           'filter': '',
                           'lookupNames': False,
                           'vrfName' : '' }

   args = dict() # pylint: disable=use-dict-literal
   for var in tcpdumpDefaultValue: # pylint: disable=consider-using-dict-items
      # Retrieve arguments' value if present in the command, or the default
      # value otherwise
      if var == 'device':
         if var in options:
            args[ var ] = formatTcpdumpOption( mode, var, options[ var ] )
         else:
            args[ var ] = tcpdumpDefaultValue[ var ]
      else:
         optionValue = options.get( var, tcpdumpDefaultValue[ var ] )
         args[var] = formatTcpdumpOption( mode, var, optionValue )

   return args

def doTcpdumpSession( mode, args ):
   options = args.get( 'options' )
   dumpFilter = args.get( 'FILTER' )
   sessionName = args.get( 'SESSION_NAME' )

   # Validate the filter
   if dumpFilter and not re.match( '[a-z]', dumpFilter[ 0 ] ):
      # pylint: disable-next=consider-using-f-string
      mode.addError( "Invalid filter: %s" % dumpFilter )
      return
   options[ 'filter' ] = dumpFilter
   deviceType = 'kernel'

   if 'device' in options:
      options[ 'vrfName' ] = DEFAULT_VRF
      if 'intf' in options[ 'device' ]:
         deviceType = 'interface'
         intf = options[ 'device' ][ 'intf' ]
         if isinstance(intf, dict) and intf[ 'intf' ] == TcpdumpLib.anyInterfaceName:
            deviceType = 'any'
            if intf[ 'vrfName' ]:
               options[ 'vrfName' ] = intf[ 'vrfName' ]
            options[ 'device' ][ 'intf' ] = TcpdumpLib.anyInterfaceName
      elif 'intfAddr' in options[ 'device' ]:
         netwkAddr = options[ 'device' ][ 'intfAddr' ].showNetworkAddr()
         if sessionName:
            deviceType = 'interfaceAddress'
            options[ 'device' ][ 'intf' ] = options[ 'device' ][ 'intfAddr' ]
            options[ 'filter' ] = dumpFilter
         elif netwkAddr:
            deviceType = 'any'
            intfId = options[ 'device' ][ 'intfAddr' ].name
            options[ 'filter' ] = TcpdumpLib.composeFilter( intfId, ipAddressDir,
                                                            ip6AddressDir,
                                                            options[ 'filter' ] )
         else:
            mode.addWarning( "Interface has no IP addresses; no packets can be " +
                             "captured" )
            return
      elif 'mirrorName' in options[ 'device' ]:
         deviceType = 'mirror'
      elif 'lanz' in options[ 'device' ]:
         deviceType = 'lanz'

   if sessionName:
      # Check if the session name is already defined
      if tcpdumpSessionConfigDir.session.get( sessionName ):
         # pylint: disable-next=consider-using-f-string
         mode.addError( "Session %s already exists" % sessionName )
         return
      # Set the session configuration in Sysdb
      opts = Tac.newInstance( "Tcpdump::Options",
             TcpdumpLib.DEFAULT_FILE_PATH_FMT % ( sessionName, sessionName ) )
      opts.deviceType = deviceType
      for k, v in options.items():
         setattr( opts, k, formatTcpdumpOption( mode, k, v ) )
      session = tcpdumpSessionConfigDir.session.newMember( sessionName, opts )
      session.enabled = True

   # Enable mode (session-less)
   else:
      nsCmdExt = CmdExtension.getCmdExtender()
      try:
         # Parse the arguments
         args = parseTcpdumpOptions( mode, options )
         # Build the command to be run following tcpdump's syntax
         deviceStatusAll = { 'interface' : intfStatusAll,
                             'interfaceAddress' : intfStatusAll,
                             'mirror' : mirroringStatus.sessionStatus,
                             'lanz' : mirroringStatus.sessionStatus }
         cmd = TcpdumpLib.tcpdumpCmdSyntax( deviceStatusAll, args, deviceType )
         netNs = TcpdumpLib.anyIntfToNetNs( allVrfStatusLocal, deviceType,
                                             args[ 'vrfName' ] )
         if not netNs:
            netNs = TcpdumpLib.getNetNsName( deviceStatusAll, intfStatusLocal,
                                             allVrfStatusLocal, args[ 'device' ],
                                             deviceType )

         if mode.session_.entityManager_.redundancyStatus().mode == 'active':
            if tcpdumpSessionConfigDir.sessionLess.get( args[ 'device' ] ):
               tcpdumpSessionConfigDir.sessionLess[ args[ 'device' ] ] += 1
            else:
               tcpdumpSessionConfigDir.sessionLess[ args[ 'device' ] ] = 1

         p = nsCmdExt.subprocessPopen( cmd, mode.session, nsName=netNs,
                                       stdout=sys.stdout, stderr=sys.stderr )
         p.communicate()

         if mode.session_.entityManager_.redundancyStatus().mode == 'active':
            tcpdumpSessionConfigDir.sessionLess[ args[ 'device' ] ] -= 1
            if tcpdumpSessionConfigDir.sessionLess[ args[ 'device' ] ] == 0:
               del tcpdumpSessionConfigDir.sessionLess[ args[ 'device' ] ]

      except TcpdumpLib.InvalidTargetException as e:
         mode.addError( str( e ) )
      except OSError as e:
         mode.addError( e.strerror )

tcpdumpIntfSharedMatchObj = object()

class InterfaceExpression( CliCommand.CliExpression ):
   expression = ( 'queue-monitor | ( monitor MIRRORING_NAME ) | '
                  '( interface-address INTF_ADDR ) | '
                  '( interface ( INTF | ( any [ VRF ] ) ) )' )
   data = {
            'queue-monitor': CliCommand.singleNode(
               matcher=CliMatcher.KeywordMatcher( 'queue-monitor',
                  helpdesc='Monitor queue length' ),
               sharedMatchObj=tcpdumpIntfSharedMatchObj ),
            'monitor': CliCommand.singleNode(
               matcher=CliMatcher.KeywordMatcher( 'monitor',
                  helpdesc='Select a monitor session' ),
               guard=tcpdumpMirroringSupportGuard,
               sharedMatchObj=tcpdumpIntfSharedMatchObj ),
            'MIRRORING_NAME': CliMatcher.DynamicNameMatcher( 
               lambda mode: mirroringStatus.sessionStatus,
               helpdesc='Monitor session name' ),
            'interface-address': CliCommand.singleNode(
               matcher=CliMatcher.KeywordMatcher( 'interface-address',
                  helpdesc='Filter on interface\'s IP addresses' ),
               sharedMatchObj=tcpdumpIntfSharedMatchObj ),
            'INTF_ADDR': IntfCli.Intf.matcher,
            'interface': CliCommand.singleNode(
               matcher=CliMatcher.KeywordMatcher( 'interface',
                  helpdesc='Select an interface to monitor (default=fabric)',
                  alternates=[ '-i' ] ),
               sharedMatchObj=tcpdumpIntfSharedMatchObj ),
            'INTF': IntfCli.Intf.matcher,
            'any': 'Monitor any interface',
            'VRF': VrfCli.VrfExprFactory( helpdesc='Specify a VRF',
                                          inclDefaultVrf=True ),
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      device = None
      if 'queue-monitor' in args:
         device = { 'lanz': True }
      elif 'monitor' in args:
         device = { 'mirrorName': args[ 'MIRRORING_NAME' ][ 0 ] }
      elif 'interface-address' in args:
         device = { 'intfAddr': args[ 'INTF_ADDR' ][ 0 ] }
      elif 'interface' in args:
         if 'any' in args:
            vrfName = args[ 'VRF' ][ 0 ] if 'VRF' in args else None
            device = { 'intf': { 'intf': 'any', 'vrfName': vrfName } }
         else:
            device = { 'intf': args[ 'INTF' ][ 0 ] }

      if device is not None:
         args[ 'device' ] = device

maxFileSizeSharedObj = object()
fileCountSharedObj = object()
packetCountSharedObj = object()

class CommonTcpOptions( CliCommand.CliExpression ):
   expression = ( '( file ( FILE | LOCAL_FILE ) ) | '
                  '( size SIZE ) | '
                  '( ( max-file-size | -C ) MAX_FILE_SIZE ) | '
                  '( ( filecount | -W ) FILECOUNT ) | '
                  '( ( packet-count | -c ) PACKET_COUNT ) | '
                  'INTF_EXPR' )
   data = {
            'file': CliCommand.singleNode( CliMatcher.KeywordMatcher( 'file',
               helpdesc='Set the output file', alternates=[ '-w' ] ) ),
            'FILE': fileMatcher,
            'LOCAL_FILE': CliCommand.Node( matcher=filePatternMatcher, hidden=True ),
            'size': CliCommand.singleNode( CliMatcher.KeywordMatcher( 'size',
               helpdesc='Set the maximum number of bytes to dump per packet',
               alternates=[ '-s' ] ) ),
            'SIZE': CliMatcher.IntegerMatcher( 1, 65536, helpdesc='bytes' ),
            'max-file-size': CliCommand.singleNode(
               CliMatcher.KeywordMatcher( 'max-file-size',
                  helpdesc='Specify the maximum size of output file' ),
               sharedMatchObj=maxFileSizeSharedObj ),
            '-C': CliCommand.singleNode(
               matcher=CliMatcher.PatternMatcher( '-C', helpname='-C',
                  helpdesc='Specify the maximum size of output file' ),
               hidden=True,
               sharedMatchObj=maxFileSizeSharedObj ),
            'MAX_FILE_SIZE': CliMatcher.IntegerMatcher( 0, 100,
               helpdesc='millions of bytes' ),
            'filecount': CliCommand.singleNode(
               CliMatcher.KeywordMatcher( 'filecount',
                  helpdesc='Specify the number of output files' ),
               sharedMatchObj=fileCountSharedObj ),
            '-W': CliCommand.singleNode(
               matcher=CliMatcher.PatternMatcher( '-W', helpname='-W',
                  helpdesc='Specify the number of output files' ),
               hidden=True,
               sharedMatchObj=fileCountSharedObj ),
            'FILECOUNT': CliMatcher.IntegerMatcher( 0, 100,
               helpdesc='number of output files' ),
            'packet-count': CliCommand.singleNode(
               CliMatcher.KeywordMatcher( 'packet-count',
                  helpdesc='Limit number of packets to capture' ),
               sharedMatchObj=packetCountSharedObj ),
            '-c': CliCommand.singleNode(
               matcher=CliMatcher.PatternMatcher( '-c', helpname='-c',
                  helpdesc='number of packets to capture' ),
               hidden=True,
               sharedMatchObj=packetCountSharedObj ),
            'PACKET_COUNT': CliMatcher.IntegerMatcher( 1, 10000,
               helpdesc='number of packets to capture' ),
            'INTF_EXPR': InterfaceExpression
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'options' in args:
         return
      options = {}
      args[ 'options' ] = options

      for k, v in ( ( 'SIZE', 'size' ), ( 'MAX_FILE_SIZE', 'maxFileSize' ),
                    ( 'FILECOUNT', 'fileCount' ), ( 'PACKET_COUNT', 'packetCount' ),
                    ( 'DURATION', 'duration' ), ( 'FILE', 'file' ),
                    ( 'LOCAL_FILE', 'file' ) ):
         if k in args:
            options[ v ] = args[ k ][ 0 ]
            del args[ k ]

      if 'device' in args:
         options[ 'device' ] = args[ 'device' ]
         del args[ 'device' ]

class TcpDumpExecOptions( CliCommand.CliExpression ):
   expression = '{ COMMON_OPTIONS | verbose | lookup-names }'
   data = {
            'COMMON_OPTIONS': CommonTcpOptions,
            'verbose': CliCommand.singleNode( verboseKwMatcher ),
            'lookup-names': CliCommand.singleNode( lookupNamesKwMatcher ),
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'verbose' in args:
         args[ 'options' ][ 'verbose' ] = True
         del args[ 'verbose' ]
      if 'lookup-names' in args:
         args[ 'options' ][ 'lookupNames' ] = True
         del args[ 'lookup-names' ]

class TcpDumpSessionOptions( CliCommand.CliExpression ):
   expression = '{ COMMON_OPTIONS | verbose | ( duration DURATION ) }'
   data = {
            'COMMON_OPTIONS': CommonTcpOptions,
            'verbose': CliCommand.singleNode( verboseKwMatcher, hidden=True ),
            'duration': CliCommand.singleKeyword( 'duration',
               helpdesc='Limit duration' ),
            'DURATION': CliMatcher.IntegerMatcher( 1, 10000, helpdesc='seconds' )
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'verbose' in args:
         args[ 'options' ][ 'verbose' ] = True
         del args[ 'verbose' ]
      if 'DURATION' in args:
         args[ 'options' ][ 'duration' ] = args[ 'DURATION' ]
         del args[ 'DURATION' ]

class TcpdumpExecCmd( CliCommand.CliCommandClass ):
   syntax = 'tcpdump [ OPTIONS ] [ filter FILTER ]'
   data = {
            'tcpdump': tcpdumpKwMatcher,
            'OPTIONS': TcpDumpExecOptions,
            'filter': filterKwMatcher,
            'FILTER': filterMatcher,
          }

   handler = doTcpdumpSession

BasicCli.ExecMode.addCommandClass( TcpdumpExecCmd )

class TcpdumpSessionCmd( CliCommand.CliCommandClass ):
   syntax = 'tcpdump session SESSION_NAME [ OPTIONS ] [ filter FILTER ]'
   noOrDefaultSyntax = 'tcpdump session SESSION_NAME ... '
   data = {
            'tcpdump': tcpdumpKwMatcher,
            'session': sessionKwMatcher,
            'SESSION_NAME': sessionNameMatcher,
            'OPTIONS': TcpDumpSessionOptions,
            'filter': filterKwMatcher,
            'FILTER': filterMatcher,
          }

   handler = doTcpdumpSession

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      # Process is stopped by the SuperServer agent
      del tcpdumpSessionConfigDir.session[ args[ 'SESSION_NAME' ] ]

BasicCli.GlobalConfigMode.addCommandClass( TcpdumpSessionCmd )

class TcpdumpSessionShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'tcpdump session SESSION_NAME shutdown'
   noOrDefaultSyntax = 'tcpdump session SESSION_NAME shutdown ...'
   data = {
            'tcpdump': tcpdumpKwMatcher,
            'session': sessionKwMatcher,
            'SESSION_NAME': sessionNameMatcher,
            'shutdown': 'Shutdown the tcpdump session'
          }

   @staticmethod
   def handler( mode, args ):
      session = tcpdumpSessionConfigDir.session.get( args[ 'SESSION_NAME' ] )
      if session is not None:
         session.enabled = False

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      sessionName = args[ 'SESSION_NAME' ]
      session = tcpdumpSessionConfigDir.session.get( sessionName )
      if session is None:
         # pylint: disable-next=consider-using-f-string
         mode.addWarning( 'Session %s doesn\'t exist' % sessionName )
      else:
         session.enabled = True

BasicCli.GlobalConfigMode.addCommandClass( TcpdumpSessionShutdownCmd )

class SessionModel( CliModel.Model ):
   enabled = CliModel.Bool( help='Enabled' )
   outputFile = CliModel.Str( help='Output file' )
   packetFilter = CliModel.Str( help='Packet filter' )
   packetCount = CliModel.Int( help='Packet count' )
   maxFileSize = CliModel.Int( help='Max file size (millions of bytes)' )
   maxPerPacketCaptureSize = CliModel.Int( help='Per-packet capture size (bytes)' )
   duration = CliModel.Int( help='Capture duration (seconds)' )
   running = CliModel.Bool( help='Indicates if capture is running' )
   errorMessage = CliModel.Str( help='Error message associated with the session' )
   interface = CliModel.Str( help='Capture source' )
   vrfName = CliModel.Str( help='VRF name', sinceRevision=2 )

class TcpdumpModel( CliModel.Model ):
   __revision__ = 2
   sessions = CliModel.Dict( keyType=str, valueType=SessionModel,
                     help='A mapping between session name and session information' )
   
   def render( self ):
      fmt = TableOutput.Format( justify='left' )
      table = TableOutput.createTable( ( 'Session', 'Enabled', 'Target',  'VRF',
                        'File', 'Filter', 'c/C/s/t', 'On' ) )
      table.formatColumns( fmt, fmt, fmt, fmt, fmt, fmt, fmt, fmt )

      errorTable = TableOutput.createTable( ( 'Session', 'Error' ) ) 
      errorTable.formatColumns( fmt, fmt )
   
      errors = False
      for sessionName in sorted( self.sessions ):
         session = self.sessions[ sessionName ]
         # pylint: disable-next=consider-using-f-string
         options = '%d/%d/%d/%d' % ( session.packetCount, 
                                     session.maxFileSize, 
                                     session.maxPerPacketCaptureSize, 
                                     session.duration )
         intf = session.interface
         vrfName = session.vrfName
         value = ( session.enabled, intf, vrfName, session.outputFile,
                session.packetFilter, options, session.running )         
         table.newRow( sessionName, *value )
         
         if session.errorMessage:
            errors = True
            value = ( session.errorMessage, )
            errorTable.newRow( sessionName, *value )

      print( table.output() )
      if errors:
         print( errorTable.output() )

class ShowTcpDumpCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show tcpdump'
   data = {
            'tcpdump': 'Show tcpdump sessions'
          }
   cliModel = TcpdumpModel

   @staticmethod
   def handler( mode, args ):
      sessions = tcpdumpSessionConfigDir
      stati = tcpdumpSessionStatusDir.session 
      model = TcpdumpModel()
      for sessionName in sessions.session:
         session = sessions.session[ sessionName ]
         sessionModel = SessionModel()
         if sessionName in stati:
            status = stati[ sessionName ]
            sessionModel.running = status.running
         else:
            sessionModel.running = False
         # Make sure the file and interface show up in the table
         eosFile = Url.filenameToUrl( TcpdumpLib.dumpFile( session ) )
         sessionModel.outputFile = eosFile
         sessionModel.interface = session.device
         if not sessionModel.interface:
            sessionModel.interface = TcpdumpLib.fabricName
         
         sessionModel.enabled = session.enabled
         sessionModel.packetFilter = session.filter
         sessionModel.packetCount = session.packetCount
         sessionModel.maxFileSize = session.maxFileSize
         sessionModel.maxPerPacketCaptureSize = session.size
         sessionModel.duration = session.duration
         sessionModel.errorMessage = ''
         sessionModel.vrfName = session.vrfName
         model.sessions[ sessionName ] = sessionModel
         
      for s in sessions.session:
         if s in stati and stati[ s ].reason:
            sessionModel = model.sessions[ s ]
            assert sessionModel
            sessionModel.errorMessage = stati[ s ].reason
      return model

BasicCli.addShowCommandClass( ShowTcpDumpCmd )

class ShowTcpDumpSessionCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show tcpdump SESSION_NAME [ MAX_PACKETS ] [ lookup-names ] '
              '[ verbose ] [ filter FILTER ]' )
   data = {
            'tcpdump': 'Show tcpdump sessions',
            'SESSION_NAME': sessionNameMatcher,
            'MAX_PACKETS': CliMatcher.IntegerMatcher( 0, 0xffffffff,
               helpdesc='Maximum number of packets (default=10)' ),
            'lookup-names': lookupNamesKwMatcher,
            'verbose': verboseKwMatcher,
            'filter': filterKwMatcher,
            'FILTER': filterMatcher,
          }

   @staticmethod
   def handler( mode, args ):
      sessionName = args[ 'SESSION_NAME' ]
      maxPackets = args.get( 'MAX_PACKETS', 10 )
      lookupNames = 'lookup-names' in args
      verbose = 'verbose' in args
      dumpFilter = args.get( 'FILTER' )
      
      if maxPackets == 0:
         maxPackets = sys.maxsize
      if sessionName not in tcpdumpSessionConfigDir.session:
         # pylint: disable-next=consider-using-f-string
         mode.addError( 'Session %s does not exist' % sessionName )
         return

      session = tcpdumpSessionConfigDir.session[ sessionName ]
      if not verbose:
         verboseFlag = False
      else:
         verboseFlag = True

      output = TcpdumpLib.getDumpOutput( session, maxPackets, lookupNames,
                                         dumpFilter, verboseFlag )
      if output:
         for line in output:
            print( line )
      else:
         print( 'Session dump is empty' )

      stati = tcpdumpSessionStatusDir.session
      if sessionName in stati and stati[ sessionName ].reason:
         mode.addWarning( stati[ sessionName ].reason )


BasicCli.addShowCommandClass( ShowTcpDumpSessionCmd )

def Plugin( entityManager ):
   global intfStatusAll, tcpdumpSessionConfigDir, tcpdumpSessionStatusDir
   global ipAddressDir, ip6AddressDir
   global mirroringStatus, mirroringHwCapability, intfStatusLocal, allVrfStatusLocal

   intfStatusAll = LazyMount.mount( entityManager, 'interface/status/all',
                                    'Interface::AllIntfStatusDir', 'r' )
   ipAddressDir = LazyMount.mount( entityManager, 'ip/status', 'Ip::Status', 'r' )
   ip6AddressDir = LazyMount.mount( entityManager, 'ip6/status', 'Ip6::Status', 'r' )
   tcpdumpSessionConfigDir = ConfigMount.mount( entityManager, 'sys/tcpdump/config', 
                                                'Tcpdump::Config', 'w' )
   tcpdumpSessionStatusDir = LazyMount.mount( entityManager, 'sys/tcpdump/status', 
                                              'Tcpdump::Status', 'r' )
   mirroringStatus = LazyMount.mount( entityManager, 'mirroring/status',
                                      'Mirroring::Status', 'r' )
   mirroringHwCapability = LazyMount.mount( entityManager, 'mirroring/hwcapability',
                                            'Mirroring::HwCapability', 'r' )
   intfStatusLocal = LazyMount.mount( entityManager,
                                      Cell.path( 'interface/status/local'),
                                      'Interface::AllIntfStatusLocalDir', 'r' )
   allVrfStatusLocal = LazyMount.mount( entityManager, 
                                        Cell.path( 'ip/vrf/status/local' ),
                                       'Ip::AllVrfStatusLocal', 'r' )

