#!/usr/bin/env python3
# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import re
import BasicCli
import CliCommand
import CliMatcher
import CliParser
from CliMode.Te import ( GlobalTeMode,
                         FlexAlgoModeBase,
                         FlexAlgoDefModeBase,
                        )
from CliParserCommon import namePattern
import CliPlugin.IntfCli as IntfCli
import CliPlugin.IpAddrMatcher as IpAddrMatcher
import CliPlugin.Ip6AddrMatcher as Ip6AddrMatcher
from CliSavePlugin.TeCliSave import flexAlgoMetricMap
import CliToken.Router
from MultiRangeRule import (
   MultiRangeMatcher,
   rangeSetFromString,
   multiRangeFromCanonicalString,
   )
import ConfigMount
import LazyMount
import Tac
from TypeFuture import TacLazyType
from Toggles import TeToggleLib

# pkgdeps: rpmwith %{_libdir}/libTe.so*

Color = TacLazyType( 'FlexAlgo::Color' )

# When just traffic-engineering bandwidth is configured,
# it would imply 75% of the link bw is available as maximum
# reservable bandwidth
IMPLIED_MAX_RESERVABLE_BW = 75

# Administrative Group (Tlv) representing a limit of 32 AGs(color) per link.
ADMIN_GROUP_COLORS_NUM = 32

teConfig = None
twampLightConfig = None

def getSrlgIdToNameMap():
   config = teConfiguration()
   srlgIdToNameMap = {}
   for srlgNameKey, srlgMapEntry in config.srlgMap.items():
      srlgIdToNameMap[ srlgMapEntry.srlgId ] = srlgNameKey
   return srlgIdToNameMap

def adminGroupToStr( adminGroup ):
   '''Given adminGroup is like 23 which means adminGroup Id 0, 1, 2 and 4
   and this function returns string like "red(0),1-2,4"
   '''
   adminGroupColl = Tac.newInstance( "TrafficEngineering::AdminGroupColl" )
   if isinstance( adminGroup, dict ):
      adminGroupColl.adminGroupVal.update( adminGroup )
   else:
      adminGroupColl.adminGroupVal[ 0 ] = adminGroup

   return teConfiguration().adminGroupCollToStr( adminGroupColl )

def addAdminGroupBitToDict( adminGroupDict, bit ):
   index = bit // 32
   adminGroup = adminGroupDict.get( index, 0 )
   adminGroup |= 1 << ( bit % 32 )
   adminGroupDict[ index ] = adminGroup

def adminGroupDecimalListToDict( agDecimalList ):
   adminGroupDict = {}
   for bit in agDecimalList:
      addAdminGroupBitToDict( adminGroupDict, bit )
   return adminGroupDict

def adminGroupDictToDecimalList( adminGroupDict ):
   agDecimalList = []
   for index, adminGroup in adminGroupDict.items():
      bit = 0
      while adminGroup:
         if adminGroup % 2:
            # Least significant bit is set, add to decimal list
            agDecimalList.append( bit + ( 32 * index ) )
         bit += 1
         adminGroup >>= 1
   return sorted( agDecimalList )

def adminGroupAndNameListToStr( adminGroupList, adminGroupNames ):
   '''Given adminGroup is like [ 1,2,4,32 ] and adminGroupNames is like
   [ 'Red', 'Blue' ] with Red being the alias for 2 and Blue being unassigned and
   this function returns the admin group string like 1,Red(2),4,Blue(unassigned),32
   '''
   config = teConfiguration()
   adminGroupDict = adminGroupDecimalListToDict( adminGroupList )
   adminGroups = []
   for adminGroupName in sorted( adminGroupNames ):
      value = config.adminGroupMap.get(adminGroupName)
      if value is None:
         adminGroups.append( f'{adminGroupName}(unassigned)' )
      else:
         addAdminGroupBitToDict( adminGroupDict, value )
   groupString = adminGroupToStr( adminGroupDict )
   if groupString:
      adminGroups.insert( 0, groupString )
   if not TeToggleLib.toggleExtendedAdminGroupEnabled():
      hexValue = 0 if not adminGroupDict else adminGroupDict[ 0 ]
      return f'{",".join( adminGroups )} ({hex(hexValue)})'
   return f'{",".join( adminGroups )}'

def strToAdminGroupDict( adminGroupString ):
   ''' This would convert "red(0),2-3,5,32" to { 0: 35, 1: 1 }
   '''
   adminGroupList = adminGroupString.split( ',' )
   adminGroup = {}
   for adminGroupNameOrNumber in adminGroupList:
      st = adminGroupNameOrNumber.find( '(' )
      posOfDash = adminGroupNameOrNumber.find( '-' )
      if st != -1:
         end = adminGroupNameOrNumber.find( ')' )
         val = int( adminGroupNameOrNumber[ st + 1 : end ] )
         addAdminGroupBitToDict( adminGroup, val )
      elif posOfDash != -1:
         lb = int( adminGroupNameOrNumber[ 0 : posOfDash ] )
         ub = int( adminGroupNameOrNumber[ posOfDash + 1 : ] )
         for i in range ( lb, ub + 1 ):
            addAdminGroupBitToDict( adminGroup, i )
      else:
         addAdminGroupBitToDict( adminGroup, int( adminGroupNameOrNumber ) )
   return adminGroup

def strToAdminGroup( adminGroupString ):
   '''This would convert "red(0), 2-3, 5" to 45 i.e (1+4+8+32)
   '''
   # This function will be removed eventually when ExtendedAdminGroup toggle is
   # cleaned up
   bitMaskDict = strToAdminGroupDict( adminGroupString )
   if 0 in bitMaskDict:
      return bitMaskDict[ 0 ]
   return 0

def adminGroupRange():
   if TeToggleLib.toggleExtendedAdminGroupEnabled():
      return ( 0, 127 )
   else:
      return ( 0, 31 )

adminGrpMatcher = MultiRangeMatcher( adminGroupRange, False,
                                     'Administrative Group Ids' )

COMMA_SEP = ','
DASH_SEP = '-'
EMPTY_SEP = ''
# To have a clear and easy differentiation, we restrict name to not start with a
# number or special characters.
# Derived from namePattern in CliParserCommon.py
adminGroupNamePattern = r'^[A-za-z][^\s|>&;`$()\\]*$'
hexRegex = re.compile( r'0x[0-9a-fA-F]{1,8}$' )
intRangeRegex = re.compile( r'(\d+|\d+-\d+)$' )
nameRegex = re.compile( adminGroupNamePattern )
INVALID_FMT = 'Invalid administrative group format'
INVALID_GROUP_NAME = 'Invalid administrative group name'

regexForAdminGroupMatcher = r'(.+)'
adminGroupMixedListMatcher = CliMatcher.PatternMatcher(
      pattern=regexForAdminGroupMatcher,
      helpdesc='Administrative Group value in hexadecimal, range or name format',
      helpname='<0x0-0xFFFFFFFF>,<0-31>,<0-31>-<0-31>,WORD' )

extendedAdminGroupMixedListMatcher = CliMatcher.PatternMatcher(
      pattern=regexForAdminGroupMatcher,
      helpdesc='Administrative Group value in hexadecimal, range or name format',
      helpname='<0x0-0xFFFFFFFF>,<0-127>,<0-127>-<0-127>,WORD' )

def convertAGHexAndRangeStrToAGValue( adGrpHexStrings, adGrpRangeString ):
   adminGroupVal = {}
   hexValue = 0
   # Process hex admin group values
   for hexStr in adGrpHexStrings:
      hexValue |= int( hexStr, 16 )
   # Hex admin group values are only specified for the first 32 bits of AG value
   adminGroupVal[ 0 ] = hexValue

   # Process range admin group values
   for bit in multiRangeFromCanonicalString( adGrpRangeString ):
      addAdminGroupBitToDict( adminGroupVal, bit )
   return adminGroupVal

def parseMixedAGListToNamesAndValues( mode, mixedAGList ):
   mixedAdGrp = parseMixedAdminGroupList( mode, mixedAGList,
                  extended=TeToggleLib.toggleExtendedAdminGroupEnabled() )
   adGrpHexStrings, adGrpRangeString, adGrpNames = mixedAdGrp
   adminGroupVal = convertAGHexAndRangeStrToAGValue( adGrpHexStrings,
                                                     adGrpRangeString )
   return ( adminGroupVal, adGrpNames ) 

# Based on the existing values in the collection and operation
# we will update the configDict key:value pairs
def updateConfigBasedOnOperation( configDict, agColl, operation ):
   if operation == "replace":
      for key in agColl.keys():
         if key in configDict and agColl[ key ] == configDict[ key ]:
            continue
         del agColl[ key ]
      for key in configDict:
         if key in agColl.keys() and agColl[ key ] == configDict[ key ]:
            continue
         agColl[ key ] = configDict[ key ]
   elif operation == "add":
      for key in configDict.keys():
         if key in agColl.keys():
            updatedValue = agColl[ key ]
            for i in range( 0, ADMIN_GROUP_COLORS_NUM ):
               if configDict[ key ] & ( 1 << i ):
                  updatedValue |= 1 << i
            agColl[ key ] = updatedValue
         else:
            agColl[ key ] = configDict[ key ]
   elif operation == "remove":
      for key in configDict.keys():
         if key not in agColl.keys():
            continue
         updatedValue = agColl[ key ]
         for i in range( 0, ADMIN_GROUP_COLORS_NUM ):
            if configDict[ key ] & ( 1 << i ):
               updatedValue = updatedValue & ~( 1 << i )
         if updatedValue == 0:
            del agColl[ key ]
         else:
            agColl[ key ] = updatedValue
def populateAdminGroupValues( adminGroupVal, config, agCollStr,
      operation='replace', indexType=None ):
   agColl = getattr( config, agCollStr )
   configDict = {}
   if indexType is not None:
      IndexType = TacLazyType( indexType, returnValueConst=True )
      configDict = { IndexType( k ): v for k, v in adminGroupVal.items() if v != 0 }
   else:
      configDict = { k: v for k, v in adminGroupVal.items() if v != 0 }

   updateConfigBasedOnOperation( configDict, agColl, operation )

# If operation is remove , we would iterate over collection and remove that  value
# If operation is add, we just need to add the new name to the collection ,
# If operation is replace, then we would first delete the words that we dont need
def populateAdminGroupNames( adGrpNames, config, agNameStr, operation='replace' ):
   agNames = getattr( config, agNameStr )

   for name in agNames:
      if operation == 'add':
         break
      if operation == 'replace' and name in adGrpNames:
         continue
      if operation == 'remove' and name not in adGrpNames:
         continue
      agNames.remove( name )

   for name in adGrpNames:
      if operation == 'remove':
         break
      if name in agNames:
         continue
      agNames.add( name )

def populateAdminGroupValuesAndNames( parsedAgResult, config, agCollStr, agNameStr,
                                      operation='replace', indexType=None ):
   adGrpHexStrings, adGrpRangeString, adGrpNames = parsedAgResult
   adminGroupVal = convertAGHexAndRangeStrToAGValue( adGrpHexStrings,
                                                     adGrpRangeString )
   populateAdminGroupValues( adminGroupVal, config, agCollStr,
                              operation=operation, indexType=indexType )
   populateAdminGroupNames( adGrpNames, config, agNameStr,
                             operation=operation )

def matchAdminGroupValuesAndNames( parsedAgResult, config, agCollStr, agNameStr ):
   adGrpHexStrings, adGrpRangeString, adGrpNames = parsedAgResult
   adminGroupVal = convertAGHexAndRangeStrToAGValue( adGrpHexStrings,
                                                      adGrpRangeString )

   agNames = getattr( config, agNameStr )
   agColl = getattr( config, agCollStr )
   configDict = { k: v for k, v in adminGroupVal.items() if v != 0 }

   if configDict != dict( agColl.items() ):
      return False
   
   if set( agNames.keys() ) != set( adGrpNames ):
      return False

   return True

def parseMixedAdminGroupList( mode, mixedStrLists, extended=False ):
   '''
   Takes a list of administrative groups of the following types in string format:
   1. Hexadecimal value: 0 - 0xFFFFFFFF
   2. Integer value or range of values (0-31 or 0-127 if extended): 1, 3, 12-18
   3. Names
   The input can contain a comabination of the above types separated by ','.
   E.g. blue,1,10-12
   will be accepted as the following list:
   mixedStrLists = [ 'blue,1,2,10-12' ]
   In addition the input pattern may contain spaces between the comma-separated
   types. In this scenario, the PatternMatcher splits the entered pattern
   based on ' ' and the list of tokens is then parsed here.
   E.g. blue, 1,2,0xA, 10-12
   will be accepted as the following list:
   mixedStrLists = [ 'blue', '1,2,0xA', '10-12' ]

   The function separates the above types and on successful parsing returns a
   tuple of the following:
   a) adGrpHexStrings -> [ '0xA' ]
   b) adGrpRangeString -> '1-2,10-12'
   c) adGrpNames -> [ 'blue' ]
   '''

   adGrpHexStrings = []
   adGrpRangeStrings = []
   adGrpNames = []
   start = 0
   end = len( mixedStrLists ) - 1
   for i, mixedList in enumerate( mixedStrLists ):
      # PatternMatcher already splits input based on space. Now process input
      # lists based on comma as separator
      tokens = mixedList.split( COMMA_SEP )
      if i == start and tokens[ start ] == EMPTY_SEP:
         # Input pattern cannot start with ','
         mode.addErrorAndStop(
            f'{INVALID_FMT}: input cannot start with a comma' )
      if i == end and tokens[ len( tokens ) - 1 ] == EMPTY_SEP:
         # Input pattern cannot end with ','
         mode.addErrorAndStop(
            f'{INVALID_FMT}: input cannot end with a comma' )
      if TeToggleLib.toggleIncrementalAdminGroupConfigEnabled():
         invalidKeywords = ""
         lowerCaseTokens = [ token.lower() for token in tokens ]
         if 'add' in lowerCaseTokens:
            invalidKeywords = invalidKeywords + "add"
         if 'remove' in lowerCaseTokens:
            if invalidKeywords != "":
               invalidKeywords = invalidKeywords + ", "
            invalidKeywords = invalidKeywords + "remove"
         if invalidKeywords != "":
            mode.addErrorAndStop(
               f'{INVALID_GROUP_NAME}: {invalidKeywords}' )
      for token in tokens:
         if token == EMPTY_SEP:
            # For tokens in the middle of the pattern that contain ' '
            # on either side or start/end with ',', skip the empty strings
            continue
         if re.match( hexRegex, token ):
            adGrpHexStrings.append( token )
         elif re.match( intRangeRegex, token ):
            # Check whether the input string range is within the upper and lower
            # admin group ID bounds (0-31 or 0-127 if extended)
            maxAdminGrpVal = 127 if extended else 31
            integers = token.split( DASH_SEP )
            for j in integers:
               if int( j ) > maxAdminGrpVal:
                  mode.addErrorAndStop( '{}: range is out of bounds <0-{}>'.format(
                     INVALID_FMT, maxAdminGrpVal ) )
            adGrpRangeStrings.append( token )
         elif re.match( nameRegex, token ):
            adGrpNames.append( token )
         else:
            # Rejected inputs:
            # E.g 1 - 3, 1- 3, 1 -3, -3
            # Any input admin group names not matching adminGroupNamePattern
            mode.addErrorAndStop( INVALID_FMT )
   # Validate range input
   adGrpRangeStr = COMMA_SEP.join( adGrpRangeStrings )
   try:
      rangeSet = rangeSetFromString( adGrpRangeStr )
   except ValueError:
      mode.addErrorAndStop( f'{INVALID_FMT}: incorrect range input' )
   adGrpRangeString = str( rangeSet )

   return ( adGrpHexStrings, adGrpRangeString, adGrpNames )

bandwidth = TacLazyType( 'TrafficEngineering::Bandwidth' )
bwUnit = TacLazyType( 'TrafficEngineering::BwUnitType' )
srlgGid = TacLazyType( 'TrafficEngineering::SrlgId' )
metric = TacLazyType( 'TrafficEngineering::Metric' )
delayUnit = TacLazyType( 'TrafficEngineering::DelayUnitType' )
srlgCost = TacLazyType( 'TrafficEngineering::SrlgCost' )
teBwFloodPercent = TacLazyType( 'TrafficEngineering::TeBwFloodPercent' )

#-------------------------------------------------------------------------------
# Adds TE-specific CLI commands to the 'config-if' mode for routed ports.
# Enable TE commands under config-if only on routed and loopback ports
#-------------------------------------------------------------------------------
def teSupportedOnIntfConfigMode( intfConfigMode ):
   # Don't configure TE on ineligible interfaces
   if intfConfigMode.intf.routingSupported() and \
         not intfConfigMode.intf.name.startswith( 'Management' ):
      return True
   return False

class RoutingProtocolTeIntfConfigModelet( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      return teSupportedOnIntfConfigMode( mode )

#-------------------------------------------------------------------------------
# Associate the RoutingProtocolIntfConfigModelet with the 'config-if' mode.
#-------------------------------------------------------------------------------
IntfCli.IntfConfigMode.addModelet( RoutingProtocolTeIntfConfigModelet )

def teConfiguration():
   return teConfig

class TeIntf( IntfCli.IntfDependentBase ):

   def setDefault( self ):
      config = teConfiguration()
      del config.intfConfig[ self.intf_.name ]

modelet = RoutingProtocolTeIntfConfigModelet

#-------------------------------------------------------------------------------
# IntfConfig for Traffic-engineering configuration, is created when one of its
# attributes is configured. It is deleted when all the attributes are at their
# defaults What this means is that after the last 'no traffic engineering...'
# command is run on the interface, we delete the intfConfig object
#------------------------------------------------------------------------------
def _getIntfConfig( intfName ):
   return teConfiguration().intfConfig.get( intfName, None )

def _getOrCreateIntfConfig( config, intfName ):
   intfConfig = config.intfConfig.get( intfName, None )
   if intfConfig is None:
      intfConfig = config.intfConfig.newMember( intfName )
   return intfConfig

def _deleteIntfConfigIfAllAttributeHaveDefaults( config, intfName ):
   '''Delete the intfConfig collection element if all the attributes of intfConfig
   have default values. This needs to be called by command handlers for 'no' form
   of the Traffic-engineering interface config commands.
   '''
   intfConfig = config.intfConfig.get( intfName, None )
   if intfConfig is None:
      return
   if ( ( intfConfig.enabled == intfConfig.enabledDefault ) and
        ( intfConfig.maxReservableBwUnit == \
              intfConfig.maxReservableBwUnitDefault ) and
        ( intfConfig.maxReservableBw == intfConfig.maxReservableBwDefault ) and
        ( not intfConfig.adminGroup ) and
        ( not intfConfig.adminGroupName ) and
        ( intfConfig.metric == intfConfig.metricDefault ) and
        ( not intfConfig.srlgIdList ) and
        ( not intfConfig.srlgNameList ) and
        ( intfConfig.minDelay == intfConfig.minDelayDefault ) and
        ( intfConfig.dynMinDelayFallback ==
          intfConfig.dynMinDelayFallbackDefault ) and
        ( intfConfig.senderProfile == intfConfig.senderProfileDefault ) ):

      del config.intfConfig[ intfName ]

#-------------------------------------------------------------------------
# Helper Class. This is used as a base class for all dependent classes of
# 'config-te' mode. When the traffic engineering is unconfigured the
# dependents can cleanup their configuration.
#-------------------------------------------------------------------------
class TeDependentBase:
   def __init__( self, mode ):
      self.mode = mode

   def setDefault( self ):
      pass

#----------------------------------------
# Routine to register dependent class
#----------------------------------------
TeModeDependents = IntfCli.DependentClassRegistry()

#------------------------------------------------------------------------------
# router traffic engineering config mode. This mode is created when the user
# enters 'router traffic engineering'
#------------------------------------------------------------------------------
class RouterGlobalTeMode( GlobalTeMode, BasicCli.ConfigModeBase ):
   name = 'Traffic-engineering global configuration'

   def __init__( self, parent, session ):
      GlobalTeMode.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouterGlobalTeModelet( CliParser.Modelet ):
   pass

RouterGlobalTeMode.addModelet( RouterGlobalTeModelet )

#--------------------------------------------------------------------------------
# [ no | default ] router traffic-engineering
#--------------------------------------------------------------------------------
class RouterTrafficEngineeringCmd( CliCommand.CliCommandClass ):
   syntax = 'router traffic-engineering'
   noOrDefaultSyntax = syntax
   data = {
      'router' : CliToken.Router.routerMatcherForConfig,
      'traffic-engineering' : 'Traffic-engineering global config',
   }

   @staticmethod
   def handler( mode, args ):
      teConfig.enabled = True
      childMode = mode.childMode( RouterGlobalTeMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config = teConfiguration()
      config.routerId = config.routerIdDefault
      config.routerIdV6 = config.routerIdV6Default
      config.srlgMap.clear()
      config.adminGroupMap.clear()
      teConfig.enabled = False
      config.teBwFloodThreshold = config.teBwFloodThresholdDefault
      config.reflectorProfile = config.reflectorProfileDefault
      config.senderProfile = config.senderProfileDefault

      for cls in TeModeDependents:
         cls( mode ).setDefault()

BasicCli.GlobalConfigMode.addCommandClass( RouterTrafficEngineeringCmd )

#--------------------------------------------------------------------------------
# [ no | default ] srlg NAME GROUP_ID
#--------------------------------------------------------------------------------
# SRLG name must contain atleast one letter
matcherSrlgName = CliMatcher.PatternMatcher( '.{0,100}[A-Za-z]+.{0,100}',
                                             helpname='WORD', helpdesc='SRLG Name' )
matcherSrlgGroupId = CliMatcher.IntegerMatcher( srlgGid.srlgIdMin, srlgGid.srlgIdMax,
                                                helpdesc='SRLG Group ID' )
matcherCost = CliMatcher.KeywordMatcher( 'cost', helpdesc='Configure SRLG cost' )
matcherSrlgCostValue = CliMatcher.IntegerMatcher( srlgCost.min, srlgCost.max,
   helpdesc='Cost value (default %d)' % srlgCost.defaultValue )

class SrlgSrlgnameGidCmd( CliCommand.CliCommandClass ):
   syntax = 'srlg NAME GROUP_ID [ cost SRLG_COST ]'
   noOrDefaultSyntax = syntax
   data = {
      'srlg' : 'Shared Risk Link Group mapping for traffic engineering',
      'NAME' : matcherSrlgName,
      'GROUP_ID' : matcherSrlgGroupId,
      'cost' : matcherCost,
      'SRLG_COST': matcherSrlgCostValue,
   }

   @staticmethod
   def handler( mode, args ):
      gid = args[ 'GROUP_ID' ]
      config = teConfiguration()
      # If the srlg name already exists, remove the associated cost before
      # any changes is made to his group id
      if args[ 'NAME' ] in config.srlgMap:
         del config.srlgCost[ config.srlgMap[ args[ 'NAME' ] ].srlgId ]
      # If a previous mapping with the same srlgId exist, delete that. Multiple
      # srlgNames mapping to the same srlgId is not allowed.
      for srlgNameKey, srlgMapEntry in config.srlgMap.items():
         if srlgMapEntry.srlgId == gid:
            del config.srlgMap[ srlgNameKey ]
            break
      srlgMapEntry = config.srlgMap.newMember( args[ 'NAME' ], srlgGid( gid ) )
      cost = args.get( 'SRLG_COST', srlgCost.defaultValue )
      config.srlgCost[ srlgGid( gid ) ] = srlgCost( cost )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if args[ 'NAME' ] in teConfiguration().srlgMap:
         # use the name to retrieve the group id of the srlg that will be delete
         del teConfiguration().srlgCost[
               teConfiguration().srlgMap[ args [ 'NAME' ] ].srlgId ]
         del teConfiguration().srlgMap[ args[ 'NAME' ] ]

RouterGlobalTeModelet.addCommandClass( SrlgSrlgnameGidCmd )

#--------------------------------------------------------------------------------
# [ no | default ] administrative-group alias NAME group GROUP_ID
#--------------------------------------------------------------------------------
matcherAdminGroupName = CliMatcher.PatternMatcher( adminGroupNamePattern,
   helpname='WORD',
   helpdesc='Administrative group name' )

class AdminGroupNameCmd( CliCommand.CliCommandClass ):
   syntax = 'administrative-group alias NAME group GROUP_ID'
   noOrDefaultSyntax = 'administrative-group alias NAME ...'
   data = {
      'administrative-group' : 'Administrative group for traffic-engineering',
      'alias' : 'Alias for administrative group',
      'NAME' : matcherAdminGroupName,
      'group' : 'Integer administrative group ID',
      'GROUP_ID' : CliMatcher.IntegerMatcher( 0, 31,
         helpdesc='Administrative group ID' ),
   }
   if TeToggleLib.toggleExtendedAdminGroupEnabled():
      data[ 'GROUP_ID' ] = CliMatcher.IntegerMatcher( 0, 127,
         helpdesc='Administrative group ID' )

   @staticmethod
   def handler( mode, args ):
      adminGroupValue = args[ 'GROUP_ID' ]

      # If a previous mapping with the same admin group ID exists, delete that.
      # Multiple admin group names mapping to the same ID is not allowed.
      for name, value in teConfig.adminGroupMap.items():
         if value == adminGroupValue:
            del teConfig.adminGroupMap[ name ]
            break

      teConfig.adminGroupMap[ args[ 'NAME' ] ] = adminGroupValue

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      adminGroupName = args[ 'NAME' ]
      del teConfig.adminGroupMap[ adminGroupName ]

RouterGlobalTeModelet.addCommandClass( AdminGroupNameCmd )

#--------------------------------------------------------------------------------
# [ no | default ] router-id ipv4 ROUTER_ID
#--------------------------------------------------------------------------------
matcherRouterId = CliMatcher.KeywordMatcher( 'router-id',
      helpdesc='Configure the router ID for traffic engineering' )

class RouterIdIpv4RouteridCmd( CliCommand.CliCommandClass ):
   syntax = 'router-id ipv4 ROUTER_ID'
   noOrDefaultSyntax = 'router-id ipv4 ...'
   data = {
      'router-id' : matcherRouterId,
      'ipv4' : 'TE router ID in IPv4 address format',
      'ROUTER_ID' : IpAddrMatcher.IpAddrMatcher(
         helpdesc='Traffic engineering router ID in IP address format' ),
   }

   @staticmethod
   def handler( mode, args ):
      teConfiguration().routerId = args[ 'ROUTER_ID' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      teConfiguration().routerId = teConfiguration().routerIdDefault

RouterGlobalTeModelet.addCommandClass( RouterIdIpv4RouteridCmd )

#--------------------------------------------------------------------------------
# router-id ipv6 ROUTER_ID
#--------------------------------------------------------------------------------
class RouterIdIpv6Routeridv6Cmd( CliCommand.CliCommandClass ):
   syntax = 'router-id ipv6 ROUTER_ID'
   noOrDefaultSyntax = 'router-id ipv6 ...'
   data = {
      'router-id' : matcherRouterId,
      'ipv6' : 'TE router ID in IPv6 address format',
      'ROUTER_ID' : Ip6AddrMatcher.Ip6AddrMatcher(
         helpdesc='Traffic engineering router ID in IPv6 address format' ),
   }

   @staticmethod
   def handler( mode, args ):
      teConfiguration().routerIdV6 = args[ 'ROUTER_ID' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      teConfiguration().routerIdV6 = teConfiguration().routerIdV6Default

RouterGlobalTeModelet.addCommandClass( RouterIdIpv6Routeridv6Cmd )

#--------------------------------------------------------------------------------
# [ no | default ] traffic-engineering
#--------------------------------------------------------------------------------
matcherTrafficEngineering = CliMatcher.KeywordMatcher( 'traffic-engineering',
      helpdesc='Configure traffic-engineering' )

class TrafficEngineeringCmd( CliCommand.CliCommandClass ):
   syntax = 'traffic-engineering'
   noOrDefaultSyntax = syntax
   data = {
      'traffic-engineering' : matcherTrafficEngineering,
   }

   @staticmethod
   def handler( mode, args ):
      intfConfig = _getOrCreateIntfConfig( teConfiguration(), mode.intf.name )
      intfConfig.enabled = True
      _deleteIntfConfigIfAllAttributeHaveDefaults( teConfiguration(),
                                                   mode.intf.name )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intfConfig = _getIntfConfig( mode.intf.name )
      if intfConfig is None:
         return
      intfConfig.enabled = intfConfig.enabledDefault
      _deleteIntfConfigIfAllAttributeHaveDefaults( teConfiguration(),
                                                   mode.intf.name )

modelet.addCommandClass( TrafficEngineeringCmd )

#--------------------------------------------------------------------------------
# [ no | default ] traffic-engineering bandwidth [ BW UNIT ]
#--------------------------------------------------------------------------------

class TrafficEngineeringBandwidthCmd( CliCommand.CliCommandClass ):
   syntax = 'traffic-engineering bandwidth [ BW UNIT ]'
   noOrDefaultSyntax = syntax
   data = {
      'traffic-engineering' : matcherTrafficEngineering,
      'bandwidth' : 'Maximum reservable bandwidth for traffic-engineering',
      'BW' : CliMatcher.IntegerMatcher( bandwidth.bandwidthMin,
         bandwidth.bandwidthMax,
         helpdesc='Maximum reservable bandwidth' ),
      'UNIT' : CliMatcher.EnumMatcher( {
         'gbps' : 'Maximum reservable bandwidth in Gbps',
         'mbps' : 'Maximum reservable bandwidth in Mbps',
         'percent' : 'Maximum reservable bandwidth as percentage of link bandwidth',
      } ),
   }

   @staticmethod
   def handler( mode, args ):
      intfConfig = _getOrCreateIntfConfig( teConfiguration(), mode.intf.name )
      if 'BW' in args:
         intfConfig.maxReservableBw = args[ 'BW' ]
         if args[ 'UNIT' ] == 'percent':
            intfConfig.maxReservableBwUnit = bwUnit.percent
         elif args[ 'UNIT' ] == 'mbps':
            intfConfig.maxReservableBwUnit = bwUnit.mbps
         elif args[ 'UNIT' ] == 'gbps':
            intfConfig.maxReservableBwUnit = bwUnit.gbps
         else:
            assert False, 'Unknown unit'
      else:
         # Just configuring 'traffic-engineering bandwidth' means bw = 75% of link bw
         intfConfig.maxReservableBw = IMPLIED_MAX_RESERVABLE_BW
         intfConfig.maxReservableBwUnit = bwUnit.percent
      _deleteIntfConfigIfAllAttributeHaveDefaults( teConfiguration(),
                                                   mode.intf.name )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intfConfig = _getIntfConfig( mode.intf.name )
      if intfConfig is None:
         return

      intfConfig.maxReservableBwUnit = intfConfig.maxReservableBwUnitDefault
      intfConfig.maxReservableBw = intfConfig.maxReservableBwDefault

      _deleteIntfConfigIfAllAttributeHaveDefaults( teConfiguration(),
                                                   mode.intf.name )

modelet.addCommandClass( TrafficEngineeringBandwidthCmd )

#------------------------------------------------------------------------------------
# [ no | default ] traffic-engineering administrative-group { GROUP_MIXED_LISTS }
#------------------------------------------------------------------------------------
class IntfTrafficEngineeringAdminGroupCmd( CliCommand.CliCommandClass ):
   syntax = 'traffic-engineering administrative-group { GROUP_MIXED_LISTS }'
   noOrDefaultSyntax = 'traffic-engineering administrative-group ...'
   data = {
      'traffic-engineering': matcherTrafficEngineering,
      'administrative-group': 'Administrative group for traffic-engineering',
      'GROUP_MIXED_LISTS': adminGroupMixedListMatcher,
   }
   if TeToggleLib.toggleIncrementalAdminGroupConfigEnabled():
      syntax = 'traffic-engineering administrative-group '\
                '[ (add|remove) ] { GROUP_MIXED_LISTS }'
      noOrDefaultSyntax = 'traffic-engineering administrative-group '\
                           '[{GROUP_MIXED_LISTS}]'
      data = {
         'traffic-engineering': matcherTrafficEngineering,
         'administrative-group': 'Administrative group for traffic-engineering',
         'add': "Add to existing administrative groups",
         'remove': "Remove from existing administrative groups",
         'GROUP_MIXED_LISTS': adminGroupMixedListMatcher,
         }
   if TeToggleLib.toggleExtendedAdminGroupEnabled():
      data[ 'GROUP_MIXED_LISTS' ] = extendedAdminGroupMixedListMatcher

   @staticmethod
   def handler( mode, args ):
      intfConfig = _getOrCreateIntfConfig( teConfiguration(), mode.intf.name )
      adminGroupMixedLists = args.get( 'GROUP_MIXED_LISTS' )
      operation = 'replace'
      if args.get( 'add' ) is not None:
         operation = 'add'
      if args.get( 'remove' ) is not None:
         operation = 'remove'

      parsedAgResult = parseMixedAdminGroupList(
         mode, adminGroupMixedLists,
         extended=TeToggleLib.toggleExtendedAdminGroupEnabled() )
      populateAdminGroupValuesAndNames( parsedAgResult, intfConfig,
                                       'adminGroup', 'adminGroupName',
                                       operation )

      _deleteIntfConfigIfAllAttributeHaveDefaults( teConfiguration(),
                                                   mode.intf.name )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intfConfig = _getIntfConfig( mode.intf.name )
      if intfConfig is None:
         return

      adminGroupMixedLists = args.get( 'GROUP_MIXED_LISTS' )
      invalidGroupError = "Invalid administrative group specification"
      
      toggleIncrementalAdminGroup = \
            TeToggleLib.toggleIncrementalAdminGroupConfigEnabled()
      toggleExtendedAdminGroup = \
            TeToggleLib.toggleExtendedAdminGroupEnabled()
     
      if toggleIncrementalAdminGroup and adminGroupMixedLists:
         parsedAgResult = parseMixedAdminGroupList(
                                 mode, adminGroupMixedLists,
                                 extended=toggleExtendedAdminGroup )
         doesConfigMatch = matchAdminGroupValuesAndNames( parsedAgResult,
                                 intfConfig, 'adminGroup',
                                 'adminGroupName' )
         if not doesConfigMatch:
            mode.addErrorAndStop( invalidGroupError )

      intfConfig.adminGroup.clear()
      intfConfig.adminGroupName.clear()
      _deleteIntfConfigIfAllAttributeHaveDefaults( teConfiguration(),
                                                   mode.intf.name )

modelet.addCommandClass( IntfTrafficEngineeringAdminGroupCmd )

#--------------------------------------------------------------------------------
# [ no | default ] traffic-engineering srlg ( NAME | GROUP_ID )
#--------------------------------------------------------------------------------
class TrafficEngineeringSrlgSrlgnameCmd( CliCommand.CliCommandClass ):
   syntax = 'traffic-engineering srlg ( NAME | GROUP_ID )'
   noOrDefaultSyntax = syntax
   data = {
      'traffic-engineering' : matcherTrafficEngineering,
      'srlg' : 'SRLG for traffic-engineering',
      'NAME' : matcherSrlgName,
      'GROUP_ID' : matcherSrlgGroupId,
   }

   @staticmethod
   def handler( mode, args ):
      intfConfig = _getOrCreateIntfConfig( teConfiguration(), mode.intf.name )
      if 'NAME' in args: # pylint: disable-msg=simplifiable-if-statement
         intfConfig.srlgNameList[ args[ 'NAME' ] ] = True
      else:
         intfConfig.srlgIdList[ srlgGid( args[ 'GROUP_ID' ] ) ] = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intfConfig = _getIntfConfig( mode.intf.name )
      if intfConfig is None:
         return

      if 'NAME' in args:
         del intfConfig.srlgNameList[ args[ 'NAME' ] ]
      else:
         del intfConfig.srlgIdList[ args[ 'GROUP_ID' ] ]
      _deleteIntfConfigIfAllAttributeHaveDefaults( teConfiguration(),
                                                   mode.intf.name )

modelet.addCommandClass( TrafficEngineeringSrlgSrlgnameCmd )

#--------------------------------------------------------------------------------
# [ no | default ] traffic-engineering metric METRIC
#--------------------------------------------------------------------------------
class IntfTrafficEngineeringMetricCmd( CliCommand.CliCommandClass ):
   syntax = 'traffic-engineering metric METRIC'
   noOrDefaultSyntax = 'traffic-engineering metric ...'
   data = {
      'traffic-engineering' : matcherTrafficEngineering,
      'metric' : 'Metric for traffic-engineering',
      'METRIC' : CliMatcher.IntegerMatcher( metric.metricMin + 1, metric.metricMax,
         helpdesc='Value of the route metric' ),
   }

   @staticmethod
   def handler( mode, args ):
      intfConfig = _getOrCreateIntfConfig( teConfiguration(), mode.intf.name )
      intfConfig.metric = args[ 'METRIC' ]
      _deleteIntfConfigIfAllAttributeHaveDefaults( teConfiguration(),
                                                   mode.intf.name )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intfConfig = _getIntfConfig( mode.intf.name )
      if intfConfig is None:
         return

      intfConfig.metric = intfConfig.metricDefault
      _deleteIntfConfigIfAllAttributeHaveDefaults( teConfiguration(),
                                                   mode.intf.name )

modelet.addCommandClass( IntfTrafficEngineeringMetricCmd )

#--------------------------------------------------------------------------------
# [ no | default ] traffic-engineering min-delay ( static <DELAY> <UNIT> )
#                    | ( dynamic twamp-light fallback DELAY UNIT )
#--------------------------------------------------------------------------------
class IntfTrafficEngineeringMinDelayCmd( CliCommand.CliCommandClass ):
   syntax = '''traffic-engineering min-delay ( static DELAY UNIT ) | 
         ( dynamic twamp-light fallback DELAY UNIT )'''

   noOrDefaultSyntax = 'traffic-engineering min-delay...'
   data = {
      'traffic-engineering' : matcherTrafficEngineering,
      'min-delay' : 'Minimum delay for traffic-engineering',
      'static' : 'Static value for traffic-engineering',
      'dynamic' : 'Link min-delay is measured dynamically using external' \
                  ' mechanism, e.g. TWAMP-Light',
      'twamp-light' : ' Link min-delay is measured by TWAMP-Light',
      'fallback' : 'Fallback min-delay value used in case when no value is' \
                   ' advertised by TWAMP-Light',
      'DELAY' : CliMatcher.IntegerMatcher( metric.metricMin + 1, metric.metricMax,
         helpdesc='Value of the minimum delay' ),
      'UNIT' : CliMatcher.EnumMatcher( {
         'milliseconds' : 'Minimum delay in milliseconds',
         'microseconds' : 'Minimum delay in microseconds',
      } ),
   }

   @staticmethod
   def handler( mode, args ):
      intfConfig = _getOrCreateIntfConfig( teConfiguration(), mode.intf.name )
      if ( args[ 'UNIT' ] == 'milliseconds' and
         args[ 'DELAY' ] * 1000 > metric.metricMax ):
         mode.addError( 'Minimum delay value exceeds %d us'
                       % metric.metricMax )
         return
      if 'static' in args:
         intfConfig.minDelayUnit = args[ 'UNIT' ]
         intfConfig.minDelay = args[ 'DELAY' ]
         intfConfig.dynMinDelayFallback = \
                  intfConfig.dynMinDelayFallbackDefault
         intfConfig.dynMinDelayFallbackUnit = \
            intfConfig.dynMinDelayFallbackUnit
      elif 'dynamic' in args:
         intfConfig.dynMinDelayFallbackUnit = args[ 'UNIT' ]
         multiplier = 1
         if args[ 'UNIT' ] == 'milliseconds':
            multiplier = 1000

         intfConfig.dynMinDelayFallback = args[ 'DELAY' ] * multiplier

         intfConfig.minDelay = intfConfig.minDelayDefault
         intfConfig.minDelayUnit = intfConfig.minDelayUnitDefault
      _deleteIntfConfigIfAllAttributeHaveDefaults( teConfiguration(),
                                                   mode.intf.name )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intfConfig = _getIntfConfig( mode.intf.name )
      if intfConfig is None:
         return

      intfConfig.minDelay = intfConfig.minDelayDefault
      intfConfig.dynMinDelayFallback = \
                  intfConfig.dynMinDelayFallbackDefault
      _deleteIntfConfigIfAllAttributeHaveDefaults( teConfiguration(),
                                                   mode.intf.name )

modelet.addCommandClass( IntfTrafficEngineeringMinDelayCmd )

#--------------------------------------------------------------------------------
# [ no | default ] traffic-engineering twamp-light sender profile <PROFILE-NAME>
#--------------------------------------------------------------------------------
namePattern = '.{1,100}'
senderNameMatcher = CliMatcher.DynamicNameMatcher(
   lambda mode: twampLightConfig.senderProfile,
   'Profile name', pattern=namePattern )

class IntfTrafficEngineeringTwampLightSenderProfCmd( CliCommand.CliCommandClass ):
   syntax = '''traffic-engineering twamp-light sender profile PROFILE_NAME'''
   noOrDefaultSyntax = 'traffic-engineering twamp-light sender profile...'

   data = {
      'traffic-engineering' : matcherTrafficEngineering,
      'twamp-light' : 'Configure TWAMP light',
      'sender' : 'Configure TWAMP light sender',
      'profile' : 'Configure TWAMP light sender profile',
      'PROFILE_NAME' : senderNameMatcher
   }

   @staticmethod
   def handler( mode, args ):
      intfConfig = _getOrCreateIntfConfig( teConfiguration(), mode.intf.name )
      intfConfig.senderProfile = args[ 'PROFILE_NAME' ]
      _deleteIntfConfigIfAllAttributeHaveDefaults( teConfiguration(),
                                                   mode.intf.name )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intfConfig = _getIntfConfig( mode.intf.name )
      if intfConfig is None:
         return

      intfConfig.senderProfile = intfConfig.senderProfileDefault
      _deleteIntfConfigIfAllAttributeHaveDefaults( teConfiguration(),
                                                   mode.intf.name )

if TeToggleLib.toggleIsisIntfTeSenderProfileEnabled():
   modelet.addCommandClass( IntfTrafficEngineeringTwampLightSenderProfCmd )

#---------------------------------
# [ no | default ] flex-algo
#---------------------------------
flexAlgoConfig = None

def delFlexAlgo():
   """Remove all configuration on 'no flex-algo'
      or 'no traffic-engineering'
   """
   flexAlgoConfig.enabled = False
   teConfig.aslaFallback = teConfig.aslaFallbackDefault
   flexAlgoConfig.nameToIdMap.clear()
   flexAlgoConfig.definition.clear()

class TeFlexAlgoSubMode( TeDependentBase ):
   def setDefault( self ):
      delFlexAlgo()

class CfgFlexAlgoCmd( CliCommand.CliCommandClass ):
   syntax = 'flex-algo'
   noOrDefaultSyntax = syntax
   data = {
      'flex-algo': 'Flexible Algorithm Configuration',
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( FlexAlgoMode )
      mode.session_.gotoChildMode( childMode )
      flexAlgoConfig.enabled = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      delFlexAlgo()

RouterGlobalTeMode.addCommandClass( CfgFlexAlgoCmd )

class FlexAlgoMode( FlexAlgoModeBase, BasicCli.ConfigModeBase ):
   name = "Flexible Algorithm Configuration"

   def __init__( self, parent, session ):
      FlexAlgoModeBase.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

#--------------------------------------------------------------------------------
# [ no | default ] link-attributes asla fallback
#--------------------------------------------------------------------------------
class AslaFallbackCmd( CliCommand.CliCommandClass ):
   syntax = 'link-attributes asla fallback'
   noOrDefaultSyntax = 'link-attributes asla fallback'
   data = {
      'link-attributes' : 'Traffic engineering attributes',
      'asla' : 'Application specific link attributes',
      'fallback' : 'Fallback to legacy traffic-engineering'
   }

   @staticmethod
   def handler( mode, args ):
      teConfig.aslaFallback = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      teConfig.aslaFallback = False

FlexAlgoMode.addCommandClass( AslaFallbackCmd )

#--------------------------------------------
# [ no | default ] flex-algo ( ( ID [ NAME ] ) | NAME )
# This command allows:
# (1) flex-algo ID NAME
# (2) flex-algo ID
# (3) flex-algo NAME
# Format (1) has to be used when defining a new algorithm. It maps a name
# to a specific algorithm ID. Formats (2) and (3) are allowed for algorithms
# that have already defined.
# This command invokes a sub-mode for the given algorithm.
#--------------------------------------------

algoIdType = TacLazyType( 'FlexAlgo::AlgoId' )
algoDef = TacLazyType( 'FlexAlgo::Definition' )
# We allow "flex-algo <algo-id>" and "flex-algo <name>". To have a clear
# and easy differentiation, we restrict flex-algo name to not start with a number.
namePattern = r'^[^\d\s|>&;`$()\\][^\s|>&;`$()\\]+'
matcherAlgoName = CliMatcher.PatternMatcher(
   namePattern, helpname='WORD',
   helpdesc='Algorithm name not starting with a number' )
matcherAlgoId = CliMatcher.IntegerMatcher( algoIdType.min,
                                           algoIdType.max,
                                           helpdesc='Algorithm identifier' )

def delFlexAlgoDef( algoId=None, name=None ):
   if algoId:
      if algoId in flexAlgoConfig.definition:
         name = flexAlgoConfig.definition[ algoId ].name
         del flexAlgoConfig.nameToIdMap[ name ]
         del flexAlgoConfig.definition[ algoId ]
         return
   if name:
      if name in flexAlgoConfig.nameToIdMap:
         defId = flexAlgoConfig.nameToIdMap[ name ]
         del flexAlgoConfig.nameToIdMap[ name ]
         del flexAlgoConfig.definition[ defId ]

def createFlexAlgoDef( mode, algoId, name ):
   """ Create a definition of an algoId and name if one doesn't exist.
   If conficting entries exist, delete them first."""
   if algoId is None and name is None:
      return None

   # We allow user to input either an algoId or a name. If only the name is provided,
   # lookup the definition collection to find the corresponding algorithm ID.
   if algoId is None:
      for defId, defEnt in flexAlgoConfig.definition.items():
         if defEnt.name == name:
            algoId = defId
            break
      else:
         mode.addError( 'Flexible algorithm %s is not defined.' % name )
         return None

   defin = flexAlgoConfig.definition.get( algoId )
   # The algorithm name has to be provided when the algorithm is first defined.
   if name is None:
      if defin:
         name = defin.name
      else:
         mode.addError( 'A name must be provided for flexible algorithm %d.' %
                        algoId )
         return None

   if defin:
      if defin.name == name:
         return defin
      # User has provided a different name for the algorithm. We do not allow this in
      # the interactive mode. User has to remove the old definition before defining
      # the new one.
      if mode.session_.interactive_:
         mode.addError( 'Flexible algorithm %d is already defined. ' % algoId +
                        'Remove the old definition before re-naming the algorithm.' )
         return None
      del flexAlgoConfig.nameToIdMap[ name ]
      del flexAlgoConfig.definition[ algoId ]


   # Before creating the new definition, check the input name to ensure that it is
   # not used by any other algorithm.
   for defId, defEnt in flexAlgoConfig.definition.items() :
      if defEnt.name == name:
         if mode.session_.interactive_:
            mode.addError( 'Flexible algorithm name %s is already used. ' % name )
            return None
         del flexAlgoConfig.nameToIdMap[ name ]
         del flexAlgoConfig.definition[ defId ]

   flexAlgoConfig.nameToIdMap[ name ] = algoId
   flexAlgoConfig.newDefinition( name, algoId )
   return flexAlgoConfig.definition[ algoId ]

class CfgFlexAlgoDefCmd( CliCommand.CliCommandClass ):
   syntax = 'flex-algo ( ( ALGO_ID [ NAME ] ) | NAME )'
   noOrDefaultSyntax = 'flex-algo ( ALGO_ID | NAME ) ...'
   data = {
      'flex-algo' : 'Definition of an algorithm',
      'ALGO_ID' : matcherAlgoId,
      'NAME' : matcherAlgoName,
   }

   @staticmethod
   def handler( mode, args ):
      algoId = args.get( 'ALGO_ID' )
      algoName = args.get( 'NAME' )
      newDef = createFlexAlgoDef( mode, algoId, algoName )
      if newDef is not None:
         childMode = mode.childMode( FlexAlgoDefMode,
                                     param=( newDef.algoId, newDef.name ) )
         childMode.definition = newDef
      else:
         childMode = mode.childMode( FlexAlgoMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      delFlexAlgoDef( args.get( 'ALGO_ID' ), args.get( 'NAME' ) )

FlexAlgoMode.addCommandClass( CfgFlexAlgoDefCmd )

class FlexAlgoDefMode( FlexAlgoDefModeBase, BasicCli.ConfigModeBase ):
   name = "Flexible Algorithm Definition"

   def __init__( self, parent, session, param ):
      FlexAlgoDefModeBase.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.definition = None

#---------------------------------
# Flex Algo:
# [ no | default ] priority <prio>
#---------------------------------

priorityType = Tac.Type( 'FlexAlgo::Priority' )

class CfgFlexAlgoPriorityCmd( CliCommand.CliCommandClass ):
   syntax = 'priority PRIORITY'
   noOrDefaultSyntax = 'priority ...'
   data = {
      'priority' : 'Priority of this definition',
      'PRIORITY' : CliMatcher.IntegerMatcher( priorityType.min, priorityType.max,
                                              helpdesc="Definition priiority" ),
      }

   @staticmethod
   def handler( mode, args ):
      mode.definition.prio = args[ 'PRIORITY' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.definition.prio = priorityType.defValue

FlexAlgoDefMode.addCommandClass( CfgFlexAlgoPriorityCmd )

#------------------------------------------------------------------------------
# Flex Algo:
# [ no | default ] administrative-group [include all RANGE] [include any RANGE]
#                                       [exclude RANGE]
#------------------------------------------------------------------------------
def agRangeListToValue( rangeList ):
   adminGroupVal = {}
   for r in rangeList:
      for bit in r.values():
         addAdminGroupBitToDict( adminGroupVal, bit )
   return adminGroupVal

class CfgFlexAlgoAdminGrpCmd( CliCommand.CliCommandClass ):
   syntax = ( 'administrative-group { [ include all INC_ALL_GROUP ] '
              '[ include any INC_ANY_GROUP ] [ exclude EXCLUDE_GROUP ] }' )
   noOrDefaultSyntax = ( 'administrative-group [ { [ include all [INC_ALL_GROUP] ] '
                         '[ include any [INC_ANY_GROUP] ] '
                         '[ exclude [EXCLUDE_GROUP] ] } ]' )
   data = {
      'administrative-group' : 'Administrative group for link coloring',
      'include' : 'Include any or all designated groups',
      'all' : 'Include all designated groups',
      'any' : 'Include any designated groups',
      'exclude' : 'Exclude designated groups',
      'INC_ALL_GROUP' : adminGrpMatcher,
      'INC_ANY_GROUP' : adminGrpMatcher,
      'EXCLUDE_GROUP' : adminGrpMatcher,
      }

   if TeToggleLib.toggleFlexAlgoAdminGroupNameEnabled():
      if TeToggleLib.toggleExtendedAdminGroupEnabled():
         data[ 'INC_ALL_GROUP' ] = extendedAdminGroupMixedListMatcher
         data[ 'INC_ANY_GROUP' ] = extendedAdminGroupMixedListMatcher
         data[ 'EXCLUDE_GROUP' ] = extendedAdminGroupMixedListMatcher
      else:
         data[ 'INC_ALL_GROUP' ] = adminGroupMixedListMatcher
         data[ 'INC_ANY_GROUP' ] = adminGroupMixedListMatcher
         data[ 'EXCLUDE_GROUP' ] = adminGroupMixedListMatcher

   @staticmethod
   def handler( mode, args ):
      if 'INC_ALL_GROUP' in args:
         if TeToggleLib.toggleFlexAlgoAdminGroupNameEnabled():
            inclAllMixedAg = args.get( 'INC_ALL_GROUP' )
            adminGroupAllVal, adGrpAllNames = parseMixedAGListToNamesAndValues( mode,
                  inclAllMixedAg )
            mode.definition.includeAllAdminGroupName.clear()
            for name in adGrpAllNames:
               mode.definition.includeAllAdminGroupName.add( name )
         else:
            adminGroupAllVal = agRangeListToValue( args[ 'INC_ALL_GROUP' ] )
         mode.definition.includeAllAdminGroup.clear()
         for i, ag in adminGroupAllVal.items():
            if ag != 0:
               mode.definition.includeAllAdminGroup[ i ] = ag
         
      if 'INC_ANY_GROUP' in args:
         if TeToggleLib.toggleFlexAlgoAdminGroupNameEnabled():
            inclAnyMixedAg = args.get( 'INC_ANY_GROUP' )
            adminGroupAnyVal, adGrpAnyNames = parseMixedAGListToNamesAndValues( mode,
                  inclAnyMixedAg )
            mode.definition.includeAnyAdminGroupName.clear()
            for name in adGrpAnyNames:
               mode.definition.includeAnyAdminGroupName.add( name )
         else:
            adminGroupAnyVal = agRangeListToValue( args[ 'INC_ANY_GROUP' ] )
         mode.definition.includeAnyAdminGroup.clear()
         for i, ag in adminGroupAnyVal.items():
            if ag != 0:
               mode.definition.includeAnyAdminGroup[ i ] = ag

      if 'EXCLUDE_GROUP' in args:
         if TeToggleLib.toggleFlexAlgoAdminGroupNameEnabled():
            excludeMixedAg = args.get( 'EXCLUDE_GROUP' )
            adminGroupExcludeVal, adGrpExNames = parseMixedAGListToNamesAndValues(
                  mode, excludeMixedAg )
            mode.definition.excludeAdminGroupName.clear()
            for name in adGrpExNames:
               mode.definition.excludeAdminGroupName.add( name )
         else:
            adminGroupExcludeVal = agRangeListToValue( args[ 'EXCLUDE_GROUP' ] )
         mode.definition.excludeAdminGroup.clear()
         for i, ag in adminGroupExcludeVal.items():
            if ag != 0:
               mode.definition.excludeAdminGroup[ i ] = ag

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      anyKw = 'any' in args
      allKw = 'all' in args
      excludeKw = 'exclude' in args
      includeAny = args.get( 'INC_ANY_GROUP' )
      includeAll = args.get( 'INC_ALL_GROUP' )
      excludeGrp = args.get( 'EXCLUDE_GROUP' )

      if ( not anyKw and not allKw and not excludeKw and not includeAny and
           not includeAll and not excludeGrp ):
         # 'no administrative-group': Flush everything
         mode.definition.includeAllAdminGroup.clear()
         mode.definition.includeAnyAdminGroup.clear()
         mode.definition.excludeAdminGroup.clear()
         mode.definition.includeAllAdminGroupName.clear()
         mode.definition.includeAnyAdminGroupName.clear()
         mode.definition.excludeAdminGroupName.clear()
         return

      if includeAll:
         if TeToggleLib.toggleFlexAlgoAdminGroupNameEnabled():
            adminGroupAllVal, adGrpAllNames = parseMixedAGListToNamesAndValues( mode,
                  includeAll )
            for name in adGrpAllNames:
               del mode.definition.includeAllAdminGroupName[ name ]
         else:
            adminGroupAllVal = agRangeListToValue( includeAll )
         for i, ag in adminGroupAllVal.items():
            if i in mode.definition.includeAllAdminGroup:
               mode.definition.includeAllAdminGroup[ i ] &= ~ag
               if mode.definition.includeAllAdminGroup[ i ] == 0:
                  del mode.definition.includeAllAdminGroup[ i ]
      elif allKw:
         mode.definition.includeAllAdminGroup.clear()
         mode.definition.includeAllAdminGroupName.clear()

      if includeAny:
         if TeToggleLib.toggleFlexAlgoAdminGroupNameEnabled():
            adminGroupAnyVal, adGrpAnyNames = parseMixedAGListToNamesAndValues( mode,
                  includeAny )
            for name in adGrpAnyNames:
               del mode.definition.includeAnyAdminGroupName[ name ]
         else:
            adminGroupAnyVal = agRangeListToValue( includeAny )
         for i, ag in adminGroupAnyVal.items():
            if i in mode.definition.includeAnyAdminGroup:
               mode.definition.includeAnyAdminGroup[ i ] &= ~ag
               if mode.definition.includeAnyAdminGroup[ i ] == 0:
                  del mode.definition.includeAnyAdminGroup[ i ]
      elif anyKw:
         mode.definition.includeAnyAdminGroup.clear()
         mode.definition.includeAnyAdminGroupName.clear()

      if excludeGrp:
         if TeToggleLib.toggleFlexAlgoAdminGroupNameEnabled():
            adminGroupExcVal, adGrpExNames = parseMixedAGListToNamesAndValues( mode,
                  excludeGrp )
            for name in adGrpExNames:
               del mode.definition.excludeAdminGroupName[ name ]
         else:
            adminGroupExcVal = agRangeListToValue( excludeGrp )
         for i, ag in adminGroupExcVal.items():
            if i in mode.definition.excludeAdminGroup:
               mode.definition.excludeAdminGroup[ i ] &= ~ag
               if mode.definition.excludeAdminGroup[ i ] == 0:
                  del mode.definition.excludeAdminGroup[ i ]
      elif excludeKw:
         mode.definition.excludeAdminGroup.clear()
         mode.definition.excludeAdminGroupName.clear()

FlexAlgoDefMode.addCommandClass( CfgFlexAlgoAdminGrpCmd )

#------------------------------------------------------------------------------
# Flex Algo:
# [ no | default ] metric ( igp-metric | min-delay | te-metric | <0-2> )
#------------------------------------------------------------------------------

metricType = Tac.Type( 'FlexAlgo::MetricType' )
metricTypeRangeEnd = metricType.teMetric
metricName = {
   'igp-metric' : 'Use the IGP metric',
   'min-delay' : 'Use the minimum link delay as the metric',
   'te-metric' : 'Use the traffic engineering metric'
}

class CfgFlexAlgoMetricCmd( CliCommand.CliCommandClass ):
   syntax = 'metric ( METRIC | METRIC_NAME )'
   noOrDefaultSyntax = 'metric ...'

   data = {
      'metric' : 'Metric for path computation',
      'METRIC' : CliMatcher.IntegerMatcher( metricType.igpMetric,
                                            metricTypeRangeEnd,
                                            helpdesc="Metric index" ),
      'METRIC_NAME' : CliMatcher.EnumMatcher( metricName ),
   }

   @staticmethod
   def handler( mode, args ):
      value = args.get( 'METRIC' )
      if value is None:
         value = flexAlgoMetricMap[ args[ 'METRIC_NAME' ] ]
      mode.definition.metric = value

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.definition.metric = mode.definition.defaultMetric

FlexAlgoDefMode.addCommandClass( CfgFlexAlgoMetricCmd )

#------------------------------------------------------------------------------
# Flex Algo:
# [ no | default ] srlg exclude { NAME | GROUP_ID }
#------------------------------------------------------------------------------

def srlgRange():
   return ( srlgGid.srlgIdMin, srlgGid.srlgIdMax )

def decodeArgs( mode, args ):
   names = args.get( 'NAME', [] )
   if 'GROUP_ID' in args:
      values = [ group for rng in args[ 'GROUP_ID' ]
                 for group in rng.values() ]
   else:
      values = []
   return ( values, names )

class CfgFlexAlgoSrlgCmd( CliCommand.CliCommandClass ):
   syntax = 'srlg exclude { NAME | GROUP_ID }'
   noOrDefaultSyntax = 'srlg [ exclude [ { NAME | GROUP_ID } ] ]'
   data = {
      'srlg' : 'Shared Risk Link Group',
      'exclude' : 'Exclude designated groups',
      'NAME' : matcherSrlgName,
      'GROUP_ID' : MultiRangeMatcher( srlgRange, False, 'Share Risk Link Groups' )
      }

   @staticmethod
   def handler( mode, args ):
      ( values, names ) = decodeArgs( mode, args )
      for val in mode.definition.srlgValueSet:
         if val not in values:
            del mode.definition.srlgValueSet[ val ]
      for val in values:
         mode.definition.srlgValueSet[ val ] = True
      for name in mode.definition.srlgNameSet:
         if name not in names:
            del mode.definition.srlgNameSet[ name ]
      for name in names:
         if name not in mode.definition.srlgNameSet:
            mode.definition.srlgNameSet[ name ] = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      ( values, names ) = decodeArgs( mode, args )
      if not values and not names:
         mode.definition.srlgValueSet.clear()
         mode.definition.srlgNameSet.clear()
      else:
         for val in values:
            del mode.definition.srlgValueSet[ val ]
         for name in names:
            del mode.definition.srlgNameSet[ name ]

FlexAlgoDefMode.addCommandClass( CfgFlexAlgoSrlgCmd )

#------------------------------------------------------------------------------
# Flex Algo:
# [ no | default ] color COLOR_VALUE
#------------------------------------------------------------------------------

class CfgFlexAlgoColorCmd( CliCommand.CliCommandClass ):
   syntax = 'color COLOR_VALUE'
   noOrDefaultSyntax = 'color ...'
   data = {
      'color' : 'Select the colored tunnel RIB for results',
      'COLOR_VALUE' : CliMatcher.IntegerMatcher( Color.min, Color.max,
                                                 helpdesc='Color value' )
   }

   @staticmethod
   def handler( mode, args ):
      mode.definition.color = args.get( 'COLOR_VALUE', mode.definition.defaultColor )
      mode.definition.colorDefined = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.definition.colorDefined = False

FlexAlgoDefMode.addCommandClass( CfgFlexAlgoColorCmd )

#-------------------------------------------------------------------------------
# igp bandwidth update threshold unreserved <0-100> percent
#-------------------------------------------------------------------------------
class CfgTeBwIgpUpdateThresholdCmd ( CliCommand.CliCommandClass ):
   syntax = 'igp bandwidth update threshold unreserved VALUE percent'
   noOrDefaultSyntax = 'igp bandwidth update threshold ...'
   data = {
      'igp' : 'IGP related configuration',
      'bandwidth' : 'Traffic Engineering interface bandwidth configuration',
      'update' : 'Flooding parameters',
      'threshold' : 'Bandwidth change threshold for flooding',
      'unreserved' : 'Calculate threshold based on available bandwidth',
      'VALUE' : CliMatcher.IntegerMatcher(
         teBwFloodPercent.teBwFloodPercentMin, teBwFloodPercent.teBwFloodPercentMax,
         helpdesc='<0-100>' ),
      'percent' : 'Percent change from advertised value'
   }

   @staticmethod
   def handler( mode, args ):
      teConfiguration().teBwFloodThreshold = args[ 'VALUE' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      teConfiguration().teBwFloodThreshold = teConfiguration().\
                                                    teBwFloodThresholdDefault

RouterGlobalTeMode.addCommandClass( CfgTeBwIgpUpdateThresholdCmd )

#-------------------------------------------------------------------------------
# twamp-light sender profile <PROFILE_NAME>
#-------------------------------------------------------------------------------
namePattern = '.{1,100}'
senderNameMatcher = CliMatcher.DynamicNameMatcher(
   lambda mode: twampLightConfig.senderProfile,
   'Profile name', pattern=namePattern )

class CfgTeTwampLightSenderProfileCmd ( CliCommand.CliCommandClass ):
   syntax = 'twamp-light sender profile PROFILE_NAME'
   noOrDefaultSyntax = 'twamp-light sender profile...'
   data = {
      'twamp-light' : 'Configure TWAMP light',
      'sender' : 'Configure TWAMP light sender',
      'profile' : 'Configure TWAMP light sender profile',
      'PROFILE_NAME' : senderNameMatcher
   }

   @staticmethod
   def handler( mode, args ):
      teConfiguration().senderProfile = args[ 'PROFILE_NAME' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      teConfiguration().senderProfile = teConfiguration().\
                                          senderProfileDefault

RouterGlobalTeMode.addCommandClass( CfgTeTwampLightSenderProfileCmd )

def Plugin( entityManager ):
   global teConfig
   global flexAlgoConfig
   global twampLightConfig

   teConfig = ConfigMount.mount( entityManager, 'te/config',
                                 'TrafficEngineering::Config', 'w' )
   flexAlgoConfig = ConfigMount.mount( entityManager, 'te/flexalgo/config',
                                       'FlexAlgo::Config', 'w' )
   twampLightConfig = LazyMount.mount(
      entityManager, 'monitor/twamp/twampLight/config',
      'Twamp::Light::Config', 'r' )
   IntfCli.Intf.registerDependentClass( TeIntf, priority=20 )
   TeModeDependents.registerDependentClass( TeFlexAlgoSubMode, priority=20 )
