# Copyright (c) 2008, 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from socket import IPPROTO_TCP

from CliPlugin import AclCli
from CliPlugin import AclCliModel
from CliPlugin import ConfigMgmtMode
from CliPlugin.SysMgrModels import ManagementTelnet
from CliPlugin.VrfCli import VrfExprFactory, DEFAULT_VRF
import AclCliLib
import BasicCli
import CliCommand
import CliMatcher
from CliMode.VrfConfig import VrfConfigMode
import ConfigMount
import CliToken.Clear
import LazyMount
import ShowCommand
import Tac


# ------------------------------
# Telnet config commands
#
# From config mode
#    management telnet    - enter telnet config mode
#    no management telnet - return telnet to its default configuration
#
# From management config mode
#    [ no | default] shutdown
#
#    idle-timeout x  - set the idle session timeout to x minutes
#    vrf VRF - enter telnet vrf config mode
#
# From telnet vrf config mode
#    [ no | default ] shutdown - disables or enables telnetd in vrf
#
#    For example:
#    management telnet
#       A
#       vrf x
#          B
#
#
#    A\B             B default             B shutdown            B no shutdown
#    A default       disabled\disabled     disabled\disabled     disabled\enabled
#    A shutdown      disabled\disabled     disabled\disabled     disabled\enabled
#    A no shutdown   enabled\enabled       enabled\disabled      enabled\enabled
#
# See AID435 for a discussion of the design of telnet, including
# support cli commands.

telnetConfig = None
aclConfig = None
aclCpConfig = None
aclStatus = None
aclCheckpoint = None

telnetShowKwMatcher = CliMatcher.KeywordMatcher( 'telnet',
                                                 helpdesc='show telnet status' )

class TelnetConfigMode( ConfigMgmtMode.ConfigMgmtMode ):
   name = "Telnet configuration"

   def __init__( self, parent, session ):
      ConfigMgmtMode.ConfigMgmtMode.__init__( self, parent, session, "telnet" )
      self.config_ = telnetConfig
      self.aclConfig_ = aclConfig
      self.aclCpConfig_ = aclCpConfig

   def shutdown( self ):
      self.config_.serverState = "disabled"

   def noShutdown( self ):
      self.config_.serverState = "enabled"

   def setIdleTimeout( self, tokenIdleTime ):
      to = Tac.Value( "Mgmt::SessionTimeout::IdleSessionTimeout" )
      to.timeout = tokenIdleTime * 60
      self.config_.idleTimeout = to

   def noIdleTimeout( self ):
      self.setIdleTimeout( self.config_.idleTimeout.defaultTimeout // 60 )

   def setIpAcl( self, args ):
      aclName = args[ 'ACL_NAME' ]
      vrfName = args.get( 'VRF' )
      AclCliLib.setServiceAcl( self, 'telnet', IPPROTO_TCP,
                               self.aclConfig_, self.aclCpConfig_,
                               aclName, 'ip', vrfName, port=[ 23 ] )

   def noIpAcl( self, args=None ):
      if args:
         aclName = args.get( 'ACL_NAME' )
         vrfName = args.get( 'VRF' )
      else:
         aclName = None
         vrfName = None
      AclCliLib.noServiceAcl( self, 'telnet', self.aclConfig_,
                              self.aclCpConfig_, aclName, 'ip', vrfName )

   def setIp6Acl( self, args ):
      aclName = args[ 'ACL_NAME' ]
      vrfName = args.get( 'VRF' )
      AclCliLib.setServiceAcl( self, 'telnet', IPPROTO_TCP,
                               self.aclConfig_, self.aclCpConfig_,
                               aclName, 'ipv6', vrfName, port=[ 23 ] )

   def noIp6Acl( self, args=None ):
      if args:
         aclName = args.get( 'ACL_NAME' )
         vrfName = args.get( 'VRF' )
      else:
         aclName = None
         vrfName = None
      AclCliLib.noServiceAcl( self, 'telnet', self.aclConfig_,
                              self.aclCpConfig_, aclName, 'ipv6', vrfName )

   def setSessionLimit( self, sessionLimit=None ):
      self.config_.sessionLimit = sessionLimit or self.config_.sessionLimitDefault

   def setSessionLimitPerHost( self, perhostLimit=None ):
      self.config_.sessionLimitPerHost = perhostLimit or \
                                         self.config_.sessionLimitPerHostDefault

#-------------------------------------------------------------------------------
# "vrf VRF" config mode
#-------------------------------------------------------------------------------
class TelnetVrfConfigMode( VrfConfigMode, BasicCli.ConfigModeBase ):
   name = "Telnet VRF Configuration"

   def __init__( self, parent, session, vrfName ):
      VrfConfigMode.__init__( self, ( vrfName, 'telnet', telnetConfig ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

#-----------------------------------------------------------------------------------
# [ no | default ] shutdown
#-----------------------------------------------------------------------------------
class ShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = 'shutdown'
   data = {
            'shutdown': 'Disable telnet',
          }
   @staticmethod
   def handler( mode, args ):
      mode.shutdown()

   @staticmethod
   def defaultHandler( mode, args ):
      mode.shutdown()

   @staticmethod
   def noHandler( mode, args ):
      mode.noShutdown()

TelnetConfigMode.addCommandClass( ShutdownCmd )

#-----------------------------------------------------------------------------------
# [ no | default ] idle-timeout IDLE_TIMEOUT
#-----------------------------------------------------------------------------------
class IdleTimeoutCmd( CliCommand.CliCommandClass ):
   syntax = 'idle-timeout IDLE_TIMEOUT'
   noOrDefaultSyntax = 'idle-timeout ...'
   data = {
            'idle-timeout': 'Set idle session timeout(minutes)',
            'IDLE_TIMEOUT': CliMatcher.IntegerMatcher( 0, 86400,
                                        helpdesc='Idle session timeout in minutes' ),
          }
   @staticmethod
   def handler( mode, args ):
      mode.setIdleTimeout( args[ 'IDLE_TIMEOUT' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.noIdleTimeout()

TelnetConfigMode.addCommandClass( IdleTimeoutCmd )

#-----------------------------------------------------------------------------------
# [ no | default ] session-limit SESSION_LIMIT
#-----------------------------------------------------------------------------------
class SessionLimitCmd( CliCommand.CliCommandClass ):
   syntax = 'session-limit SESSION_LIMIT'
   noOrDefaultSyntax = 'session-limit ...'
   data = {
            'session-limit': 'Set maximum number of telnet sessions',
            'SESSION_LIMIT': CliMatcher.IntegerMatcher( 1, 20,
                                      helpdesc='Maximum number of telnet sessions' ),
          }
   @staticmethod
   def handler( mode, args ):
      mode.setSessionLimit( args[ 'SESSION_LIMIT' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.setSessionLimit()

TelnetConfigMode.addCommandClass( SessionLimitCmd )

#-----------------------------------------------------------------------------------
# [ no | default ] session-limit per-host PERHOST_LIMIT
#-----------------------------------------------------------------------------------
class SessionLimitPerHostCmd( CliCommand.CliCommandClass ):
   syntax = 'session-limit per-host PERHOST_LIMIT'
   noOrDefaultSyntax = 'session-limit per-host ...'
   data = {
            'session-limit': 'Set maximum number of telnet sessions',
            'per-host': 'Set maximum number of telnet sessions from a single host',
            'PERHOST_LIMIT': CliMatcher.IntegerMatcher( 1, 20,
                                      helpdesc='Maximum number of telnet sessions' ),
          }
   @staticmethod
   def handler( mode, args ):
      mode.setSessionLimitPerHost( args[ 'PERHOST_LIMIT' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.setSessionLimitPerHost()

TelnetConfigMode.addCommandClass( SessionLimitPerHostCmd )

#-----------------------------------------------------------------------------------
# [ no | default ] ip access-group NAME [ vrf VRF ] in
#-----------------------------------------------------------------------------------
vrfWithNameExpr = VrfExprFactory(
      helpdesc='Configure the VRF in which to apply the access control list' )
class IpAccessGroup( CliCommand.CliCommandClass ):
   syntax = 'ip access-group ACL_NAME [ VRF ] in'
   noOrDefaultSyntax = 'ip access-group [ ACL_NAME ] [ VRF ] in'
   data = {
            'ip': 'Telnet IP configuration',
            'access-group': 'Configure access control list',
            'ACL_NAME': AclCli.standardIpAclNameMatcher,
            'VRF': vrfWithNameExpr,
            'in': 'Inbound packets'

          }
   handler = TelnetConfigMode.setIpAcl
   noOrDefaultHandler = TelnetConfigMode.noIpAcl

TelnetConfigMode.addCommandClass( IpAccessGroup )

#-----------------------------------------------------------------------------------
# [ no | default ] ipv6 access-group NAME [ vrf VRF ] in
#-----------------------------------------------------------------------------------
ipv6AclNameMatcher = CliMatcher.DynamicNameMatcher(
                            lambda mode : AclCliLib.getAclNames( aclConfig, 'ipv6' ),
                            'Standard access-list name' )
class Ipv6AccessGroup( CliCommand.CliCommandClass ):
   syntax = 'ipv6 access-group ACL_NAME [ VRF ] in'
   noOrDefaultSyntax = 'ipv6 access-group [ ACL_NAME ] [ VRF ] in'
   data = {
            'ipv6': 'Telnet IPv6 configuration',
            'access-group': 'Configure access control list',
            'ACL_NAME': AclCli.standardIp6AclNameMatcher,
            'VRF': vrfWithNameExpr,
            'in': 'Inbound packets'

          }
   handler = TelnetConfigMode.setIp6Acl
   noOrDefaultHandler = TelnetConfigMode.noIp6Acl

TelnetConfigMode.addCommandClass( Ipv6AccessGroup )

#-----------------------------------------------------------------------------------
# [ no | default ] vrf VRF
#-----------------------------------------------------------------------------------
class VrfModeCmd( CliCommand.CliCommandClass ):
   syntax = 'VRF'
   noOrDefaultSyntax = syntax
   data = {
            'VRF': VrfExprFactory( helpdesc='Enter VRF sub-mode',
                                   inclDefaultVrf=True ),
          }
   @staticmethod
   def handler( mode, args ):
      vrfName = args[ 'VRF' ]
      childMode = mode.childMode( TelnetVrfConfigMode, vrfName=vrfName )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vrfName = args[ 'VRF' ]
      del telnetConfig.vrfConfig[ vrfName ]

TelnetConfigMode.addCommandClass( VrfModeCmd )

#-----------------------------------------------------------------------------------
# Telnet VRF sub mode
# [ no | default ] shutdown
#-----------------------------------------------------------------------------------
class VrfShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = 'shutdown'
   data = {
            'shutdown': 'Disable telnet',
          }
   @staticmethod
   def handler( mode, args ):
      TelnetVrfConfigMode.shutdown( mode )

   @staticmethod
   def defaultHandler( mode, args ):
      TelnetVrfConfigMode.defaultShutdown( mode )

   @staticmethod
   def noHandler( mode, args ):
      TelnetVrfConfigMode.noShutdown( mode )

TelnetVrfConfigMode.addCommandClass( VrfShutdownCmd )

#-----------------------------------------------------------------------------------
# [ no | default ] management telnet
# in global config mode
#-----------------------------------------------------------------------------------
class ManagmentTelnet( CliCommand.CliCommandClass ):
   syntax = 'management telnet'
   noOrDefaultSyntax = 'management telnet'
   data = {
            'management': ConfigMgmtMode.managementKwMatcher,
            'telnet': 'Configure telnet'
          }
   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( TelnetConfigMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      childMode = mode.childMode( TelnetConfigMode )
      childMode.shutdown()
      childMode.noIdleTimeout()
      childMode.setSessionLimit()
      childMode.setSessionLimitPerHost()
      for vrf in childMode.aclCpConfig_.cpConfig[ 'ip' ].serviceAcl:
         childMode.noIpAcl( { 'VRF': vrf } )
      for vrf in childMode.aclCpConfig_.cpConfig[ 'ipv6' ].serviceAcl:
         childMode.noIp6Acl( { 'VRF': vrf } )
      childMode.removeComment()
      telnetConfig.vrfConfig.clear()

BasicCli.GlobalConfigMode.addCommandClass( ManagmentTelnet )

#----------------------------------------------------------------
# "clear management telnet counters ( ip | ipv6 ) access-list"
#----------------------------------------------------------------
class ClearIpAclCounters( CliCommand.CliCommandClass ):
   syntax = 'clear management telnet counters ( ip | ipv6 ) access-list'
   data = { 'clear': CliToken.Clear.clearKwNode,
            'management': ConfigMgmtMode.managementClearKwMatcher,
            'telnet': 'Clear Telnet statistics',
            'counters': 'Connection Counters',
            'ip': AclCli.ipKwForClearServiceAclMatcher,
            'ipv6': AclCli.ipv6KwMatcherForClearServiceAcl,
            'access-list': AclCli.accessListKwMatcherForServiceAcl }

   @staticmethod
   def handler( mode, args ):
      if 'ip' in args:
         aclType = 'ip'
      elif 'ipv6' in args:
         aclType = 'ipv6'
      else:
         assert False
      AclCli.clearServiceAclCounters( mode, aclStatus, aclCheckpoint, aclType )

BasicCli.EnableMode.addCommandClass( ClearIpAclCounters )

#-------------------------------------------------------------------------------
# show management telnet [ vrf VRF ]
#-------------------------------------------------------------------------------
class ShowManagementTelnet( ShowCommand.ShowCliCommandClass ):
   syntax = 'show management telnet [ VRF ]'
   data = {
            'management': ConfigMgmtMode.managementShowKwMatcher,
            'telnet': telnetShowKwMatcher,
            'VRF': VrfExprFactory( helpdesc='Use a specific VRF' ),
         }
   cliModel = ManagementTelnet

   @staticmethod
   def handler( mode, args ):
      vrfName = args.get( 'VRF', '' )
      ret = ManagementTelnet()
      if vrfName:
         ret.vrfName = vrfName
         if vrfName in telnetConfig.vrfConfig:
            serverState = telnetConfig.vrfConfig[ vrfName ].serverState
            ret.globalDefault = ( serverState == 'globalDefault' )
            if ret.globalDefault:
               ret.serverState = telnetConfig.serverState
            else:
               ret.serverState = serverState
         else:
            # pylint: disable-next=consider-using-f-string
            mode.addError( "VRF %s not found" % vrfName )
            return ret
      else:
         ret.serverState = telnetConfig.serverState
         if DEFAULT_VRF in telnetConfig.vrfConfig:
            defVrfState = telnetConfig.vrfConfig [ DEFAULT_VRF ].serverState
            if defVrfState != "globalDefault":
               ret.serverState = defVrfState
      ret.maxTelnetSessions = telnetConfig.sessionLimit
      ret.maxTelnetSessionsPerHost = telnetConfig.sessionLimitPerHost
      return ret

BasicCli.addShowCommandClass( ShowManagementTelnet )

#-------------------------------------------------------------------------------
# show management telnet [ip|ipv6] access-list [ACL] [summary]
#-------------------------------------------------------------------------------
class ShowManagementTelnetAcl( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show management telnet'
              '('
              ' ( ip access-list [ IPACLNAME ] ) | '
              ' ( ipv6 access-list [ IPV6ACLNAME ] ) '
              ')' )
   data = {
            'management': ConfigMgmtMode.managementShowKwMatcher,
            'telnet': telnetShowKwMatcher,
            'ip': AclCli.ipKwForShowServiceAcl,
            'ipv6': AclCli.ipv6KwForShowServiceAcl,
            'access-list': AclCli.accessListKwMatcherForServiceAcl,
            'IPACLNAME': AclCli.ipAclNameExpression,
            'IPV6ACLNAME': AclCli.ip6AclNameExpression
          }

   cliModel = AclCliModel.AllAclList
   privileged = True

   @staticmethod
   def handler( mode, args ):
      aclType = 'ip' if 'ip' in args else 'ipv6'
      name = args[ '<aclNameExpr>' ]
      return AclCli.showServiceAcl( mode, aclCpConfig, aclStatus,
                                    aclCheckpoint, aclType, name,
                                    supressVrf=False, serviceName='telnet' )

BasicCli.addShowCommandClass( ShowManagementTelnetAcl )

def Plugin( entityManager ):
   global telnetConfig, aclConfig, aclCpConfig
   global aclCheckpoint
   global aclStatus
   telnetConfig = ConfigMount.mount( entityManager, "mgmt/telnet/config",
                                     "Mgmt::Telnet::Config", "w" )
   aclConfig = ConfigMount.mount( entityManager, "acl/config/cli",
                                  "Acl::Input::Config", "w" )
   aclCpConfig = ConfigMount.mount( entityManager, "acl/cpconfig/cli",
                                  "Acl::Input::CpConfig", "w" )
   aclStatus = LazyMount.mount( entityManager, "acl/status/all",
                                "Acl::Status", "r" )
   aclCheckpoint = LazyMount.mount( entityManager, "acl/checkpoint",
                                   "Acl::CheckpointStatus", "w" )
