# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

from __future__ import absolute_import, division, print_function

import Tac
import CliMatcher, CliGlobal
import ConfigMount, LazyMount
import datetime
import Ark
import re


gv = CliGlobal.CliGlobal( 
   dict( trackingConfig=None, trackingStatus=None, trackedByStatus=None ) )

matcherTrack = CliMatcher.KeywordMatcher( 'track',
      helpdesc='Configure a tracked object' )
matcherTrackForShow = CliMatcher.KeywordMatcher( 'track',
      helpdesc='Show a tracked object information' )
matcherObjName = CliMatcher.PatternMatcher( pattern='[a-zA-Z0-9]+',
      helpdesc='Tracked object name', helpname='WORD' )

#-------------------------------------------------------------------------------
# The "track OBJNAME interface INTF line-protocol" command
#-------------------------------------------------------------------------------
def trackInterfaceObject( mode, args ):
   objName = args[ 'OBJNAME' ]
   if objName in gv.trackingConfig.listBoolObjectConfig or \
      objName in gv.trackingConfig.listThreshObjectConfig:
      mode.addError( "Tracked object %s already exists and is not of "
                     "type interface" % (objName) )
   else:
      intf = args[ 'INTF' ]
      if objName not in gv.trackingConfig.interfaceObjectConfig:
         gv.trackingConfig.newInterfaceObjectConfig( objName )
      gv.trackingConfig.interfaceObjectConfig[ objName ].intfName = intf.name

                                           
#-------------------------------------------------------------------------------
# The "( no | default ) track OBJNAME"  command
#-------------------------------------------------------------------------------
def noOrDefaultTrackObject( mode, args ):
   objName = args[ 'OBJNAME' ]
   if objName in gv.trackingConfig.interfaceObjectConfig:
      del gv.trackingConfig.interfaceObjectConfig[ objName ]
   if objName in gv.trackingConfig.listBoolObjectConfig:
      del gv.trackingConfig.listBoolObjectConfig[ objName ]
   if objName in gv.trackingConfig.listThreshObjectConfig:
      del gv.trackingConfig.listThreshObjectConfig[ objName ]

#-------------------------------------------------------------------------------
# The "show track [ brief ]" command
#-------------------------------------------------------------------------------
def showTrackBrief( mode, brief ):
   if brief:
      outputbrief = []
      for objName, objStatus in gv.trackingStatus.trackedObjectStatus.items():
         trackedObjState = objStatus.objState
         outputbrief.append( [ objName, trackedObjState ] )
      for eachone in sorted( outputbrief ):   
         print( "Tracked object %s is %s" % ( eachone[0], eachone[1] ) )
   else:
      output = []
      for objName in gv.trackingStatus.trackedObjectStatus:
         output.append( objName )
      for eachone in sorted( output ):
         showTrackedObject( mode, eachone )
         print( "" )

#-------------------------------------------------------------------------------
# The "show track OBJNAME" command
#-------------------------------------------------------------------------------
def showTrackedObject( mode, objName ):
   trackedObjStatus = gv.trackingStatus.trackedObjectStatus.get( objName )
   if not trackedObjStatus:
      mode.addError( "Tracked object %s does not exist" % (objName) )
      return

   trackedObjState = trackedObjStatus.objState
   print( "Tracked object %s is %s" % (objName, trackedObjState) )

   config = gv.trackingConfig.interfaceObjectConfig.get( objName )
   if config:
      intfName = config.intfName
      print( "   Interface %s line-protocol" % (intfName) )
   if trackedObjStatus.numChanges > 0 : 
      timestring = Ark.timestampToStr( trackedObjStatus.lastChangeTime ) 
   else: 
      timestring = str(datetime.timedelta(0))   
   print( "      %d change, last change time was %s" % \
         ( trackedObjStatus.numChanges, timestring ) )
   print( "   Tracked by:" )
   output = []
   for eachone in gv.trackedByStatus.trackedBy:
      if objName == eachone.objName:
         output.append( eachone )
   for element in sorted(output):
      print( "      %s" % ( re.sub( '%2[Ff]', '/', element.displayString ) ) )

#-------------------------------------------------------------------------------
# The "show track [ brief | OBJNAME ]" command
#-------------------------------------------------------------------------------
def showTrackCommand ( mode, args ):
   name = args.get( 'OBJNAME' )
   if name is None:
      showTrackBrief( mode, brief='brief' in args )
   else:
      showTrackedObject( mode, objName=name )

def Plugin( entityManager ):
   gv.trackingConfig = ConfigMount.mount( entityManager, 'tracking/config',
                                       'Tracking::Config', 'w' )
   gv.trackingStatus = LazyMount.mount( entityManager, 'tracking/status',
                                     'Tracking::Status', 'r' )
   gv.trackedByStatus = LazyMount.mount( entityManager, 'tracking/trackedBy/status',
                                      'Tracking::TrackedByStatus', 'r' )
